// Construit et imaginé par Jean-François Avart
(() => {
  // ============================================================
  // DÉTECTION MODE DEMO
  // ============================================================
  // Le mode DEMO est activé via une meta tag dans index.html:
  // <meta name="csi-mode" content="demo">
  const DEMO_MODE = (() => {
    const meta = document.querySelector('meta[name="csi-mode"]');
    return meta && meta.content === 'demo';
  })();

  // URL du site pour télécharger l'application complète
  const DEMO_DOWNLOAD_URL = 'https://cyber-assistant.com/client_web.zip';
  const DEMO_WEBSITE_URL = 'https://cyber-assistant.com';
  const DEMO_CONTACT_EMAIL = 'cyber-assistant-contact@proton.me';

  // Configuration mode DEMO
  const DEMO_AUTO_RELOAD_INTERVAL = 60 * 60 * 1000; // 1 heure en millisecondes
  let demoAutoReloadTimer = null;

  // Log mode DEMO si activé
  if (DEMO_MODE) {
    console.log('[CSI] Mode DÉMONSTRATION activé');
    console.log('[CSI DEMO] Rechargement automatique des données toutes les heures');
  }

  /**
   * Affiche un message de blocage pour les fonctions désactivées en mode démo
   * @param {string} featureName - Nom de la fonctionnalité bloquée
   */
  function showDemoBlockedMessage(featureName = 'Cette fonction') {
    if (typeof showToast2 === 'function') {
      showToast2(`${featureName} est désactivée en mode démonstration.`, 'warning');
    } else {
      alert(`${featureName} est désactivée en mode démonstration.`);
    }
  }

  /**
   * Démarre le timer de rechargement automatique des données démo
   */
  function startDemoAutoReload() {
    if (!DEMO_MODE) return;

    // Arrêter tout timer existant
    if (demoAutoReloadTimer) {
      clearInterval(demoAutoReloadTimer);
    }

    // Démarrer le rechargement automatique toutes les heures
    demoAutoReloadTimer = setInterval(() => {
      console.log('[CSI DEMO] Rechargement automatique des données de démonstration');
      if (typeof loadDemoDataSilent === 'function') {
        loadDemoDataSilent();
        if (typeof showToast2 === 'function') {
          showToast2('Données de démonstration rechargées automatiquement.', 'info');
        }
      }
    }, DEMO_AUTO_RELOAD_INTERVAL);

    console.log('[CSI DEMO] Timer de rechargement automatique activé (1h)');
  }

  /**
   * Arrête le timer de rechargement automatique
   */
  function stopDemoAutoReload() {
    if (demoAutoReloadTimer) {
      clearInterval(demoAutoReloadTimer);
      demoAutoReloadTimer = null;
      console.log('[CSI DEMO] Timer de rechargement automatique désactivé');
    }
  }

  // ============================================================
  // PROTECTIONS ANTI-ABUS (MODE DEMO)
  // ============================================================

  // Rate limiting pour les actions fréquentes
  const demoRateLimiter = {
    actions: new Map(),
    maxActions: 100,        // Max 100 actions par fenêtre
    windowMs: 60000,        // Fenêtre de 1 minute
    cooldownMs: 5000,       // Pause de 5 secondes si dépassé

    /**
     * Vérifie si une action est autorisée (rate limiting)
     * @param {string} actionType - Type d'action (save, create, update, etc.)
     * @returns {boolean} - true si l'action est autorisée
     */
    checkAction(actionType = 'default') {
      if (!DEMO_MODE) return true;

      const now = Date.now();
      const key = actionType;

      if (!this.actions.has(key)) {
        this.actions.set(key, { count: 0, windowStart: now, blocked: false });
      }

      const action = this.actions.get(key);

      // Réinitialiser la fenêtre si expirée
      if (now - action.windowStart > this.windowMs) {
        action.count = 0;
        action.windowStart = now;
        action.blocked = false;
      }

      // Vérifier le cooldown
      if (action.blocked && now - action.blockedAt < this.cooldownMs) {
        return false;
      } else if (action.blocked) {
        action.blocked = false;
      }

      // Vérifier le rate limit
      if (action.count >= this.maxActions) {
        action.blocked = true;
        action.blockedAt = now;
        console.warn('[CSI DEMO] Rate limit atteint pour:', actionType);
        return false;
      }

      action.count++;
      return true;
    },

    /**
     * Affiche un message de rate limit
     */
    showRateLimitMessage() {
      if (typeof showToast2 === 'function') {
        showToast2('Trop d\'actions en peu de temps. Veuillez patienter quelques secondes.', 'warning');
      }
    }
  };

  // Limite de taille des données en mode démo (protection mémoire)
  const DEMO_MAX_ITEMS = {
    controls: 200,
    actions: 500,
    risks: 200,
    threats: 200,
    nonconformities: 100,
    audits: 50,
    documents: 100,
    stakeholders: 100,
    criticalAssets: 100,
    projectRisks: 100
  };

  /**
   * Vérifie les limites de données en mode démo
   * @param {string} dataType - Type de données
   * @param {number} currentCount - Nombre actuel d'éléments
   * @returns {boolean} - true si on peut ajouter
   */
  function checkDemoDataLimit(dataType, currentCount) {
    if (!DEMO_MODE) return true;

    const limit = DEMO_MAX_ITEMS[dataType];
    if (!limit) return true;

    if (currentCount >= limit) {
      showDemoBlockedMessage(`Limite atteinte (${limit} ${dataType} maximum en mode démo)`);
      return false;
    }
    return true;
  }

  /**
   * Protection contre le localStorage flooding
   * @returns {boolean} - true si on peut sauvegarder
   */
  function checkDemoStorageLimit() {
    if (!DEMO_MODE) return true;

    try {
      // Estimer la taille actuelle du localStorage
      let totalSize = 0;
      for (let key in localStorage) {
        if (localStorage.hasOwnProperty(key)) {
          totalSize += localStorage[key].length * 2; // UTF-16 = 2 bytes par char
        }
      }

      // Limite à 4MB en mode démo (sur les 5MB typiques)
      const MAX_DEMO_STORAGE = 4 * 1024 * 1024;
      if (totalSize > MAX_DEMO_STORAGE) {
        console.warn('[CSI DEMO] Limite de stockage atteinte');
        showDemoBlockedMessage('Limite de stockage atteinte en mode démo');
        return false;
      }
      return true;
    } catch (e) {
      return true; // En cas d'erreur, autoriser
    }
  }

  const DOMAIN_BY_PREFIX = {
    '5': 'Organisationnel',
    '6': 'Humain',
    '7': 'Physique',
    '8': 'Technologique'
  };
  const normalizeCapabilityKey = (value) =>
    value
      .normalize('NFD')
      .replace(/[\u0300-\u036f]/g, '')
      .toLowerCase()
      .replace(/[\s'']+/g, "_")
      .replace(/_+/g, "_");
  const getIso27002DomainFromValue = (value) => {
    const text = String(value || "").trim();
    if (!text) return "";
    const match = text.match(/(^|[^0-9])([5-8])\s*(?:\.|$)/);
    if (!match) return "";
    return DOMAIN_BY_PREFIX[match[2]] || "";
  };
  const inferIso27002Category = (control = {}) => {
    const fromNumber = getIso27002DomainFromValue(control.numero);
    if (fromNumber) return fromNumber;
    const fromReference = getIso27002DomainFromValue(control.reference);
    if (fromReference) return fromReference;
    const title = String(control.title || "").trim();
    if (title && typeof iso27002_data_default !== "undefined") {
      const match = iso27002_data_default.find((item) => item.numero === control.numero || item.titre === title);
      if (match) {
        const prefix = String(match.numero || "").split(".")[0];
        return DOMAIN_BY_PREFIX[prefix] || "";
      }
    }
    return "Non classé";
  };

  // Génération d'ID unique avec crypto.randomUUID() et fallback
  const generateId = (prefix = '') => {
    let uuid;
    if (typeof crypto !== 'undefined' && typeof crypto.randomUUID === 'function') {
      uuid = crypto.randomUUID();
    } else {
      // Fallback pour navigateurs anciens
      uuid = 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, (c) => {
        const r = Math.random() * 16 | 0;
        const v = c === 'x' ? r : (r & 0x3 | 0x8);
        return v.toString(16);
      });
    }
    return prefix ? `${prefix}-${uuid}` : uuid;
  };

  // Accès sécurisé aux éléments DOM
  const safeSetText = (id, text) => {
    const el = document.getElementById(id);
    if (el) el.textContent = text;
  };

  const safeSetHTML = (id, html) => {
    const el = document.getElementById(id);
    if (el) el.innerHTML = html;
  };

  const safeAddClass = (id, className) => {
    const el = document.getElementById(id);
    if (el) el.classList.add(className);
  };

  const safeRemoveClass = (id, className) => {
    const el = document.getElementById(id);
    if (el) el.classList.remove(className);
  };

  const OPERATIONAL_CAPABILITIES = [
    "Gouvernance",
    "Gestion_des_actifs",
    "Protection_des_informations",
    "Sécurité_des_ressources_humaines",
    "Sécurité_physique",
    "Sécurité_système_et_réseau",
    "Sécurité_des_applications",
    "Configuration_sécurisée",
    "Gestion_des_identités_et_des_accès",
    "Gestion_des_menaces_et_des_vulnérabilités",
    "Continuité",
    "Sécurité_des_relations_fournisseurs",
    "Réglementation_et_conformité",
    "Gestion_des_événements_de_sécurité_de_l'information",
    "Assurance_de_sécurité_de_l'information"
  ].map((label) => ({
    key: normalizeCapabilityKey(label),
    label: label.replace(/_/g, " ")
  }));
  const NIS2_CATEGORIES = [
    'ID.AM', 'ID.BE', 'ID.GV', 'ID.RA', 'ID.RM', 'ID.SC',
    'PR.AC', 'PR.AT', 'PR.DS', 'PR.IP', 'PR.MA', 'PR.PT',
    'DE.AE', 'DE.CM', 'DE.DP',
    'RS.RP', 'RS.CO', 'RS.AN', 'RS.MI', 'RS.IM',
    'RC.RP', 'RC.IM', 'RC.CO'
  ];
  let currentTheme = "poudre";
  const THEME_STORAGE_KEY = "smsi_theme";
  const THEME_OPTIONS = [
    { value: "light", label: "Clair" },
    { value: "dark", label: "Sombre" },
    { value: "poudre", label: "Poudre" },
    { value: "bleu", label: "Bleu" }
  ];
  // Lire le thème par défaut depuis la balise meta (pour branding) ou utiliser poudre
  const getDefaultTheme = () => {
    const metaTheme = document.querySelector('meta[name="csi-default-theme"]');
    if (metaTheme && metaTheme.content) {
      return metaTheme.content;
    }
    return "poudre";
  };
  const DEFAULT_THEME = getDefaultTheme();
  const normalizeTheme = (value) => {
    if (!value) return DEFAULT_THEME;
    return THEME_OPTIONS.some((opt) => opt.value === value) ? value : DEFAULT_THEME;
  };
  const DOCUMENT_REVIEW_ITEMS = [
    { type: "chapter", title: "4 - Contexte de l'organisation" },
    { reference: "4.1 Compréhension de l'organisation et de son contexte", description: "L'organisation doit déterminer les enjeux externes et internes pertinents qui ont une incidence sur sa capacité à atteindre les résultats attendus de son SMSI." },
    { reference: "4.2 Compréhension des besoins et attentes des parties intéressées", description: "L'organisation doit déterminer les parties intéressées pertinentes, leurs exigences, et lesquelles de ces exigences seront traitées par le SMSI." },
    { reference: "4.3 Détermination du domaine d'application du SMSI", description: "L'organisation doit déterminer les limites et l'applicabilité du SMSI pour en établir le domaine d'application. Celui-ci doit être disponible sous forme d'information documentée." },
    { reference: "4.4 Système de management de la sécurité de l'information", description: "L'organisation doit établir, mettre en œuvre, tenir à jour et améliorer en continu un SMSI, y compris les processus nécessaires et leurs interactions." },
    { type: "chapter", title: "5 - Leadership" },
    { reference: "5.1 Leadership et engagement", description: "La direction doit faire preuve de leadership et d'engagement en s'assurant de la compatibilité de la politique et des objectifs, de l'intégration du SMSI, de la disponibilité des ressources, etc." },
    { reference: "5.2 Politique", description: "La direction doit établir une politique de sécurité de l'information appropriée, incluant des objectifs et un engagement d'amélioration continue. Elle doit être documentée, communiquée et disponible." },
    { reference: "5.3 Rôles, responsabilités et autorités", description: "La direction doit s'assurer que les responsabilités et autorités des rôles pertinents pour la sécurité de l'information sont attribuées, communiquées et comprises." },
    { type: "chapter", title: "6 - Planification" },
    { reference: "6.1.1 Actions face aux risques et opportunités (Généralités)", description: "L'organisation doit déterminer les risques et opportunités qui nécessitent d'être abordés pour assurer l'atteinte des résultats, prévenir les effets indésirables et permettre l'amélioration continue." },
    { reference: "6.1.2 Appréciation des risques de sécurité de l'information", description: "Définir et appliquer un processus d'appréciation des risques (critères, identification, analyse, évaluation). Conserver des informations documentées." },
    { reference: "6.1.3 Traitement des risques de sécurité de l'information", description: "Définir et appliquer un processus de traitement des risques, produire une Déclaration d'Applicabilité (DdA) et un plan de traitement des risques. Conserver des informations documentées." },
    { reference: "6.2 Objectifs de sécurité de l'information et plans pour les atteindre", description: "Établir des objectifs de sécurité de l'information mesurables, cohérents avec la politique, et planifier les actions pour les atteindre. Conserver des informations documentées." },
    { reference: "6.3 Planification des modifications", description: "Lorsque des modifications du SMSI sont nécessaires, elles doivent être réalisées de façon planifiée." },
    { type: "chapter", title: "7 - Supports" },
    { reference: "7.1 Ressources", description: "L'organisation doit déterminer et fournir les ressources nécessaires pour l'établissement, la mise en œuvre, la tenue à jour et l'amélioration continue du SMSI." },
    { reference: "7.2 Compétences", description: "Déterminer les compétences nécessaires, s'assurer que les personnes sont compétentes, et conserver des informations documentées appropriées comme preuves." },
    { reference: "7.3 Sensibilisation", description: "Les personnes travaillant sous le contrôle de l'organisation doivent être sensibilisées à la politique SSI, à leur contribution, et aux implications en cas de non-conformité." },
    { reference: "7.4 Communication", description: "L'organisation doit déterminer les besoins de communication interne et externe pertinents pour le SMSI (quoi, quand, avec qui, comment)." },
    { reference: "7.5.1 Informations documentées (Généralités)", description: "Le SMSI doit inclure les informations documentées exigées par la norme et celles jugées nécessaires par l'organisation." },
    { reference: "7.5.2 Création et mise à jour", description: "Lors de la création et mise à jour, l'organisation doit s'assurer de l'identification, du format, du support, de la revue et de l'approbation appropriés." },
    { reference: "7.5.3 Contrôle des informations documentées", description: "Les informations documentées doivent être contrôlées (disponibilité, protection, distribution, stockage, contrôle des modifications, conservation, etc.)." },
    { type: "chapter", title: "8 - Fonctionnement" },
    { reference: "8.1 Planification et contrôle opérationnels", description: "Planifier, mettre en œuvre et contrôler les processus pour satisfaire aux exigences, contrôler les modifications et les processus externalisés." },
    { reference: "8.2 Appréciation des risques de sécurité de l'information", description: "Réaliser des appréciations des risques à des intervalles planifiés ou lors de changements significatifs. Conserver les informations documentées sur les résultats." },
    { reference: "8.3 Traitement des risques de sécurité de l'information", description: "Mettre en œuvre le plan de traitement des risques. Conserver les informations documentées sur les résultats du traitement." },
    { type: "chapter", title: "9 - Évaluation de la performance" },
    { reference: "9.1 Surveillance, mesurage, analyse et évaluation", description: "Déterminer ce qui doit être surveillé et mesuré, les méthodes, quand effectuer les actions et par qui. Conserver les informations documentées." },
    { reference: "9.2.1 Audit interne (Généralités)", description: "Réaliser des audits internes à des intervalles planifiés pour vérifier la conformité aux exigences de l'organisation et de la norme." },
    { reference: "9.2.2 Programme d'audit interne", description: "Planifier, établir, mettre en œuvre et tenir à jour un programme d'audit. Définir les critères, assurer l'objectivité et communiquer les résultats." },
    { reference: "9.3.1 Revue de direction (Généralités)", description: "La direction doit procéder à la revue du SMSI à des intervalles planifiés pour s'assurer qu'il est toujours approprié, adapté et efficace." },
    { reference: "9.3.2 Éléments d'entrée de la revue de direction", description: "La revue doit prendre en considération une liste définie d'éléments d'entrée (actions précédentes, changements, retours sur la performance, résultats d'audit, etc.)." },
    { reference: "9.3.3 Résultats des revues de direction", description: "Les résultats doivent inclure les décisions relatives aux opportunités d'amélioration et aux changements à apporter. Conserver des informations documentées." },
    { type: "chapter", title: "10 - Amélioration" },
    { reference: "10.1 Amélioration continue", description: "L'organisation doit continuellement améliorer la pertinence, l'adéquation et l'efficacité du SMSI." },
    { reference: "10.2 Non-conformité et action corrective", description: "Réagir aux non-conformités, évaluer la nécessité d'actions correctives, les mettre en œuvre et en vérifier l'efficacité. Conserver les informations documentées." }
  ];
  const createDefaultDocumentReview = (source = []) => {
    const map = new Map((source || []).map(item => [item.reference, item]));
    return DOCUMENT_REVIEW_ITEMS.filter(it => it.reference).map(it => ({
      reference: it.reference,
      justification: map.get(it.reference)?.justification || "",
      tool: map.get(it.reference)?.tool || ""
    }));
  };
  function parseCssColor(value) {
    const raw = (value || "").trim();
    if (!raw) return null;
    if (raw.startsWith("#")) {
      const hex = raw.slice(1);
      if (hex.length === 3) {
        const r = parseInt(hex[0] + hex[0], 16);
        const g = parseInt(hex[1] + hex[1], 16);
        const b = parseInt(hex[2] + hex[2], 16);
        return { r, g, b };
      }
      if (hex.length === 6) {
        const r = parseInt(hex.slice(0, 2), 16);
        const g = parseInt(hex.slice(2, 4), 16);
        const b = parseInt(hex.slice(4, 6), 16);
        return { r, g, b };
      }
      return null;
    }
    const match = raw.match(/rgba?\(([^)]+)\)/i);
    if (!match) return null;
    const parts = match[1].split(",").map((p) => parseFloat(p.trim()));
    if (parts.length < 3 || parts.some((p) => Number.isNaN(p))) return null;
    return { r: parts[0], g: parts[1], b: parts[2] };
  }
  function withAlpha(color, alpha) {
    const rgb = parseCssColor(color);
    if (!rgb) return color;
    return `rgba(${Math.round(rgb.r)}, ${Math.round(rgb.g)}, ${Math.round(rgb.b)}, ${alpha})`;
  }
  function getThemeVars() {
    const target = document.body || document.documentElement;
    const styles = getComputedStyle(target);
    const readVar = (name, fallback) => styles.getPropertyValue(name).trim() || fallback;
    const textColor = readVar("--color-text", "#2d3e4a");
    const textSecondary = readVar("--color-text-secondary", "#758592");
    const borderColor = readVar("--color-border", "#e9ecef");
    const surface = readVar("--color-surface", "#ffffff");
    const primary = readVar("--color-primary", "#1fb8cd");
    const secondary = readVar("--color-secondary", "#5d878f");
    const success = readVar("--color-success", "#32c086");
    const warning = readVar("--color-warning", "#ffc185");
    const danger = readVar("--color-danger", "#db4545");
    const info = readVar("--color-info", "#5d878f");
    return {
      textColor,
      textSecondary,
      borderColor,
      surface,
      primary,
      secondary,
      success,
      warning,
      danger,
      info
    };
  }
  function updateThemeToggleUi() {
    const btn = document.getElementById("themeToggleBtn");
    if (!btn) return;
    const isDark = currentTheme === "dark";
    btn.classList.toggle("theme-toggle--dark", isDark);
    btn.innerHTML = isDark
      ? '<i class="fas fa-sun"></i><span class="theme-toggle__label">Mode clair</span>'
      : '<i class="fas fa-moon"></i><span class="theme-toggle__label">Mode sombre</span>';
  }
  function setTheme(mode) {
    currentTheme = normalizeTheme(mode);
    document.body.classList.toggle("theme-dark", currentTheme === "dark");
    document.body.classList.toggle("theme-light", currentTheme === "light");
    document.body.classList.toggle("theme-poudre", currentTheme === "poudre");
    document.body.classList.toggle("theme-bleu", currentTheme === "bleu");
    updateThemeToggleUi();
    try {
      if (typeof Chart !== "undefined") {
        const { textColor } = getThemeVars();
        Chart.defaults.color = textColor;
      }
    } catch (e) {
      console.warn("Impossible de mettre à jour le thème des graphiques", e);
    }
    if (typeof updateCharts === "function") {
      updateCharts();
    }
    try {
      localStorage.setItem(THEME_STORAGE_KEY, currentTheme);
    } catch (e) {
      console.warn("Impossible de sauvegarder le thème", e);
    }
  }
  function toggleTheme() {
    setTheme(currentTheme === "dark" ? "light" : "dark");
  }
  function downloadJsonBackup(content, name = "smsi_backup.json") {
    try {
      const blob = new Blob([content], { type: "application/json" });
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = name;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
    } catch (e) {
      console.error("Téléchargement de secours impossible", e);
    }
  }
  function initThemeToggle() {
    // Lire le thème par défaut depuis la balise meta (pour branding/packaging)
    const metaTheme = document.querySelector('meta[name="csi-default-theme"]');
    const brandDefaultTheme = metaTheme && metaTheme.content ? metaTheme.content : null;

    const saved = (() => {
      try {
        return localStorage.getItem(THEME_STORAGE_KEY);
      } catch (e) {
        return null;
      }
    })();

    // Si un thème de marque est défini et pas de thème sauvegardé, utiliser le thème de marque
    const defaultToUse = brandDefaultTheme || DEFAULT_THEME;
    const initial = normalizeTheme(saved || defaultToUse);
    setTheme(initial);
    const btn = document.getElementById("themeToggleBtn");
    if (btn) btn.addEventListener("click", toggleTheme);
  }
  const DEFAULT_NIS2_PHASES = [
    { id: "phase-1", label: "Phase 1 (2026)", order: 1 },
    { id: "phase-2", label: "Phase 2 (2027)", order: 2 },
    { id: "phase-3", label: "Phase 3 (2028)", order: 3 },
    { id: "phase-unplanned", label: "Hors phase", order: 99, fallback: true }
  ];
  const DEFAULT_NIS2_MATURITY = Object.fromEntries(NIS2_CATEGORIES.map(c => [c, 0]));
  const DEFAULT_SOCLE_PILLARS = [
    {
      id: "socle-pillar-1",
      title: "Continuité & Reprise",
      subtitle: "Résilience, actifs critiques, reprise d'activité",
      items: [
        "BCMS opérationnel",
        "RTO/RPO définis et testés",
        "DRP industrialisé",
        "Cartographie des actifs critiques",
        "Plan de résilience organisationnelle"
      ],
      owner: "Chef de projet : à désigner",
      icon: "fa-shield-halved"
    },
    {
      id: "socle-pillar-2",
      title: "Traçabilité transversale",
      subtitle: "Visibilité bout-en-bout et réaction SOC",
      items: [
        "Collecte des traces applicatives et techniques",
        "SIEM centralisé",
        "SOAR pour l'orchestration",
        "Couverture SOC et use-cases NIS2"
      ],
      owner: "Chef de projet : à désigner",
      icon: "fa-chart-line"
    },
    {
      id: "socle-pillar-3",
      title: "Chaîne d'approvisionnement",
      subtitle: "Sécurité des tiers et marchés publics",
      items: [
        "Contrôle de la sous-traitance",
        "Exigences de sécurité cloud",
        "Clauses et critères cyber dans les marchés",
        "Audits et suivi des sous-traitants"
      ],
      owner: "Chef de projet : à désigner",
      icon: "fa-handshake"
    },
    {
      id: "socle-pillar-4",
      title: "Identification & Authentification",
      subtitle: "Contrôle d'accès renforcé et comptes à privilèges",
      items: [
        "MFA généralisé sur les accès sensibles",
        "PAM pour comptes à privilèges",
        "IAM avec règles de moindre privilège",
        "Référentiel des identités et habilitations"
      ],
      owner: "Chef de projet : à désigner",
      icon: "fa-user-shield"
    },
    {
      id: "socle-pillar-5",
      title: "Maîtrise des risques",
      subtitle: "Détection, vulnérabilités et qualité code",
      items: [
        "Programme de gestion des vulnérabilités",
        "Détection et réponse aux incidents",
        "Qualité du code et DevSecOps",
        "Tests d'intrusion et audits techniques"
      ],
      owner: "Chef de projet : à désigner",
      icon: "fa-triangle-exclamation"
    }
  ];
  const buildNis2Maturity = (source = {}) => {
    const result = Object.assign({}, DEFAULT_NIS2_MATURITY);
    NIS2_CATEGORIES.forEach(cat => {
      const val = source[cat];
      if (typeof val === 'number') {
        result[cat] = val;
      } else {
        const legacy = source[cat.split('.')[0]];
        if (typeof legacy === 'number') result[cat] = legacy;
      }
    });
    return result;
  };
  const cloneDefaultSoclePillars = () => DEFAULT_SOCLE_PILLARS.map((pillar) => ({
    id: pillar.id,
    title: pillar.title,
    subtitle: pillar.subtitle,
    items: [...pillar.items],
    owner: pillar.owner,
    icon: pillar.icon
  }));
  const normalizeSoclePillars = (pillars = []) => {
    const source = Array.isArray(pillars) ? pillars : [];
    return DEFAULT_SOCLE_PILLARS.map((fallback, idx) => {
      const current = source[idx] || {};
      return {
        id: typeof current.id === "string" ? current.id : fallback.id,
        title: typeof current.title === "string" ? current.title : fallback.title,
        subtitle: typeof current.subtitle === "string" ? current.subtitle : fallback.subtitle,
        items: Array.isArray(current.items) ? current.items.filter(Boolean) : [...fallback.items],
        owner: typeof current.owner === "string" ? current.owner : fallback.owner,
        icon: typeof current.icon === "string" ? current.icon : fallback.icon
      };
    });
  };
  const DEFAULT_NIS2_PLAN = {
    phases: [...DEFAULT_NIS2_PHASES],
    maturity: {
      labels: [
        "Gestion d'actifs",
        "Environnement opérationnel",
        "Gouvernance",
        "Evaluation des risques",
        "Stratégie de gestion des risques",
        "Gestion des risques liés à la chaine d'approvisionnement",
        "Gestion des identités, authentification, contrôle d'accès",
        "Sensibilisation et formation",
        "Sécurité des données",
        "Processus et procédures de protection de l'information",
        "Maintenance",
        "Technologie de protection",
        "Anomalies et événements",
        "Surveillance continue",
        "Processus de détection",
        "Planification de la réponse",
        "Communications",
        "Analyse",
        "Atténuation",
        "Améliorations",
        "Planification du rétablissement"
      ],
      current: Array(21).fill(0),
      target2027: 0,
      target2028: 0
    },
    statuses: [],
    budgetBlocks: [
      { id: "nis2-budget-1", title: "Budget total mise en conformité" },
      { id: "nis2-budget-2", title: "Budget 2026" },
      { id: "nis2-budget-3", title: "Budget Infrastructure" },
      { id: "nis2-budget-4", title: "Budget SSI" }
    ],
    customBlocks: [
      { id: "nis2-highlight-1", title: "Budget engagé", value: 0, format: "currency" },
      { id: "nis2-highlight-2", title: "Progression globale", value: 0, format: "percent" },
      { id: "nis2-highlight-3", title: "Actions terminées", value: 0, format: "percent" }
    ],
    domains: [
      {
        title: "Gouvernance & responsabilité",
        objective: "",
        pillar: 5,
        urgency: "standard",
        actions: [
          { text: "Formaliser la gouvernance NIS2 et le comité de pilotage.", phase: 1, status: "in_progress" },
          { text: "Définir la cartographie des rôles et responsabilités SSI.", phase: 1, status: "todo" },
          { text: "Mettre en place un reporting trimestriel des risques majeurs.", phase: 2, status: "todo" },
          { text: "Valider la feuille de route NIS2 auprès de la direction.", phase: 2, status: "todo" }
        ]
      },
      {
        title: "Gestion des risques (SMSI)",
        objective: "",
        pillar: 5,
        urgency: "standard",
        actions: [
          { text: "Actualiser la méthodologie d'analyse des risques et les critères.", phase: 1, status: "in_progress" },
          { text: "Lancer la campagne d'évaluation des risques métiers critiques.", phase: 1, status: "todo" },
          { text: "Prioriser le plan de traitement et définir les échéances.", phase: 2, status: "todo" }
        ]
      },
      {
        title: "Gestion des incidents & notification",
        objective: "",
        pillar: 2,
        urgency: "high",
        actions: [
          { text: "Revoir la chaîne d'escalade et l'astreinte cyber.", phase: 1, status: "todo" },
          { text: "Préparer des modèles de notification interne et externe.", phase: 1, status: "todo" },
          { text: "Organiser un exercice de gestion d'incident majeur.", phase: 2, status: "todo" }
        ]
      },
      {
        title: "Continuité d’activité & gestion de crise (BCP/DRP)",
        objective: "",
        pillar: 1,
        urgency: "high",
        actions: [
          { text: "Mettre à jour la BIA et les dépendances critiques.", phase: 1, status: "in_progress" },
          { text: "Définir les scénarios de reprise et les RTO/RPO cibles.", phase: 1, status: "todo" },
          { text: "Tester un plan de reprise sur un service prioritaire.", phase: 2, status: "todo" },
          { text: "Mettre à jour le plan de communication de crise.", phase: 2, status: "todo" }
        ]
      },
      {
        title: "Sécurité de la chaîne d’approvisionnement",
        objective: "",
        pillar: 3,
        urgency: "high",
        actions: [
          { text: "Classifier les fournisseurs critiques et leurs exigences.", phase: 1, status: "todo" },
          { text: "Déployer un questionnaire de sécurité tiers.", phase: 1, status: "todo" },
          { text: "Planifier des audits périodiques des prestataires.", phase: 2, status: "todo" }
        ]
      },
      {
        title: "Sécurité dans l’acquisition, le développement & la maintenance",
        objective: "",
        pillar: 5,
        urgency: "high",
        actions: [
          { text: "Intégrer les exigences de sécurité dans les cahiers des charges.", phase: 1, status: "todo" },
          { text: "Mettre en place des revues de sécurité avant mise en production.", phase: 2, status: "todo" },
          { text: "Suivre les vulnérabilités des dépendances logicielles.", phase: 2, status: "in_progress" }
        ]
      },
      {
        title: "Traçabilité et auditabilité",
        objective: "",
        pillar: 2,
        urgency: "medium",
        actions: [
          { text: "Définir la politique de journalisation centralisée.", phase: 1, status: "todo" },
          { text: "Mettre en place des alertes et corrélations prioritaires.", phase: 2, status: "todo" },
          { text: "Réaliser un audit technique annuel des périmètres critiques.", phase: 3, status: "todo" }
        ]
      },
      {
        title: "Gestion des vulnérabilités",
        objective: "",
        pillar: 5,
        urgency: "high",
        actions: [
          { text: "Établir le calendrier des scans et des correctifs.", phase: 1, status: "in_progress" },
          { text: "Mettre en place un processus d'escalade des vulnérabilités critiques.", phase: 1, status: "todo" },
          { text: "Suivre les indicateurs de remédiation mensuels.", phase: 2, status: "todo" }
        ]
      },
      {
        title: "Sensibilisation & formation",
        objective: "",
        pillar: 5,
        urgency: "standard",
        actions: [
          { text: "Construire un parcours de formation par population.", phase: 1, status: "todo" },
          { text: "Lancer des campagnes de sensibilisation ciblées.", phase: 1, status: "in_progress" },
          { text: "Mesurer l'efficacité via quiz et simulations.", phase: 2, status: "todo" }
        ]
      },
      {
        title: "Cryptographie",
        objective: "",
        pillar: 4,
        urgency: "medium",
        actions: [
          { text: "Recenser les usages cryptographiques (données et flux).", phase: 1, status: "todo" },
          { text: "Définir les standards d'algorithmes et de rotation.", phase: 2, status: "todo" },
          { text: "Mettre en place la gestion du cycle de vie des certificats.", phase: 2, status: "todo" }
        ]
      },
      {
        title: "Sécurité physique",
        objective: "",
        pillar: 4,
        urgency: "standard",
        actions: [
          { text: "Mettre à jour les zones et niveaux d'accès physiques.", phase: 1, status: "todo" },
          { text: "Renforcer la surveillance des locaux sensibles.", phase: 2, status: "todo" },
          { text: "Tester les procédures d'accès d'urgence.", phase: 3, status: "todo" }
        ]
      },
      {
        title: "Gestion des accès au SI",
        objective: "",
        pillar: 4,
        urgency: "standard",
        actions: [
          { text: "Appliquer le principe du moindre privilège sur les comptes.", phase: 1, status: "todo" },
          { text: "Déployer l'authentification multifacteur sur les comptes à risque.", phase: 2, status: "todo" },
          { text: "Planifier les revues d'habilitations trimestrielles.", phase: 3, status: "todo" }
        ]
      }
    ]
  };
  const clampPercent = (value, fallback = 0) => {
    const num = parseFloat(value);
    if (Number.isFinite(num)) return Math.max(0, Math.min(100, Math.round(num)));
    return Math.max(0, Math.min(100, fallback || 0));
  };
  const parseLooseNumber = (value) => {
    if (typeof value === "number") return value;
    if (value === null || value === void 0) return NaN;
    const str = String(value).replace(/\s/g, "").replace(",", ".").replace(/[^0-9.\-]/g, "");
    const num = parseFloat(str);
    return Number.isFinite(num) ? num : NaN;
  };
  const cloneDefaultNis2Plan = (options = {}) => {
    const includeActions = options.includeActions === true;
    const clamp = typeof clampPercent === "function"
      ? clampPercent
      : (value, fallback = 0) => {
          const num = parseFloat(value);
          if (Number.isFinite(num)) return Math.max(0, Math.min(100, Math.round(num)));
          return Math.max(0, Math.min(100, fallback || 0));
        };
    const parseLoose = typeof parseLooseNumber === "function"
      ? parseLooseNumber
      : (value) => {
          if (typeof value === "number") return value;
          if (value === null || value === void 0) return NaN;
          const str = String(value).replace(/\s/g, "").replace(",", ".").replace(/[^0-9.\-]/g, "");
          const num = parseFloat(str);
          return Number.isFinite(num) ? num : NaN;
        };
    return {
      phases: [...DEFAULT_NIS2_PLAN.phases],
      maturity: {
        labels: [...DEFAULT_NIS2_PLAN.maturity.labels],
        current: [...DEFAULT_NIS2_PLAN.maturity.current],
        target2027: DEFAULT_NIS2_PLAN.maturity.target2027,
        target2028: DEFAULT_NIS2_PLAN.maturity.target2028
      },
      statuses: DEFAULT_NIS2_PLAN.statuses.map((s, idx) => ({
        id: s.id || `nis2-status-${idx + 1}`,
        text: s.text,
        status: s.status
      })),
      budgetBlocks: (DEFAULT_NIS2_PLAN.budgetBlocks || []).map((b, idx) => ({
        id: b.id || `nis2-budget-${idx + 1}`,
        title: b.title || `Budget ${idx + 1}`
      })),
      customBlocks: (DEFAULT_NIS2_PLAN.customBlocks || []).map((b, idx) => {
        const format = (b.format || "currency").toLowerCase();
        const rawValue = typeof b.value !== "undefined" ? b.value : b.percent;
        const value = format === "currency"
          ? (() => {
              const num = parseLoose(rawValue);
              return Number.isFinite(num) ? num : 0;
            })()
          : clamp(rawValue, 0);
        return {
          id: b.id || `nis2-highlight-${idx + 1}`,
          title: b.title,
          value,
          format
        };
      }),
      domains: DEFAULT_NIS2_PLAN.domains.map((d, dIndex) => {
        const actions = includeActions ? (d.actions || []).map((a, aIndex) => ({
          id: a.id || `nis2-act-${dIndex + 1}-${aIndex + 1}`,
          text: a.text,
          phase: typeof a.phase === "number" ? a.phase : parseInt(a.phase, 10) || 1,
          status: a.status || "todo",
          owner: a.owner || "",
          workType: a.workType || "initiative",
          budget: typeof a.budget === "number" ? String(a.budget) : a.budget || "",
          budget2026: typeof a.budget2026 === "number" ? String(a.budget2026) : a.budget2026 || "",
          budget2027: typeof a.budget2027 === "number" ? String(a.budget2027) : a.budget2027 || "",
          budget2028: typeof a.budget2028 === "number" ? String(a.budget2028) : a.budget2028 || "",
          budgetCategory: a.budgetCategory || "divers",
          progress: typeof a.progress === "number" ? Math.max(0, Math.min(100, a.progress)) : a.status === "done" ? 100 : 0
        })) : [];
        return {
          id: d.id || `nis2-domain-${dIndex + 1}`,
          title: d.title,
          objective: d.objective,
          pillar: d.pillar,
          urgency: d.urgency || "standard",
          actions
        };
      })
    };
  };
  const getEmptyNis2Plan = () => {
    const plan = cloneDefaultNis2Plan();
    const labels = Array.isArray(plan.maturity?.labels) ? plan.maturity.labels : [];
    plan.maturity = {
      labels,
      current: labels.map(() => 0),
      target2027: 0,
      target2028: 0
    };
    plan.statuses = [];
    plan.budgetBlocks = [];
    plan.customBlocks = [];
    plan.domains = (plan.domains || []).map((domain) => ({
      ...domain,
      actions: []
    }));
    return plan;
  };
  const normalizeNis2Plan = (plan) => {
    const defaults = cloneDefaultNis2Plan();
    const src = plan && typeof plan === "object" ? plan : {};
    const clamp = typeof clampPercent === "function"
      ? clampPercent
      : (value, fallback = 0) => {
          const num = parseFloat(value);
          if (Number.isFinite(num)) return Math.max(0, Math.min(100, Math.round(num)));
          return Math.max(0, Math.min(100, fallback || 0));
        };
    const parseLoose = typeof parseLooseNumber === "function"
      ? parseLooseNumber
      : (value) => {
          if (typeof value === "number") return value;
          if (value === null || value === void 0) return NaN;
          const str = String(value).replace(/\s/g, "").replace(",", ".").replace(/[^0-9.\-]/g, "");
          const num = parseFloat(str);
          return Number.isFinite(num) ? num : NaN;
        };
    const phases = Array.isArray(src.phases) && src.phases.length ? src.phases.map((p, idx) => ({
      id: p.id || `phase-${idx + 1}`,
      label: p.label || p.name || `Phase ${idx + 1}`,
      order: typeof p.order === "number" ? p.order : parseInt(p.order, 10) || idx + 1,
      fallback: !!p.fallback
    })) : defaults.phases;
    appData.nis2Plan = appData.nis2Plan || {};
    appData.nis2Plan.phases = phases;
    const normalizedPhases = ensureNis2Phases();
    const maturity = {
      labels: Array.isArray(src?.maturity?.labels) && src.maturity.labels.length ? src.maturity.labels : defaults.maturity.labels,
      current: Array.isArray(src?.maturity?.current) && src.maturity.current.length ? src.maturity.current : defaults.maturity.current,
      target2027: typeof src?.maturity?.target2027 === "number" ? src.maturity.target2027 : defaults.maturity.target2027,
      target2028: typeof src?.maturity?.target2028 === "number" ? src.maturity.target2028 : defaults.maturity.target2028
    };
    let statuses = Array.isArray(src.statuses) ? src.statuses.map((s, idx) => {
      const fallbackText = defaults.statuses.length
        ? defaults.statuses[idx % defaults.statuses.length].text
        : `Statut ${idx + 1}`;
      return {
        id: s.id || `nis2-status-${idx + 1}`,
        text: s.text || fallbackText,
        status: s.status || "todo"
      };
    }) : defaults.statuses;
    const budgetBlocks = (Array.isArray(src.budgetBlocks) && src.budgetBlocks.length ? src.budgetBlocks : defaults.budgetBlocks || []).map((b, idx) => ({
      id: b.id || `nis2-budget-${idx + 1}`,
      title: b.title || defaults.budgetBlocks?.[idx % (defaults.budgetBlocks.length || 1)]?.title || `Budget ${idx + 1}`
    }));
    const sourceBlocks = Array.isArray(src.customBlocks) && src.customBlocks.length ? src.customBlocks : defaults.customBlocks;
    const customBlocks = sourceBlocks.map((b, idx) => {
      const fallback = defaults.customBlocks[idx % defaults.customBlocks.length] || defaults.customBlocks[0] || {};
      let format = (b.format || fallback.format || "currency").toLowerCase();
      if (idx === 0 && format !== "currency") format = "currency";
      const rawValue = typeof b.value !== "undefined" ? b.value : (typeof b.percent !== "undefined" ? b.percent : (typeof fallback.value !== "undefined" ? fallback.value : fallback.percent));
      const fallbackValue = typeof fallback.value !== "undefined" ? fallback.value : fallback.percent;
      const value = format === "currency"
        ? (() => {
            const num = parseLoose(rawValue);
            const fb = parseLoose(fallbackValue);
            return Number.isFinite(num) ? num : (Number.isFinite(fb) ? fb : 0);
          })()
        : clamp(rawValue ?? fallbackValue ?? 0, clamp(fallbackValue ?? 0, 0));
      return {
        id: b.id || `nis2-highlight-${idx + 1}`,
        title: b.title || fallback.title || `Bloc ${idx + 1}`,
        value,
        format
      };
    });
    const domainsSource = Array.isArray(src.domains) ? src.domains : defaults.domains;
    const domains = domainsSource.map((d, dIndex) => {
      const templateDomain = defaults.domains[dIndex % defaults.domains.length];
      const templateActions = templateDomain.actions || [];
      return {
        id: d.id || `nis2-domain-${dIndex + 1}`,
        title: d.title || templateDomain.title,
        objective: d.objective || templateDomain.objective,
        pillar: d.pillar || templateDomain.pillar,
        urgency: d.urgency || "standard",
        actions: Array.isArray(d.actions) ? d.actions.map((a, aIndex) => {
          const fallback = templateActions[aIndex % templateActions.length] || templateActions[0] || {};
          const phase = normalizePhaseId(typeof a.phase === "undefined" ? fallback.phase : a.phase);
          const progress = typeof a.progress === "number" ? Math.max(0, Math.min(100, a.progress)) : a.status === "done" ? 100 : 0;
          const budget = typeof a.budget === "number" ? String(a.budget) : a.budget || fallback.budget || "";
          const budget2026 = typeof a.budget2026 === "number" ? String(a.budget2026) : a.budget2026 || fallback.budget2026 || "";
          const budget2027 = typeof a.budget2027 === "number" ? String(a.budget2027) : a.budget2027 || fallback.budget2027 || "";
          const budget2028 = typeof a.budget2028 === "number" ? String(a.budget2028) : a.budget2028 || fallback.budget2028 || "";
          const budgetCategory = a.budgetCategory || fallback.budgetCategory || "divers";
          return {
            id: a.id || `nis2-act-${dIndex + 1}-${aIndex + 1}`,
            text: a.text || fallback.text || "",
            phase,
            status: a.status || fallback.status || "todo",
            owner: a.owner || "",
            workType: a.workType || fallback.workType || "initiative",
            budget,
            budget2026,
            budget2027,
            budget2028,
            budgetCategory,
            progress
          };
        }) : []
      };
    });
    return { phases: normalizedPhases, maturity, statuses, budgetBlocks, customBlocks, domains };
  };
  if (typeof window !== 'undefined') {
    window.DOMAIN_BY_PREFIX = DOMAIN_BY_PREFIX;
    window.OPERATIONAL_CAPABILITIES = OPERATIONAL_CAPABILITIES;
    window.NIS2_CATEGORIES = NIS2_CATEGORIES;
  }
  // modules/dashboard.js
  function initDashboard() {
    updateDashboard();
  }
  function updateDashboard() {
    updateKPIs();
    updateCharts();
    updateControlsLabel();
  }
  function validateDataQuality() {
    const issues = [];
    const collectDuplicates = (items, label, key = "title") => {
      const map = new Map();
      items.forEach((it) => {
        const val = (it[key] || "").trim().toLowerCase();
        if (!val) return;
        map.set(val, (map.get(val) || 0) + 1);
      });
      const dups = [...map.entries()].filter(([, count]) => count > 1);
      dups.forEach(([val, count]) => issues.push(`Doublon ${label}: "${val}" (${count}x)`));
    };
    collectDuplicates(appData.actions, "action");
    collectDuplicates(appData.risks, "risque");
    collectDuplicates(appData.documents, "document", "type");

    const controlsMissingSubcat = appData.controls.filter(c => !c.subcategory || (Array.isArray(c.subcategory) && c.subcategory.length === 0));
    if (controlsMissingSubcat.length) {
      issues.push(`${controlsMissingSubcat.length} contrôle(s) sans sous-catégorie opérationnelle`);
    }

    const badLinks = (appData.documents || []).filter((d) => {
      if (!d.link) return false;
      try {
        const url = new URL(d.link);
        return !["http:", "https:"].includes(url.protocol);
      } catch (e) {
        return true;
      }
    });
    badLinks.forEach((doc) => issues.push(`Lien document invalide: ${doc.link || "(vide)"}`));

    if (issues.length === 0) {
      showToast2("Aucune incohérence détectée", "success");
      return;
    }
    const body = issues.map((i) => `- ${i}`).join("\n");
    openModal("Qualité des données", [
      { name: "issues", label: "Points détectés", type: "textarea", value: body, readOnly: true }
    ], () => true);
  }
  function getHistoryLimit() {
    try {
      const raw = localStorage.getItem(HISTORY_LIMIT_STORAGE_KEY);
      const value = parseInt(raw, 10);
      if (Number.isFinite(value) && value >= 1 && value <= 30) return value;
    } catch (e) {
      // no-op
    }
    return HISTORY_LIMIT_DEFAULT;
  }
  function normalizeHistorySnapshot(entry) {
    const cleanLabel = typeof entry?.label === "string" ? entry.label.trim() : "";
    const data = typeof entry?.data === "string" ? entry.data : "";
    const tag = entry?.tag === "auto" ? "auto" : "manuel";
    const size = typeof entry?.size === "number" ? entry.size : (data ? data.length : 0);
    return {
      ...entry,
      label: cleanLabel || "Snapshot",
      tag,
      size,
      data
    };
  }
  function getHistorySnapshots() {
    try {
      const raw = localStorage.getItem(HISTORY_KEY);
      const list = raw ? JSON.parse(raw) : [];
      if (!Array.isArray(list)) return [];
      return list.map(normalizeHistorySnapshot);
    } catch (e) {
      return [];
    }
  }
  function saveHistorySnapshots(list) {
    try {
      const limit = getHistoryLimit();
      const normalized = Array.isArray(list) ? list.map(normalizeHistorySnapshot) : [];
      normalized.sort((a, b) => {
        const aTime = Date.parse(a.date || "") || 0;
        const bTime = Date.parse(b.date || "") || 0;
        return aTime - bTime;
      });
      localStorage.setItem(HISTORY_KEY, JSON.stringify(normalized.slice(-limit)));
      refreshStorageHealth();
    } catch (e) {
      console.warn("Impossible de sauvegarder l'historique", e);
    }
  }
  function addHistorySnapshot(label = "Snapshot", options = {}) {
    try {
      ensureNis2Plan();
      ensureCriticalAssets();
      ensureProjectRisks();
      const data = JSON.stringify(appData);
      const cleanLabel = typeof label === "string" ? label.trim() : "Snapshot";
      const tag = options?.tag === "auto" ? "auto" : "manuel";
      const entry = {
        id: generateId("snapshot"),
        label: cleanLabel || "Snapshot",
        date: new Date().toISOString(),
        data,
        size: data.length,
        tag
      };
      const list = getHistorySnapshots();
      list.push(entry);
      saveHistorySnapshots(list);
      showToast2("Snapshot enregistré", "success");
      renderHistoryList();
    } catch (e) {
      showToast2("Impossible de créer le snapshot: " + e.message, "error");
    }
  }
  function formatSnapshotSize(bytes) {
    const size = Number(bytes);
    if (!Number.isFinite(size) || size <= 0) return "0 KB";
    if (size < 1024) return `${size} o`;
    const kb = size / 1024;
    if (kb < 1024) return `${kb.toFixed(1)} Ko`;
    const mb = kb / 1024;
    return `${mb.toFixed(1)} Mo`;
  }
  function renderHistoryList() {
    const container = document.getElementById("historyList");
    if (!container) return;
    const searchInput = document.getElementById("historySearch");
    const tagFilter = document.getElementById("historyTagFilter");
    const searchQuery = NORMALIZE(searchInput ? searchInput.value.trim() : "");
    const selectedTag = tagFilter ? tagFilter.value : "all";
    const list = getHistorySnapshots().slice().sort((a, b) => {
      const aTime = Date.parse(a.date || "") || 0;
      const bTime = Date.parse(b.date || "") || 0;
      return bTime - aTime;
    });
    const filtered = list.filter((snap) => {
      const tagMatch = selectedTag === "all" || snap.tag === selectedTag;
      if (!tagMatch) return false;
      if (!searchQuery) return true;
      return NORMALIZE(snap.label).includes(searchQuery);
    });
    container.innerHTML = "";
    if (!list.length) {
      container.textContent = "Aucun snapshot enregistré";
      const meta = document.getElementById("historyMeta");
      if (meta) {
        meta.innerHTML = "";
        const empty = document.createElement("div");
        empty.textContent = "Historique vide.";
        meta.appendChild(empty);
      }
      return;
    }
    const meta = document.getElementById("historyMeta");
    if (meta) {
      meta.innerHTML = "";
      const countLine = document.createElement("div");
      const limit = getHistoryLimit();
      countLine.textContent = `Snapshots conservés: ${list.length}/${limit}`;
      const visibleLine = document.createElement("div");
      visibleLine.textContent = `Affichés: ${filtered.length}`;
      const latest = list[0];
      const latestLine = document.createElement("div");
      const latestDate = latest?.date ? new Date(latest.date).toLocaleString() : "N/A";
      latestLine.textContent = `Dernier snapshot: ${latestDate}`;
      const totalSize = list.reduce((sum, snap) => {
        const size = typeof snap.size === "number" ? snap.size : (snap.data ? snap.data.length : 0);
        return sum + (Number.isFinite(size) ? size : 0);
      }, 0);
      const sizeLine = document.createElement("div");
      sizeLine.textContent = `Taille cumulée: ${formatSnapshotSize(totalSize)}`;
      meta.append(countLine, visibleLine, latestLine, sizeLine);
    }
    if (!filtered.length) {
      container.textContent = "Aucun snapshot ne correspond aux filtres.";
      return;
    }
    filtered.forEach((snap) => {
      const item = document.createElement("div");
      item.className = "history-item";
      const date = snap.date ? new Date(snap.date).toLocaleString() : "N/A";
      const size = typeof snap.size === "number" ? snap.size : (snap.data ? snap.data.length : 0);
      const sizeLabel = formatSnapshotSize(size);
      const tagLabel = snap.tag === "auto" ? "Auto" : "Manuel";
      item.innerHTML = `
        <div class="history-item__info">
          <div class="history-item__headline">
            <span class="history-item__tag history-item__tag--${escapeHtml(snap.tag)}">${tagLabel}</span>
            <div class="history-item__title">${escapeHtml(snap.label || "Snapshot")}</div>
          </div>
          <div class="history-item__meta">${escapeHtml(date)} • ${escapeHtml(sizeLabel)}</div>
        </div>
        <div class="history-item__actions">
          <button class="btn btn--outline btn--sm" data-action="restore" data-id="${snap.id}">Restaurer</button>
          <button class="btn btn--outline btn--sm" data-action="download" data-id="${snap.id}">Exporter</button>
          <button class="btn btn--outline btn--sm" data-action="delete" data-id="${snap.id}">Supprimer</button>
        </div>
      `;
      container.appendChild(item);
    });
    container.querySelectorAll("[data-action]").forEach((btn) => {
      btn.addEventListener("click", (e) => {
        const id = e.currentTarget.dataset.id;
        const action = e.currentTarget.dataset.action;
        const snap = getHistorySnapshots().find((s) => s.id === id);
        if (!snap) return;
        if (action === "restore") {
          if (!confirm("Restaurer ce snapshot ? Les données actuelles seront remplacées.")) return;
          try {
            addHistorySnapshot("Backup avant restauration", { tag: "auto" });
          } catch (err) {
            console.warn("Snapshot de sécurité non créé", err);
          }
          localStorage.setItem("smsi_data", snap.data);
          showToast2("Snapshot restauré, rechargement...", "success");
          setTimeout(() => location.reload(), 300);
        } else if (action === "download") {
          downloadJsonBackup(snap.data, `smsi_snapshot_${snap.id}.json`);
        } else if (action === "delete") {
          if (!confirm("Supprimer ce snapshot ?")) return;
          const nextList = getHistorySnapshots().filter((s) => s.id !== snap.id);
          saveHistorySnapshots(nextList);
          renderHistoryList();
        }
      });
    });
  }
  function clearHistorySnapshots() {
    if (!confirm("Supprimer tout l'historique des snapshots ?")) return;
    try {
      localStorage.removeItem(HISTORY_KEY);
      renderHistoryList();
      refreshStorageHealth();
      showToast2("Historique des snapshots vidé", "info");
    } catch (e) {
      console.error("Erreur suppression historique", e);
      showToast2("Impossible de supprimer l'historique", "error");
    }
  }
  function refreshSyncMeta() {
    const meta = document.getElementById("syncMeta");
    if (!meta) return;
    meta.innerHTML = "";
    let lastPull = "";
    let lastPush = "";
    try {
      lastPull = localStorage.getItem("smsi_last_server_pull") || "";
      lastPush = localStorage.getItem("smsi_last_server_push") || "";
    } catch (e) {
      return;
    }
    const formatDate = (value) => {
      const ts = Date.parse(value || "");
      return Number.isFinite(ts) ? new Date(ts).toLocaleString() : "Jamais";
    };
    const rows = [
      { label: "Dernier chargement serveur", value: formatDate(lastPull) },
      { label: "Dernière publication serveur", value: formatDate(lastPush) }
    ];
    rows.forEach((row) => {
      const line = document.createElement("div");
      line.textContent = `${row.label}: ${row.value}`;
      meta.appendChild(line);
    });
  }
  window.addHistorySnapshot = addHistorySnapshot;
  window.refreshSyncMeta = refreshSyncMeta;
  window.refreshStorageHealth = refreshStorageHealth;
  function shouldShowWelcome(hasCachedData) {
    if (hasCachedData) return false;
    if (typeof shouldAutoUnlockForTests === "function" && shouldAutoUnlockForTests()) return false;
    try {
      return !localStorage.getItem(WELCOME_STORAGE_KEY);
    } catch (e) {
      return false;
    }
  }
  // =====================================================
  // WIZARD DE CONFIGURATION AU PREMIER LANCEMENT
  // =====================================================

  const WIZARD_IMAGE_URL = "data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 400 300'%3E%3Cdefs%3E%3ClinearGradient id='bg' x1='0%25' y1='0%25' x2='100%25' y2='100%25'%3E%3Cstop offset='0%25' style='stop-color:%23e8d5e0'/%3E%3Cstop offset='50%25' style='stop-color:%23d4c4d9'/%3E%3Cstop offset='100%25' style='stop-color:%23c9d4e8'/%3E%3C/linearGradient%3E%3ClinearGradient id='cloud' x1='0%25' y1='0%25' x2='100%25' y2='100%25'%3E%3Cstop offset='0%25' style='stop-color:%23ffffff;stop-opacity:0.9'/%3E%3Cstop offset='100%25' style='stop-color:%23f0f4f8;stop-opacity:0.8'/%3E%3C/linearGradient%3E%3C/defs%3E%3Crect fill='url(%23bg)' width='400' height='300'/%3E%3Cellipse cx='200' cy='120' rx='140' ry='80' fill='url(%23cloud)' stroke='%23ffffff' stroke-width='3'/%3E%3Crect x='110' y='80' width='180' height='100' rx='8' fill='%23ffffff' fill-opacity='0.95' stroke='%23e0e0e0'/%3E%3Ccircle cx='120' cy='90' r='3' fill='%23ff6b6b'/%3E%3Ccircle cx='130' cy='90' r='3' fill='%23ffd93d'/%3E%3Ccircle cx='140' cy='90' r='3' fill='%2332d296'/%3E%3Crect x='125' y='110' width='50' height='55' rx='4' fill='%23f8f9fa' stroke='%23e9ecef'/%3E%3Crect x='185' y='110' width='50' height='55' rx='4' fill='%23f8f9fa' stroke='%23e9ecef'/%3E%3Crect x='245' y='110' width='40' height='55' rx='4' fill='%23f8f9fa' stroke='%23e9ecef'/%3E%3Cpath d='M150 125 L150 145 L140 145 L150 155 L160 145 L150 145' fill='none' stroke='%231fb8cd' stroke-width='2'/%3E%3Ccircle cx='210' cy='137' r='12' fill='none' stroke='%231fb8cd' stroke-width='2'/%3E%3Ccircle cx='210' cy='137' r='4' fill='%231fb8cd'/%3E%3Cpath d='M260 125 L270 125 L270 150 L260 150 M265 130 L265 145' fill='none' stroke='%231fb8cd' stroke-width='2'/%3E%3Cpath d='M180 230 C180 200 200 190 200 190 L200 190' fill='none' stroke='%23d4a5a5' stroke-width='8' stroke-linecap='round'/%3E%3Cellipse cx='200' cy='250' rx='60' ry='8' fill='%23c9c9c9' fill-opacity='0.3'/%3E%3C/svg%3E";

  function showSetupWizard(onComplete) {
    // Skip if already shown or not first launch
    const isFirstLang = window.CSI_i18n && window.CSI_i18n.isFirstLaunch();
    const isFirstWelcome = shouldShowWelcome(false);

    if (!isFirstLang && !isFirstWelcome) {
      if (onComplete) onComplete();
      return;
    }

    if (document.getElementById("setupWizardOverlay")) {
      if (onComplete) onComplete();
      return;
    }

    let currentStep = 0;
    const steps = [];

    // Step 1: Language selection (if needed)
    if (isFirstLang) {
      steps.push({
        id: "language",
        title: "Bienvenue / Welcome / Welkom",
        subtitle: "Choisissez votre langue / Choose your language / Kies uw taal",
        content: () => {
          const langs = window.CSI_i18n.getAvailableLanguages();
          return `
            <div class="wizard-language-grid">
              ${Object.entries(langs).map(([code, info]) => `
                <button class="wizard-lang-btn ${code === 'fr' ? 'wizard-lang-btn--selected' : ''}" type="button" data-lang="${code}">
                  <span class="wizard-lang-btn__flag">${info.flag}</span>
                  <span class="wizard-lang-btn__name">${info.name}</span>
                  <span class="wizard-lang-btn__check"><i class="fas fa-check"></i></span>
                </button>
              `).join("")}
            </div>
            <p class="wizard-hint">Vous pourrez modifier ce choix dans les paramètres.</p>
          `;
        },
        onEnter: (stepEl) => {
          const langBtns = stepEl.querySelectorAll(".wizard-lang-btn");
          langBtns.forEach(btn => {
            btn.addEventListener("click", () => {
              langBtns.forEach(b => b.classList.remove("wizard-lang-btn--selected"));
              btn.classList.add("wizard-lang-btn--selected");
            });
          });
        },
        onLeave: (stepEl) => {
          const selected = stepEl.querySelector(".wizard-lang-btn--selected");
          if (selected && window.CSI_i18n) {
            const lang = selected.dataset.lang;
            window.CSI_i18n.setLanguage(lang);
            window.CSI_i18n.markLanguageChosen();
            window.CSI_i18n.translatePage();
            updateWizardTexts();
          }
        },
        canProceed: () => true
      });
    }

    // Step 2: Welcome & Demo data
    if (isFirstWelcome) {
      steps.push({
        id: "welcome",
        title: "Bienvenue sur Cyber-Assistant",
        subtitle: "Votre outil de pilotage SMSI conforme ISO 27001:2022 et NIS2",
        content: () => `
          <div class="wizard-welcome-content">
            <div class="wizard-features">
              <div class="wizard-feature">
                <div class="wizard-feature__icon"><i class="fas fa-shield-halved"></i></div>
                <div class="wizard-feature__text">
                  <strong>ISO 27001 & NIS2</strong>
                  <span>Pilotez votre conformité en un seul outil</span>
                </div>
              </div>
              <div class="wizard-feature">
                <div class="wizard-feature__icon"><i class="fas fa-chart-pie"></i></div>
                <div class="wizard-feature__text">
                  <strong>Tableaux de bord</strong>
                  <span>Suivez vos indicateurs et votre maturité</span>
                </div>
              </div>
              <div class="wizard-feature">
                <div class="wizard-feature__icon"><i class="fas fa-file-export"></i></div>
                <div class="wizard-feature__text">
                  <strong>Exports & rapports</strong>
                  <span>Générez PDF, Excel et JSON en un clic</span>
                </div>
              </div>
            </div>
          </div>
        `,
        onEnter: () => {},
        onLeave: () => {
          try { localStorage.setItem(WELCOME_STORAGE_KEY, "1"); } catch (e) {}
        },
        canProceed: () => true
      });

      // Step 3: Demo data choice
      steps.push({
        id: "demo",
        title: "Données de démonstration",
        subtitle: "Souhaitez-vous charger un jeu de données pour découvrir l'application ?",
        content: () => `
          <div class="wizard-demo-choice">
            <button class="wizard-choice-card wizard-choice-card--selected" type="button" data-choice="demo">
              <div class="wizard-choice-card__icon"><i class="fas fa-play-circle"></i></div>
              <div class="wizard-choice-card__content">
                <strong>Découvrir avec des exemples</strong>
                <span>Idéal pour comprendre l'application rapidement</span>
              </div>
              <span class="wizard-choice-card__badge">Recommandé</span>
            </button>
            <button class="wizard-choice-card" type="button" data-choice="empty">
              <div class="wizard-choice-card__icon"><i class="fas fa-file-circle-plus"></i></div>
              <div class="wizard-choice-card__content">
                <strong>Commencer à vide</strong>
                <span>Pour les utilisateurs expérimentés</span>
              </div>
            </button>
          </div>
          <p class="wizard-hint"><i class="fas fa-sync-alt"></i> Ce choix est réversible : vous pourrez réinitialiser à tout moment via Paramètres.</p>
        `,
        onEnter: (stepEl) => {
          const cards = stepEl.querySelectorAll(".wizard-choice-card");
          cards.forEach(card => {
            card.addEventListener("click", () => {
              cards.forEach(c => c.classList.remove("wizard-choice-card--selected"));
              card.classList.add("wizard-choice-card--selected");
            });
          });
        },
        onLeave: (stepEl) => {
          const selected = stepEl.querySelector(".wizard-choice-card--selected");
          if (selected && selected.dataset.choice === "demo") {
            loadDemoData({ allowReadOnly: true });
          }
        },
        canProceed: () => true
      });
    }

    // If no steps, just complete
    if (steps.length === 0) {
      if (onComplete) onComplete();
      return;
    }

    // Create wizard overlay
    const overlay = document.createElement("div");
    overlay.id = "setupWizardOverlay";
    overlay.className = "wizard-overlay";

    const renderStep = () => {
      const step = steps[currentStep];
      const isFirst = currentStep === 0;
      const isLast = currentStep === steps.length - 1;
      const progress = ((currentStep + 1) / steps.length) * 100;

      overlay.innerHTML = `
        <div class="wizard-modal" role="dialog" aria-modal="true" aria-labelledby="wizardTitle">
          <div class="wizard-modal__header">
            <div class="wizard-progress">
              <div class="wizard-progress__bar" style="width: ${progress}%"></div>
            </div>
            <div class="wizard-steps-indicator">
              ${steps.map((s, i) => `
                <span class="wizard-step-dot ${i < currentStep ? 'wizard-step-dot--done' : ''} ${i === currentStep ? 'wizard-step-dot--active' : ''}" data-step="${i}">
                  ${i < currentStep ? '<i class="fas fa-check"></i>' : i + 1}
                </span>
              `).join("")}
            </div>
          </div>

          <div class="wizard-modal__body">
            <div class="wizard-modal__image">
              <img src="${WIZARD_IMAGE_URL}" alt="Cyber-Assistant" class="wizard-hero-image" />
              <div class="wizard-modal__image-overlay">
                <i class="fas fa-shield-alt"></i>
                <span>Cyber-Assistant</span>
              </div>
            </div>

            <div class="wizard-modal__content">
              <div class="wizard-content-inner">
                <h2 class="wizard-title" id="wizardTitle">${step.title}</h2>
                <p class="wizard-subtitle">${step.subtitle}</p>
                <div class="wizard-step-content" data-step-id="${step.id}">
                  ${step.content()}
                </div>
              </div>

              <div class="wizard-modal__footer">
                ${!isFirst ? `
                  <button class="btn btn--outline" type="button" data-wizard-action="prev">
                    <i class="fas fa-arrow-left"></i> Précédent
                  </button>
                ` : '<div></div>'}
                <button class="btn btn--primary" type="button" data-wizard-action="${isLast ? 'finish' : 'next'}">
                  ${isLast ? '<i class="fas fa-rocket"></i> Démarrer' : 'Suivant <i class="fas fa-arrow-right"></i>'}
                </button>
              </div>
            </div>
          </div>
        </div>
      `;

      // Bind step enter
      const stepEl = overlay.querySelector(`[data-step-id="${step.id}"]`);
      if (step.onEnter) step.onEnter(stepEl);

      // Bind navigation
      const prevBtn = overlay.querySelector('[data-wizard-action="prev"]');
      const nextBtn = overlay.querySelector('[data-wizard-action="next"]');
      const finishBtn = overlay.querySelector('[data-wizard-action="finish"]');

      if (prevBtn) {
        prevBtn.addEventListener("click", () => {
          if (currentStep > 0) {
            currentStep--;
            renderStep();
          }
        });
      }

      if (nextBtn) {
        nextBtn.addEventListener("click", () => {
          const step = steps[currentStep];
          if (step.canProceed()) {
            const stepEl = overlay.querySelector(`[data-step-id="${step.id}"]`);
            if (step.onLeave) step.onLeave(stepEl);
            currentStep++;
            renderStep();
          }
        });
      }

      if (finishBtn) {
        finishBtn.addEventListener("click", () => {
          const step = steps[currentStep];
          const stepEl = overlay.querySelector(`[data-step-id="${step.id}"]`);
          if (step.onLeave) step.onLeave(stepEl);
          closeWizard();
        });
      }
    };

    const updateWizardTexts = () => {
      // Update texts based on selected language
      const lang = window.CSI_i18n ? window.CSI_i18n.getLanguage() : 'fr';
      if (lang === 'en') {
        if (steps[1]) {
          steps[1].title = "Welcome to Cyber-Assistant";
          steps[1].subtitle = "Your ISMS management tool compliant with ISO 27001:2022 and NIS2";
        }
        if (steps[2]) {
          steps[2].title = "Demo Data";
          steps[2].subtitle = "Would you like to load sample data to explore the application?";
        }
      } else if (lang === 'nl') {
        if (steps[1]) {
          steps[1].title = "Welkom bij Cyber-Assistent";
          steps[1].subtitle = "Uw ISMS-beheertool conform ISO 27001:2022 en NIS2";
        }
        if (steps[2]) {
          steps[2].title = "Demogegevens";
          steps[2].subtitle = "Wilt u voorbeeldgegevens laden om de applicatie te verkennen?";
        }
      }
    };

    const closeWizard = () => {
      overlay.classList.remove("is-visible");
      setTimeout(() => {
        overlay.remove();
        queueEditPasswordSetup();
        if (onComplete) onComplete();
      }, 300);
    };

    document.body.appendChild(overlay);
    renderStep();
    requestAnimationFrame(() => overlay.classList.add("is-visible"));
  }

  // Keep old functions for backwards compatibility but redirect to wizard
  function showLanguageModal(onComplete) {
    showSetupWizard(onComplete);
  }

  function showWelcomeModal() {
    if (document.getElementById("welcomeOverlay") || document.getElementById("setupWizardOverlay")) return;
    try {
      localStorage.setItem(WELCOME_STORAGE_KEY, "1");
    } catch (e) {
      // Ignore storage errors to avoid blocking UI.
    }
    const overlay = document.createElement("div");
    overlay.id = "welcomeOverlay";
    overlay.className = "welcome-overlay";
    overlay.innerHTML = `
      <div class="welcome-modal" role="dialog" aria-modal="true" aria-labelledby="welcomeTitle">
        <button class="welcome-modal__close" type="button" aria-label="Fermer" data-welcome-action="close">
          <i class="fas fa-times"></i>
        </button>
        <div class="welcome-modal__badge">
          <i class="fas fa-star"></i>
          Première ouverture
        </div>
        <h2 class="welcome-modal__title" id="welcomeTitle">Bienvenue sur Cyber-Assistant&nbsp;!</h2>
        <p class="welcome-modal__lead">
          Pour débuter, vous pouvez charger les données de démonstration en cliquant ici.
        </p>
        <div class="welcome-modal__actions">
          <button class="btn btn--primary btn--sm" type="button" data-welcome-action="demo">
            <i class="fas fa-bolt"></i>
            Charger les données de démonstration
          </button>
          <button class="btn btn--outline btn--sm" type="button" data-welcome-action="close">
            Commencer sans démo
          </button>
        </div>
        <div class="welcome-modal__note">
          <p>Pour réinitialiser les données et commencer à travailler, rendez-vous ensuite dans les paramètres.</p>
        </div>
        <div class="welcome-modal__callout">
          <i class="fas fa-rocket"></i>
          <span>C'est le moment de faire tourner votre SMSI&nbsp;! :-)</span>
        </div>
        <p class="welcome-modal__browser-note">Il est recommandé d'utiliser l'application avec Firefox.</p>
      </div>
    `;
    document.body.appendChild(overlay);
    requestAnimationFrame(() => overlay.classList.add("is-visible"));
    const close = () => {
      overlay.classList.remove("is-visible");
      setTimeout(() => overlay.remove(), 200);
      document.removeEventListener("keydown", onKeydown);
      queueEditPasswordSetup();
    };
    const onKeydown = (event) => {
      if (event.key === "Escape") close();
    };
    document.addEventListener("keydown", onKeydown);
    overlay.addEventListener("click", (event) => {
      if (event.target === overlay) close();
    });
    overlay.querySelectorAll("[data-welcome-action=\"close\"]").forEach((btn) => {
      btn.addEventListener("click", close);
    });
    const demoBtn = overlay.querySelector("[data-welcome-action=\"demo\"]");
    if (demoBtn) {
      demoBtn.addEventListener("click", () => {
        close();
        loadDemoData({ allowReadOnly: true });
      });
    }
  }
  function updateKPIs() {
    const soaEntries = Array.isArray(appData.soa) ? appData.soa : [];
    const soaMap = new Map(soaEntries.map((entry) => [entry.controlId, entry]));
    const applicableControls = appData.controls.filter((c) => {
      const entry = soaMap.get(c.id);
      if (!entry) return true;
      return entry.applicable !== false;
    });
    const applicableCount = applicableControls.length || 0;
    const implementedControls = applicableControls.filter((c) => ["implemente", "verifie"].includes(c.status)).length;
    const conformityRate = applicableCount > 0 ? Math.round(implementedControls / applicableCount * 100) : 0;
    const activeActions = appData.actions.filter((a) => !["termine", "annule"].includes(a.status)).length;
    const criticalRisks = appData.risks.filter((r) => r.level === "Critique").length;
    const openNCs = appData.nonconformities.length;

    // KPI principaux
    safeSetText("conformityRate", conformityRate + "%");
    safeSetText("implementedControls", implementedControls + "/" + applicableCount);
    safeSetText("totalActions", activeActions);
    safeSetText("criticalRisks", criticalRisks);
    safeSetText("openNonConformities", openNCs);

    // KPI secondaires
    const projectRisks = appData.projectRisks || [];
    const projectRisksTotal = projectRisks.length;
    const projectRisksCompleted = projectRisks.filter(r => r.status === 'green').length;
    const projectRisksInProgress = projectRisks.filter(r => r.status === 'yellow').length;
    safeSetText("projectRisksCount", projectRisksTotal);
    const detailEl = document.getElementById("projectRisksDetail");
    if (detailEl && projectRisksTotal > 0) {
      detailEl.textContent = `${projectRisksCompleted} terminées · ${projectRisksInProgress} en cours`;
    }

    const criticalAssets = appData.criticalAssets || [];
    safeSetText("criticalAssetsCount", criticalAssets.length);

    const threats = appData.threats || [];
    safeSetText("threatsCount", threats.length);

    const audits = appData.audits || [];
    safeSetText("auditsCount", audits.length);

    const documents = appData.documents || [];
    safeSetText("documentsCount", documents.length);

    // Couverture SoA
    const soaApplicable = soaEntries.filter(e => e.applicable !== false).length;
    const soaCoverage = applicableCount > 0 ? Math.round(soaApplicable / applicableCount * 100) : 0;
    safeSetText("soaCoverage", soaCoverage + "%");

    // Mettre à jour le Score Global et les alertes
    updateScoreGlobal(conformityRate);
    updateDashboardAlerts();
    updateSparklines();
    updateActivityTimeline();
  }

  // Score Global SMSI
  function updateScoreGlobal(isoConformity) {
    // Calcul des scores composants
    const soaEntries = Array.isArray(appData.soa) ? appData.soa : [];
    const soaMap = new Map(soaEntries.map((entry) => [entry.controlId, entry]));
    const applicableControls = appData.controls.filter((c) => {
      const entry = soaMap.get(c.id);
      if (!entry) return true;
      return entry.applicable !== false;
    });

    // Score ISO 27001
    const isoScore = isoConformity;
    safeSetText("scoreIso27001", isoScore + "%");

    // Score NIS2 (moyenne des maturités NIS2)
    const nis2Mat = appData.targetMaturity?.nis2 || {};
    const nis2Values = Object.values(nis2Mat).filter(v => typeof v === 'number');
    const nis2Avg = nis2Values.length > 0 ? nis2Values.reduce((a, b) => a + b, 0) / nis2Values.length : 0;
    const nis2Score = Math.round((nis2Avg / 5) * 100);
    safeSetText("scoreNis2", nis2Score + "%");

    // Score Risques maîtrisés
    const totalRisks = appData.risks.length;
    const criticalGraveRisks = appData.risks.filter(r => r.level === "Critique" || r.level === "Grave").length;
    const risksControlled = totalRisks > 0 ? Math.round(((totalRisks - criticalGraveRisks) / totalRisks) * 100) : 100;
    safeSetText("scoreRisks", risksControlled + "%");

    // Score global pondéré
    const globalScore = Math.round(isoScore * 0.4 + nis2Score * 0.3 + risksControlled * 0.3);

    // Lancer l'animation au chargement
    animateScoreGauge(globalScore);
  }

  function animateScoreGauge(globalScore) {
    const arc = document.getElementById("scoreGlobalArc");
    const valueEl = document.getElementById("scoreGlobalValue");
    const gaugeContainer = document.querySelector(".score-global__gauge");

    if (!arc || !valueEl) return;

    const circumference = 2 * Math.PI * 54; // r=54

    // Couleur dynamique selon le score
    let strokeColor = '#ef4444'; // Rouge < 40
    if (globalScore >= 70) strokeColor = '#22c55e'; // Vert
    else if (globalScore >= 40) strokeColor = '#f59e0b'; // Orange

    // Reset pour l'animation
    arc.style.transition = 'none';
    arc.style.strokeDashoffset = circumference;
    arc.style.stroke = strokeColor;

    // Forcer le reflow
    arc.getBoundingClientRect();

    // Animation fluide de la jauge (1.5s)
    arc.style.transition = 'stroke-dashoffset 1.5s cubic-bezier(0.4, 0, 0.2, 1)';
    const targetOffset = circumference - (globalScore / 100) * circumference;
    arc.style.strokeDashoffset = targetOffset;

    // Animation du compteur
    animateValue(valueEl, 0, globalScore, 1500);

    // Effet pulse sur le conteneur
    if (gaugeContainer) {
      gaugeContainer.classList.remove('gauge-pulse');
      void gaugeContainer.offsetWidth; // Force reflow
      gaugeContainer.classList.add('gauge-pulse');
    }
  }

  function animateValue(element, start, end, duration) {
    const startTime = performance.now();
    const update = (currentTime) => {
      const elapsed = currentTime - startTime;
      const progress = Math.min(elapsed / duration, 1);
      const easeOut = 1 - Math.pow(1 - progress, 3);
      const current = Math.round(start + (end - start) * easeOut);
      element.textContent = current;
      if (progress < 1) requestAnimationFrame(update);
    };
    requestAnimationFrame(update);
  }

  // Alertes dynamiques
  function updateDashboardAlerts() {
    const alertsList = document.getElementById("alertsList");
    if (!alertsList) return;

    const alerts = [];

    // Risques critiques
    const criticalRisks = appData.risks.filter(r => r.level === "Critique");
    if (criticalRisks.length > 0) {
      alerts.push({
        icon: 'critical',
        iconClass: 'fas fa-skull-crossbones',
        text: `${criticalRisks.length} risque(s) critique(s) à traiter`,
        badge: 'Urgent',
        target: 'risks'
      });
    }

    // Actions en retard
    const today = new Date().toISOString().split('T')[0];
    const overdueActions = appData.actions.filter(a => {
      if (['termine', 'annule'].includes(a.status)) return false;
      return a.dueDate && a.dueDate < today;
    });
    if (overdueActions.length > 0) {
      alerts.push({
        icon: 'warning',
        iconClass: 'fas fa-clock',
        text: `${overdueActions.length} action(s) en retard`,
        badge: 'Retard',
        target: 'actions'
      });
    }

    // Audits à venir (30 jours)
    const in30Days = new Date();
    in30Days.setDate(in30Days.getDate() + 30);
    const upcomingAudits = (appData.audits || []).filter(a => {
      if (!a.date) return false;
      const auditDate = new Date(a.date);
      return auditDate >= new Date() && auditDate <= in30Days;
    });
    if (upcomingAudits.length > 0) {
      alerts.push({
        icon: 'info',
        iconClass: 'fas fa-calendar-check',
        text: `${upcomingAudits.length} audit(s) dans les 30 jours`,
        badge: 'Planifié',
        target: 'audits'
      });
    }

    // Projets à risque (status red)
    const projectsAtRisk = (appData.projectRisks || []).filter(p => p.status === 'red');
    if (projectsAtRisk.length > 0) {
      alerts.push({
        icon: 'purple',
        iconClass: 'fas fa-diagram-project',
        text: `${projectsAtRisk.length} projet(s) à risque élevé`,
        badge: 'Projets',
        target: 'project-risks'
      });
    }

    // Rendu des alertes
    if (alerts.length === 0) {
      alertsList.innerHTML = '<div class="dashboard-alerts__empty"><i class="fas fa-check-circle"></i>&nbsp;Aucune alerte en cours</div>';
    } else {
      alertsList.innerHTML = alerts.map(a => `
        <div class="alert-item" data-nav-target="${a.target}" role="button" tabindex="0">
          <div class="alert-item__icon alert-item__icon--${a.icon}">
            <i class="${a.iconClass}"></i>
          </div>
          <span class="alert-item__text">${a.text}</span>
          <span class="alert-item__badge">${a.badge}</span>
        </div>
      `).join('');

      // Event listeners pour navigation
      alertsList.querySelectorAll('.alert-item[data-nav-target]').forEach(item => {
        item.addEventListener('click', () => {
          const target = item.getAttribute('data-nav-target');
          // Utiliser la même logique que setupKpiCardNavigation
          if (typeof switchModule === "function") {
            switchModule(target);
          } else {
            const navBtn = document.querySelector(`.nav__item[data-tab="${target}"]`);
            if (navBtn) navBtn.click();
            else if (typeof loadModuleContent === "function") loadModuleContent(target);
          }
        });
        item.addEventListener('keypress', (e) => {
          if (e.key === "Enter" || e.key === " ") {
            e.preventDefault();
            item.click();
          }
        });
      });
    }
  }

  // Sparklines mini-graphiques
  function updateSparklines() {
    // Historique des métriques (stocké dans appData)
    if (!appData.kpiHistory) {
      appData.kpiHistory = {
        conformity: [],
        controls: [],
        actions: [],
        risks: [],
        nc: []
      };
    }

    // Récupérer les valeurs actuelles
    const soaEntries = Array.isArray(appData.soa) ? appData.soa : [];
    const soaMap = new Map(soaEntries.map((entry) => [entry.controlId, entry]));
    const applicableControls = appData.controls.filter((c) => {
      const entry = soaMap.get(c.id);
      if (!entry) return true;
      return entry.applicable !== false;
    });
    const implementedCount = applicableControls.filter((c) => ["implemente", "verifie"].includes(c.status)).length;
    const conformityRate = applicableControls.length > 0 ? Math.round(implementedCount / applicableControls.length * 100) : 0;
    const activeActions = appData.actions.filter((a) => !["termine", "annule"].includes(a.status)).length;
    const criticalRisks = appData.risks.filter((r) => r.level === "Critique").length;
    const openNCs = appData.nonconformities.length;

    // Ajouter au historique (max 6 points)
    const addToHistory = (arr, val) => {
      arr.push(val);
      if (arr.length > 6) arr.shift();
    };

    // Ne mettre à jour l'historique qu'une fois par session (éviter les doublons)
    const lastUpdate = appData.kpiHistory._lastUpdate || 0;
    const now = Date.now();
    if (now - lastUpdate > 60000) { // 1 minute minimum entre updates
      addToHistory(appData.kpiHistory.conformity, conformityRate);
      addToHistory(appData.kpiHistory.controls, implementedCount);
      addToHistory(appData.kpiHistory.actions, activeActions);
      addToHistory(appData.kpiHistory.risks, criticalRisks);
      addToHistory(appData.kpiHistory.nc, openNCs);
      appData.kpiHistory._lastUpdate = now;
    }

    // Dessiner les sparklines
    drawSparkline('sparkConformity', appData.kpiHistory.conformity, '#0f9b7a');
    drawSparkline('sparkControls', appData.kpiHistory.controls, '#22c55e');
    drawSparkline('sparkActions', appData.kpiHistory.actions, '#f59e0b');
    drawSparkline('sparkRisks', appData.kpiHistory.risks, '#ef4444', true);
    drawSparkline('sparkNC', appData.kpiHistory.nc, '#2563eb', true);
  }

  function drawSparkline(canvasId, data, color, invertTrend = false) {
    const canvas = document.getElementById(canvasId);
    if (!canvas || data.length < 2) return;

    const ctx = canvas.getContext('2d');
    const width = canvas.width;
    const height = canvas.height;
    const padding = 2;

    ctx.clearRect(0, 0, width, height);

    const min = Math.min(...data);
    const max = Math.max(...data);
    const range = max - min || 1;

    const points = data.map((val, i) => ({
      x: padding + (i / (data.length - 1)) * (width - padding * 2),
      y: height - padding - ((val - min) / range) * (height - padding * 2)
    }));

    // Ligne
    ctx.beginPath();
    ctx.strokeStyle = color;
    ctx.lineWidth = 1.5;
    ctx.lineCap = 'round';
    ctx.lineJoin = 'round';

    points.forEach((p, i) => {
      if (i === 0) ctx.moveTo(p.x, p.y);
      else ctx.lineTo(p.x, p.y);
    });
    ctx.stroke();

    // Point final
    const last = points[points.length - 1];
    ctx.beginPath();
    ctx.fillStyle = color;
    ctx.arc(last.x, last.y, 2.5, 0, Math.PI * 2);
    ctx.fill();
  }
  function updateControlsLabel() {
    const navLabel = document.getElementById("controlsNavLabel");
    if (navLabel) navLabel.textContent = "Contrôles ISO27002";
  }
  function updateCharts() {
    if (typeof Chart === "undefined") {
      console.error("Chart.js non disponible");
      showToast("Erreur: Biblioth\xE8que de graphiques non disponible", "error");
      return;
    }
    updateCapabilityRadarChart();
    charts.nis2Radar = updateNis2RadarChart();
    updateActionsChart();
    updateRisksChart();
    updateCmmChart();
    updateRiskMatrix();
    // Nouveaux graphiques compacts
    updateActionsStatusChart();
    updateControlsDomainChart();
    updateProjectRisksChart();
  }

  // Graphique Statut des Actions (Doughnut)
  function updateActionsStatusChart() {
    const ctx = document.getElementById("actionsStatusChart");
    if (!ctx) return;
    if (charts.actionsStatus) charts.actionsStatus.destroy();
    const themeVars = getThemeVars();

    const statusCounts = {
      'a_faire': 0,
      'en_cours': 0,
      'termine': 0,
      'annule': 0
    };

    appData.actions.forEach(a => {
      if (statusCounts.hasOwnProperty(a.status)) {
        statusCounts[a.status]++;
      }
    });

    const data = [statusCounts.a_faire, statusCounts.en_cours, statusCounts.termine, statusCounts.annule];
    const total = data.reduce((a, b) => a + b, 0);

    if (total === 0) {
      ctx.parentElement.innerHTML = '<div style="display:flex;align-items:center;justify-content:center;height:100%;color:var(--color-text-light);font-size:0.875rem;">Aucune action</div>';
      return;
    }

    const colors = [
      withAlpha(themeVars.warning, 0.8),
      withAlpha(themeVars.info, 0.8),
      withAlpha(themeVars.success, 0.8),
      withAlpha('#9ca3af', 0.6)
    ];
    const labels = ["À faire", "En cours", "Terminées", "Annulées"];

    charts.actionsStatus = new Chart(ctx, {
      type: "doughnut",
      data: {
        labels: labels,
        datasets: [{
          data: data,
          backgroundColor: colors,
          borderWidth: 0
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        cutout: '60%',
        plugins: {
          legend: { display: false },
          tooltip: {
            callbacks: {
              label: (ctx) => `${ctx.label}: ${ctx.raw} (${Math.round(ctx.raw / total * 100)}%)`
            }
          }
        }
      }
    });

    // Mini-légende
    renderMiniLegend('actionsStatusLegend', labels, colors, data);
  }

  // Graphique Contrôles par Domaine (Bar horizontal)
  function updateControlsDomainChart() {
    const ctx = document.getElementById("controlsDomainChart");
    if (!ctx) return;
    if (charts.controlsDomain) charts.controlsDomain.destroy();
    const themeVars = getThemeVars();

    const domains = ['Organisationnel', 'Humain', 'Physique', 'Technologique'];
    const domainData = domains.map(domain => {
      const controls = appData.controls.filter(c => c.category === domain);
      const implemented = controls.filter(c => ['implemente', 'verifie'].includes(c.status)).length;
      return {
        domain: domain.substring(0, 4) + '.',
        total: controls.length,
        implemented: implemented
      };
    });

    charts.controlsDomain = new Chart(ctx, {
      type: "bar",
      data: {
        labels: domainData.map(d => d.domain),
        datasets: [
          {
            label: 'Implémentés',
            data: domainData.map(d => d.implemented),
            backgroundColor: withAlpha(themeVars.success, 0.8),
            borderRadius: 4
          },
          {
            label: 'Total',
            data: domainData.map(d => d.total - d.implemented),
            backgroundColor: withAlpha(themeVars.secondary, 0.3),
            borderRadius: 4
          }
        ]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        indexAxis: 'y',
        scales: {
          x: {
            stacked: true,
            display: false
          },
          y: {
            stacked: true,
            grid: { display: false },
            ticks: { color: themeVars.textColor, font: { size: 10 } }
          }
        },
        plugins: {
          legend: { display: false },
          tooltip: {
            callbacks: {
              title: (items) => domains[items[0].dataIndex],
              label: (ctx) => ctx.datasetIndex === 0
                ? `Implémentés: ${ctx.raw}`
                : `Restants: ${ctx.raw}`
            }
          }
        }
      }
    });

    // Mini-légende
    const totalImpl = domainData.reduce((a, d) => a + d.implemented, 0);
    const totalAll = domainData.reduce((a, d) => a + d.total, 0);
    renderMiniLegend('controlsDomainLegend',
      ['Implémentés', 'Restants'],
      [withAlpha(themeVars.success, 0.8), withAlpha(themeVars.secondary, 0.3)],
      [totalImpl, totalAll - totalImpl]
    );
  }

  // Graphique Projets Risques (Doughnut avec statuts)
  function updateProjectRisksChart() {
    const ctx = document.getElementById("projectRisksChart");
    if (!ctx) return;
    if (charts.projectRisks) charts.projectRisks.destroy();
    const themeVars = getThemeVars();

    const projectRisks = appData.projectRisks || [];
    const statusCounts = {
      'green': 0,
      'yellow': 0,
      'red': 0
    };

    projectRisks.forEach(p => {
      if (statusCounts.hasOwnProperty(p.status)) {
        statusCounts[p.status]++;
      }
    });

    const data = [statusCounts.green, statusCounts.yellow, statusCounts.red];
    const labels = ["Validé", "En cours", "À risque"];
    const colors = [
      withAlpha(themeVars.success, 0.85),
      withAlpha(themeVars.warning, 0.85),
      withAlpha(themeVars.danger, 0.85)
    ];
    const total = data.reduce((a, b) => a + b, 0);

    if (total === 0) {
      ctx.parentElement.innerHTML = '<div style="display:flex;align-items:center;justify-content:center;height:100%;color:var(--color-text-light);font-size:0.875rem;">Aucun projet</div>';
      const legend = document.getElementById('projectRisksLegend');
      if (legend) legend.innerHTML = '';
      return;
    }

    charts.projectRisks = new Chart(ctx, {
      type: "doughnut",
      data: {
        labels: labels,
        datasets: [{
          data: data,
          backgroundColor: colors,
          borderWidth: 0
        }]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        cutout: '55%',
        plugins: {
          legend: { display: false },
          tooltip: {
            callbacks: {
              label: (ctx) => `${ctx.label}: ${ctx.raw} projet(s)`
            }
          }
        }
      }
    });

    // Mini-légende
    renderMiniLegend('projectRisksLegend', labels, colors, data);
  }

  // Fonction helper pour générer les mini-légendes
  function renderMiniLegend(containerId, labels, colors, values) {
    const container = document.getElementById(containerId);
    if (!container) return;

    container.innerHTML = labels.map((label, i) => {
      if (values[i] === 0) return '';
      return `
        <span class="chart-mini-legend__item">
          <span class="chart-mini-legend__dot" style="background:${colors[i]}"></span>
          <span>${label}</span>
          <span class="chart-mini-legend__value">${values[i]}</span>
        </span>
      `;
    }).join('');
  }

  // Timeline Activité Récente
  function updateActivityTimeline() {
    const container = document.getElementById('activityTimeline');
    if (!container) return;

    // Collecter toutes les activités avec timestamp
    const activities = [];

    // Ajouter les actions récentes
    (appData.actions || []).forEach(a => {
      if (a.createdAt || a.updatedAt) {
        activities.push({
          type: 'action',
          icon: 'fa-tasks',
          iconClass: 'action',
          title: a.title || 'Action',
          meta: getStatusLabel(a.status),
          time: a.updatedAt || a.createdAt
        });
      }
    });

    // Ajouter les risques
    (appData.risks || []).forEach(r => {
      if (r.createdAt || r.updatedAt) {
        activities.push({
          type: 'risk',
          icon: 'fa-exclamation-triangle',
          iconClass: 'risk',
          title: r.title || 'Risque',
          meta: r.level || '',
          time: r.updatedAt || r.createdAt
        });
      }
    });

    // Ajouter les audits
    (appData.audits || []).forEach(a => {
      if (a.createdAt || a.date) {
        activities.push({
          type: 'audit',
          icon: 'fa-clipboard-check',
          iconClass: 'audit',
          title: a.title || 'Audit',
          meta: a.scope || '',
          time: a.createdAt || a.date
        });
      }
    });

    // Ajouter les NC
    (appData.nonconformities || []).forEach(nc => {
      if (nc.createdAt || nc.date) {
        activities.push({
          type: 'nc',
          icon: 'fa-exclamation-circle',
          iconClass: 'nc',
          title: nc.title || 'Non-conformité',
          meta: nc.source || '',
          time: nc.createdAt || nc.date
        });
      }
    });

    // Trier par date décroissante
    activities.sort((a, b) => new Date(b.time) - new Date(a.time));

    // Prendre les 8 plus récentes
    const recent = activities.slice(0, 8);

    if (recent.length === 0) {
      container.innerHTML = `
        <div class="activity-empty">
          <i class="fas fa-inbox"></i>
          <span>Aucune activité récente</span>
        </div>
      `;
      return;
    }

    container.innerHTML = recent.map(a => `
      <div class="activity-item">
        <div class="activity-item__icon activity-item__icon--${a.iconClass}">
          <i class="fas ${a.icon}"></i>
        </div>
        <div class="activity-item__content">
          <div class="activity-item__title">${escapeHtml(a.title)}</div>
          <div class="activity-item__meta">${escapeHtml(a.meta)}</div>
        </div>
        <div class="activity-item__time">${formatRelativeTime(a.time)}</div>
      </div>
    `).join('');
  }

  function getStatusLabel(status) {
    const labels = {
      'a_faire': 'À faire',
      'en_cours': 'En cours',
      'termine': 'Terminé',
      'annule': 'Annulé'
    };
    return labels[status] || status || '';
  }

  function formatRelativeTime(dateStr) {
    if (!dateStr) return '';
    const date = new Date(dateStr);
    const now = new Date();
    const diffMs = now - date;
    const diffMins = Math.floor(diffMs / 60000);
    const diffHours = Math.floor(diffMs / 3600000);
    const diffDays = Math.floor(diffMs / 86400000);

    if (diffMins < 1) return "À l'instant";
    if (diffMins < 60) return `${diffMins}min`;
    if (diffHours < 24) return `${diffHours}h`;
    if (diffDays < 7) return `${diffDays}j`;
    if (diffDays < 30) return `${Math.floor(diffDays / 7)}sem`;
    return date.toLocaleDateString('fr-FR', { day: 'numeric', month: 'short' });
  }

  function updateConformityChart() {
    const ctx = document.getElementById("conformityChart");
    if (!ctx) return;
    if (charts.conformity) charts.conformity.destroy();
    const themeVars = getThemeVars();
    const categories = [...new Set(appData.controls.map((c) => c.category))];
    const data = categories.map((category) => {
      const filtered = appData.controls.filter((c) => c.category === category);
      const implemented = filtered.filter((c) => ["implemente", "verifie"].includes(c.status)).length;
      return filtered.length > 0 ? Math.round(implemented / filtered.length * 100) : 0;
    });
    const counts = categories.map((cat) => appData.controls.filter((c) => c.category === cat).length);
    const labels = categories.map((cat, i) => `${cat} (${counts[i]})`);
    const palette = [themeVars.primary, themeVars.secondary, themeVars.warning, themeVars.success, themeVars.info];
    const colors = categories.map((_, i) => withAlpha(palette[i % palette.length], 0.75));
    charts.conformity = new Chart(ctx, {
      type: "doughnut",
      data: { labels, datasets: [{ data, backgroundColor: colors, borderWidth: 1, borderColor: themeVars.surface }] },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
          legend: {
            position: "bottom",
            labels: { font: { size: 12 }, padding: 20 }
          },
          tooltip: {
            callbacks: {
              label: (context) => `${context.label}: ${context.raw}% de conformit\xE9`
            }
          }
        },
        cutout: "60%"
      }
    });
  }
  function updateActionsChart() {
    const ctx = document.getElementById("actionsChart");
    if (!ctx) return;
    if (charts.actions) charts.actions.destroy();
    const themeVars = getThemeVars();
    const gridColor = withAlpha(themeVars.borderColor, 0.6);
    const priorities = ["critical", "high", "medium", "low"];
    const priorityLabels = ["Critique", "Haute", "Moyenne", "Faible"];
    const currentData = priorities.map((p) => appData.actions.filter((a) => a.priority === p).length);
    const months = ["Janvier", "F\xE9vrier", "Mars", "Avril", "Mai", "Juin"];
    if (!appData.actionsHistory) appData.actionsHistory = {};
    priorities.forEach((priority, index) => {
      if (!Array.isArray(appData.actionsHistory[priority])) appData.actionsHistory[priority] = [];
      appData.actionsHistory[priority].push(currentData[index]);
      if (appData.actionsHistory[priority].length > months.length) {
        appData.actionsHistory[priority].shift();
      }
    });
    saveData();
    const datasets = priorities.map((priority, index) => {
      const history = appData.actionsHistory[priority] || [];
      const colors = {
        critical: withAlpha(themeVars.danger, 0.75),
        high: withAlpha(themeVars.warning, 0.75),
        medium: withAlpha(themeVars.secondary, 0.75),
        low: withAlpha(themeVars.success, 0.75)
      };
      return {
        label: priorityLabels[index],
        data: history.slice(-months.length),
        backgroundColor: colors[priority],
        borderColor: withAlpha(colors[priority], 1),
        borderWidth: 2,
        tension: 0.3
      };
    });
    charts.actions = new Chart(ctx, {
      type: "line",
      data: { labels: months, datasets },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
          legend: { position: "bottom" },
          tooltip: { mode: "index", intersect: false }
        },
        scales: {
          y: {
            beginAtZero: true,
            ticks: { precision: 0, color: themeVars.textColor },
            grid: { color: gridColor },
            title: { display: true, text: "Nombre d'actions", color: themeVars.textColor }
          },
          x: {
            ticks: { color: themeVars.textColor },
            grid: { color: gridColor },
            title: { display: true, text: "Mois", color: themeVars.textColor }
          }
        }
      }
    });
  }
  function updateRisksChart() {
    const ctx = document.getElementById("riskChart");
    if (!ctx) return;
    if (charts.risks) charts.risks.destroy();
    const themeVars = getThemeVars();
    const levels = ["Critique", "Grave", "Significatif", "Mineur", "Insignifiant"];
    const data = levels.map((level) => appData.risks.filter((r) => r.level === level).length);
    const colors = [
      withAlpha(themeVars.danger, 0.75),
      withAlpha(themeVars.warning, 0.75),
      withAlpha(themeVars.primary, 0.7),
      withAlpha(themeVars.success, 0.7),
      withAlpha(themeVars.info, 0.7)
    ];
    charts.risks = new Chart(ctx, {
      type: "pie",
      data: { labels: levels, datasets: [{ data, backgroundColor: colors, borderWidth: 1, borderColor: themeVars.surface }] },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
          legend: { position: "bottom" },
          tooltip: {
            callbacks: {
              label: (context) => `${context.label}: ${context.raw} risque(s)`
            }
          }
        }
      }
    });
  }

  function updateCmmChart() {
    const ctx = document.getElementById("cmmChart");
    if (!ctx) return;
    if (charts.cmm) charts.cmm.destroy();
    const themeVars = getThemeVars();
    const levels = ["Initial", "R\xE9p\xE9table", "D\xE9fini", "G\xE9r\xE9", "Optimis\xE9"];
    const data = levels.map((lvl) => appData.controls.filter((c) => c.cmm === lvl).length);
    const colors = [
      withAlpha(themeVars.primary, 0.7),
      withAlpha(themeVars.success, 0.7),
      withAlpha(themeVars.warning, 0.7),
      withAlpha(themeVars.secondary, 0.7),
      withAlpha(themeVars.danger, 0.7)
    ];
    charts.cmm = new Chart(ctx, {
      type: "pie",
      data: { labels: levels, datasets: [{ data, backgroundColor: colors, borderWidth: 1, borderColor: themeVars.surface }] },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: { legend: { position: "bottom" } }
      }
    });
  }

  function updateCapabilityRadarChart() {
    const ctx = document.getElementById("capabilityRadarChart");
    if (!ctx) return;
    if (charts.capabilityRadar) charts.capabilityRadar.destroy();
    const themeVars = typeof getThemeVars === "function" ? getThemeVars() : (() => {
      const target = document.body || document.documentElement;
      const styles = getComputedStyle(target);
      return {
        textColor: styles.getPropertyValue("--color-text").trim() || "#2d3e4a",
        textSecondary: styles.getPropertyValue("--color-text-secondary").trim() || "#758592",
        borderColor: styles.getPropertyValue("--color-border").trim() || "#e9ecef",
        primary: styles.getPropertyValue("--color-primary").trim() || "#1fb8cd",
        warning: styles.getPropertyValue("--color-warning").trim() || "#ffc185"
      };
    })();
    const { textColor, textSecondary, borderColor, primary, warning } = themeVars;
    const gridColor = withAlpha(borderColor, 0.7);
    const capabilities = OPERATIONAL_CAPABILITIES;
    const cmmValues = {
      Initial: 1,
      "R\xE9p\xE9table": 2,
      "D\xE9fini": 3,
      "G\xE9r\xE9": 4,
      "Optimis\xE9": 5
    };
    const validControls = appData.controls.filter(c =>
      Array.isArray(c.subcategory) ? c.subcategory.length > 0 : c.subcategory
    );
    const missingCount = appData.controls.length - validControls.length;
    if (missingCount > 0) {
      console.warn(`Contrôles sans sous-catégorie ignorés: ${missingCount}`);
    }
    // Légende détaillée (numérotation assurée par <ol>)
    const renderLegend = typeof renderRadarLegend === "function" ? renderRadarLegend : () => {};
    renderLegend("capabilityRadarLegend", capabilities.map((cap) => cap.label));
    const data = capabilities.map((cap) => {
      const controls = validControls.filter((c) =>
        Array.isArray(c.subcategory) ? c.subcategory.includes(cap.key) : c.subcategory === cap.key
      );
      if (controls.length === 0) return 0;
      const sum = controls.reduce(
        (acc, c) => acc + (cmmValues[c.cmm] || 0),
        0
      );
      return +(sum / controls.length).toFixed(2);
    });
    const targetValue = Number(appData.targetMaturity?.iso27002) || 0;
    const targetData = capabilities.map(() => targetValue);
    charts.capabilityRadar = new Chart(ctx, {
      type: "radar",
      data: {
        labels: capabilities.map((cap) => cap.label),
        datasets: [
          {
            label: "Cible",
            data: targetData,
            backgroundColor: withAlpha(warning, 0.08),
            borderColor: withAlpha(warning, 0.55),
            borderWidth: 3,
            pointRadius: 2,
            borderDash: [4, 4],
            fill: true
          },
          {
            label: "Moyenne CMM",
            data,
            backgroundColor: withAlpha(primary, 0.2),
            borderColor: withAlpha(primary, 1),
            borderWidth: 3,
            pointBackgroundColor: withAlpha(primary, 1),
            pointRadius: 3,
            pointHoverRadius: 5
          }
        ]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        scales: {
          r: {
            beginAtZero: true,
            suggestedMin: 0,
            suggestedMax: 5,
            ticks: {
              stepSize: 1,
              color: textColor,
              backdropColor: "transparent",
              font: { size: 12 }
            },
            grid: { color: gridColor },
            angleLines: { color: gridColor },
            pointLabels: {
              display: true,
              color: textSecondary || borderColor,
              font: { size: 11, weight: "400" },
              callback: (_label, index) => `${index + 1}`
            }
          }
        },
        plugins: { legend: { display: false } }
      }
    });
  }

  function updateNis2RadarChart() {
    const ctx = document.getElementById("nis2RadarChart");
    if (!ctx) return null;
    if (charts.nis2Radar) charts.nis2Radar.destroy();
    const themeVars = typeof getThemeVars === "function" ? getThemeVars() : (() => {
      const target = document.body || document.documentElement;
      const styles = getComputedStyle(target);
      return {
        textColor: styles.getPropertyValue("--color-text").trim() || "#2d3e4a",
        textSecondary: styles.getPropertyValue("--color-text-secondary").trim() || "#758592",
        borderColor: styles.getPropertyValue("--color-border").trim() || "#e9ecef",
        secondary: styles.getPropertyValue("--color-secondary").trim() || "#5d878f",
        warning: styles.getPropertyValue("--color-warning").trim() || "#ffc185"
      };
    })();
    const { textColor, textSecondary, borderColor, secondary, warning } = themeVars;
    const gridColor = withAlpha(borderColor, 0.7);
    const averages = appData.targetMaturity?.nis2 || {};
    const labelsFull = NIS2_CATEGORIES.map((k) => {
      const full = CATEGORY_LABELS[k] || k;
      return full.split(" (")[0].trim();
    });
    const renderLegend = typeof renderRadarLegend === "function" ? renderRadarLegend : () => {};
    renderLegend("nis2RadarLegend", labelsFull);
    const categoryData = NIS2_CATEGORIES.map((k) => Number(averages[k]) || 0);
    const globalTarget = Number(appData.targetMaturity?.nis2Target) || 0;
    const globalData = NIS2_CATEGORIES.map(() => globalTarget);
    return new Chart(ctx, {
      type: "radar",
      data: {
        labels: labelsFull,
        datasets: [
          {
            label: "Cible globale",
            data: globalData,
            backgroundColor: withAlpha(warning, 0.08),
            borderColor: withAlpha(warning, 0.55),
            borderWidth: 3,
            pointRadius: 2,
            borderDash: [4, 4],
            fill: true
          },
          {
            label: "Maturit\xE9 moyenne",
            data: categoryData,
            backgroundColor: withAlpha(secondary, 0.2),
            borderColor: withAlpha(secondary, 1),
            borderWidth: 3,
            pointBackgroundColor: withAlpha(secondary, 1),
            pointRadius: 3,
            pointHoverRadius: 5,
            fill: true
          }
        ]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        scales: {
          r: {
            beginAtZero: true,
            suggestedMin: 0,
            suggestedMax: 5,
            ticks: {
              stepSize: 1,
              color: textColor,
              backdropColor: "transparent",
              font: { size: 12 }
            },
            grid: { color: gridColor },
            angleLines: { color: gridColor },
            pointLabels: {
              display: true,
              color: textSecondary || borderColor,
              font: { size: 14, weight: "400" },
              callback: (_label, index) => `${index + 1}`
            }
          }
        },
        plugins: { legend: { display: false } }
      }
    });
    renderLegend("nis2RadarLegend", labelsFull);
  }
  function updateRiskMatrix() {
    const matrix = document.getElementById("matrixGrid");
    if (!matrix) return;
    matrix.innerHTML = "";
    for (let impact = 5; impact >= 1; impact--) {
      for (let prob = 1; prob <= 5; prob++) {
        const cell = document.createElement("div");
        cell.className = "matrix-cell";
        const score = impact * prob;
        let level = "";
        if (score >= 20) level = "critique";
        else if (score >= 15) level = "grave";
        else if (score >= 10) level = "significatif";
        else if (score >= 5) level = "mineur";
        else level = "insignifiant";
        cell.classList.add(`matrix-cell--${level}`);
        const label = document.createElement("div");
        label.className = "matrix-cell__label";
        label.textContent = score;
        cell.title = `${IMPACT_LEVEL_LABELS[impact]} / ${PROBABILITY_LEVEL_LABELS[prob]}`;
        cell.appendChild(label);
        const count = appData.risks.filter((r) => r.impact === impact && r.probability === prob).length;
        if (count > 0) {
          const countElem = document.createElement("div");
          countElem.className = "matrix-cell__count";
          countElem.textContent = count;
          cell.appendChild(countElem);
        }
        matrix.appendChild(cell);
      }
    }
  }
  window.updateDashboard = updateDashboard;
  window.updateRiskMatrix = updateRiskMatrix;

  // modules/search.js
  function switchModule(moduleId) {
    const navBtn = document.querySelector(`.nav__item[data-tab="${moduleId}"]`);
    if (navBtn) {
      navBtn.click();
    } else {
      loadModuleContent(moduleId);
    }
  }
  function getSearchIndex() {
    const entries = [];
    const push = (type, label, module, action) => {
      entries.push({ type, label: label || "", module, action });
    };
    (appData.actions || []).forEach(a => push("Action", a.title, "actions", () => editAction(a.id)));
    (appData.risks || []).forEach(r => push("Risque", r.title, "risks", () => editRisk(r.id)));
    (appData.threats || []).forEach(t => push("Menace", t.title, "threats-section", () => editThreat(t.id)));
    (appData.nonconformities || []).forEach(nc => push("NC", nc.title, "nonconformities", () => editNC(nc.id)));
    (appData.audits || []).forEach(a => push("Audit", a.title, "audits", () => editAudit(a.id)));
    (appData.reviews || []).forEach(r => push("Revue", r.inputs || r.decisions || r.participants, "managementReviews", () => editReview(r.id)));
    (appData.stakeholders || []).forEach(s => push("Partie prenante", s.name, "stakeholders", () => editStakeholder(s.id)));
    (appData.documents || []).forEach(d => push("Document", d.type, "documents", () => editDocument(d.id)));
    (appData.controls || []).forEach(c => push("Contrôle", c.title, "controls", () => editControl(c.id)));
    (appData.nis2Controls || []).forEach(c => push("Contrôle NIS2", c.description || c.categorie, "nis2-controls", () => editNis2Control ? editNis2Control(c.id) : null));
    (appData.kpis || []).forEach(k => push("KPI", k.title, "kpis", () => editKpi(k.id)));
    (appData.criticalAssets || []).forEach(a => push("Actif critique", getCriticalAssetLabel(a), "critical-assets", () => openCriticalAssetModal(a.id)));
    (appData.projectRisks || []).forEach(r => push("Risque projet", r.projectName || r.reportNumber, "project-risks", () => openProjectRiskModal(r.id)));
    return entries;
  }
  function renderSearchResults(results) {
    const container = document.getElementById("globalSearchResults");
    if (!container) return;
    container.innerHTML = "";
    if (!results.length) {
      container.textContent = "Aucun résultat";
      return;
    }
    results.slice(0, 20).forEach((res) => {
      const btn = document.createElement("button");
      btn.type = "button";
      btn.className = "global-search__item";
      btn.innerHTML = `
        <span class="global-search__type">${escapeHtml(res.type)}</span>
        <span class="global-search__label">${escapeHtml(res.label)}</span>
        <span class="global-search__module">${escapeHtml(res.module)}</span>
      `;
      btn.addEventListener("click", () => {
        switchModule(res.module);
        setTimeout(() => {
          if (typeof res.action === "function") res.action();
        }, 200);
      });
      container.appendChild(btn);
    });
  }
  function performGlobalSearch(query) {
    const q = NORMALIZE(query);
    if (!q) {
      renderSearchResults([]);
      return;
    }
    const results = getSearchIndex().filter((r) => NORMALIZE(r.label).includes(q));
    renderSearchResults(results);
  }
  function setupGlobalSearch() {
    const input = document.getElementById("globalSearchInput");
    const btn = document.getElementById("globalSearchBtn");
    if (!input || !btn) return;
    input.addEventListener("input", () => performGlobalSearch(input.value));
    input.addEventListener("keydown", (e) => {
      if (e.key === "Enter") {
        e.preventDefault();
        performGlobalSearch(input.value);
      }
    });
    btn.addEventListener("click", () => performGlobalSearch(input.value));
  }

  // modules/readOnlyMode.js
  const EDIT_PASSWORD_PROMPT = "Mot de passe admin :";
  const EDIT_PASSWORD_HASH = "15a4c04ed176b88620ebbe26e1f730cfdfa4d9e2bef419ec5cc2e438e80c1b51";
  function toUtf8Bytes(value) {
    if (typeof TextEncoder !== "undefined") {
      return new TextEncoder().encode(value);
    }
    const encoded = unescape(encodeURIComponent(value));
    const result = new Uint8Array(encoded.length);
    for (let i = 0; i < encoded.length; i++) {
      result[i] = encoded.charCodeAt(i);
    }
    return result;
  }
  function rotr(value, bits) {
    return value >>> bits | value << 32 - bits;
  }
  function sha256Fallback(bytes) {
    const K = [
      0x428a2f98, 0x71374491, 0xb5c0fbcf, 0xe9b5dba5, 0x3956c25b, 0x59f111f1, 0x923f82a4, 0xab1c5ed5,
      0xd807aa98, 0x12835b01, 0x243185be, 0x550c7dc3, 0x72be5d74, 0x80deb1fe, 0x9bdc06a7, 0xc19bf174,
      0xe49b69c1, 0xefbe4786, 0x0fc19dc6, 0x240ca1cc, 0x2de92c6f, 0x4a7484aa, 0x5cb0a9dc, 0x76f988da,
      0x983e5152, 0xa831c66d, 0xb00327c8, 0xbf597fc7, 0xc6e00bf3, 0xd5a79147, 0x06ca6351, 0x14292967,
      0x27b70a85, 0x2e1b2138, 0x4d2c6dfc, 0x53380d13, 0x650a7354, 0x766a0abb, 0x81c2c92e, 0x92722c85,
      0xa2bfe8a1, 0xa81a664b, 0xc24b8b70, 0xc76c51a3, 0xd192e819, 0xd6990624, 0xf40e3585, 0x106aa070,
      0x19a4c116, 0x1e376c08, 0x2748774c, 0x34b0bcb5, 0x391c0cb3, 0x4ed8aa4a, 0x5b9cca4f, 0x682e6ff3,
      0x748f82ee, 0x78a5636f, 0x84c87814, 0x8cc70208, 0x90befffa, 0xa4506ceb, 0xbef9a3f7, 0xc67178f2
    ];
    const H = [
      0x6a09e667, 0xbb67ae85, 0x3c6ef372, 0xa54ff53a,
      0x510e527f, 0x9b05688c, 0x1f83d9ab, 0x5be0cd19
    ];
    const bitLen = bytes.length * 8;
    const withOne = bytes.length + 1;
    let paddedLength = withOne;
    while (paddedLength % 64 !== 56) paddedLength++;
    const totalLength = paddedLength + 8;
    const msg = new Uint8Array(totalLength);
    msg.set(bytes);
    msg[bytes.length] = 0x80;
    const view = new DataView(msg.buffer);
    const high = Math.floor(bitLen / 0x100000000);
    const low = bitLen >>> 0;
    view.setUint32(totalLength - 8, high, false);
    view.setUint32(totalLength - 4, low, false);
    const W = new Uint32Array(64);
    for (let i = 0; i < msg.length; i += 64) {
      for (let j = 0; j < 16; j++) {
        W[j] = view.getUint32(i + j * 4, false);
      }
      for (let j = 16; j < 64; j++) {
        const s0 = rotr(W[j - 15], 7) ^ rotr(W[j - 15], 18) ^ W[j - 15] >>> 3;
        const s1 = rotr(W[j - 2], 17) ^ rotr(W[j - 2], 19) ^ W[j - 2] >>> 10;
        W[j] = (W[j - 16] + s0 + W[j - 7] + s1) >>> 0;
      }
      let a = H[0];
      let b = H[1];
      let c = H[2];
      let d = H[3];
      let e = H[4];
      let f = H[5];
      let g = H[6];
      let h = H[7];
      for (let j = 0; j < 64; j++) {
        const S1 = rotr(e, 6) ^ rotr(e, 11) ^ rotr(e, 25);
        const ch = e & f ^ ~e & g;
        const temp1 = (h + S1 + ch + K[j] + W[j]) >>> 0;
        const S0 = rotr(a, 2) ^ rotr(a, 13) ^ rotr(a, 22);
        const maj = a & b ^ a & c ^ b & c;
        const temp2 = (S0 + maj) >>> 0;
        h = g;
        g = f;
        f = e;
        e = (d + temp1) >>> 0;
        d = c;
        c = b;
        b = a;
        a = (temp1 + temp2) >>> 0;
      }
      H[0] = (H[0] + a) >>> 0;
      H[1] = (H[1] + b) >>> 0;
      H[2] = (H[2] + c) >>> 0;
      H[3] = (H[3] + d) >>> 0;
      H[4] = (H[4] + e) >>> 0;
      H[5] = (H[5] + f) >>> 0;
      H[6] = (H[6] + g) >>> 0;
      H[7] = (H[7] + h) >>> 0;
    }
    return Array.from(H).map((val) => val.toString(16).padStart(8, "0")).join("");
  }
  async function hashEditPassword(raw) {
    const data = toUtf8Bytes(raw || "");
    const cryptoObj = (typeof crypto !== "undefined" && crypto) || (typeof window !== "undefined" ? window.crypto : undefined);
    if (cryptoObj && cryptoObj.subtle && typeof cryptoObj.subtle.digest === "function") {
      const hashBuffer = await cryptoObj.subtle.digest("SHA-256", data);
      return Array.from(new Uint8Array(hashBuffer)).map((b) => b.toString(16).padStart(2, "0")).join("");
    }
    return sha256Fallback(data);
  }
  function hashEditPasswordSync(raw) {
    return sha256Fallback(toUtf8Bytes(raw || ""));
  }
  const EDIT_PASSWORD_ALLOWED_HASHES = [EDIT_PASSWORD_HASH, "06898c84c85f504230dcf2664c054e62899be54a9d8daa319ef0cca8f6b2ae3d"].filter(Boolean);
  const EDIT_PASSWORD_STORAGE_KEY = "smsi_edit_password_hash";
  const MIN_EDIT_PASSWORD_LENGTH = 18;
  const EDIT_PASSWORD_REGEX = {
    upper: /[A-Z]/,
    lower: /[a-z]/,
    special: /[^A-Za-z0-9\s]/
  };
  let editPasswordUnlocked = false;
  function getEditPasswordPolicyStatus(password = "", minLength = MIN_EDIT_PASSWORD_LENGTH) {
    const value = String(password || "");
    const checks = [
      { key: "length", label: `Au moins ${minLength} caractères`, ok: value.length >= minLength },
      { key: "upper", label: "Au moins une majuscule", ok: EDIT_PASSWORD_REGEX.upper.test(value) },
      { key: "lower", label: "Au moins une minuscule", ok: EDIT_PASSWORD_REGEX.lower.test(value) },
      { key: "special", label: "Au moins un caractère spécial", ok: EDIT_PASSWORD_REGEX.special.test(value) }
    ];
    const missing = checks.filter((check) => !check.ok).map((check) => check.label);
    return { checks, missing, isValid: missing.length === 0 };
  }
  function createPasswordPolicyList(minLength = MIN_EDIT_PASSWORD_LENGTH) {
    const wrap = document.createElement("div");
    wrap.className = "password-policy";
    const title = document.createElement("p");
    title.className = "password-policy__title";
    title.textContent = "Exigences du mot de passe :";
    const list = document.createElement("ul");
    list.className = "password-policy__list";
    getEditPasswordPolicyStatus("", minLength).checks.forEach((check) => {
      const item = document.createElement("li");
      item.className = "password-policy__item";
      item.dataset.policyKey = check.key;
      item.dataset.label = check.label;
      list.appendChild(item);
    });
    wrap.append(title, list);
    updatePasswordPolicyList(list, "", minLength);
    return { wrap, list };
  }
  function updatePasswordPolicyList(list, password, minLength = MIN_EDIT_PASSWORD_LENGTH) {
    if (!list) return false;
    const status = getEditPasswordPolicyStatus(password, minLength);
    status.checks.forEach((check) => {
      const item = list.querySelector(`[data-policy-key="${check.key}"]`);
      if (!item) return;
      const label = item.dataset.label || check.label;
      item.textContent = check.ok ? `OK: ${label}` : `Manque: ${label}`;
      item.classList.toggle("password-policy__item--ok", check.ok);
      item.classList.toggle("password-policy__item--missing", !check.ok);
    });
    return status.isValid;
  }
  function getStoredEditPasswordHash() {
    try {
      const stored = localStorage.getItem(EDIT_PASSWORD_STORAGE_KEY);
      return typeof stored === "string" ? stored : "";
    } catch (e) {
      return "";
    }
  }
  function setStoredEditPasswordHash(hash) {
    try {
      if (hash) localStorage.setItem(EDIT_PASSWORD_STORAGE_KEY, hash);
      else localStorage.removeItem(EDIT_PASSWORD_STORAGE_KEY);
    } catch (e) {
      // Ignore storage errors
    }
  }
  function getAllowedEditPasswordHashes() {
    const stored = getStoredEditPasswordHash();
    if (stored && stored.length >= 64) return [stored];
    return EDIT_PASSWORD_ALLOWED_HASHES;
  }
  function isLocalhost() {
    try {
      return ["localhost", "127.0.0.1", "", "::1"].includes(window.location.hostname);
    } catch (e) {
      return false;
    }
  }
  async function verifyEditPassword(pwd) {
    try {
      const hash = await hashEditPassword(pwd);
      return getAllowedEditPasswordHashes().includes(hash);
    } catch (e) {
      console.warn("Impossible de vérifier le mot de passe (crypto manquante)", e);
      return false;
    }
  }
  function shouldPromptEditPasswordSetup() {
    if (typeof shouldAutoUnlockForTests === "function" && shouldAutoUnlockForTests()) return false;
    return !getStoredEditPasswordHash();
  }
  function queueEditPasswordSetup() {
    if (!shouldPromptEditPasswordSetup()) return;
    setTimeout(() => {
      if (!shouldPromptEditPasswordSetup()) return;
      openEditPasswordSetupModal({ allowSkip: false });
    }, 250);
  }
  function openEditPasswordSetupModal(options = {}) {
    if (!shouldPromptEditPasswordSetup()) return;
    const overlay = document.getElementById("modalOverlay");
    if (overlay && overlay.classList.contains("modal-overlay--visible")) return;
    const minLength = typeof options.minLength === "number" ? options.minLength : MIN_EDIT_PASSWORD_LENGTH;
    const allowSkip = options.allowSkip !== false;
    const cancelLabel = allowSkip ? "Plus tard" : "Annuler";
    openModal("Définir le mot de passe", [
      { name: "password", label: "Nouveau mot de passe", type: "password", value: "" },
      { name: "confirm", label: "Confirmation", type: "password", value: "" }
    ], (data) => {
      const password = data.password || "";
      const confirm = data.confirm || "";
      if (!password) {
        showToast2("Veuillez saisir un mot de passe.", "warning");
        return false;
      }
      const policy = getEditPasswordPolicyStatus(password, minLength);
      if (!policy.isValid) {
        const detail = policy.missing.length ? `Exigences manquantes : ${policy.missing.join(", ")}.` : "";
        showToast2(`Mot de passe non conforme. ${detail}`.trim(), "warning");
        return false;
      }
      if (password !== confirm) {
        showToast2("Les mots de passe ne correspondent pas.", "error");
        return false;
      }
      const hash = hashEditPasswordSync(password);
      const isFirstPasswordSetup = !localStorage.getItem("smsi_password_setup_done");
      setStoredEditPasswordHash(hash);
      editPasswordUnlocked = true;
      showToast2("Mot de passe enregistré", "success");
      if (isFirstPasswordSetup) {
        localStorage.setItem("smsi_password_setup_done", "1");
        setTimeout(() => {
          showToast2("L'application démarre en mode lecture seule. Cliquez sur le cadenas 🔓 pour passer en mode édition.", "info", 8000);
        }, 2000);
      }
      return true;
    }, null, {
      allowReadOnly: true,
      saveLabel: "Enregistrer",
      cancelLabel,
      hideCancel: !allowSkip,
      hideClose: !allowSkip
    });
    const modalBody = document.getElementById("modalBody");
    const passwordInput = document.getElementById("modal-input-password");
    if (modalBody && passwordInput) {
      const { wrap, list } = createPasswordPolicyList(minLength);
      modalBody.appendChild(wrap);
      const update = () => updatePasswordPolicyList(list, passwordInput.value, minLength);
      passwordInput.addEventListener("input", update);
      update();
    }
  }
  let editModeEnabled = false;
  const HISTORY_KEY = "smsi_history";
  const HISTORY_LIMIT_DEFAULT = 10;
  const HISTORY_LIMIT_STORAGE_KEY = "smsi_history_limit";
  const WELCOME_STORAGE_KEY = "smsi_welcome_seen";
  const SYNC_CONFIG_KEY = "smsi_server_sync_cfg";
  const SYNC_PASSWORD_SESSION_KEY = "smsi_server_sync_pwd_session";
  const SYNC_PASSWORD_PERSIST_KEY = "smsi_server_sync_pwd_persist";
  const NORMALIZE = (str) => (str || "").normalize("NFD").replace(/[\u0300-\u036f]/g, "").toLowerCase();
  function shortenLegendLabel(label) {
    const clean = (label || "").replace(/\s+/g, " ").trim();
    const maxLen = 38;
    if (clean.length <= maxLen) return clean;
    const truncated = clean.slice(0, maxLen);
    const lastSpace = truncated.lastIndexOf(" ");
    const base = lastSpace > 18 ? truncated.slice(0, lastSpace) : truncated;
    return `${base.trim()}…`;
  }
  function renderRadarLegend(targetId, items) {
    const container = document.getElementById(targetId);
    if (!container) return;
    if (!Array.isArray(items) || items.length === 0) {
      container.innerHTML = "";
      return;
    }
    container.classList.add("chart-legend--two-cols");
    const list = document.createElement("ol");
    items.forEach((text) => {
      const li = document.createElement("li");
      li.textContent = shortenLegendLabel(text);
      list.appendChild(li);
    });
    container.innerHTML = "";
    container.appendChild(list);
  }
  function attachLegendTooltip(targetId) {
    const btn = document.querySelector(`[data-legend-toggle="${targetId}"]`);
    const legend = document.getElementById(targetId);
    if (!btn || !legend) return;
    const show = () => legend.removeAttribute("hidden");
    const hide = () => legend.setAttribute("hidden", "");
    ["mouseenter", "focus"].forEach((evt) => btn.addEventListener(evt, show));
    ["mouseleave", "blur"].forEach((evt) => btn.addEventListener(evt, hide));
    ["mouseenter", "mouseleave"].forEach((evt) => legend.addEventListener(evt, (e) => {
      if (evt === "mouseenter") show();
      else hide();
    }));
    hide();
  }
  const EDIT_LOCK_IDS = [
    "addControlBtn",
    "addActionBtn",
    "addNis2DomainBtn",
    "addNis2ActionBtn",
    "addNis2StatusBtn",
    "addRiskBtn",
    "addThreatBtn",
    "addNcBtn",
    "addAuditBtn",
    "addReviewBtn",
    "addStakeholderBtn",
    "addDocumentBtn",
    "addKpiBtn",
    "addProjectRiskBtn",
    "settingsBtn",
    "serverSyncBtn",
    "settingsImport",
    "settingsReset",
    "settingsDemo",
    "serverSyncDownload",
    "serverSyncUpload"
  ];
  function isReadOnlyMode() {
    return !editModeEnabled;
  }
  function showReadOnlyNotice(actionLabel = "modifier") {
    try {
      showToast2(`Mode lecture seule : activez le mode édition pour ${actionLabel}.`, "warning");
    } catch (e) {
      console.warn("Mode lecture seule actif");
    }
  }
  function refreshReadOnlyLocks() {
    const disabled = isReadOnlyMode();
    document.body.classList.toggle("mode-readonly", disabled);
    document.body.classList.toggle("mode-edit", !disabled);
    document.querySelectorAll("[data-readonly-lock]").forEach((el) => {
      const isPermanent = el.dataset?.permanentReadonly === "true";
      if (isPermanent) {
        el.disabled = true;
        return;
      }
      if ("disabled" in el) {
        el.disabled = disabled;
      } else if (disabled) {
        el.setAttribute("aria-disabled", "true");
      } else if (el.getAttribute("aria-disabled") === "true") {
        el.removeAttribute("aria-disabled");
      }
    });
    const saveBtn = document.getElementById("modalSave");
    const deleteBtn = document.getElementById("modalDelete");
    if (saveBtn) {
      saveBtn.disabled = disabled;
      saveBtn.textContent = disabled ? "Lecture seule" : "Sauvegarder";
    }
    if (deleteBtn) {
      deleteBtn.disabled = disabled || deleteBtn.classList.contains("hidden");
    }
  }
  function updateModeBannerUi() {
    const banner = document.getElementById("modeBanner");
    const status = document.getElementById("modeBannerStatus");
    const subtitle = document.getElementById("modeBannerSubtitle");
    if (!banner) return;
    banner.classList.toggle("mode-banner--readonly", isReadOnlyMode());
    banner.classList.toggle("mode-banner--edit", !isReadOnlyMode());
    if (status) {
      status.innerHTML = isReadOnlyMode()
        ? '<i class="fas fa-lock"></i> Mode lecture seule'
        : '<i class="fas fa-pen"></i> Mode édition';
    }
    if (subtitle) {
      subtitle.textContent = isReadOnlyMode()
        ? ""
        : "Mode édition actif : vous pouvez modifier les données et publier vers le serveur.";
    }
  }
  function updateEditModeButtonUi() {
    const btn = document.getElementById("editModeBtn");
    if (!btn) return;
    if (isReadOnlyMode()) {
      btn.textContent = "Mode édition";
      btn.classList.remove("btn--primary");
      btn.classList.add("btn--outline");
    } else {
      btn.textContent = "Quitter édition";
      btn.classList.add("btn--primary");
      btn.classList.remove("btn--outline");
    }
  }
  function setEditMode(enabled, options = {}) {
    editModeEnabled = !!enabled;
    refreshReadOnlyLocks();
    updateModeBannerUi();
    updateEditModeButtonUi();
    if (typeof renderNis2Plan === "function") {
      try {
        renderNis2Plan();
      } catch (e) {
        console.warn("Impossible de rafraîchir le plan NIS2 après changement de mode", e);
      }
    }
    if (options.silent) return;
    showToast2(enabled ? "Mode édition activé" : "Mode lecture seule activé", enabled ? "success" : "info");
  }
  function shouldAutoUnlockForTests() {
    return isLocalhost() && !!(typeof navigator !== "undefined" && navigator.webdriver);
  }

  // Modale sécurisée pour la saisie du mot de passe (masqué)
  function showPasswordModal(title, message) {
    return new Promise((resolve) => {
      const overlay = document.getElementById("modalOverlay");
      const modalTitle = document.getElementById("modalTitle");
      const modalBody = document.getElementById("modalBody");
      const saveBtn = document.getElementById("modalSave");
      const cancelBtn = document.getElementById("modalCancel");
      const closeBtn = document.getElementById("modalClose");
      const deleteBtn = document.getElementById("modalDelete");

      if (!overlay || !modalTitle || !modalBody || !saveBtn || !cancelBtn) {
        resolve(null);
        return;
      }

      modalTitle.textContent = title;
      modalBody.innerHTML = "";

      // Message d'instruction
      if (message) {
        const messageEl = document.createElement("p");
        messageEl.className = "modal__message";
        messageEl.textContent = message;
        modalBody.appendChild(messageEl);
      }

      // Champ mot de passe
      const label = document.createElement("label");
      label.textContent = "Mot de passe";
      label.htmlFor = "modal-password-input";
      label.className = "modal__label";

      const inputWrapper = document.createElement("div");
      inputWrapper.className = "password-input-wrapper";
      inputWrapper.style.cssText = "position: relative; display: flex; align-items: center;";

      const input = document.createElement("input");
      input.type = "password";
      input.id = "modal-password-input";
      input.className = "form-control";
      input.autocomplete = "current-password";
      input.placeholder = "Entrez le mot de passe";
      input.style.paddingRight = "2.5rem";

      // Bouton pour afficher/masquer le mot de passe
      const toggleBtn = document.createElement("button");
      toggleBtn.type = "button";
      toggleBtn.className = "password-toggle-btn";
      toggleBtn.setAttribute("aria-label", "Afficher le mot de passe");
      toggleBtn.style.cssText = "position: absolute; right: 0.5rem; background: none; border: none; cursor: pointer; padding: 0.25rem; color: var(--color-text-muted, #6c757d); display: flex; align-items: center; justify-content: center;";
      toggleBtn.innerHTML = '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path><circle cx="12" cy="12" r="3"></circle></svg>';

      toggleBtn.addEventListener("click", () => {
        const isPassword = input.type === "password";
        input.type = isPassword ? "text" : "password";
        toggleBtn.setAttribute("aria-label", isPassword ? "Masquer le mot de passe" : "Afficher le mot de passe");
        toggleBtn.innerHTML = isPassword
          ? '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M17.94 17.94A10.07 10.07 0 0 1 12 20c-7 0-11-8-11-8a18.45 18.45 0 0 1 5.06-5.94M9.9 4.24A9.12 9.12 0 0 1 12 4c7 0 11 8 11 8a18.5 18.5 0 0 1-2.16 3.19m-6.72-1.07a3 3 0 1 1-4.24-4.24"></path><line x1="1" y1="1" x2="23" y2="23"></line></svg>'
          : '<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path><circle cx="12" cy="12" r="3"></circle></svg>';
        input.focus();
      });

      inputWrapper.appendChild(input);
      inputWrapper.appendChild(toggleBtn);

      modalBody.appendChild(label);
      modalBody.appendChild(inputWrapper);

      // Configuration des boutons
      saveBtn.textContent = "Valider";
      saveBtn.disabled = false;
      cancelBtn.textContent = "Annuler";
      cancelBtn.style.display = "";
      if (closeBtn) closeBtn.style.display = "";
      if (deleteBtn) deleteBtn.style.display = "none";

      // Fermer la modale
      function close(result) {
        overlay.classList.remove("modal-overlay--visible");
        modalTitle.textContent = "";
        modalBody.innerHTML = "";
        saveBtn.replaceWith(saveBtn.cloneNode(true));
        cancelBtn.replaceWith(cancelBtn.cloneNode(true));
        if (closeBtn) closeBtn.replaceWith(closeBtn.cloneNode(true));
        resolve(result);
      }

      // Événements
      document.getElementById("modalSave").addEventListener("click", () => {
        close(input.value || null);
      });

      document.getElementById("modalCancel").addEventListener("click", () => {
        close(null);
      });

      if (closeBtn) {
        document.getElementById("modalClose").addEventListener("click", () => {
          close(null);
        });
      }

      // Entrée pour valider
      input.addEventListener("keydown", (e) => {
        if (e.key === "Enter") {
          e.preventDefault();
          close(input.value || null);
        } else if (e.key === "Escape") {
          e.preventDefault();
          close(null);
        }
      });

      // Afficher la modale
      overlay.classList.add("modal-overlay--visible");

      // Focus sur le champ après ouverture
      setTimeout(() => input.focus(), 100);
    });
  }

  async function requireEditUnlock() {
    // Mode DEMO: pas de mot de passe requis
    if (DEMO_MODE) {
      editPasswordUnlocked = true;
      setEditMode(true, { silent: true });
      return true;
    }
    if (shouldAutoUnlockForTests()) {
      editPasswordUnlocked = true;
      setEditMode(true, { silent: true });
      return true;
    }
    if (editPasswordUnlocked) return true;
    if (typeof document === "undefined") {
      showReadOnlyNotice();
      return false;
    }
    const input = await showPasswordModal("Authentification requise", EDIT_PASSWORD_PROMPT);
    if (!input) {
      showReadOnlyNotice();
      return false;
    }
    const ok = await verifyEditPassword(input);
    if (!ok) {
      showToast2("Mot de passe incorrect", "error");
      return false;
    }
    editPasswordUnlocked = true;
    const syncPwdInput = document.getElementById("serverSyncPassword");
    if (syncPwdInput) syncPwdInput.value = input;
    return true;
  }
  function ensureEditMode(actionLabel = "modifier") {
    if (!isReadOnlyMode()) return true;
    showReadOnlyNotice(actionLabel);
    return false;
  }
  function initReadOnlyBanner() {
    EDIT_LOCK_IDS.forEach((id) => {
      const el = document.getElementById(id);
      if (el) el.dataset.readonlyLock = "true";
    });
    // En mode DEMO : édition active par défaut (pas de mot de passe)
    // Sinon : démarrage par défaut en mode lecture seule
    if (DEMO_MODE) {
      setEditMode(true, { silent: true });
    } else {
      setEditMode(false, { silent: true });
    }
    updateModeBannerUi();
    updateEditModeButtonUi();
  }
  async function handleEditModeButton() {
    if (isReadOnlyMode()) {
      const ok = await requireEditUnlock();
      if (!ok) return;
      setEditMode(true);
    } else {
      setEditMode(false);
    }
  }
  // ============================================================
  // Système de licence serveur
  // ============================================================

  // Clé secrète pour la validation des licences (HMAC)
  // Cette clé doit correspondre à celle utilisée pour générer les licences
  const LICENSE_SECRET_KEY = "LINA-LAB-2025-CYBER-ASSISTANT-LICENSE-KEY";

  /**
   * Calcule le hash SHA-256 d'une chaîne (legacy - version 1.0)
   * @param {string} message
   * @returns {Promise<string>} hash hexadécimal
   */
  async function sha256Hash(message) {
    const encoder = new TextEncoder();
    const data = encoder.encode(message);
    const hashBuffer = await crypto.subtle.digest('SHA-256', data);
    const hashArray = Array.from(new Uint8Array(hashBuffer));
    return hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
  }

  /**
   * Calcule le HMAC-SHA256 d'une chaîne (version 1.1)
   * @param {string} message - Le message à signer
   * @param {string} secret - La clé secrète
   * @returns {Promise<string>} HMAC hexadécimal
   */
  async function hmacSha256(message, secret) {
    const encoder = new TextEncoder();
    const keyData = encoder.encode(secret);
    const msgData = encoder.encode(message);

    // Importer la clé pour HMAC
    const cryptoKey = await crypto.subtle.importKey(
      'raw',
      keyData,
      { name: 'HMAC', hash: 'SHA-256' },
      false,
      ['sign']
    );

    // Générer le HMAC
    const signature = await crypto.subtle.sign('HMAC', cryptoKey, msgData);
    const hashArray = Array.from(new Uint8Array(signature));
    return hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
  }

  /**
   * Valide la signature d'un fichier de licence
   * Supporte les deux versions de signature pour la rétrocompatibilité:
   * - Version 1.0: SHA-256(content + key)
   * - Version 1.1: HMAC-SHA256(content, key)
   * @param {Object} licenseData - Données de la licence
   * @param {string} signature - Signature à vérifier
   * @param {Object} meta - Métadonnées de la licence (optionnel)
   * @returns {Promise<boolean>}
   */
  async function validateLicenseSignature(licenseData, signature, meta) {
    try {
      // Reconstruit le contenu signé (ordre déterministe)
      const content = JSON.stringify({
        type: licenseData.type,
        organization: licenseData.organization,
        issuedAt: licenseData.issuedAt,
        expiresAt: licenseData.expiresAt || null,
        features: licenseData.features || ['server-sync']
      });

      // Déterminer le type de signature
      const signatureType = meta && meta.signatureType;
      const version = meta && meta.version;

      let expectedSignature;
      if (signatureType === 'hmac-sha256' || version === '1.1') {
        // Nouvelle méthode HMAC-SHA256 (version 1.1)
        expectedSignature = await hmacSha256(content, LICENSE_SECRET_KEY);
      } else {
        // Ancienne méthode SHA256 simple (version 1.0 - rétrocompatibilité)
        expectedSignature = await sha256Hash(content + LICENSE_SECRET_KEY);
      }

      return signature === expectedSignature;
    } catch (e) {
      console.error('[CSI License] Erreur de validation:', e);
      return false;
    }
  }

  /**
   * Vérifie si une licence est expirée
   * @param {Object} licenseData
   * @returns {boolean}
   */
  function isLicenseExpired(licenseData) {
    if (!licenseData.expiresAt) return false; // Licence permanente
    return new Date(licenseData.expiresAt) < new Date();
  }

  /**
   * Vérifie si le mode serveur est débloqué (licence valide)
   * @returns {boolean}
   */
  function isServerModeUnlocked() {
    const Storage = window.CSI_Storage;
    if (!Storage) return false;

    if (!Storage.isServerLicenseActivated()) return false;

    const license = Storage.getServerLicense();
    if (!license) return false;

    // Vérifier l'expiration
    if (isLicenseExpired(license)) {
      Storage.deactivateServerLicense();
      return false;
    }

    return true;
  }

  /**
   * Affiche la modale d'upload de licence
   */
  function showLicenseUploadModal() {
    const overlay = document.createElement('div');
    overlay.className = 'license-modal-overlay';
    overlay.innerHTML = `
      <div class="license-modal">
        <div class="license-modal__header">
          <h3><i class="fas fa-key"></i> Activation du Mode Serveur</h3>
          <button class="license-modal__close" title="Fermer">&times;</button>
        </div>
        <div class="license-modal__body">
          <p class="license-modal__intro">
            Le mode serveur nécessite une clé d'activation car un accompagnement est proposé pour l'installation et l'adaptation de l'application à votre administration ou entreprise.
          </p>

          <div class="license-modal__dropzone" id="licenseDropzone">
            <i class="fas fa-cloud-upload-alt"></i>
            <p>Glissez votre fichier ici<br>ou cliquez pour sélectionner</p>
            <input type="file" id="licenseFileInput" accept=".csi-license,.json" style="display:none">
          </div>

          <p class="license-modal__help">
            <i class="fas fa-info-circle"></i>
            Pour obtenir une clé d'activation et bénéficier d'un accompagnement personnalisé, <a href="mailto:cyber-assistant-contact@proton.me?subject=Demande%20de%20cl%C3%A9%20d'activation%20Cyber-Assistant">Contactez LINA LAB</a>.
          </p>

          <div class="license-modal__status" id="licenseStatus" style="display:none">
            <div class="license-modal__status-icon"></div>
            <div class="license-modal__status-text"></div>
          </div>

          <div class="license-modal__info" id="licenseInfo" style="display:none">
            <h4>Informations de la clé d'activation</h4>
            <dl>
              <dt>Organisation</dt>
              <dd id="licenseOrg">-</dd>
              <dt>Type</dt>
              <dd id="licenseType">-</dd>
              <dt>Émise le</dt>
              <dd id="licenseIssued">-</dd>
              <dt>Expire le</dt>
              <dd id="licenseExpires">-</dd>
            </dl>
          </div>
        </div>
        <div class="license-modal__footer">
          <button class="btn btn--outline license-modal__cancel">Annuler</button>
          <button class="btn btn--primary license-modal__activate" id="licenseActivateBtn" disabled>
            <i class="fas fa-check"></i> Activer
          </button>
        </div>
      </div>
    `;
    document.body.appendChild(overlay);

    // Variables pour stocker la licence en cours de validation
    let pendingLicense = null;

    // Fermer la modale
    const closeModal = () => {
      overlay.remove();
    };

    overlay.querySelector('.license-modal__close').addEventListener('click', closeModal);
    overlay.querySelector('.license-modal__cancel').addEventListener('click', closeModal);
    overlay.addEventListener('click', (e) => {
      if (e.target === overlay) closeModal();
    });

    // Zone de dépôt
    const dropzone = overlay.querySelector('#licenseDropzone');
    const fileInput = overlay.querySelector('#licenseFileInput');
    const activateBtn = overlay.querySelector('#licenseActivateBtn');

    // En mode démo, désactiver la zone de dépôt et le bouton Activer
    if (DEMO_MODE) {
      dropzone.style.opacity = '0.5';
      dropzone.style.pointerEvents = 'none';
      dropzone.style.cursor = 'not-allowed';
      dropzone.innerHTML = `
        <i class="fas fa-lock"></i>
        <p>Fonctionnalité désactivée<br>en mode démonstration</p>
      `;
      activateBtn.disabled = true;
      activateBtn.style.opacity = '0.5';
      activateBtn.style.cursor = 'not-allowed';
      return; // Ne pas attacher les événements d'upload
    }

    dropzone.addEventListener('click', () => fileInput.click());
    dropzone.addEventListener('dragover', (e) => {
      e.preventDefault();
      dropzone.classList.add('license-modal__dropzone--dragover');
    });
    dropzone.addEventListener('dragleave', () => {
      dropzone.classList.remove('license-modal__dropzone--dragover');
    });
    dropzone.addEventListener('drop', (e) => {
      e.preventDefault();
      dropzone.classList.remove('license-modal__dropzone--dragover');
      const file = e.dataTransfer.files[0];
      if (file) processLicenseFile(file);
    });
    fileInput.addEventListener('change', (e) => {
      const file = e.target.files[0];
      if (file) processLicenseFile(file);
    });

    // Afficher le statut
    const showStatus = (type, message) => {
      const statusEl = overlay.querySelector('#licenseStatus');
      const iconEl = statusEl.querySelector('.license-modal__status-icon');
      const textEl = statusEl.querySelector('.license-modal__status-text');

      statusEl.style.display = 'flex';
      statusEl.className = `license-modal__status license-modal__status--${type}`;

      const icons = {
        loading: '<i class="fas fa-spinner fa-spin"></i>',
        success: '<i class="fas fa-check-circle"></i>',
        error: '<i class="fas fa-exclamation-circle"></i>',
        warning: '<i class="fas fa-exclamation-triangle"></i>'
      };
      iconEl.innerHTML = icons[type] || icons.loading;
      textEl.textContent = message;
    };

    // Traitement du fichier de licence
    const processLicenseFile = async (file) => {
      showStatus('loading', 'Vérification de la clé d\'activation...');
      overlay.querySelector('#licenseInfo').style.display = 'none';
      overlay.querySelector('#licenseActivateBtn').disabled = true;
      pendingLicense = null;

      try {
        const content = await file.text();
        let licenseFile;

        try {
          licenseFile = JSON.parse(content);
        } catch (e) {
          showStatus('error', 'Format de fichier invalide. Le fichier doit être au format JSON.');
          return;
        }

        // Vérifier la structure
        if (!licenseFile.license || !licenseFile.signature) {
          showStatus('error', 'Structure de clé invalide. Fichier corrompu ou incorrect.');
          return;
        }

        const licenseData = licenseFile.license;
        const signature = licenseFile.signature;
        const meta = licenseFile.meta;

        // Valider la signature (supporte v1.0 SHA256 et v1.1 HMAC-SHA256)
        const isValid = await validateLicenseSignature(licenseData, signature, meta);
        if (!isValid) {
          showStatus('error', 'Signature invalide. Cette clé d\'activation n\'est pas authentique.');
          return;
        }

        // Vérifier l'expiration
        if (isLicenseExpired(licenseData)) {
          showStatus('warning', 'Cette clé d\'activation a expiré le ' + new Date(licenseData.expiresAt).toLocaleDateString('fr-FR'));
          return;
        }

        // Clé valide !
        showStatus('success', 'Clé d\'activation valide ! Vous pouvez activer le mode serveur.');
        pendingLicense = licenseFile;

        // Afficher les infos
        const infoEl = overlay.querySelector('#licenseInfo');
        infoEl.style.display = 'block';
        infoEl.querySelector('#licenseOrg').textContent = licenseData.organization || '-';
        infoEl.querySelector('#licenseType').textContent = licenseData.type === 'permanent' ? 'Clé permanente' : 'Clé temporaire';
        infoEl.querySelector('#licenseIssued').textContent = new Date(licenseData.issuedAt).toLocaleDateString('fr-FR');
        infoEl.querySelector('#licenseExpires').textContent = licenseData.expiresAt
          ? new Date(licenseData.expiresAt).toLocaleDateString('fr-FR')
          : 'Jamais (permanente)';

        overlay.querySelector('#licenseActivateBtn').disabled = false;

      } catch (e) {
        console.error('[CSI License] Erreur:', e);
        showStatus('error', 'Erreur lors de la lecture du fichier.');
      }
    };

    // Activation de la licence
    overlay.querySelector('#licenseActivateBtn').addEventListener('click', () => {
      if (!pendingLicense) return;

      const Storage = window.CSI_Storage;
      if (Storage) {
        Storage.activateServerLicense(pendingLicense.license);
        showToast2('Mode serveur activé ! La fonctionnalité est maintenant disponible.', 'success');
        closeModal();

        // Recharger pour activer le mode serveur
        setTimeout(() => {
          window.location.reload();
        }, 1500);
      }
    });
  }

  /**
   * Handler du bouton Sync Serveur
   */
  function handleServerSyncButton() {
    if (!ensureEditMode("utiliser la synchronisation serveur")) {
      refreshReadOnlyLocks();
      return;
    }

    // Vérifier si le mode serveur est débloqué (licence valide)
    if (isServerModeUnlocked()) {
      // Licence valide - ouvrir le panneau de synchronisation
      if (typeof window.toggleServerSyncPanel === "function") {
        window.toggleServerSyncPanel();
      } else {
        showToast2("Panneau de synchronisation en cours de chargement...", "info");
      }
    } else {
      // Pas de licence - afficher la modale d'upload
      showLicenseUploadModal();
    }
  }

  // ============================================================
  // Panneau de synchronisation serveur
  // ============================================================

  let syncPanelCreated = false;
  let syncPanel = null;
  let syncPanelOverlay = null;

  /**
   * Crée le panneau de synchronisation serveur
   */
  function createServerSyncPanel() {
    if (syncPanelCreated) return;
    syncPanelCreated = true;

    // Overlay
    syncPanelOverlay = document.createElement('div');
    syncPanelOverlay.className = 'sync-panel-overlay';
    syncPanelOverlay.addEventListener('click', closeServerSyncPanel);
    document.body.appendChild(syncPanelOverlay);

    // Panel
    syncPanel = document.createElement('div');
    syncPanel.id = 'serverSyncPanel';
    syncPanel.className = 'sync-panel';
    syncPanel.innerHTML = `
      <div class="sync-panel__header">
        <h3><i class="fas fa-cloud"></i> Synchronisation Serveur</h3>
        <button class="sync-panel__close" title="Fermer">&times;</button>
      </div>
      <div class="sync-panel__body">
        <div class="sync-panel__section">
          <div class="sync-panel__status" id="syncPanelStatus">
            <div class="sync-panel__status-icon sync-panel__status-icon--ok">
              <i class="fas fa-check"></i>
            </div>
            <div class="sync-panel__status-text">
              <div class="sync-panel__status-label">Prêt</div>
              <div class="sync-panel__status-detail">Aucune synchronisation en cours</div>
            </div>
          </div>
        </div>

        <div class="sync-panel__section">
          <div class="sync-panel__section-title">Actions</div>
          <div class="sync-panel__actions">
            <button class="sync-panel__btn" id="serverSyncDownload">
              <i class="fas fa-cloud-arrow-down"></i>
              Charger depuis le serveur
            </button>
            <button class="sync-panel__btn sync-panel__btn--primary" id="serverSyncUpload">
              <i class="fas fa-cloud-arrow-up"></i>
              Publier sur le serveur
            </button>
          </div>
        </div>

        <div class="sync-panel__section">
          <div class="sync-panel__section-title">Informations</div>
          <div class="sync-panel__meta" id="syncPanelMeta">
            <div class="sync-panel__meta-row">
              <span>Dernier chargement</span>
              <span id="syncMetaLastPull">-</span>
            </div>
            <div class="sync-panel__meta-row">
              <span>Dernière publication</span>
              <span id="syncMetaLastPush">-</span>
            </div>
            <div class="sync-panel__meta-row">
              <span>Mode</span>
              <span id="syncMetaMode">Serveur</span>
            </div>
          </div>
        </div>
      </div>
      <div class="sync-panel__footer">
        <p class="sync-panel__footer-note">
          Les données sont synchronisées avec le serveur configuré dans les paramètres.
        </p>
      </div>
    `;
    document.body.appendChild(syncPanel);

    // Event listeners
    syncPanel.querySelector('.sync-panel__close').addEventListener('click', closeServerSyncPanel);

    // Download button
    syncPanel.querySelector('#serverSyncDownload').addEventListener('click', handleSyncDownload);

    // Upload button
    syncPanel.querySelector('#serverSyncUpload').addEventListener('click', handleSyncUpload);
  }

  function openServerSyncPanel() {
    createServerSyncPanel();
    syncPanel.classList.add('sync-panel--visible');
    syncPanelOverlay.classList.add('sync-panel-overlay--visible');
    updateSyncPanelMeta();
    updateSyncPanelStatus('ready');
  }

  function closeServerSyncPanel() {
    if (syncPanel) syncPanel.classList.remove('sync-panel--visible');
    if (syncPanelOverlay) syncPanelOverlay.classList.remove('sync-panel-overlay--visible');
  }

  /**
   * Bascule l'affichage du panneau de synchronisation
   */
  function toggleServerSyncPanel() {
    createServerSyncPanel();
    if (syncPanel.classList.contains('sync-panel--visible')) {
      closeServerSyncPanel();
    } else {
      openServerSyncPanel();
    }
  }

  function updateSyncPanelStatus(status, message) {
    if (!syncPanel) return;
    const statusEl = syncPanel.querySelector('#syncPanelStatus');
    if (!statusEl) return;

    let iconClass = 'sync-panel__status-icon--ok';
    let iconHtml = '<i class="fas fa-check"></i>';
    let label = 'Prêt';
    let detail = message || 'Aucune synchronisation en cours';

    switch (status) {
      case 'syncing':
        iconClass = 'sync-panel__status-icon--syncing';
        iconHtml = '<i class="fas fa-sync-alt"></i>';
        label = 'Synchronisation...';
        detail = message || 'Transfert en cours';
        break;
      case 'success':
        iconClass = 'sync-panel__status-icon--ok';
        iconHtml = '<i class="fas fa-check"></i>';
        label = 'Succès';
        detail = message || 'Opération terminée';
        break;
      case 'error':
        iconClass = 'sync-panel__status-icon--error';
        iconHtml = '<i class="fas fa-exclamation-triangle"></i>';
        label = 'Erreur';
        detail = message || 'Une erreur est survenue';
        break;
    }

    statusEl.innerHTML = `
      <div class="sync-panel__status-icon ${iconClass}">
        ${iconHtml}
      </div>
      <div class="sync-panel__status-text">
        <div class="sync-panel__status-label">${label}</div>
        <div class="sync-panel__status-detail">${detail}</div>
      </div>
    `;
  }

  function updateSyncPanelMeta() {
    if (!syncPanel) return;
    const Storage = window.CSI_Storage;
    if (!Storage) return;

    const timestamps = Storage.getSyncTimestamps();
    const pullEl = syncPanel.querySelector('#syncMetaLastPull');
    const pushEl = syncPanel.querySelector('#syncMetaLastPush');

    const formatDate = (dateStr) => {
      if (!dateStr) return '-';
      const date = new Date(dateStr);
      if (isNaN(date.getTime())) return '-';
      return date.toLocaleDateString('fr-FR', { day: '2-digit', month: '2-digit', hour: '2-digit', minute: '2-digit' });
    };

    if (pullEl) pullEl.textContent = formatDate(timestamps.lastPull);
    if (pushEl) pushEl.textContent = formatDate(timestamps.lastPush);
  }

  async function handleSyncDownload() {
    const SyncManager = window.CSI_SyncManager;
    if (!SyncManager) {
      updateSyncPanelStatus('error', 'Module de synchronisation non chargé');
      return;
    }

    const downloadBtn = syncPanel.querySelector('#serverSyncDownload');
    if (downloadBtn) downloadBtn.disabled = true;
    updateSyncPanelStatus('syncing', 'Chargement depuis le serveur...');

    try {
      const result = await SyncManager.pullFromServer();
      if (result && result.success) {
        updateSyncPanelStatus('success', 'Données chargées avec succès');
        updateSyncPanelMeta();
        if (typeof window.refreshAllTables === 'function') window.refreshAllTables();
        if (typeof window.updateDashboard === 'function') window.updateDashboard();
      } else {
        updateSyncPanelStatus('error', result ? result.message : 'Échec du chargement');
      }
    } catch (err) {
      console.error('[CSI Sync] Download error:', err);
      updateSyncPanelStatus('error', 'Erreur: ' + (err.message || 'Échec du chargement'));
    } finally {
      if (downloadBtn) downloadBtn.disabled = false;
    }
  }

  async function handleSyncUpload() {
    const SyncManager = window.CSI_SyncManager;
    if (!SyncManager) {
      updateSyncPanelStatus('error', 'Module de synchronisation non chargé');
      return;
    }

    const uploadBtn = syncPanel.querySelector('#serverSyncUpload');
    if (uploadBtn) uploadBtn.disabled = true;
    updateSyncPanelStatus('syncing', 'Publication sur le serveur...');

    try {
      const result = await SyncManager.forceServerSync();
      if (result && result.success) {
        updateSyncPanelStatus('success', 'Données publiées avec succès');
        updateSyncPanelMeta();
      } else {
        updateSyncPanelStatus('error', result ? result.message : 'Échec de la publication');
      }
    } catch (err) {
      console.error('[CSI Sync] Upload error:', err);
      updateSyncPanelStatus('error', 'Erreur: ' + (err.message || 'Échec de la publication'));
    } finally {
      if (uploadBtn) uploadBtn.disabled = false;
    }
  }

  // Exposer les fonctions pour usage externe
  window.isServerModeUnlocked = isServerModeUnlocked;
  window.showLicenseUploadModal = showLicenseUploadModal;
  window.toggleServerSyncPanel = toggleServerSyncPanel;

  window.refreshReadOnlyLocks = refreshReadOnlyLocks;
  window.isReadOnlyMode = isReadOnlyMode;
  window.ensureEditMode = ensureEditMode;
  window.showReadOnlyNotice = showReadOnlyNotice;
  window.handleEditModeButton = handleEditModeButton;
  window.handleServerSyncButton = handleServerSyncButton;
  window.getThemeVars = typeof getThemeVars === "function"
    ? getThemeVars
    : () => ({ textColor: "#2d3e4a", borderColor: "#e9ecef", surface: "#ffffff" });

  // modules/modal.js
  function openModal(title, fields, onSave, onDelete = null, options = {}) {
    const overlay = document.getElementById("modalOverlay");
    const modalTitle = document.getElementById("modalTitle");
    const modalBody = document.getElementById("modalBody");
    const saveBtn = document.getElementById("modalSave");
    const cancelBtn = document.getElementById("modalCancel");
    const closeBtn = document.getElementById("modalClose");
    const deleteBtn = document.getElementById("modalDelete");
    const allowReadOnly = options.allowReadOnly === true;
    const saveLabel = options.saveLabel || "Sauvegarder";
    const cancelLabel = options.cancelLabel || "Annuler";
    const deleteLabel = options.deleteLabel || "Supprimer";
    const hideCancel = options.hideCancel === true;
    const hideClose = options.hideClose === true;
    const message = options.message || "";
    const lockedInitially = isReadOnlyMode() && !allowReadOnly;
    if (!overlay || !modalTitle || !modalBody || !saveBtn || !cancelBtn || !closeBtn || !deleteBtn) return;
    modalTitle.textContent = title;
    modalBody.innerHTML = "";
    if (message) {
      const messageEl = document.createElement("p");
      messageEl.className = "modal__message";
      messageEl.textContent = message;
      modalBody.appendChild(messageEl);
    }
    fields.forEach((f) => {
      const label = document.createElement("label");
      label.textContent = f.label;
      label.htmlFor = `modal-input-${f.name}`;
      label.className = "modal__label";
      let input;
      let fieldWrap;
      if (f.type === "textarea") {
        input = document.createElement("textarea");
        input.value = f.value || "";
      } else if (f.type === "select" || f.type === "multiselect") {
        input = document.createElement("select");
        if (f.type === "multiselect") input.multiple = true;
        (f.options || []).forEach((opt) => {
          const option = document.createElement("option");
          if (typeof opt === "object") {
            option.value = opt.value;
            option.textContent = opt.label;
          } else {
            option.value = opt;
            option.textContent = opt;
          }
          const val = Array.isArray(f.value) ? f.value : [f.value];
          if (val && val.includes(option.value)) option.selected = true;
          input.appendChild(option);
        });
      } else if (f.type === "checkbox") {
        input = document.createElement("input");
        input.type = "checkbox";
        input.checked = !!f.value;
        input.className = "modal__checkbox";
        label.classList.add("modal__label--checkbox");
        fieldWrap = document.createElement("div");
        fieldWrap.className = "modal__checkbox-row";
      } else {
        input = document.createElement("input");
        input.type = f.type || "text";
        input.value = f.value || "";
      }
      input.dataset.readonlyLock = "true";
      if (f.readOnly) input.dataset.permanentReadonly = "true";
      const lockedField = (lockedInitially && !f.allowInReadOnly) || f.readOnly;
      if (lockedField) input.disabled = true;
      if (f.help) input.title = f.help;
      // Validation côté client
      if (f.required) input.required = true;
      if (f.minLength) input.minLength = f.minLength;
      if (f.maxLength) input.maxLength = f.maxLength;
      if (f.pattern) input.pattern = f.pattern;
      if (f.min !== undefined) input.min = f.min;
      if (f.max !== undefined) input.max = f.max;
      if (f.placeholder) input.placeholder = f.placeholder;
      input.id = `modal-input-${f.name}`;
      if (f.type === "checkbox") {
        if (fieldWrap) {
          fieldWrap.append(input, label);
          modalBody.appendChild(fieldWrap);
        }
      } else {
        input.className = "form-control";
        modalBody.appendChild(label);
        modalBody.appendChild(input);
        // Hint d'aide sous le champ
        if (f.help) {
          const hint = document.createElement("small");
          hint.className = "form-hint";
          hint.textContent = f.help;
          modalBody.appendChild(hint);
        }
      }
    });
    function close() {
      overlay.classList.remove("modal-overlay--visible");
      saveBtn.removeEventListener("click", handleSave);
      cancelBtn.removeEventListener("click", close);
      closeBtn.removeEventListener("click", close);
      deleteBtn.removeEventListener("click", handleDelete);
    }
    function handleSave() {
      if (isReadOnlyMode() && !allowReadOnly) {
        showReadOnlyNotice("enregistrer");
        return;
      }
      const data = {};
      fields.forEach((f) => {
        const inp = document.getElementById(`modal-input-${f.name}`);
        if (!inp) {
          data[f.name] = "";
          return;
        }
        if (f.type === "checkbox") {
          data[f.name] = inp.checked;
        } else if (f.type === "multiselect") {
          data[f.name] = Array.from(inp.selectedOptions).map((o) => o.value);
        } else {
          data[f.name] = inp.value.trim();
        }
      });
      if (onSave(data) !== false) {
        close();
      }
    }
    function handleDelete() {
      if (lockedInitially && !allowReadOnly) {
        showReadOnlyNotice("supprimer");
        return;
      }
      // Confirmation avant suppression
      if (!confirm("Êtes-vous sûr de vouloir supprimer cet élément ? Cette action est irréversible.")) {
        return;
      }
      if (onDelete && onDelete() !== false) {
        close();
        showToast2("Élément supprimé", "success");
      }
    }
    cancelBtn.textContent = cancelLabel;
    deleteBtn.textContent = deleteLabel;
    if (hideCancel) {
      cancelBtn.classList.add("hidden");
      cancelBtn.disabled = true;
    } else {
      cancelBtn.classList.remove("hidden");
      cancelBtn.disabled = false;
    }
    if (hideClose) {
      closeBtn.classList.add("hidden");
      closeBtn.disabled = true;
    } else {
      closeBtn.classList.remove("hidden");
      closeBtn.disabled = false;
    }
    if (lockedInitially) {
      saveBtn.disabled = true;
      deleteBtn.disabled = true;
      saveBtn.textContent = "Lecture seule";
    } else {
      saveBtn.disabled = false;
      saveBtn.textContent = saveLabel;
    }
    saveBtn.addEventListener("click", handleSave);
    cancelBtn.addEventListener("click", close);
    closeBtn.addEventListener("click", close);
    if (onDelete) {
      deleteBtn.classList.remove("hidden");
      deleteBtn.disabled = lockedInitially;
      deleteBtn.addEventListener("click", handleDelete);
    } else {
      deleteBtn.classList.add("hidden");
      deleteBtn.disabled = true;
    }
    overlay.classList.add("modal-overlay--visible");
    // Focus sur le premier champ de formulaire
    requestAnimationFrame(() => {
      const firstInput = modalBody.querySelector("input:not([type='checkbox']), textarea, select");
      if (firstInput) {
        firstInput.focus();
      }
      // Callback afterOpen pour du contenu personnalisé
      if (typeof options.afterOpen === "function") {
        options.afterOpen();
      }
    });
    // Focus trap - garder le focus dans la modale
    function handleKeydown(e) {
      if (e.key === "Escape") {
        close();
        return;
      }
      if (e.key === "Tab") {
        const modal = overlay.querySelector(".modal");
        const focusable = modal.querySelectorAll('button:not([disabled]), input:not([disabled]), select:not([disabled]), textarea:not([disabled]), [tabindex]:not([tabindex="-1"])');
        const first = focusable[0];
        const last = focusable[focusable.length - 1];
        if (e.shiftKey && document.activeElement === first) {
          e.preventDefault();
          last.focus();
        } else if (!e.shiftKey && document.activeElement === last) {
          e.preventDefault();
          first.focus();
        }
      }
    }
    overlay.addEventListener("keydown", handleKeydown);
    // Nettoyer le listener à la fermeture
    const originalClose = close;
    close = function() {
      overlay.removeEventListener("keydown", handleKeydown);
      originalClose();
    };
  }
  function closeModal() {
    const overlay = document.getElementById("modalOverlay");
    const modalTitle = document.getElementById("modalTitle");
    const modalBody = document.getElementById("modalBody");
    if (!overlay || !modalTitle || !modalBody) return;
    overlay.classList.remove("modal-overlay--visible");
    modalTitle.textContent = "";
    modalBody.innerHTML = "";
  }
  window.openModal = openModal;
  window.closeModal = closeModal;
  window.showPasswordModal = showPasswordModal;

  // modules/kpis.js
  function loadKpiTable() {
    const tbody = document.getElementById("kpiTableBody");
    if (!tbody) return;
    tbody.innerHTML = "";
    appData.kpis.forEach((kpi) => {
      const row = document.createElement("tr");
      row.innerHTML = `
            <td>${escapeHtml(kpi.title)}</td>
            <td>${escapeHtml(kpi.description)}</td>
            <td><input type="text" class="form-control" data-readonly-lock="true" ${isReadOnlyMode() ? "disabled" : ""} value="${escapeHtml(kpi.comments)}" onchange="updateKpiComment('${kpi.id}', this.value)"></td>
            <td><div class="progress-bar"><div class="progress-bar__fill" style="width: ${kpi.progress}%"></div></div><span>${kpi.progress}%</span></td>
        `;
      row.addEventListener("click", (e) => {
        if (e.target.closest(".icon-btn--danger")) return;
        editKpi(kpi.id);
      });
      tbody.appendChild(row);
    });
  }
  function updateKpiComment(id, value) {
    if (!ensureEditMode("modifier un KPI")) {
      refreshReadOnlyLocks();
      return;
    }
    const kpi = appData.kpis.find((k) => k.id === id);
    if (kpi) {
      kpi.comments = value;
      saveData();
    }
  }
  function editKpi(id) {
    const kpi = appData.kpis.find((k) => k.id === id);
    if (!kpi) return;
    openModal("Modifier un KPI", [
      { name: "title", label: "Titre", value: kpi.title },
      { name: "description", label: "Description", value: kpi.description },
      { name: "comments", label: "Commentaire", value: kpi.comments },
      { name: "progress", label: "Progression (%)", type: "number", value: kpi.progress }
    ], (data) => {
      kpi.title = data.title;
      kpi.description = data.description;
      kpi.comments = data.comments;
      kpi.progress = parseInt(data.progress, 10) || 0;
      saveData();
      loadKpiTable();
    }, () => deleteKpi(id));
  }
  function addKpi() {
    openModal("Ajouter un KPI", [
      { name: "title", label: "Titre" },
      { name: "description", label: "Description" },
      { name: "comments", label: "Commentaire" },
      { name: "progress", label: "Progression (%)", type: "number", value: 0 }
    ], (data) => {
      if (!data.title) return false;
      const id = generateId("kpi");
      appData.kpis.push({
        id,
        title: data.title,
        description: data.description || "",
        comments: data.comments || "",
        progress: parseInt(data.progress, 10) || 0
      });
      saveData();
      loadKpiTable();
      updateDashboard();
    });
  }
  function deleteKpi(id) {
    if (!confirm("Supprimer cet indicateur ?")) return;
    appData.kpis = appData.kpis.filter((k) => k.id !== id);
    saveData();
    loadKpiTable();
  }
  window.loadKpiTable = loadKpiTable;
  window.updateKpiComment = updateKpiComment;
  window.editKpi = editKpi;
  window.addKpi = addKpi;
  window.deleteKpi = deleteKpi;
  window.toggleTheme = toggleTheme;

  // modules/utils.js
  function escapeHtml(str) {
    if (str === null || str === void 0) return "";
    return String(str).replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/"/g, "&quot;").replace(/'/g, "&#39;");
  }
  const CSV_DELIMITER = ";";
  const CSV_BOM = "\uFEFF";
  const EXCEL_XML_BOM = "\uFEFF";
  function encodeUtf16Le(text) {
    const buffer = new ArrayBuffer(text.length * 2);
    const view = new DataView(buffer);
    for (let i = 0; i < text.length; i++) {
      view.setUint16(i * 2, text.charCodeAt(i), true);
    }
    return buffer;
  }
  function normalizeExportFilename(label) {
    const safe = NORMALIZE(label || "export").replace(/[^a-z0-9]+/g, "_").replace(/^_+|_+$/g, "");
    return safe || "export";
  }
  function getExportDateStamp() {
    return (/* @__PURE__ */ new Date()).toISOString().slice(0, 10);
  }
  function cleanExportText(value) {
    if (value === null || value === void 0) return "";
    return String(value).replace(/\s+/g, " ").trim();
  }
  function escapeCsvValue(value) {
    const text = cleanExportText(value);
    if (!text) return "";
    const needsQuotes = text.includes(CSV_DELIMITER) || text.includes('"') || /[\r\n]/.test(text);
    const escaped = text.replace(/"/g, '""');
    return needsQuotes ? `"${escaped}"` : escaped;
  }
  function buildCsvContent(headers, rows) {
    const lines = [];
    lines.push(`sep=${CSV_DELIMITER}`);
    if (headers.length) lines.push(headers.map(escapeCsvValue).join(CSV_DELIMITER));
    rows.forEach((row) => {
      lines.push(row.map(escapeCsvValue).join(CSV_DELIMITER));
    });
    return CSV_BOM + lines.join("\r\n");
  }
  function downloadCsvFile(label, headers, rows) {
    // Bloquer l'export en mode démo
    if (DEMO_MODE) {
      showDemoBlockedMessage('L\'export Excel/CSV');
      return;
    }
    if (!rows.length) {
      showToast2("Aucune donnée à exporter", "warning");
      return;
    }
    try {
      const csvContent = buildCsvContent(headers, rows);
      const filename = `${normalizeExportFilename(label)}_${getExportDateStamp()}.csv`;
      const blob = new Blob([encodeUtf16Le(csvContent)], { type: "text/csv;charset=utf-16le;" });
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
      showToast2("Export Excel réussi", "success");
    } catch (error) {
      console.error("Erreur lors de l'export Excel:", error);
      showToast2("Impossible d'exporter. Vérifiez que des données existent.", "error");
    }
  }
  function buildMultiSectionCsvContent(sections) {
    const lines = [];
    lines.push(`sep=${CSV_DELIMITER}`);
    (sections || []).forEach((section, index) => {
      if (!section || !Array.isArray(section.headers)) return;
      if (index > 0) lines.push("");
      lines.push(escapeCsvValue(section.title || "Section"));
      lines.push(section.headers.map(escapeCsvValue).join(CSV_DELIMITER));
      (section.rows || []).forEach((row) => {
        lines.push(row.map(escapeCsvValue).join(CSV_DELIMITER));
      });
    });
    return CSV_BOM + lines.join("\r\n");
  }
  function downloadCsvContent(label, csvContent) {
    if (!csvContent) {
      showToast2("Aucune donnée à exporter", "warning");
      return;
    }
    try {
      const filename = `${normalizeExportFilename(label)}_${getExportDateStamp()}.csv`;
      const blob = new Blob([encodeUtf16Le(csvContent)], { type: "text/csv;charset=utf-16le;" });
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
      showToast2("Export Excel réussi", "success");
    } catch (error) {
      console.error("Erreur lors de l'export Excel:", error);
      showToast2("Impossible d'exporter. Vérifiez que des données existent.", "error");
    }
  }
  function stripInvalidXmlChars(text) {
    return String(text).replace(/[\x00-\x08\x0B\x0C\x0E-\x1F]/g, "");
  }
  function escapeExcelXml(value) {
    return stripInvalidXmlChars(value)
      .replace(/&/g, "&amp;")
      .replace(/</g, "&lt;")
      .replace(/>/g, "&gt;")
      .replace(/"/g, "&quot;")
      .replace(/'/g, "&#39;");
  }
  function sanitizeExcelString(value) {
    const text = String(value);
    return /^[=+\-@]/.test(text) ? `'${text}` : text;
  }
  function sanitizeSheetName(name) {
    const cleaned = cleanExportText(name || "Section").replace(/[\\/?*\[\]:]/g, " ").trim();
    const safe = cleaned || "Section";
    return safe.length > 31 ? safe.slice(0, 31) : safe;
  }
  function getExcelCellData(value) {
    if (value === null || value === void 0) return { type: "String", value: "" };
    if (typeof value === "number" && Number.isFinite(value)) return { type: "Number", value: String(value) };
    if (typeof value === "boolean") return { type: "Boolean", value: value ? "1" : "0" };
    if (value instanceof Date && Number.isFinite(value.getTime())) {
      return { type: "DateTime", value: value.toISOString() };
    }
    const text = sanitizeExcelString(String(value));
    return { type: "String", value: text };
  }
  function buildExcelRowXml(values, columnCount, isHeader) {
    const rowValues = Array.isArray(values) ? values : [];
    const padCount = Math.max(0, columnCount - rowValues.length);
    const padded = rowValues.concat(Array(padCount).fill(""));
    const cells = padded.map((cellValue) => {
      const cell = getExcelCellData(cellValue);
      const style = isHeader ? ' ss:StyleID="sHeader"' : "";
      return `<Cell${style}><Data ss:Type="${cell.type}">${escapeExcelXml(cell.value)}</Data></Cell>`;
    }).join("");
    return `<Row>${cells}</Row>`;
  }
  function buildExcelWorkbookXml(sections) {
    const usedNames = new Set();
    const workbookParts = [];
    workbookParts.push('<?xml version="1.0" encoding="UTF-8"?>');
    workbookParts.push('<?mso-application progid="Excel.Sheet"?>');
    workbookParts.push('<Workbook xmlns="urn:schemas-microsoft-com:office:spreadsheet" xmlns:o="urn:schemas-microsoft-com:office:office" xmlns:x="urn:schemas-microsoft-com:office:excel" xmlns:ss="urn:schemas-microsoft-com:office:spreadsheet" xmlns:html="http://www.w3.org/TR/REC-html40">');
    workbookParts.push('<Styles><Style ss:ID="sHeader"><Font ss:Bold="1"/><Interior ss:Color="#D9E1F2" ss:Pattern="Solid"/></Style></Styles>');
    (sections || []).forEach((section) => {
      if (!section || !Array.isArray(section.headers)) return;
      const baseName = sanitizeSheetName(section.title || "Section");
      let sheetName = baseName;
      if (usedNames.has(sheetName)) {
        let index = 2;
        while (index < 1000) {
          const suffix = ` (${index})`;
          const candidateBase = baseName.slice(0, 31 - suffix.length);
          const candidate = `${candidateBase}${suffix}`;
          if (!usedNames.has(candidate)) {
            sheetName = candidate;
            break;
          }
          index += 1;
        }
      }
      usedNames.add(sheetName);
      const rows = Array.isArray(section.rows) ? section.rows : [];
      const rowLengths = rows.map((row) => Array.isArray(row) ? row.length : 0);
      const columnCount = Math.max(1, section.headers.length || 0, ...rowLengths);
      const totalRows = (section.headers.length ? 1 : 0) + rows.length;
      const rowCount = Math.max(1, totalRows);
      workbookParts.push(`<Worksheet ss:Name="${escapeExcelXml(sheetName)}">`);
      workbookParts.push(`<Table ss:ExpandedColumnCount="${columnCount}" ss:ExpandedRowCount="${rowCount}" x:FullColumns="1" x:FullRows="1">`);
      if (section.headers.length) {
        workbookParts.push(buildExcelRowXml(section.headers, columnCount, true));
      }
      rows.forEach((row) => {
        workbookParts.push(buildExcelRowXml(row, columnCount, false));
      });
      workbookParts.push("</Table>");
      workbookParts.push('<WorksheetOptions xmlns="urn:schemas-microsoft-com:office:excel"><FreezePanes/><FrozenNoSplit/><SplitHorizontal>1</SplitHorizontal><TopRowBottomPane>1</TopRowBottomPane><ActivePane>2</ActivePane></WorksheetOptions>');
      workbookParts.push("</Worksheet>");
    });
    workbookParts.push("</Workbook>");
    return workbookParts.join("");
  }
  function downloadExcelWorkbook(label, sections) {
    if (!sections || !sections.length) {
      showToast2("Aucune donnée à exporter", "warning");
      return;
    }
    try {
      const xmlContent = buildExcelWorkbookXml(sections);
      const filename = `${normalizeExportFilename(label)}_${getExportDateStamp()}.xls`;
      const blob = new Blob([EXCEL_XML_BOM + xmlContent], { type: "application/vnd.ms-excel;charset=utf-8;" });
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
      showToast2("Export Excel réussi", "success");
    } catch (error) {
      console.error("Erreur lors de l'export Excel:", error);
      showToast2("Impossible d'exporter. Vérifiez que des données existent.", "error");
    }
  }
  function isEmptyExportRow(row) {
    if (!row) return true;
    if (row.querySelector(".nis2-empty")) return true;
    const cells = row.querySelectorAll("td");
    if (cells.length === 1 && cells[0].getAttribute("colspan")) return true;
    return cells.length === 0;
  }
  function exportTableToCsv(tableId, label) {
    const table = document.getElementById(tableId);
    if (!table) {
      showToast2("Table d'export introuvable", "error");
      return;
    }
    const headerCells = Array.from(table.querySelectorAll("thead th"));
    const headers = headerCells.map((cell) => cleanExportText(cell.textContent));
    const rows = Array.from(table.querySelectorAll("tbody tr"))
      .filter((row) => !isEmptyExportRow(row))
      .map((row) => Array.from(row.querySelectorAll("td")).map((cell) => cleanExportText(cell.textContent)));
    downloadCsvFile(label, headers, rows);
  }
  function getNis2ProgramExportData() {
    ensureNis2Plan();
    const headers = [
      "Domaine",
      "Pilier",
      "Urgence",
      "Objectif",
      "Action",
      "Type",
      "Phase",
      "Statut",
      "Responsable",
      "Progression (%)",
      "Budget 2026",
      "Budget 2027",
      "Budget 2028",
      "Budget total",
      "Catégorie budget"
    ];
    const rows = [];
    const domains = appData.nis2Plan.domains || [];
    domains.forEach((domain, index) => {
      const domainName = domain.title || `Domaine ${index + 1}`;
      const urgencyLabel = NIS2_URGENCY_LABELS[domain.urgency] || NIS2_URGENCY_LABELS.standard;
      const pillarLabel = domain.pillar ? `Pilier ${domain.pillar}` : "";
      const actions = Array.isArray(domain.actions) ? domain.actions : [];
      if (!actions.length) {
        rows.push([
          domainName,
          pillarLabel,
          urgencyLabel,
          domain.objective || "",
          "",
          "",
          "",
          "",
          "",
          "",
          "",
          "",
          "",
          "",
          ""
        ]);
        return;
      }
      actions.forEach((action) => {
        const statusKey = action.status || "todo";
        const statusLabel = (NIS2_STATUS_META[statusKey] || NIS2_STATUS_META.todo).label;
        const typeLabel = NIS2_ACTION_TYPE_LABELS[action.workType] || NIS2_ACTION_TYPE_LABELS.initiative;
        const phaseLabel = getNis2PhaseLabel(action.phase);
        const progress = Number.isFinite(action.progress)
          ? Math.max(0, Math.min(100, Math.round(action.progress)))
          : statusKey === "done" ? 100 : 0;
        const budget26 = parseBudgetValue(action.budget2026);
        const budget27 = parseBudgetValue(action.budget2027);
        const budget28 = parseBudgetValue(action.budget2028);
        const budgetTotal = getActionBudgetTotal(action);
        rows.push([
          domainName,
          pillarLabel,
          urgencyLabel,
          domain.objective || "",
          action.text || "",
          typeLabel,
          phaseLabel,
          statusLabel,
          action.owner || "",
          Number.isFinite(progress) ? progress : "",
          Number.isFinite(budget26) ? budget26 : "",
          Number.isFinite(budget27) ? budget27 : "",
          Number.isFinite(budget28) ? budget28 : "",
          Number.isFinite(budgetTotal) ? budgetTotal : "",
          action.budgetCategory || ""
        ]);
      });
    });
    return { headers, rows };
  }
  function exportNis2ProgramExcel(label) {
    const { headers, rows } = getNis2ProgramExportData();
    downloadCsvFile(label, headers, rows);
  }
  function getNis2SocleExportData() {
    const headers = ["Pilier", "Titre", "Sous-titre", "Élément", "Responsable"];
    const pillars = normalizeSoclePillars(appData.soclePillars);
    const rows = [];
    pillars.forEach((pillar, index) => {
      const pillarLabel = `Pilier ${index + 1}`;
      const items = Array.isArray(pillar.items) ? pillar.items : [];
      if (!items.length) {
        rows.push([
          pillarLabel,
          pillar.title || "",
          pillar.subtitle || "",
          "",
          pillar.owner || ""
        ]);
        return;
      }
      items.forEach((item) => {
        rows.push([
          pillarLabel,
          pillar.title || "",
          pillar.subtitle || "",
          item,
          pillar.owner || ""
        ]);
      });
    });
    return { headers, rows };
  }
  function exportNis2SocleExcel(label) {
    const { headers, rows } = getNis2SocleExportData();
    downloadCsvFile(label, headers, rows);
  }
  function exportCustomExcel(key, label) {
    switch (key) {
      case "nis2-program":
        exportNis2ProgramExcel(label);
        break;
      case "nis2-socle":
        exportNis2SocleExcel(label);
        break;
      default:
        showToast2("Export Excel non supporté pour cette section", "error");
        break;
    }
  }
  function setupExcelExportButtons() {
    document.querySelectorAll("[data-export-table]").forEach((btn) => {
      btn.addEventListener("click", () => {
        const tableId = btn.dataset.exportTable || "";
        const label = btn.dataset.exportName || "export";
        exportTableToCsv(tableId, label);
      });
    });
    document.querySelectorAll("[data-export-custom]").forEach((btn) => {
      btn.addEventListener("click", () => {
        const key = btn.dataset.exportCustom || "";
        const label = btn.dataset.exportName || "export";
        exportCustomExcel(key, label);
      });
    });
  }
  function getAllSectionExportData() {
    ensureNis2Plan();
    ensureCriticalAssets();
    ensureProjectRisks();
    const getOptionLabel = (options, value) => {
      if (!Array.isArray(options)) return value || "";
      const match = options.find((opt) => {
        if (!opt) return false;
        if (typeof opt === "object") return opt.value === value;
        return opt === value;
      });
      if (!match) return value || "";
      return typeof match === "object" ? match.label : match;
    };
    const formatDate = (value, fallback = "N/A") => {
      if (!value) return fallback;
      const date = new Date(value);
      return Number.isNaN(date.getTime()) ? value : date.toLocaleDateString();
    };
    const formatProjectRiskDate = (value) => {
      if (!value) return "—";
      const date = new Date(value);
      return Number.isNaN(date.getTime()) ? value : date.toLocaleDateString();
    };
    const getCapabilityLabel = (key) => {
      if (!key) return "";
      const cap = OPERATIONAL_CAPABILITIES.find((c) => c.key === key);
      return cap ? cap.label : String(key).replace(/_/g, " ");
    };
    const formatCapabilities = (value) => {
      if (Array.isArray(value)) {
        return value.map((item) => getCapabilityLabel(item)).filter(Boolean).join(", ");
      }
      return getCapabilityLabel(value);
    };
    const assetLookup = getCriticalAssetLookup();
    const getCriticalAssetLabels = (ids) => {
      const list = Array.isArray(ids) ? ids.filter(Boolean) : [];
      if (!list.length) return [];
      return list.map((id) => {
        const asset = assetLookup.get(id);
        return asset ? getCriticalAssetLabel(asset) : id;
      }).filter(Boolean);
    };
    const getActionLinkLabels = (action) => {
      if (!action) return [];
      const type = action.linkType || (action.controlId ? "control" : "");
      const ids = getActionLinkIds(action);
      if (type === "control") {
        const id = ids[0];
        if (!id) return [];
        const ctrl = appData.controls.find((c) => c.id === id);
        return [ctrl ? ctrl.reference : id];
      }
      if (type === "risk") {
        return ids.map((id) => {
          const risk = appData.risks.find((r) => r.id === id);
          return risk ? risk.title : id;
        }).filter(Boolean);
      }
      if (type === "nc") {
        const id = ids[0];
        if (!id) return [];
        const nc = appData.nonconformities.find((n) => n.id === id);
        return [nc ? nc.title : id];
      }
      if (type === "audit") {
        const id = ids[0];
        if (!id) return [];
        const audit = appData.audits.find((a) => a.id === id);
        return [audit ? audit.title : id];
      }
      if (type === "threat") {
        const id = ids[0];
        if (!id) return [];
        const threat = appData.threats.find((t) => t.id === id);
        return [threat ? threat.title : id];
      }
      if (type === "continuous") {
        return ids.length ? ids.map(() => "Amélioration continue") : [];
      }
      return [];
    };
    const sections = [];
    const sortedControls = [...appData.controls].sort((a, b) => String(a.reference || "").localeCompare(String(b.reference || ""), void 0, { numeric: true }));
    const controlRows = sortedControls.map((control) => {
      const actionIds = appData.actions
        .filter((a) => a.linkType === "control" && a.linkId === control.id)
        .map((a) => a.id)
        .join(", ");
      return [
        control.numero || "",
        control.title || "",
        control.measure || control.description || "",
        formatCapabilities(control.subcategory),
        getOptionLabel(CONTROL_STATUS_OPTIONS, control.status),
        actionIds || "Aucune",
        control.cmm || ""
      ];
    });
    sections.push({
      key: "controls",
      title: "Contrôles ISO27002",
      headers: ["Num", "Titre", "Description", "Capacité", "Statut", "Actions liées", "Niveau de maturité CMM"],
      rows: controlRows
    });
    const nis2Rows = (appData.nis2Controls || []).map((control) => [
      control.fonction || "",
      control.categorie || "",
      control.description || "",
      getOptionLabel(CONTROL_STATUS_OPTIONS, control.status),
      control.cmm || "",
      control.justification || ""
    ]);
    sections.push({
      key: "nis2-controls",
      title: "Contrôles NIS2",
      headers: ["Fonction", "Catégorie", "Description", "Statut", "Maturité CMM", "Justification"],
      rows: nis2Rows
    });
    const soaEntries = Array.isArray(appData.soa) ? appData.soa : [];
    const soaRows = sortedControls.map((control) => {
      const entry = soaEntries.find((e) => e.controlId === control.id);
      const applicable = entry ? entry.applicable : true;
      return [
        control.reference || "",
        control.title || "",
        applicable ? "Applicable" : "Non applicable",
        (entry && entry.justification) || ""
      ];
    });
    sections.push({
      key: "soa",
      title: "Déclaration d'applicabilité",
      headers: ["Référence", "Titre", "Applicabilité", "Justification"],
      rows: soaRows
    });
    const docEntries = Array.isArray(appData.documentReview) ? appData.documentReview : [];
    const docRows = [];
    DOCUMENT_REVIEW_ITEMS.forEach((item) => {
      if (item.type === "chapter") {
        docRows.push([item.title || "", "", "", ""]);
        return;
      }
      const entry = docEntries.find((e) => e.reference === item.reference) || {};
      docRows.push([
        item.reference || "",
        item.description || "",
        entry.justification || "",
        entry.tool || ""
      ]);
    });
    sections.push({
      key: "docReview",
      title: "Revue documentaire 27001",
      headers: ["Référence", "Exigence", "Justification", "Outil / preuve"],
      rows: docRows
    });
    const actionRows = (appData.actions || []).map((action) => {
      const linkLabels = getActionLinkLabels(action);
      const linkText = linkLabels.length ? linkLabels.join(", ") : "N/A";
      const assetLabels = getCriticalAssetLabels(action.criticalAssetIds);
      const assetText = assetLabels.length ? assetLabels.join(", ") : "Aucun";
      const progressRaw = parseInt(action.progress, 10);
      const progress = Number.isFinite(progressRaw) ? Math.max(0, Math.min(100, progressRaw)) : 0;
      return [
        action.id || "",
        action.title || "",
        linkText,
        assetText,
        getOptionLabel(ACTION_PRIORITY_OPTIONS, action.priority),
        `${progress}%`,
        action.dueDate ? formatDate(action.dueDate, "N/A") : "N/A",
        action.assignedTo ? action.assignedTo : "N/A",
        Array.isArray(action.comments) ? action.comments.length : 0
      ];
    });
    sections.push({
      key: "actions",
      title: "Plan d'actions",
      headers: ["ID", "Action", "Lien", "Actifs critiques", "Priorité", "Avancement", "Échéance", "Responsable", "Commentaires"],
      rows: actionRows
    });
    const riskRows = (appData.risks || []).map((risk) => {
      const impactLabel = IMPACT_LEVEL_LABELS[risk.impact] || risk.impact || "";
      const probLabel = PROBABILITY_LEVEL_LABELS[risk.probability] || risk.probability || "";
      const score = Number.isFinite(risk.score)
        ? risk.score
        : (parseInt(risk.impact, 10) || 0) * (parseInt(risk.probability, 10) || 0);
      const level = risk.level || calculateRiskLevel(score);
      const linkedActions = appData.actions.filter((a) => actionLinksToRisk(a, risk.id));
      const actionsText = linkedActions.length
        ? linkedActions.map((a) => a.title || a.id).join(", ")
        : "Aucune";
      const assetLabels = getCriticalAssetLabels(risk.criticalAssetIds);
      const assetText = assetLabels.length ? assetLabels.join(", ") : "Aucun";
      const codirDate = risk.codirDate ? formatDate(risk.codirDate, risk.codirDate) : "";
      return [
        risk.id || "",
        risk.title || "",
        risk.description || "",
        impactLabel,
        probLabel,
        score,
        level,
        actionsText,
        assetText,
        risk.treatment || "",
        codirDate,
        Array.isArray(risk.comments) ? risk.comments.length : 0
      ];
    });
    sections.push({
      key: "risks",
      title: "Risques SMSI",
      headers: ["ID", "Risque", "Description", "Impact", "Probabilité", "Score", "Niveau", "Actions liées", "Actifs critiques", "Traitement", "Date CODIR", "Commentaires"],
      rows: riskRows
    });
    const threatRows = (appData.threats || []).map((threat) => {
      const impactLabel = IMPACT_LEVEL_LABELS[threat.impact] || threat.impact || "";
      const probLabel = PROBABILITY_LEVEL_LABELS[threat.likelihood] || threat.likelihood || "";
      const score = Number.isFinite(threat.score)
        ? threat.score
        : (parseInt(threat.impact, 10) || 0) * (parseInt(threat.likelihood, 10) || 0);
      const level = threat.level || calculateRiskLevel(score);
      const linkedActionIds = appData.actions
        .filter((a) => a.linkType === "threat" && a.linkId === threat.id)
        .map((a) => a.id)
        .join(", ");
      const assetLabels = getCriticalAssetLabels(threat.criticalAssetIds);
      const assetText = assetLabels.length ? assetLabels.join(", ") : "Aucun";
      return [
        threat.title || "",
        threat.description || "",
        impactLabel,
        probLabel,
        score,
        level,
        THREAT_STATUS_LABELS[threat.status] || threat.status || "",
        linkedActionIds || "Aucune",
        assetText
      ];
    });
    sections.push({
      key: "threats",
      title: "Menaces",
      headers: ["Titre", "Description", "Impact", "Probabilité", "Score", "Niveau", "Statut", "Actions liées", "Actifs critiques"],
      rows: threatRows
    });
    const auditMap = new Map((appData.audits || []).map((audit) => [audit.id, audit.title || audit.id]));
    const ncRows = (appData.nonconformities || []).map((nc) => {
      const relatedActions = (appData.actions || []).filter((a) => (a.ncIds || []).includes(nc.id) || (a.linkType === "nc" && a.linkId === nc.id));
      const actionsText = relatedActions.length ? relatedActions.map((a) => a.title || a.id).join(", ") : "Aucune";
      const auditText = auditMap.get(nc.auditId) || "Aucun";
      const assetLabels = getCriticalAssetLabels(nc.criticalAssetIds);
      const assetText = assetLabels.length ? assetLabels.join(", ") : "Aucun";
      return [
        nc.title || "",
        nc.type || "",
        nc.status || "",
        nc.detectionDate ? formatDate(nc.detectionDate, "N/A") : "N/A",
        nc.assignedTo ? nc.assignedTo : "N/A",
        auditText,
        actionsText,
        assetText,
        nc.description || ""
      ];
    });
    sections.push({
      key: "nonconformities",
      title: "Non-conformités",
      headers: ["Titre", "Type", "Statut", "Date détection", "Responsable", "Audit lié", "Actions liées", "Actifs critiques", "Opérations"],
      rows: ncRows
    });
    const auditRows = (appData.audits || []).map((audit) => {
      const linkedActionIds = (appData.actions || []).filter((a) => a.linkType === "audit" && a.linkId === audit.id).map((a) => a.id).join(", ");
      return [
        audit.title || "",
        audit.type || "",
        audit.scope || "",
        audit.plannedDate ? formatDate(audit.plannedDate, "N/A") : "N/A",
        audit.auditor ? audit.auditor : "N/A",
        audit.status || "",
        linkedActionIds || "Aucune"
      ];
    });
    sections.push({
      key: "audits",
      title: "Audits",
      headers: ["Titre", "Type", "Périmètre", "Date planifiée", "Auditeur", "Statut", "Actions liées"],
      rows: auditRows
    });
    const reviewRows = (appData.reviews || []).map((review) => [
      review.date ? formatDate(review.date, "N/A") : "N/A",
      review.participants || "",
      review.inputs || "",
      review.decisions || "",
      Array.isArray(review.comments) ? review.comments.length : 0
    ]);
    sections.push({
      key: "reviews",
      title: "Revues de Direction",
      headers: ["Date", "Participants", "Entrées", "Décisions", "Commentaires"],
      rows: reviewRows
    });
    const stakeholderRows = (appData.stakeholders || []).map((st) => [
      st.name || "",
      st.role || "",
      st.contact || "",
      st.notes || ""
    ]);
    sections.push({
      key: "stakeholders",
      title: "Parties prenantes",
      headers: ["Nom", "Rôle", "Contact", "Commentaires"],
      rows: stakeholderRows
    });
    const documentRows = (appData.documents || []).map((doc) => [
      doc.category || "",
      doc.type || "",
      doc.tool || "",
      doc.link || ""
    ]);
    sections.push({
      key: "documents",
      title: "Structure documentaire",
      headers: ["Niveau", "Type", "Outil", "Lien"],
      rows: documentRows
    });
    const kpiRows = (appData.kpis || []).map((kpi) => {
      const progressRaw = parseInt(kpi.progress, 10);
      const progress = Number.isFinite(progressRaw) ? Math.max(0, Math.min(100, progressRaw)) : 0;
      return [
        kpi.title || "",
        kpi.description || "",
        kpi.comments || "",
        `${progress}%`
      ];
    });
    sections.push({
      key: "kpis",
      title: "KPI",
      headers: ["Titre", "Description", "Commentaire", "Progression"],
      rows: kpiRows
    });
    const nis2ProgramData = getNis2ProgramExportData();
    sections.push({
      key: "nis2-program",
      title: "Programme NIS2",
      headers: nis2ProgramData.headers,
      rows: nis2ProgramData.rows
    });
    const nis2SocleData = getNis2SocleExportData();
    sections.push({
      key: "nis2-socle",
      title: "Socle de sécurité",
      headers: nis2SocleData.headers,
      rows: nis2SocleData.rows
    });
    const assetRows = (appData.criticalAssets || []).slice().sort((a, b) => (a.priority || 0) - (b.priority || 0)).map((asset, index) => {
      const tags = Array.isArray(asset.productTags) ? asset.productTags.filter(Boolean) : [];
      const productParts = [];
      if (asset.productCode) productParts.push(asset.productCode);
      if (asset.productName) productParts.push(asset.productName);
      let productLabel = productParts.join(" - ");
      if (tags.length) {
        const tagLabel = tags.join(", ");
        productLabel = productLabel ? `${productLabel} (Tags: ${tagLabel})` : `Tags: ${tagLabel}`;
      }
      const availability = asset.availability ? asset.availability : "—";
      const availabilityLabel = asset.availabilityNote ? `${availability} (${asset.availabilityNote})` : availability;
      const conf = asset.confidentiality || "—";
      const impact = asset.impact || "—";
      const dacp = asset.dacpSensitive || "—";
      const mfaParts = [];
      if (asset.mfaStatus) mfaParts.push(asset.mfaStatus);
      if (asset.mfaDetail) mfaParts.push(asset.mfaDetail);
      const mfaLabel = mfaParts.length ? mfaParts.join(" - ") : "—";
      return [
        asset.priority || index + 1,
        asset.beneficiary || "",
        productLabel || "",
        asset.rto || "—",
        asset.rpo || "—",
        availabilityLabel,
        conf,
        impact,
        dacp,
        mfaLabel
      ];
    });
    sections.push({
      key: "critical-assets",
      title: "Actifs critiques",
      headers: ["Priorité", "Bénéficiaire", "Produit", "RTO", "RPO", "Disponibilité", "Conf.", "Impact", "DACP sens.", "Obligation MFA"],
      rows: assetRows
    });
    const projectRows = (appData.projectRisks || []).map((risk) => {
      const statusMeta = getProjectRiskStatusMeta(risk.status);
      const revision = risk.revisionNumber || "";
      return [
        risk.projectName || "",
        risk.beneficiary || "",
        risk.reportNumber || "",
        revision,
        risk.description || "",
        risk.analysisLink || "—",
        formatProjectRiskDate(risk.completionDate),
        statusMeta.label
      ];
    });
    sections.push({
      key: "project-risks",
      title: "Risques projets",
      headers: ["Nom du Projet", "Bénéficiaire", "Numéro du rapport", "Nombre de révision", "Description projet", "Lien vers l'analyse des risques", "Date de réalisation", "Statut"],
      rows: projectRows
    });
    return sections;
  }
  function openGlobalExcelExportModal() {
    const sections = getAllSectionExportData();
    if (!sections.length) {
      showToast2("Aucune donnée à exporter", "warning");
      return;
    }
    const fields = sections.map((section) => ({
      name: `export_${section.key}`,
      label: `${section.title} (${section.rows.length})`,
      type: "checkbox",
      value: true,
      allowInReadOnly: true
    }));
    openModal("Export global Excel", fields, (data) => {
      const selected = sections.filter((section) => data[`export_${section.key}`]);
      if (!selected.length) {
        showToast2("Sélectionnez au moins une section.", "warning");
        return false;
      }
      const label = appData.title ? `${appData.title}_export_global` : "export_global";
      downloadExcelWorkbook(label, selected);
      return true;
    }, null, {
      saveLabel: "Exporter Excel",
      cancelLabel: "Annuler",
      allowReadOnly: true,
      message: "Sélectionnez les sections à exporter. Chaque section sera un onglet Excel."
    });
    const modalBody = document.getElementById("modalBody");
    if (modalBody) {
      const toolbar = document.createElement("div");
      toolbar.className = "settings-actions modal__toolbar";
      const selectAllBtn = document.createElement("button");
      selectAllBtn.type = "button";
      selectAllBtn.className = "btn btn--outline btn--sm";
      selectAllBtn.textContent = "Tout sélectionner";
      const clearBtn = document.createElement("button");
      clearBtn.type = "button";
      clearBtn.className = "btn btn--outline btn--sm";
      clearBtn.textContent = "Tout désélectionner";
      toolbar.append(selectAllBtn, clearBtn);
      const messageEl = modalBody.querySelector(".modal__message");
      if (messageEl && messageEl.parentNode) {
        messageEl.parentNode.insertBefore(toolbar, messageEl.nextSibling);
      } else if (modalBody.firstChild) {
        modalBody.insertBefore(toolbar, modalBody.firstChild);
      } else {
        modalBody.appendChild(toolbar);
      }
      const setAll = (checked) => {
        fields.forEach((field) => {
          const input = document.getElementById(`modal-input-${field.name}`);
          if (input && input.type === "checkbox") input.checked = checked;
        });
      };
      selectAllBtn.addEventListener("click", () => setAll(true));
      clearBtn.addEventListener("click", () => setAll(false));
    }
  }
  function exportAllSectionsExcel() {
    try {
      openGlobalExcelExportModal();
    } catch (error) {
      console.error("Erreur lors de l'export global Excel:", error);
      showToast2("Impossible d'exporter. Vérifiez que des données existent.", "error");
    }
  }
  function exportData() {
    // Bloquer l'export en mode démo
    if (DEMO_MODE) {
      showDemoBlockedMessage('L\'export des données');
      return;
    }
    try {
      ensureNis2Plan();
      ensureCriticalAssets();
      ensureProjectRisks();
      const data = {
        controls: appData.controls,
        actions: appData.actions,
        criticalAssets: appData.criticalAssets,
        criticalAssetsMeta: appData.criticalAssetsMeta,
        projectRisks: appData.projectRisks,
        risks: appData.risks,
        threats: appData.threats,
        nonconformities: appData.nonconformities,
        audits: appData.audits,
        kpis: appData.kpis,
        reviews: appData.reviews,
        stakeholders: appData.stakeholders,
        documents: appData.documents,
        documentReview: appData.documentReview,
        nis2Controls: appData.nis2Controls,
        nis2Plan: normalizeNis2Plan(appData.nis2Plan),
        soclePillars: appData.soclePillars,
        targetMaturity: appData.targetMaturity,
        soa: appData.soa,
        nextActionId: appData.nextActionId,
        title: appData.title,
        exportDate: (/* @__PURE__ */ new Date()).toISOString(),
        version: "1.0"
      };
      const blob = new Blob([JSON.stringify(data, null, 2)], { type: "application/json" });
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      const cryptoObj = (typeof crypto !== "undefined" && crypto) || (typeof window !== "undefined" ? window.crypto : void 0);
      const exportId = cryptoObj && typeof cryptoObj.randomUUID === "function" ? cryptoObj.randomUUID() : Math.random().toString(36).slice(2);
      a.download = `SMSI_Export_${(/* @__PURE__ */ new Date()).toISOString().slice(0, 10)}_${exportId}.smsi`;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
      showToast2("Export r\xE9ussi", "success");
    } catch (error) {
      console.error("Erreur lors de l'export:", error);
      showToast2("Impossible d'exporter les données.", "error");
    }
  }
  function formatImportDate(data) {
    const candidates = [
      data?.exportDate,
      data?.lastUpdated,
      data?.updatedAt,
      data?.lastModified
    ];
    for (const candidate of candidates) {
      if (!candidate) continue;
      const ts = Date.parse(candidate);
      if (Number.isFinite(ts)) return new Date(ts).toLocaleString();
    }
    return "Inconnue";
  }
  function buildImportSummary(data, sourceLabel) {
    const controlsCount = Array.isArray(data?.controls) ? data.controls.length : 0;
    const actionsCount = Array.isArray(data?.actions) ? data.actions.length : 0;
    const risksCount = Array.isArray(data?.risks) ? data.risks.length : 0;
    const projectRisksCount = Array.isArray(data?.projectRisks) ? data.projectRisks.length : 0;
    const nis2ControlsCount = Array.isArray(data?.nis2Controls) ? data.nis2Controls.length : 0;
    const hasNis2Plan = !!data?.nis2Plan;
    const dateLabel = formatImportDate(data);
    const lines = [
      `Source: ${sourceLabel}`,
      `Contrôles ISO: ${controlsCount}`,
      `Contrôles NIS2: ${nis2ControlsCount}`,
      `Actions: ${actionsCount}`,
      `Risques: ${risksCount}`,
      `Risques projets: ${projectRisksCount}`,
      `Plan NIS2: ${hasNis2Plan ? 'Oui (avec budgets)' : 'Non (sera initialisé)'}`,
      `Dernière MAJ: ${dateLabel}`,
      ""
    ];
    if (!hasNis2Plan && nis2ControlsCount > 0) {
      lines.push("⚠️ Ancien format détecté: le plan NIS2 et les budgets");
      lines.push("   seront initialisés avec les valeurs par défaut.");
      lines.push("");
    }
    lines.push("Attention: cette opération va écraser les données locales.");
    return lines.join("\n");
  }
  function applyImportedData(data) {
    try {
      addHistorySnapshot("Backup avant import", { tag: "auto" });
    } catch (err) {
      console.warn("Snapshot de sécurité non créé", err);
    }
    // Normaliser les contrôles (gestion de l'ancien format avec selected/justification)
    appData.controls = normalizeControls((data.controls || []).map(c => {
      const { selected, justification, ...rest } = c;
      if (typeof rest.justificationControl === "undefined") rest.justificationControl = "";
      return rest;
    }));

    // Normaliser SOA (compatibilité avec ancien format à 3 champs)
    if (Array.isArray(data.soa)) {
      appData.soa = normalizeSoa(data.soa);
    } else {
      appData.soa = normalizeSoa((data.controls || []).map(c => ({
        controlId: c.id,
        applicable: c.selected !== undefined ? c.selected : true,
        justification: c.justification || ""
      })));
    }

    // Normaliser toutes les autres sections avec compatibilité ascendante
    appData.actions = normalizeActions(data.actions || []);
    appData.criticalAssets = normalizeCriticalAssets(data.criticalAssets || []);
    appData.criticalAssetsMeta = normalizeCriticalAssetsMeta(data.criticalAssetsMeta);
    appData.projectRisks = normalizeProjectRisks(data.projectRisks || []);
    appData.risks = normalizeRisks(data.risks || []);
    ensureRiskIds();
    appData.threats = normalizeThreats(data.threats || []);
    appData.nonconformities = normalizeNonconformities(data.nonconformities || []);
    appData.audits = normalizeAudits(data.audits || []);
    appData.kpis = normalizeKpis(data.kpis || []);
    appData.reviews = normalizeReviews(data.reviews || []);
    appData.stakeholders = normalizeStakeholders(data.stakeholders || []);
    appData.documents = normalizeDocuments(data.documents || []);
    appData.documentReview = createDefaultDocumentReview(data.documentReview || []);
    const tm = data.targetMaturity || {};
    appData.targetMaturity = {
      iso27002: typeof tm.iso27002 === "number" ? tm.iso27002 : parseInt(tm.iso27002, 10) || 3,
      nis2: buildNis2Maturity(tm.nis2 || {}),
      nis2Target: typeof tm.nis2Target === "number" ? tm.nis2Target : parseInt(tm.nis2Target, 10) || 3
    };
    const nis2PlanDefaults = getEmptyNis2Plan();
    let nis2PlanMigrated = false;
    if (data.nis2Plan) {
      const plan = data.nis2Plan;
      appData.nis2Plan = {
        phases: Array.isArray(plan.phases) && plan.phases.length ? plan.phases : nis2PlanDefaults.phases,
        maturity: {
          labels: Array.isArray(plan.maturity?.labels) && plan.maturity.labels.length ? plan.maturity.labels : nis2PlanDefaults.maturity.labels,
          current: Array.isArray(plan.maturity?.current) && plan.maturity.current.length ? plan.maturity.current : nis2PlanDefaults.maturity.current,
          target2027: typeof plan.maturity?.target2027 === "number" ? plan.maturity.target2027 : nis2PlanDefaults.maturity.target2027,
          target2028: typeof plan.maturity?.target2028 === "number" ? plan.maturity.target2028 : nis2PlanDefaults.maturity.target2028
        },
        statuses: Array.isArray(plan.statuses) ? plan.statuses : nis2PlanDefaults.statuses,
        budgetBlocks: Array.isArray(plan.budgetBlocks) ? plan.budgetBlocks : nis2PlanDefaults.budgetBlocks,
        customBlocks: Array.isArray(plan.customBlocks) ? plan.customBlocks : nis2PlanDefaults.customBlocks,
        domains: Array.isArray(plan.domains) ? plan.domains : nis2PlanDefaults.domains
      };
    } else {
      // Migration automatique des anciens formats sans nis2Plan
      nis2PlanMigrated = true;
      appData.nis2Plan = cloneDefaultNis2Plan({ includeActions: true });

      // Tenter de récupérer la maturité depuis nis2Controls si disponible
      if (Array.isArray(data.nis2Controls) && data.nis2Controls.length > 0) {
        // Mapper les CMM des contrôles NIS2 vers des valeurs numériques
        const cmmToValue = { 'Initial': 1, 'Répétable': 2, 'Défini': 3, 'Géré': 4, 'Optimisé': 5 };
        const maturityValues = data.nis2Controls.map(ctrl => {
          const cmmVal = cmmToValue[ctrl.cmm] || 0;
          return cmmVal;
        });
        // Calculer une moyenne globale pour target2027/2028
        const avgMaturity = maturityValues.length > 0
          ? Math.round(maturityValues.reduce((a, b) => a + b, 0) / maturityValues.length)
          : 0;
        appData.nis2Plan.maturity.target2027 = Math.min(avgMaturity + 1, 5);
        appData.nis2Plan.maturity.target2028 = Math.min(avgMaturity + 2, 5);
      }

      console.log('[CSI Migration] Ancien format détecté - Plan NIS2 créé avec valeurs par défaut');
    }
    appData.soclePillars = normalizeSoclePillars(data.soclePillars);
    ensureNis2Plan();
    ensureCriticalAssets();
    normalizeCriticalAssetLinks(appData.actions);
    normalizeCriticalAssetLinks(appData.risks);
    normalizeCriticalAssetLinks(appData.threats);
    normalizeCriticalAssetLinks(appData.nonconformities);
    const defaultNis2 = loadNis2Controls();
    if (Array.isArray(data.nis2Controls)) {
      appData.nis2Controls = defaultNis2.map(d => {
        const found = data.nis2Controls.find(c => c.id === d.id) || {};
        return { ...d, ...found };
      });
    } else {
      appData.nis2Controls = defaultNis2;
    }
    appData.nextActionId = data.nextActionId || 0;
    appData.title = data.title || appData.title;
    updateAppTitle();
    warnMissingSubcategories();
    saveData();
    updateDashboard();
    loadKpiTable();
    if (nis2PlanMigrated) {
      showToast2("Import réussi - Plan NIS2 initialisé (ancien format)", "info");
      setTimeout(() => {
        showToast2("Personnalisez le plan NIS2 et les budgets dans l'onglet Plan NIS2", "info");
      }, 2500);
    } else {
      showToast2("Import réussi", "success");
    }
    setTimeout(() => {
      location.reload();
    }, nis2PlanMigrated ? 4500 : 100);
  }
  function importData() {
    // Bloquer l'import en mode démo
    if (DEMO_MODE) {
      showDemoBlockedMessage('L\'import des données');
      return;
    }
    if (!ensureEditMode("importer des données")) return;
    const fileInput = document.getElementById("fileInput");
    fileInput.click();
    fileInput.onchange = function(event) {
      const file = event.target.files[0];
      if (!file) return;
      const reader = new FileReader();
      reader.onload = function(e) {
        try {
          const data = JSON.parse(e.target.result);
          if (!data.controls || !Array.isArray(data.controls)) throw new Error("Format de fichier invalide");
          const summary = buildImportSummary(data, file.name ? `Fichier ${file.name}` : "Fichier importé");
          openModal("Confirmer l'import", [
            { name: "summary", label: "Résumé avant import", type: "textarea", value: summary, readOnly: true }
          ], () => {
            try {
              applyImportedData(data);
            } catch (error) {
              console.error("Erreur lors de l'import:", error);
              showToast2("Impossible d'importer le fichier. Vérifiez son format.", "error");
              return false;
            }
          }, null, { saveLabel: "Importer", cancelLabel: "Annuler" });
        } catch (error) {
          console.error("Erreur lors de l'import:", error);
          showToast2("Impossible d'importer le fichier. Vérifiez son format.", "error");
        } finally {
          fileInput.value = "";
        }
      };
      reader.readAsText(file);
    };
  }
  function runTestSuite() {
    showToast2("Exécution des tests en cours...", "info");

    const tests = [];
    let passed = 0;
    let failed = 0;

    // Helper pour ajouter un test
    function addTest(name, fn) {
      tests.push({ name, fn });
    }

    // Test 1: Vérification de la structure appData
    addTest("Structure appData valide", () => {
      return appData && Array.isArray(appData.controls) && Array.isArray(appData.soa);
    });

    // Test 2: Fonction isControlNonApplicable existe
    addTest("Fonction isControlNonApplicable existe", () => {
      return typeof isControlNonApplicable === "function";
    });

    // Test 3: Fonction getSoaJustification existe
    addTest("Fonction getSoaJustification existe", () => {
      return typeof getSoaJustification === "function";
    });

    // Test 4: Fonction navigateToSoaControl existe
    addTest("Fonction navigateToSoaControl existe", () => {
      return typeof navigateToSoaControl === "function";
    });

    // Test 5: Test isControlNonApplicable avec contrôle applicable
    addTest("isControlNonApplicable retourne false pour contrôle applicable", () => {
      // Créer une entrée SoA temporaire pour le test
      const testControlId = "TEST_CONTROL_APPLICABLE";
      const originalSoa = [...appData.soa];
      appData.soa.push({ controlId: testControlId, applicable: true, justification: "" });
      const result = isControlNonApplicable(testControlId) === false;
      appData.soa = originalSoa;
      return result;
    });

    // Test 6: Test isControlNonApplicable avec contrôle non applicable
    addTest("isControlNonApplicable retourne true pour contrôle non applicable", () => {
      const testControlId = "TEST_CONTROL_NON_APPLICABLE";
      const originalSoa = [...appData.soa];
      appData.soa.push({ controlId: testControlId, applicable: false, justification: "Test" });
      const result = isControlNonApplicable(testControlId) === true;
      appData.soa = originalSoa;
      return result;
    });

    // Test 7: Test getSoaJustification
    addTest("getSoaJustification retourne la justification", () => {
      const testControlId = "TEST_CONTROL_JUSTIF";
      const originalSoa = [...appData.soa];
      appData.soa.push({ controlId: testControlId, applicable: false, justification: "Ma justification test" });
      const result = getSoaJustification(testControlId) === "Ma justification test";
      appData.soa = originalSoa;
      return result;
    });

    // Test 8: Contrôles ISO existent
    addTest("Contrôles ISO 27002 présents", () => {
      return appData.controls.length > 0;
    });

    // Test 9: Chaque contrôle a un ID
    addTest("Chaque contrôle ISO a un ID", () => {
      return appData.controls.every(c => c.id && typeof c.id === "string");
    });

    // Test 10: Entrées SoA correspondent aux contrôles
    addTest("Entrées SoA valides (controlId existe)", () => {
      const controlIds = appData.controls.map(c => c.id);
      return appData.soa.every(s => controlIds.includes(s.controlId));
    });

    // Test 11: Navigation module dashboard fonctionne
    addTest("Module dashboard existe", () => {
      return document.getElementById("dashboard") !== null;
    });

    // Test 12: Module contrôles existe
    addTest("Module contrôles ISO existe", () => {
      return document.getElementById("controls") !== null;
    });

    // Test 13: Module SoA existe
    addTest("Module SoA existe", () => {
      return document.getElementById("soa") !== null;
    });

    // Test 14: Table contrôles body existe
    addTest("Table contrôles body existe", () => {
      return document.getElementById("controlsTableBody") !== null;
    });

    // Test 15: Table SoA body existe
    addTest("Table SoA body existe", () => {
      return document.getElementById("soaTableBody") !== null;
    });

    // Test 16: Fonction openModal existe
    addTest("Fonction openModal existe", () => {
      return typeof openModal === "function";
    });

    // Test 17: Fonction closeModal existe
    addTest("Fonction closeModal existe", () => {
      return typeof closeModal === "function";
    });

    // Test 18: Fonction loadControlsTable existe
    addTest("Fonction loadControlsTable existe", () => {
      return typeof loadControlsTable === "function";
    });

    // Test 19: Fonction loadSoaTable existe
    addTest("Fonction loadSoaTable existe", () => {
      return typeof loadSoaTable === "function";
    });

    // Test 20: Fonction escapeHtml existe
    addTest("Fonction escapeHtml existe", () => {
      return typeof escapeHtml === "function";
    });

    // Test 21: Fonction editSoaRisks existe
    addTest("Fonction editSoaRisks existe", () => {
      return typeof editSoaRisks === "function";
    });

    // Test 22: Fonction renderSoaRiskBadges existe
    addTest("Fonction renderSoaRiskBadges existe", () => {
      return typeof renderSoaRiskBadges === "function";
    });

    // Test 23: Entrées SoA ont linkedRiskIds
    addTest("Entrées SoA supportent linkedRiskIds", () => {
      if (appData.soa.length === 0) return true;
      // Vérifier qu'au moins une entrée a linkedRiskIds défini
      return appData.soa.some(s => Array.isArray(s.linkedRiskIds));
    });

    // Test 24: renderSoaRiskBadges retourne HTML valide
    addTest("renderSoaRiskBadges retourne HTML pour risques vides", () => {
      const result = renderSoaRiskBadges([]);
      return result.includes("Aucun");
    });

    // Exécuter les tests
    setTimeout(() => {
      tests.forEach(test => {
        try {
          if (test.fn()) {
            passed++;
          } else {
            failed++;
            console.error(`Test échoué: ${test.name}`);
          }
        } catch (e) {
          failed++;
          console.error(`Test en erreur: ${test.name}`, e);
        }
      });

      const total = tests.length;
      if (failed === 0) {
        showToast2(`Tests terminés : ${passed}/${total} réussis`, "success");
      } else {
        showToast2(`Tests terminés : ${passed}/${total} réussis, ${failed} échoué(s)`, "warning");
      }
    }, 500);
  }
  function showToast2(message, type = "info") {
    const container = document.getElementById("toastContainer");
    if (!container) return;
    const toast = document.createElement("div");
    toast.className = `toast toast--${type}`;
    const icons = { success: "check-circle", error: "exclamation-triangle", warning: "exclamation-circle", info: "info-circle" };
    toast.innerHTML = `<div class="toast__icon"><i class="fas fa-${icons[type]}"></i></div><div class="toast__content"><div class="toast__message">${message}</div></div><button class="toast__close"><i class="fas fa-times"></i></button>`;
    container.appendChild(toast);
    setTimeout(() => {
      toast.classList.add("toast--visible");
    }, 10);
    const closeBtn = toast.querySelector(".toast__close");
    closeBtn.addEventListener("click", () => {
      toast.classList.remove("toast--visible");
      setTimeout(() => {
        container.removeChild(toast);
      }, 300);
    });
    setTimeout(() => {
      if (container.contains(toast)) {
        toast.classList.remove("toast--visible");
        setTimeout(() => {
          if (container.contains(toast)) container.removeChild(toast);
        }, 300);
      }
    }, 5e3);
  }
  let missingSubcategoryWarned = false;
  function warnMissingSubcategories() {
    const missing = appData.controls.filter(c =>
      !c.subcategory || (Array.isArray(c.subcategory) && c.subcategory.length === 0)
    );
    if (missing.length > 0 && !missingSubcategoryWarned) {
      missingSubcategoryWarned = true;
      showToast2(`Sous-catégorie manquante pour ${missing.length} contrôle(s)`, "warning");
    }
  }
  function hideLoadingSpinner2() {
    const spinner = document.getElementById("loadingSpinner");
    if (spinner) spinner.classList.add("hidden");
  }

  function updateAppTitle() {
    const el = document.getElementById("appTitle");
    if (el) {
      el.innerHTML = '<i class="fas fa-shield-alt"></i> ' + escapeHtml(appData.title);
    }
  }

  function openManual() {
    const navItems = document.querySelectorAll(".nav__item");
    const modules = document.querySelectorAll(".module");
    navItems.forEach(nav => nav.classList.remove("nav__item--active"));
    modules.forEach(m => m.classList.remove("module--active"));
    const manual = document.getElementById("manual");
    if (manual) manual.classList.add("module--active");
  }

  function openBugReport() {
    const presentationUrl = "https://cyber-assistant.com";
    const win = window.open(presentationUrl, "_blank");
    if (win) win.opener = null;
  }

  function loadSyncConfig() {
    let cfg = {};
    try {
      cfg = JSON.parse(localStorage.getItem(SYNC_CONFIG_KEY)) || {};
    } catch (e) {
      cfg = {};
    }
    if (!cfg || typeof cfg !== "object") cfg = {};
    if (cfg.password) {
      try {
        localStorage.setItem(SYNC_PASSWORD_PERSIST_KEY, cfg.password);
      } catch (e) {
        // Ignore storage errors
      }
      cfg.rememberPassword = true;
      delete cfg.password;
      try {
        localStorage.setItem(SYNC_CONFIG_KEY, JSON.stringify(cfg));
      } catch (e) {
        // Ignore storage errors
      }
    }
    if (typeof cfg.rememberPassword !== "boolean") {
      let hasPersist = false;
      try {
        hasPersist = !!localStorage.getItem(SYNC_PASSWORD_PERSIST_KEY);
      } catch (e) {
        hasPersist = false;
      }
      cfg.rememberPassword = hasPersist;
    }
    return cfg;
  }

  function getStoredSyncPassword(cfg = null) {
    let sessionPwd = "";
    try {
      sessionPwd = sessionStorage.getItem(SYNC_PASSWORD_SESSION_KEY) || "";
    } catch (e) {
      sessionPwd = "";
    }
    if (sessionPwd) return sessionPwd;
    const currentCfg = cfg || loadSyncConfig();
    if (currentCfg.rememberPassword) {
      try {
        return localStorage.getItem(SYNC_PASSWORD_PERSIST_KEY) || "";
      } catch (e) {
        return "";
      }
    }
    return "";
  }

  function saveSyncPassword(value, remember) {
    try {
      if (value) sessionStorage.setItem(SYNC_PASSWORD_SESSION_KEY, value);
      else sessionStorage.removeItem(SYNC_PASSWORD_SESSION_KEY);
    } catch (e) {
      // Ignore storage errors
    }
    try {
      if (remember && value) localStorage.setItem(SYNC_PASSWORD_PERSIST_KEY, value);
      else localStorage.removeItem(SYNC_PASSWORD_PERSIST_KEY);
    } catch (e) {
      // Ignore storage errors
    }
  }

  // Store for settings section content generators
  let settingsSectionGenerators = null;
  let currentSettingsSection = "general";

  function renderSettingsPage() {
    const container = document.getElementById("settingsContent");
    if (!container) return;
    container.innerHTML = "";

    // Create main layout with sidebar
    const layout = document.createElement("div");
    layout.className = "settings-layout";

    // Create sidebar navigation
    const sidebar = document.createElement("nav");
    sidebar.className = "settings-sidebar";
    const nav = document.createElement("ul");
    nav.className = "settings-nav";

    // Define navigation items
    const navItems = [
      { id: "general", icon: "fa-sliders", label: "Général" },
      { id: "security", icon: "fa-shield-halved", label: "Sécurité" },
      { id: "data", icon: "fa-database", label: "Données" },
      { id: "storage", icon: "fa-hard-drive", label: "Stockage" }
    ];
    // Add sync if unlocked
    if (isServerModeUnlocked()) {
      navItems.push({ id: "sync", icon: "fa-cloud", label: "Sync Pro", badge: "PRO" });
    }
    navItems.push(
      { id: "advanced", icon: "fa-flask-vial", label: "Avancé" },
      { id: "reset", icon: "fa-triangle-exclamation", label: "Zone Danger", danger: true }
    );

    navItems.forEach(item => {
      const li = document.createElement("li");
      li.className = "settings-nav__item";
      if (item.danger) li.classList.add("settings-nav__item--danger");
      if (item.id === currentSettingsSection) li.classList.add("settings-nav__item--active");
      li.dataset.section = item.id;

      const icon = document.createElement("span");
      icon.className = "settings-nav__icon";
      icon.innerHTML = `<i class="fas ${item.icon}"></i>`;

      const label = document.createElement("span");
      label.textContent = item.label;

      li.appendChild(icon);
      li.appendChild(label);

      if (item.badge) {
        const badge = document.createElement("span");
        badge.className = "settings-nav__badge";
        badge.textContent = item.badge;
        li.appendChild(badge);
      }

      li.addEventListener("click", () => {
        currentSettingsSection = item.id;
        document.querySelectorAll(".settings-nav__item").forEach(el => el.classList.remove("settings-nav__item--active"));
        li.classList.add("settings-nav__item--active");
        renderSettingsContent();
      });

      nav.appendChild(li);
    });

    sidebar.appendChild(nav);

    // Create content area
    const contentArea = document.createElement("div");
    contentArea.id = "settingsContentArea";
    contentArea.className = "settings-content-area";

    layout.appendChild(sidebar);
    layout.appendChild(contentArea);
    container.appendChild(layout);

    // Initialize section generators
    initSettingsSectionGenerators();

    // Render current section
    renderSettingsContent();
  }

  function renderSettingsContent() {
    const contentArea = document.getElementById("settingsContentArea");
    if (!contentArea || !settingsSectionGenerators) return;

    contentArea.classList.add("settings-content-area--transitioning");

    setTimeout(() => {
      contentArea.innerHTML = "";
      const generator = settingsSectionGenerators[currentSettingsSection];
      if (generator) {
        generator(contentArea);
      }
      contentArea.classList.remove("settings-content-area--transitioning");
      refreshReadOnlyLocks();
    }, 100);
  }

  function createSettingsCard({ title, description, icon, tone, status }) {
    const card = document.createElement("div");
    card.className = "settings-card";
    if (tone) card.classList.add(`settings-card--${tone}`);

    const header = document.createElement("div");
    header.className = "settings-card__header";

    const iconWrap = document.createElement("div");
    iconWrap.className = "settings-card__icon-wrap";
    iconWrap.innerHTML = `<i class="fas ${icon}"></i>`;

    const headerText = document.createElement("div");
    headerText.className = "settings-card__header-text";

    const titleEl = document.createElement("h3");
    titleEl.className = "settings-card__title";
    titleEl.textContent = title;

    const descEl = document.createElement("p");
    descEl.className = "settings-card__desc";
    descEl.textContent = description;

    headerText.appendChild(titleEl);
    headerText.appendChild(descEl);

    header.appendChild(iconWrap);
    header.appendChild(headerText);

    if (status !== undefined) {
      const statusEl = document.createElement("div");
      statusEl.className = `settings-card__status${status ? " settings-card__status--ok" : ""}`;
      header.appendChild(statusEl);
    }

    const body = document.createElement("div");
    body.className = "settings-card__body";

    card.appendChild(header);
    card.appendChild(body);

    return { card, body };
  }

  function initSettingsSectionGenerators() {
    settingsSectionGenerators = {
      general: renderGeneralSection,
      security: renderSecuritySection,
      data: renderDataSection,
      storage: renderStorageSection,
      sync: renderSyncSection,
      advanced: renderAdvancedSection,
      reset: renderResetSection
    };
  }

  // Settings helper functions
  function createSettingsInputGroup({ id, label, value = "", type = "text", placeholder = "", help = "", readonly = true }) {
    const wrap = document.createElement("div");
    wrap.className = "settings-form-group";
    const lbl = document.createElement("label");
    lbl.className = "settings-form-label";
    lbl.setAttribute("for", id);
    lbl.textContent = label;
    const input = document.createElement("input");
    input.id = id;
    input.type = type;
    input.className = "form-control";
    input.value = value || "";
    if (placeholder) input.placeholder = placeholder;
    if (readonly) input.dataset.readonlyLock = "true";
    wrap.appendChild(lbl);
    wrap.appendChild(input);
    if (help) {
      const helpText = document.createElement("p");
      helpText.className = "settings-form-help";
      helpText.textContent = help;
      wrap.appendChild(helpText);
    }
    return { wrap, input };
  }

  function createSettingsPasswordGroup({ id, label, value = "", placeholder = "", help = "" }) {
    const wrap = document.createElement("div");
    wrap.className = "settings-form-group";
    const lbl = document.createElement("label");
    lbl.className = "settings-form-label";
    lbl.setAttribute("for", id);
    lbl.textContent = label;
    const row = document.createElement("div");
    row.className = "settings-input-row";
    const input = document.createElement("input");
    input.id = id;
    input.type = "password";
    input.className = "form-control";
    input.value = value || "";
    if (placeholder) input.placeholder = placeholder;
    input.dataset.readonlyLock = "true";
    const toggle = document.createElement("button");
    toggle.type = "button";
    toggle.className = "btn btn--outline btn--sm settings-toggle-btn";
    toggle.innerHTML = '<i class="fas fa-eye"></i>';
    toggle.setAttribute("aria-pressed", "false");
    toggle.addEventListener("click", () => {
      const isHidden = input.type === "password";
      input.type = isHidden ? "text" : "password";
      toggle.innerHTML = isHidden ? '<i class="fas fa-eye-slash"></i>' : '<i class="fas fa-eye"></i>';
      toggle.setAttribute("aria-pressed", String(isHidden));
    });
    row.append(input, toggle);
    wrap.appendChild(lbl);
    wrap.appendChild(row);
    if (help) {
      const helpText = document.createElement("p");
      helpText.className = "settings-form-help";
      helpText.textContent = help;
      wrap.appendChild(helpText);
    }
    return { wrap, input, toggle };
  }

  function createSettingsSelectGroup({ id, label, value = "", options = [], help = "" }) {
    const wrap = document.createElement("div");
    wrap.className = "settings-form-group";
    const lbl = document.createElement("label");
    lbl.className = "settings-form-label";
    lbl.setAttribute("for", id);
    lbl.textContent = label;
    const select = document.createElement("select");
    select.id = id;
    select.className = "form-control";
    select.dataset.readonlyLock = "true";
    options.forEach((opt) => {
      const option = document.createElement("option");
      option.value = opt.value;
      option.textContent = opt.label;
      select.appendChild(option);
    });
    select.value = value || "";
    wrap.appendChild(lbl);
    wrap.appendChild(select);
    if (help) {
      const helpText = document.createElement("p");
      helpText.className = "settings-form-help";
      helpText.textContent = help;
      wrap.appendChild(helpText);
    }
    return { wrap, select };
  }

  // === SECTION: GENERAL ===
  function renderGeneralSection(container) {
    const legacyTitle = "Système de Management de la Sécurité";
    const safeTitle = !appData.title || appData.title === legacyTitle ? "Cyber-Assistant" : appData.title;
    if (appData.title !== safeTitle) {
      appData.title = safeTitle;
      updateAppTitle();
    }

    const { card, body } = createSettingsCard({
      title: "Configuration générale",
      description: "Personnalisez le titre, le thème et les cibles de maturité.",
      icon: "fa-sliders"
    });

    const formGrid = document.createElement("div");
    formGrid.className = "settings-form-grid";

    const { wrap: titleWrap, input: titleInput } = createSettingsInputGroup({
      id: "settingsTitle",
      label: "Titre de l'application",
      value: safeTitle
    });
    titleWrap.classList.add("settings-form-group--full");

    const { wrap: themeWrap, select: themeSelect } = createSettingsSelectGroup({
      id: "settingsTheme",
      label: "Thème",
      value: currentTheme,
      options: THEME_OPTIONS
    });

    const langOptions = window.CSI_i18n ? Object.entries(window.CSI_i18n.getAvailableLanguages()).map(([code, info]) => ({
      value: code,
      label: `${info.flag} ${info.name}`
    })) : [
      { value: "fr", label: "🇫🇷 Français" },
      { value: "en", label: "🇬🇧 English" },
      { value: "nl", label: "🇳🇱 Nederlands" }
    ];
    const currentLang = window.CSI_i18n ? window.CSI_i18n.getLanguage() : "fr";
    const { wrap: langWrap, select: langSelect } = createSettingsSelectGroup({
      id: "settingsLanguage",
      label: "Langue / Language",
      value: currentLang,
      options: langOptions
    });

    const { wrap: isoWrap, input: isoInput } = createSettingsInputGroup({
      id: "settingsIsoTarget",
      label: "ISO 27002 - Cible",
      type: "number",
      value: appData.targetMaturity.iso27002
    });
    isoInput.min = "1";
    isoInput.max = "5";

    const { wrap: nis2Wrap, input: nis2Input } = createSettingsInputGroup({
      id: "settingsNis2Target",
      label: "NIS2 - Cible",
      type: "number",
      value: appData.targetMaturity.nis2Target,
      help: "Valeur cible (1-5) pour toutes les catégories."
    });
    nis2Input.min = "1";
    nis2Input.max = "5";

    formGrid.append(titleWrap, themeWrap, langWrap, isoWrap, nis2Wrap);
    body.appendChild(formGrid);

    const actions = document.createElement("div");
    actions.className = "settings-card__actions";
    const btnSave = document.createElement("button");
    btnSave.id = "settingsSave";
    btnSave.className = "btn btn--primary btn--sm";
    btnSave.textContent = "Enregistrer les paramètres";
    btnSave.dataset.readonlyLock = "true";
    actions.appendChild(btnSave);
    body.appendChild(actions);

    btnSave.addEventListener("click", () => {
      if (!ensureEditMode("enregistrer les paramètres")) {
        refreshReadOnlyLocks();
        return;
      }
      const newTitle = titleInput.value.trim();
      if (newTitle) {
        appData.title = newTitle;
        updateAppTitle();
      }
      if (themeSelect.value) setTheme(themeSelect.value);
      const isoTarget = parseInt(isoInput.value, 10);
      if (!isNaN(isoTarget)) appData.targetMaturity.iso27002 = isoTarget;
      const nis2Target = parseInt(nis2Input.value, 10);
      if (!isNaN(nis2Target)) appData.targetMaturity.nis2Target = nis2Target;
      if (!appData.targetMaturity.nis2) {
        appData.targetMaturity.nis2 = Object.assign({}, DEFAULT_NIS2_MATURITY);
      }
      const newLang = langSelect.value;
      if (newLang && window.CSI_i18n && newLang !== window.CSI_i18n.getLanguage()) {
        window.CSI_i18n.setLanguage(newLang);
        window.CSI_i18n.translatePage();
      }
      saveData();
      updateDashboard();
      showToast2("Paramètres enregistrés", "success");
      btnSave.classList.add("btn--success-flash");
      setTimeout(() => btnSave.classList.remove("btn--success-flash"), 600);
    });

    container.appendChild(card);

    // === Second card: Display options ===
    const { card: displayCard, body: displayBody } = createSettingsCard({
      title: "Options d'affichage",
      description: "Personnalisez l'apparence et la densité de l'interface.",
      icon: "fa-desktop"
    });

    const displayGrid = document.createElement("div");
    displayGrid.className = "settings-form-grid";

    // Compact mode toggle
    const compactWrap = document.createElement("div");
    compactWrap.className = "settings-form-group settings-form-group--full";
    const compactLabel = document.createElement("label");
    compactLabel.className = "settings-checkbox";
    const compactInput = document.createElement("input");
    compactInput.type = "checkbox";
    compactInput.id = "settingsCompactMode";
    compactInput.checked = document.body.classList.contains("compact-mode");
    compactLabel.appendChild(compactInput);
    compactLabel.appendChild(document.createTextNode(" Mode compact"));
    const compactHelp = document.createElement("p");
    compactHelp.className = "settings-field-help";
    compactHelp.textContent = "Réduit l'espacement et la taille des éléments pour afficher plus d'informations.";
    compactWrap.appendChild(compactLabel);
    compactWrap.appendChild(compactHelp);

    compactInput.addEventListener("change", () => {
      document.body.classList.toggle("compact-mode", compactInput.checked);
      try {
        localStorage.setItem("smsi_compact_mode", compactInput.checked ? "1" : "0");
      } catch (e) {
        console.warn("Impossible de sauvegarder le mode compact", e);
      }
      showToast2(compactInput.checked ? "Mode compact activé" : "Mode compact désactivé", "info");
    });

    displayGrid.appendChild(compactWrap);
    displayBody.appendChild(displayGrid);
    container.appendChild(displayCard);
  }

  // === SECTION: SECURITY ===
  function renderSecuritySection(container) {
    const { card, body } = createSettingsCard({
      title: "Mot de passe du mode édition",
      description: `Minimum ${MIN_EDIT_PASSWORD_LENGTH} caractères, 1 majuscule, 1 minuscule, 1 caractère spécial.`,
      icon: "fa-shield-halved",
      status: !!getStoredEditPasswordHash()
    });

    const formGrid = document.createElement("div");
    formGrid.className = "settings-form-grid";

    const { wrap: editPwdWrap, input: editPwdInput } = createSettingsPasswordGroup({
      id: "settingsEditPassword",
      label: "Nouveau mot de passe",
      placeholder: `Minimum ${MIN_EDIT_PASSWORD_LENGTH} caractères`
    });

    const { wrap: editPwdConfirmWrap, input: editPwdConfirmInput } = createSettingsPasswordGroup({
      id: "settingsEditPasswordConfirm",
      label: "Confirmation",
      placeholder: "Ressaisir le mot de passe"
    });

    formGrid.append(editPwdWrap, editPwdConfirmWrap);
    body.appendChild(formGrid);

    const { wrap: policyWrap, list: policyList } = createPasswordPolicyList();
    body.appendChild(policyWrap);

    const passwordStatus = document.createElement("p");
    passwordStatus.className = "settings-form-help";
    passwordStatus.textContent = getStoredEditPasswordHash()
      ? "Mot de passe personnalisé actif."
      : "Mot de passe par défaut actif.";
    body.appendChild(passwordStatus);

    const updatePolicy = () => updatePasswordPolicyList(policyList, editPwdInput.value);
    editPwdInput.addEventListener("input", updatePolicy);
    updatePolicy();

    const actions = document.createElement("div");
    actions.className = "settings-card__actions";
    const btnUpdatePassword = document.createElement("button");
    btnUpdatePassword.className = "btn btn--primary btn--sm";
    btnUpdatePassword.textContent = "Mettre à jour le mot de passe";
    btnUpdatePassword.dataset.readonlyLock = "true";
    actions.appendChild(btnUpdatePassword);
    body.appendChild(actions);

    btnUpdatePassword.addEventListener("click", () => {
      if (!ensureEditMode("modifier le mot de passe")) {
        refreshReadOnlyLocks();
        return;
      }
      const password = editPwdInput.value || "";
      const confirm = editPwdConfirmInput.value || "";
      if (!password) {
        showToast2("Veuillez saisir un mot de passe.", "warning");
        return;
      }
      const policy = getEditPasswordPolicyStatus(password);
      if (!policy.isValid) {
        const detail = policy.missing.length ? `Exigences manquantes : ${policy.missing.join(", ")}.` : "";
        showToast2(`Mot de passe non conforme. ${detail}`.trim(), "warning");
        return;
      }
      if (password !== confirm) {
        showToast2("Les mots de passe ne correspondent pas.", "error");
        return;
      }
      btnUpdatePassword.disabled = true;
      const initialLabel = btnUpdatePassword.textContent;
      btnUpdatePassword.textContent = "Enregistrement...";
      hashEditPassword(password)
        .then((hash) => {
          setStoredEditPasswordHash(hash);
          editPasswordUnlocked = true;
          editPwdInput.value = "";
          editPwdConfirmInput.value = "";
          passwordStatus.textContent = "Mot de passe personnalisé actif.";
          showToast2("Mot de passe mis à jour", "success");
        })
        .catch((err) => {
          console.error("Impossible d'enregistrer le mot de passe", err);
          showToast2("Impossible d'enregistrer le mot de passe", "error");
        })
        .finally(() => {
          btnUpdatePassword.disabled = false;
          btnUpdatePassword.textContent = initialLabel;
        });
    });

    container.appendChild(card);
  }

  // === SECTION: DATA (Backup, Export, Snapshots) ===
  function renderDataSection(container) {
    // Backup Card
    const { card: backupCard, body: backupBody } = createSettingsCard({
      title: "Sauvegarde",
      description: "Exportez ou importez vos données SMSI.",
      icon: "fa-download"
    });

    const quickActions = document.createElement("div");
    quickActions.className = "quick-actions";

    const exportAction = document.createElement("div");
    exportAction.className = "quick-action";
    exportAction.id = "settingsExport";
    exportAction.innerHTML = `
      <div class="quick-action__icon"><i class="fas fa-download"></i></div>
      <span class="quick-action__label">Exporter</span>
      <span class="quick-action__hint">.smsi</span>
    `;
    exportAction.addEventListener("click", exportData);

    const importAction = document.createElement("div");
    importAction.className = "quick-action";
    importAction.id = "settingsImport";
    importAction.dataset.readonlyLock = "true";
    importAction.innerHTML = `
      <div class="quick-action__icon"><i class="fas fa-upload"></i></div>
      <span class="quick-action__label">Importer</span>
      <span class="quick-action__hint">.smsi / .json</span>
    `;
    importAction.addEventListener("click", importData);

    const pdfAction = document.createElement("div");
    pdfAction.className = "quick-action";
    pdfAction.id = "settingsPdfReport";
    pdfAction.innerHTML = `
      <div class="quick-action__icon"><i class="fas fa-file-pdf"></i></div>
      <span class="quick-action__label">Rapport PDF</span>
      <span class="quick-action__hint">Complet</span>
    `;
    pdfAction.addEventListener("click", () => {
      if (typeof generatePdfReportWithFeedback === "function") {
        generatePdfReportWithFeedback();
      } else {
        showToast2("Génération PDF indisponible.", "error");
      }
    });

    const excelAction = document.createElement("div");
    excelAction.className = "quick-action";
    excelAction.id = "settingsExcelAll";
    excelAction.innerHTML = `
      <div class="quick-action__icon"><i class="fas fa-file-excel"></i></div>
      <span class="quick-action__label">Export Excel</span>
      <span class="quick-action__hint">Global</span>
    `;
    excelAction.addEventListener("click", exportAllSectionsExcel);

    quickActions.append(exportAction, importAction, pdfAction, excelAction);
    backupBody.appendChild(quickActions);
    container.appendChild(backupCard);

    // Snapshots Card
    const { card: snapshotCard, body: snapshotBody } = createSettingsCard({
      title: "Historique & Snapshots",
      description: "Créez des points de sauvegarde locaux.",
      icon: "fa-history"
    });

    const snapshotGrid = document.createElement("div");
    snapshotGrid.className = "settings-form-grid";

    const { wrap: snapshotLabelWrap, input: snapshotLabelInput } = createSettingsInputGroup({
      id: "snapshotLabel",
      label: "Nom du snapshot",
      placeholder: "Ex: comité sécurité T2",
      readonly: false
    });

    snapshotGrid.appendChild(snapshotLabelWrap);
    snapshotBody.appendChild(snapshotGrid);

    const snapshotActions = document.createElement("div");
    snapshotActions.className = "settings-card__actions";

    const btnValidate = document.createElement("button");
    btnValidate.id = "settingsValidate";
    btnValidate.className = "btn btn--outline btn--sm";
    btnValidate.innerHTML = '<i class="fas fa-search"></i> Analyser';
    btnValidate.addEventListener("click", validateDataQuality);

    const btnSnapshot = document.createElement("button");
    btnSnapshot.id = "settingsSnapshot";
    btnSnapshot.className = "btn btn--primary btn--sm";
    btnSnapshot.innerHTML = '<i class="fas fa-save"></i> Créer';
    btnSnapshot.dataset.readonlyLock = "true";
    btnSnapshot.addEventListener("click", () => {
      const label = snapshotLabelInput.value.trim() || "Snapshot manuel";
      addHistorySnapshot(label);
      snapshotLabelInput.value = "";
    });

    const btnSnapshotClear = document.createElement("button");
    btnSnapshotClear.className = "btn btn--outline btn--sm";
    btnSnapshotClear.textContent = "Nettoyer";
    btnSnapshotClear.dataset.readonlyLock = "true";
    btnSnapshotClear.addEventListener("click", clearHistorySnapshots);

    snapshotActions.append(btnValidate, btnSnapshot, btnSnapshotClear);
    snapshotBody.appendChild(snapshotActions);

    // Snapshots Timeline
    const historyList = document.createElement("div");
    historyList.id = "historyList";
    historyList.className = "snapshots-timeline";
    snapshotBody.appendChild(historyList);

    container.appendChild(snapshotCard);
    renderHistoryList();
  }

  // === SECTION: STORAGE ===
  function renderStorageSection(container) {
    const { card, body } = createSettingsCard({
      title: "Stockage & Santé",
      description: "Surveillance de l'occupation localStorage.",
      icon: "fa-hard-drive"
    });

    const gaugeWrap = document.createElement("div");
    gaugeWrap.style.marginBottom = "var(--spacing-md)";

    const gaugeLabel = document.createElement("div");
    gaugeLabel.className = "settings-form-label";
    gaugeLabel.textContent = "Occupation localStorage";
    gaugeLabel.style.marginBottom = "var(--spacing-xs)";

    const gauge = document.createElement("div");
    gauge.className = "storage-gauge";
    const gaugeBar = document.createElement("div");
    gaugeBar.id = "storageUsageBar";
    gaugeBar.className = "storage-gauge__fill";
    gauge.appendChild(gaugeBar);

    const gaugeText = document.createElement("div");
    gaugeText.id = "storageUsageText";
    gaugeText.className = "settings-form-help";
    gaugeText.style.marginTop = "var(--spacing-xs)";

    gaugeWrap.append(gaugeLabel, gauge, gaugeText);
    body.appendChild(gaugeWrap);

    const stats = document.createElement("div");
    stats.className = "storage-stats";

    const createStat = (label, id) => {
      const stat = document.createElement("div");
      stat.className = "storage-stat";
      const value = document.createElement("div");
      value.id = id;
      value.className = "storage-stat__value";
      value.textContent = "—";
      const lbl = document.createElement("div");
      lbl.className = "storage-stat__label";
      lbl.textContent = label;
      stat.append(value, lbl);
      return stat;
    };

    stats.append(
      createStat("Taille données", "storageDatasetSize"),
      createStat("Dernier autosave", "storageLastSave")
    );
    body.appendChild(stats);

    const warning = document.createElement("div");
    warning.id = "storageHealthWarning";
    warning.className = "danger-warning";
    warning.hidden = true;
    warning.innerHTML = `
      <i class="fas fa-exclamation-triangle danger-warning__icon"></i>
      <span class="danger-warning__text">Espace de stockage faible. Exportez vos données.</span>
    `;
    body.appendChild(warning);

    container.appendChild(card);
    refreshStorageHealth();
  }

  // === SECTION: SYNC ===
  function renderSyncSection(container) {
    const { card, body } = createSettingsCard({
      title: "Synchronisation serveur",
      description: "Partagez un snapshot via l'API /data/export et /data/import.",
      icon: "fa-cloud"
    });

    const cfg = loadSyncConfig();
    const storedSyncPassword = getStoredSyncPassword(cfg);

    const formGrid = document.createElement("div");
    formGrid.className = "settings-form-grid";

    const { wrap: urlWrap, input: urlInput } = createSettingsInputGroup({
      id: "syncBaseUrl",
      label: "URL de base (optionnel)",
      value: cfg.baseUrl || "",
      placeholder: "https://mon-domaine/app/"
    });

    const { wrap: userWrap, input: userInput } = createSettingsInputGroup({
      id: "syncUser",
      label: "Utilisateur",
      value: cfg.user || "admin",
      placeholder: "admin"
    });

    const { wrap: passWrap, input: passInput } = createSettingsPasswordGroup({
      id: "syncPassword",
      label: "Mot de passe",
      value: storedSyncPassword || "",
      placeholder: "SYNC_PASSWORD"
    });

    formGrid.append(urlWrap, userWrap, passWrap);
    body.appendChild(formGrid);

    const rememberWrap = document.createElement("div");
    rememberWrap.className = "settings-form-group";
    const rememberLabel = document.createElement("label");
    rememberLabel.className = "settings-checkbox";
    const rememberInput = document.createElement("input");
    rememberInput.id = "syncRememberPassword";
    rememberInput.type = "checkbox";
    rememberInput.checked = !!cfg.rememberPassword;
    rememberInput.dataset.readonlyLock = "true";
    rememberLabel.appendChild(rememberInput);
    rememberLabel.appendChild(document.createTextNode(" Mémoriser le mot de passe"));
    rememberWrap.appendChild(rememberLabel);
    body.appendChild(rememberWrap);

    const actions = document.createElement("div");
    actions.className = "settings-card__actions";

    const btnSaveSync = document.createElement("button");
    btnSaveSync.className = "btn btn--primary btn--sm";
    btnSaveSync.textContent = "Enregistrer";
    btnSaveSync.dataset.readonlyLock = "true";

    const btnSyncTest = document.createElement("button");
    btnSyncTest.className = "btn btn--outline btn--sm";
    btnSyncTest.innerHTML = '<i class="fas fa-stethoscope"></i> Tester';
    btnSyncTest.dataset.readonlyLock = "true";

    const btnSyncPanel = document.createElement("button");
    btnSyncPanel.className = "btn btn--outline btn--sm";
    btnSyncPanel.textContent = "Ouvrir Sync";
    btnSyncPanel.dataset.readonlyLock = "true";

    actions.append(btnSaveSync, btnSyncTest, btnSyncPanel);
    body.appendChild(actions);

    const syncStatus = document.createElement("div");
    syncStatus.id = "syncTestStatus";
    syncStatus.className = "settings-sync-status";
    syncStatus.textContent = "Aucun test exécuté.";
    body.appendChild(syncStatus);

    const syncMeta = document.createElement("div");
    syncMeta.id = "syncMeta";
    syncMeta.className = "settings-meta";
    body.appendChild(syncMeta);

    // Event handlers
    btnSaveSync.addEventListener("click", () => {
      const newCfg = {
        baseUrl: urlInput.value.trim(),
        user: (userInput.value || "admin").trim(),
        rememberPassword: rememberInput.checked
      };
      try {
        localStorage.setItem(SYNC_CONFIG_KEY, JSON.stringify(newCfg));
        saveSyncPassword(passInput.value, rememberInput.checked);
        showToast2("Configuration synchronisation sauvegardée", "success");
        refreshSyncMeta();
      } catch (e) {
        showToast2("Impossible d'enregistrer la configuration", "error");
      }
    });

    btnSyncPanel.addEventListener("click", () => {
      if (typeof window.toggleServerSyncPanel === "function") {
        window.toggleServerSyncPanel();
      } else {
        showToast2("Panneau Sync serveur indisponible", "warning");
      }
    });

    const resolveSyncUrl = (path) => {
      const base = urlInput.value.trim() || window.location.href;
      try { return new URL(path, base).toString(); } catch { return null; }
    };

    const getAuthHeader = () => {
      const user = (userInput.value || "admin").trim();
      const pass = passInput.value || "";
      return "Basic " + btoa(`${user}:${pass}`);
    };

    const fetchWithTimeout = async (url, options = {}, timeoutMs = 8000) => {
      const controller = new AbortController();
      const timer = setTimeout(() => controller.abort(), timeoutMs);
      try {
        return await fetch(url, { ...options, signal: controller.signal });
      } finally {
        clearTimeout(timer);
      }
    };

    btnSyncTest.addEventListener("click", async () => {
      syncStatus.textContent = "Test en cours...";
      const rows = [];
      const exportUrl = resolveSyncUrl("./data/export");
      const importUrl = resolveSyncUrl("./data/import");

      if (!exportUrl || !importUrl) {
        syncStatus.innerHTML = '<span style="color:var(--color-danger)">URL invalide</span>';
        return;
      }

      try {
        const start = performance.now();
        const res = await fetchWithTimeout(exportUrl, { cache: "no-store" });
        const duration = Math.round(performance.now() - start) + "ms";
        if (res.ok) {
          rows.push(`<div style="color:var(--color-success)">GET /data/export: OK (${duration})</div>`);
        } else {
          rows.push(`<div style="color:var(--color-danger)">GET /data/export: HTTP ${res.status} (${duration})</div>`);
        }
      } catch (err) {
        rows.push(`<div style="color:var(--color-danger)">GET /data/export: Erreur réseau</div>`);
      }

      try {
        const start = performance.now();
        const res = await fetchWithTimeout(importUrl, {
          method: "POST",
          headers: { "Content-Type": "application/json", Authorization: getAuthHeader() },
          body: "{}"
        });
        const duration = Math.round(performance.now() - start) + "ms";
        if (res.status === 401) {
          rows.push(`<div style="color:var(--color-danger)">POST /data/import: Auth refusée (${duration})</div>`);
        } else if (res.status === 400) {
          rows.push(`<div style="color:var(--color-success)">POST /data/import: Auth OK (${duration})</div>`);
        } else {
          rows.push(`<div style="color:var(--color-warning)">POST /data/import: HTTP ${res.status} (${duration})</div>`);
        }
      } catch (err) {
        rows.push(`<div style="color:var(--color-danger)">POST /data/import: Erreur réseau</div>`);
      }

      syncStatus.innerHTML = rows.join("");
    });

    container.appendChild(card);
    refreshSyncMeta();
  }

  // === SECTION: ADVANCED ===
  function renderAdvancedSection(container) {
    // Tests Card
    const { card: testCard, body: testBody } = createSettingsCard({
      title: "Tests",
      description: "Vérification rapide des fonctions principales.",
      icon: "fa-flask-vial"
    });

    const testActions = document.createElement("div");
    testActions.className = "settings-card__actions";
    const btnTests = document.createElement("button");
    btnTests.id = "settingsTests";
    btnTests.className = "btn btn--secondary btn--sm";
    btnTests.innerHTML = '<i class="fas fa-flask"></i> Lancer les tests';
    btnTests.addEventListener("click", runTestSuite);
    testActions.appendChild(btnTests);
    testBody.appendChild(testActions);
    container.appendChild(testCard);

    // Demo Card
    const { card: demoCard, body: demoBody } = createSettingsCard({
      title: "Données de démonstration",
      description: "Charge un jeu de données complet pour découvrir l'application.",
      icon: "fa-database"
    });

    const demoActions = document.createElement("div");
    demoActions.className = "settings-card__actions";
    const btnDemo = document.createElement("button");
    btnDemo.id = "settingsDemo";
    btnDemo.className = "btn btn--outline btn--sm";
    btnDemo.innerHTML = '<i class="fas fa-database"></i> Charger les données démo';
    btnDemo.dataset.readonlyLock = "true";
    btnDemo.addEventListener("click", loadDemoData);
    demoActions.appendChild(btnDemo);
    demoBody.appendChild(demoActions);
    container.appendChild(demoCard);
  }

  // === SECTION: RESET ===
  function renderResetSection(container) {
    // En mode DEMO, afficher une section spécifique pour recharger les données démo
    if (DEMO_MODE) {
      const { card, body } = createSettingsCard({
        title: "Réinitialisation des données démo",
        description: "Recharge les données de démonstration originales.",
        icon: "fa-sync-alt",
        tone: "warning"
      });

      const info = document.createElement("div");
      info.className = "info-box info-box--warning";
      info.innerHTML = `
        <i class="fas fa-info-circle info-box__icon"></i>
        <div class="info-box__text">
          <strong>Mode démonstration</strong><br>
          Les données sont automatiquement rechargées toutes les heures.
          Utilisez ce bouton pour réinitialiser manuellement.
        </div>
      `;
      body.appendChild(info);

      const actions = document.createElement("div");
      actions.className = "settings-card__actions";
      const btnReloadDemo = document.createElement("button");
      btnReloadDemo.id = "settingsReloadDemo";
      btnReloadDemo.className = "btn btn--warning btn--sm";
      btnReloadDemo.innerHTML = '<i class="fas fa-sync-alt"></i> Recharger les données démo';
      btnReloadDemo.addEventListener("click", () => {
        if (confirm("Recharger les données de démonstration ? Vos modifications seront perdues.")) {
          loadDemoDataSilent();
          showToast2("Données de démonstration rechargées.", "success");
        }
      });
      actions.appendChild(btnReloadDemo);
      body.appendChild(actions);

      container.appendChild(card);
      return;
    }

    // Mode normal : réinitialisation complète
    const { card, body } = createSettingsCard({
      title: "Réinitialisation",
      description: "Réinitialise l'application aux paramètres d'origine.",
      icon: "fa-triangle-exclamation",
      tone: "danger"
    });

    const warning = document.createElement("div");
    warning.className = "danger-warning";
    warning.innerHTML = `
      <i class="fas fa-exclamation-triangle danger-warning__icon"></i>
      <div class="danger-warning__text">
        <strong>Attention :</strong> Cette action supprimera toutes vos données.
        Un snapshot de sécurité sera créé automatiquement avant la réinitialisation.
      </div>
    `;
    body.appendChild(warning);

    const actions = document.createElement("div");
    actions.className = "settings-card__actions";
    const btnReset = document.createElement("button");
    btnReset.id = "settingsReset";
    btnReset.className = "btn btn--danger btn--sm";
    btnReset.innerHTML = '<i class="fas fa-trash"></i> Réinitialiser l\'application';
    btnReset.dataset.readonlyLock = "true";
    btnReset.addEventListener("click", resetApplication);
    actions.appendChild(btnReset);
    body.appendChild(actions);

    container.appendChild(card);
  }

  // ========================================
  // SETTINGS PAGE OPEN FUNCTION
  // ========================================

  function openSettings() {
    if (!ensureEditMode("ouvrir les paramètres")) {
      refreshReadOnlyLocks();
      return;
    }
    closeModal();
    const navItems = document.querySelectorAll(".nav__item");
    const modules = document.querySelectorAll(".module");
    navItems.forEach(nav => nav.classList.remove("nav__item--active"));
    modules.forEach(m => m.classList.remove("module--active"));
    const settings = document.getElementById("settings");
    if (settings) settings.classList.add("module--active");
    renderSettingsPage();
  }
  async function resetApplication() {
    if (!ensureEditMode("réinitialiser l'application")) return;
    const runReset = async () => {
      try {
        try {
          addHistorySnapshot("Backup avant réinitialisation", { tag: "auto" });
        } catch (err) {
          console.warn("Snapshot de sécurité non créé", err);
        }
        appData.controls = await loadISO27002Controls();
        appData.soa = appData.controls.map(c => ({ controlId: c.id, applicable: true, justification: "" }));
        appData.nis2Controls = loadNis2Controls();
        appData.nis2Plan = getEmptyNis2Plan();
        appData.soclePillars = cloneDefaultSoclePillars();
        appData.actions = [];
        appData.actionsHistory = {};
        appData.criticalAssets = [];
        appData.criticalAssetsMeta = getDefaultCriticalAssetsMeta();
        appData.projectRisks = [];
        appData.risks = [];
        appData.threats = [];
        appData.nonconformities = [];
        appData.audits = [];
        appData.kpis = [];
        appData.reviews = [];
        appData.stakeholders = [];
        appData.documents = [];
        appData.targetMaturity = {
          iso27002: 3.5,
          nis2: Object.assign({}, DEFAULT_NIS2_MATURITY),
          nis2Target: 0
        };
        appData.nextActionId = 0;
        const tbody = document.getElementById("controlsTableBody");
        if (tbody) tbody.innerHTML = "";
        saveData();
        loadControlsTable();
        loadNis2ControlsTable();
        loadSoaTable();
        loadActionsTable();
        loadCriticalAssetsTable();
        loadProjectRisksTable();
        loadRisksTable();
        updateRiskMatrix();
        loadThreatsTable();
        loadNonConformitiesTable();
        loadAuditsTable();
        loadReviewsTable();
        loadStakeholdersTable();
        loadDocumentsTable();
        loadKpiTable();
        updateActionsChart();
        updateDashboard();
        showToast2("Réinitialisation terminée", "success");
      } catch (error) {
        console.error("Erreur lors de la réinitialisation:", error);
        showToast2("Impossible de réinitialiser les données.", "error");
      }
    };
    openModal(
      "Confirmer la réinitialisation",
      [],
      () => {
        runReset();
      },
      null,
      {
        saveLabel: "Oui",
        cancelLabel: "Non",
        hideClose: true,
        message: "Vous êtes certain de vouloir tout supprimer ?"
      }
    );
  }

  /**
   * Charge les données de démonstration silencieusement (pour mode DEMO)
   * @param {boolean} silent - Si true, pas de confirmation ni de vérification edit mode
   */
  function loadDemoDataSilent() {
    try {
      appData = JSON.parse(JSON.stringify(demoData));
      window.appData = appData;
      appData.soclePillars = normalizeSoclePillars(appData.soclePillars);
      ensureNis2Plan();
      ensureCriticalAssets();
      ensureProjectRisks();
      normalizeCriticalAssetLinks(appData.actions);
      normalizeCriticalAssetLinks(appData.risks);
      normalizeCriticalAssetLinks(appData.threats);
      normalizeCriticalAssetLinks(appData.nonconformities);
      appData.targetMaturity = {
        iso27002: 4,
        nis2: buildNis2Maturity(demoNis2Maturity),
        nis2Target: 4
      };
      if (!appData.actionsHistory) appData.actionsHistory = {};
      NIS2_CATEGORIES.forEach(cat => {
        if (typeof appData.targetMaturity.nis2[cat] !== 'number') {
          appData.targetMaturity.nis2[cat] = 0;
        }
      });
      const defaultNis2 = loadNis2Controls();
      if (Array.isArray(appData.nis2Controls)) {
        appData.nis2Controls = defaultNis2.map(d => {
          const found = appData.nis2Controls.find(c => c.id === d.id) || {};
          return { ...d, ...found };
        });
      } else {
        appData.nis2Controls = defaultNis2;
      }
      saveData();
      console.log('[CSI DEMO] Données de démonstration chargées');
      return true;
    } catch (e) {
      console.error('[CSI DEMO] Erreur chargement données démo:', e);
      return false;
    }
  }
  window.loadDemoDataSilent = loadDemoDataSilent;

  function loadDemoData(options = {}) {
    const allowReadOnly = options.allowReadOnly === true;
    if (!allowReadOnly && !ensureEditMode("charger les données de démonstration")) return;
    if (!confirm("Charger les données de démonstration ? Les données actuelles seront écrasées."))
      return;
    try {
      try {
        addHistorySnapshot("Backup avant données de démonstration", { tag: "auto" });
      } catch (err) {
        console.warn("Snapshot de sécurité non créé", err);
      }
      appData = JSON.parse(JSON.stringify(demoData));
      window.appData = appData;
      appData.soclePillars = normalizeSoclePillars(appData.soclePillars);
      ensureNis2Plan();
      ensureCriticalAssets();
      ensureProjectRisks();
      normalizeCriticalAssetLinks(appData.actions);
      normalizeCriticalAssetLinks(appData.risks);
      normalizeCriticalAssetLinks(appData.threats);
      normalizeCriticalAssetLinks(appData.nonconformities);
      appData.targetMaturity = {
        iso27002: 4,
        nis2: buildNis2Maturity(demoNis2Maturity),
        nis2Target: 4
      };
      if (!appData.actionsHistory) appData.actionsHistory = {};
      if (!appData.targetMaturity) {
        appData.targetMaturity = { iso27002: 3, nis2: Object.assign({}, DEFAULT_NIS2_MATURITY), nis2Target: 3 };
      } else if (!appData.targetMaturity.nis2) {
        appData.targetMaturity.nis2 = Object.assign({}, DEFAULT_NIS2_MATURITY);
      } else {
        NIS2_CATEGORIES.forEach(cat => {
          if (typeof appData.targetMaturity.nis2[cat] !== 'number') {
            appData.targetMaturity.nis2[cat] = 0;
          }
        });
      }
      if (typeof appData.targetMaturity.nis2Target !== 'number') {
        const v = parseInt(appData.targetMaturity.nis2Target, 10);
        appData.targetMaturity.nis2Target = isNaN(v) ? 3 : v;
      }
      const defaultNis2 = loadNis2Controls();
      if (Array.isArray(appData.nis2Controls)) {
        appData.nis2Controls = defaultNis2.map(d => {
          const found = appData.nis2Controls.find(c => c.id === d.id) || {};
          return { ...d, ...found };
        });
      } else {
        appData.nis2Controls = defaultNis2;
      }
      appData.nextActionId = Array.isArray(appData.actions) ? appData.actions.length : 0;
      saveData();
      loadControlsTable();
      loadNis2ControlsTable();
      loadSoaTable();
      loadActionsTable();
      loadCriticalAssetsTable();
      loadProjectRisksTable();
      loadRisksTable();
      updateRiskMatrix();
      loadThreatsTable();
      loadNonConformitiesTable();
      loadAuditsTable();
      loadReviewsTable();
      loadStakeholdersTable();
      loadDocumentsTable();
      loadKpiTable();
      updateActionsChart();
      updateDashboard();
      showToast2("Données de démonstration chargées", "success");
    } catch (error) {
      console.error("Erreur lors du chargement des données de démonstration:", error);
      showToast2("Impossible de charger les données.", "error");
    }
  }
  window.exportData = exportData;
  window.importData = importData;
  window.runTestSuite = runTestSuite;
  window.showToast = showToast2;
  window.showToast2 = showToast2;
  window.hideLoadingSpinner = hideLoadingSpinner2;
  window.openSettings = openSettings;
  window.resetApplication = resetApplication;
  window.loadDemoData = loadDemoData;
  window.updateAppTitle = updateAppTitle;
  window.escapeHtml = escapeHtml;
  window.saveData = saveData;

  // iso27002-data.js

var iso27002_data_default = [
  {
    "numero": "5.1",
    "titre": "Politiques de sécurité de l'information",
    "mesure de sécurité": "Il convient de définir une politique de sécurité de l'information ainsi que des politiques spécifiques par thématique, de les faire approuver par la direction, de les publier, de les communiquer aux personnels et parties intéressées concernés en attestant de leur prise de connaissance, et de les examiner à intervalles planifiés ainsi qu'en cas de changements significatifs.",
    "capability": [
      "Gouvernance"
    ]
  },
  {
    "numero": "5.2",
    "titre": "Fonctions et responsabilités liées à la sécurité de l'information",
    "mesure de sécurité": "Il convient de définir et d'attribuer les rôles et responsabilités en matière de sécurité de l'information en fonction des besoins de l'organisation.",
    "capability": [
      "Gouvernance"
    ]
  },
  {
    "numero": "5.3",
    "titre": "Séparation des tâches",
    "mesure de sécurité": "Il convient de séparer les tâches et les domaines de responsabilité incompatibles.",
    "capability": [
      "Gouvernance"
    ]
  },
  {
    "numero": "5.4",
    "titre": "Responsabilités de la direction",
    "mesure de sécurité": "Il convient que la direction exige de l'ensemble du personnel l'application de la sécurité de l'information conformément à la politique de sécurité de l'information établie, aux politiques thématiques spécifiques et aux procédures de l'organisation.",
    "capability": [
      "Gouvernance"
    ]
  },
  {
    "numero": "5.5",
    "titre": "Contacts avec les autorités",
    "mesure de sécurité": "Il convient d'établir et de maintenir des contacts avec les autorités appropriées (p. ex. autorités réglementaires ou forces de l'ordre) en matière de sécurité de l'information, afin de pouvoir agir efficacement en cas d'incident de sécurité de l'information.",
    "capability": [
      "Gouvernance"
    ]
  },
  {
    "numero": "5.6",
    "titre": "Contacts avec des groupes d’intérêt spécifiques",
    "mesure de sécurité": "Il convient que l'organisation établisse et maintienne des contacts avec des groupes d’intérêt spécifiques, forums spécialisés ou associations professionnelles pertinents en sécurité de l'information, afin de se tenir informée des bonnes pratiques et des menaces émergentes.",
    "capability": [
      "Gouvernance"
    ]
  },
  {
    "numero": "5.7",
    "titre": "Renseignements sur les menaces",
    "mesure de sécurité": "Il convient de collecter et d'analyser les renseignements relatifs aux menaces pour la sécurité de l'information, afin de produire des renseignements exploitables sur ces menaces (intelligence des menaces).",
    "capability": [
      "Gouvernance",
      "Gestion_des_menaces_et_des_vulnérabilités"
    ]
  },
  {
    "numero": "5.8",
    "titre": "Sécurité de l'information dans la gestion de projet",
    "mesure de sécurité": "Il convient d'intégrer la sécurité de l'information dans la gestion de projet, afin de s'assurer que les risques de sécurité liés aux projets et aux livrables sont pris en compte dès le début et tout au long du projet.",
    "capability": [
      "Gouvernance"
    ]
  },
  {
    "numero": "5.9",
    "titre": "Inventaire des informations et autres actifs associés",
    "mesure de sécurité": "Il convient d'établir et de tenir à jour un inventaire des informations et des autres actifs associés, à un niveau de granularité approprié, afin de savoir en permanence quels actifs doivent être protégés.",
    "capability": [
      "Gestion_des_actifs"
    ]
  },
  {
    "numero": "5.10",
    "titre": "Utilisation correcte des informations et autres actifs associés",
    "mesure de sécurité": "Il convient d'identifier, de documenter et de mettre en œuvre des règles d'utilisation acceptable des informations et autres actifs associés, et de s'assurer que ces règles sont communiquées aux utilisateurs concernés.",
    "capability": [
      "Gestion_des_actifs"
    ]
  },
  {
    "numero": "5.11",
    "titre": "Restitution des actifs",
    "mesure de sécurité": "Il convient de s'assurer que tous les actifs de l'organisation en possession du personnel sont restitués lors de la cessation ou du changement d'emploi de ce personnel, conformément aux conditions établies.",
    "capability": [
      "Gestion_des_actifs"
    ]
  },
  {
    "numero": "5.12",
    "titre": "Classification des informations",
    "mesure de sécurité": "Il convient de classifier les informations en fonction de leur valeur, de leur sensibilité et de leur criticité, conformément aux besoins métier de l'organisation en matière de sécurité de l'information.",
    "capability": [
      "Gestion_des_actifs"
    ]
  },
  {
    "numero": "5.13",
    "titre": "Marquage des informations",
    "mesure de sécurité": "Il convient de marquer les informations conformément à leur classification, au moyen de procédés d'étiquetage appropriés, de sorte que leur niveau de sensibilité soit clairement indiqué.",
    "capability": [
      "Gestion_des_actifs"
    ]
  },
  {
    "numero": "5.14",
    "titre": "Transfert des informations",
    "mesure de sécurité": "Il convient de protéger le transfert des informations par des politiques, des procédures et des mesures de sécurité appropriées, afin de prévenir tout accès, altération ou perte non autorisés pendant le transfert.",
    "capability": [
      "Protection_des_informations"
    ]
  },
  {
    "numero": "5.15",
    "titre": "Contrôle d'accès",
    "mesure de sécurité": "Il convient de contrôler l'accès aux informations et aux autres actifs associés, de façon à ce que seules les entités autorisées (utilisateurs, processus, dispositifs) puissent y accéder, conformément aux exigences de sécurité de l'organisation.",
    "capability": [
      "Gestion_des_identités_et_des_accès"
    ]
  },
  {
    "numero": "5.16",
    "titre": "Gestion des identités",
    "mesure de sécurité": "Il convient de gérer les identités des utilisateurs et autres entités tout au long de leur cycle de vie (création, gestion, suppression), de manière à permettre une authentification et un contrôle d'accès appropriés.",
    "capability": [
      "Gestion_des_identités_et_des_accès"
    ]
  },
  {
    "numero": "5.17",
    "titre": "Informations d'authentification",
    "mesure de sécurité": "Il convient de gérer et de protéger de manière appropriée les informations d'authentification (ex. mots de passe, clés, dispositifs d'authentification) pendant tout leur cycle de vie, afin d'empêcher tout accès non autorisé.",
    "capability": [
      "Gestion_des_identités_et_des_accès"
    ]
  },
  {
    "numero": "5.18",
    "titre": "Droits d'accès",
    "mesure de sécurité": "Il convient de gérer l'octroi, la modification et la révocation des droits d'accès aux informations et autres actifs associés, conformément aux politiques de sécurité et au principe du moindre privilège, et de tenir à jour les enregistrements correspondants.",
    "capability": [
      "Gestion_des_identités_et_des_accès"
    ]
  },
  {
    "numero": "5.19",
    "titre": "Sécurité de l'information dans les relations avec les fournisseurs",
    "mesure de sécurité": "Il convient de définir et de mettre en œuvre des exigences de sécurité de l'information pour les relations avec les fournisseurs, afin d'assurer la protection des informations échangées ou auxquelles les fournisseurs ont accès.",
    "capability": [
      "Sécurité_des_relations_fournisseurs"
    ]
  },
  {
    "numero": "5.20",
    "titre": "Sécurité de l'information dans les accords avec les fournisseurs",
    "mesure de sécurité": "Il convient d'inclure des exigences de sécurité de l'information dans les accords conclus avec les fournisseurs, de façon à formaliser les obligations de chaque partie en matière de protection des informations.",
    "capability": [
      "Sécurité_des_relations_fournisseurs"
    ]
  },
  {
    "numero": "5.21",
    "titre": "Gestion de la sécurité de l'information dans la chaîne d'approvisionnement TIC",
    "mesure de sécurité": "Il convient de mettre en place des processus pour gérer la sécurité de l'information dans la chaîne d'approvisionnement des technologies de l'information et de la communication (TIC), en abordant les risques de sécurité liés aux fournisseurs de produits et services TIC.",
    "capability": [
      "Sécurité_des_relations_fournisseurs"
    ]
  },
  {
    "numero": "5.22",
    "titre": "Surveillance, révision et gestion des changements des services fournisseurs",
    "mesure de sécurité": "Il convient de surveiller et de réviser régulièrement les services fournis par les fournisseurs, et de gérer les changements apportés à ces services, afin de s'assurer que les contrôles de sécurité convenus restent efficaces et que les exigences de sécurité de l'information continuent d'être respectées.",
    "capability": [
      "Sécurité_des_relations_fournisseurs"
    ]
  },
  {
    "numero": "5.23",
    "titre": "Sécurité de l'information dans l'utilisation de services en nuage",
    "mesure de sécurité": "Il convient d'établir des processus pour l'acquisition, l'utilisation, la gestion et la cessation des services en nuage, conformément aux exigences de sécurité de l'information de l'organisation.",
    "capability": [
      "Sécurité_des_relations_fournisseurs"
    ]
  },
  {
    "numero": "5.24",
    "titre": "Planification et préparation de la gestion des incidents de sécurité de l'information",
    "mesure de sécurité": "Il convient de planifier et de préparer la gestion des incidents de sécurité de l'information en élaborant des procédures et des plans d'intervention, de façon à pouvoir réagir efficacement aux incidents et limiter leurs impacts.",
    "capability": [
      "Réglementation_et_conformité"
    ]
  },
  {
    "numero": "5.25",
    "titre": "Évaluation des événements de sécurité de l'information et prise de décision",
    "mesure de sécurité": "Il convient d'évaluer les événements de sécurité de l'information et de décider s'ils doivent être considérés comme des incidents de sécurité de l'information, afin de déclencher, le cas échéant, les procédures de réponse appropriées.",
    "capability": [
      "Gestion_des_événements_de_sécurité_de_l_information"
    ]
  },
  {
    "numero": "5.26",
    "titre": "Réponse aux incidents de sécurité de l'information",
    "mesure de sécurité": "Il convient de répondre aux incidents de sécurité de l'information en appliquant les procédures de gestion des incidents définies, afin de maîtriser l'incident, d'en minimiser l'impact et d'en informer les parties concernées dans les meilleurs délais.",
    "capability": [
      "Gestion_des_événements_de_sécurité_de_l_information"
    ]
  },
  {
    "numero": "5.27",
    "titre": "Tirer des enseignements des incidents de sécurité de l'information",
    "mesure de sécurité": "Il convient de tirer des enseignements de chaque incident de sécurité de l'information en analysant ses causes et son traitement, puis de mettre en œuvre les améliorations nécessaires des contrôles de sécurité pour éviter qu'il ne se reproduise.",
    "capability": [
      "Gestion_des_événements_de_sécurité_de_l_information"
    ]
  },
  {
    "numero": "5.28",
    "titre": "Collecte des preuves",
    "mesure de sécurité": "Il convient d'établir des procédures pour la collecte, la conservation, l'analyse et la présentation des preuves relatives aux incidents de sécurité de l'information, de façon à pouvoir soutenir d'éventuelles actions juridiques ou enquêtes ultérieures.",
    "capability": [
      "Gestion_des_événements_de_sécurité_de_l_information"
    ]
  },
  {
    "numero": "5.29",
    "titre": "Sécurité de l'information pendant une perturbation",
    "mesure de sécurité": "Il convient de s'assurer que la sécurité de l'information est maintenue pendant une perturbation des activités de l'organisation, en prévoyant des mesures de continuité appropriées pour protéger les informations et actifs critiques durant cette perturbation.",
    "capability": [
      "Continuité"
    ]
  },
  {
    "numero": "5.30",
    "titre": "Préparation des TIC pour la continuité d'activité",
    "mesure de sécurité": "Il convient de planifier, de mettre en œuvre, de maintenir et de tester la préparation des technologies de l'information et de la communication (TIC) en vue de la continuité d'activité, sur la base des objectifs de continuité d'activité et des exigences de continuité des TIC.",
    "capability": [
      "Continuité"
    ]
  },
  {
    "numero": "5.31",
    "titre": "Exigences légales, statutaires, réglementaires et contractuelles",
    "mesure de sécurité": "Il convient d'identifier et de documenter les exigences légales, statutaires, réglementaires et contractuelles applicables à la sécurité de l'information, ainsi que la démarche adoptée par l'organisation pour satisfaire à ces exigences.",
    "capability": [
      "Réglementation_et_conformité"
    ]
  },
  {
    "numero": "5.32",
    "titre": "Droits de propriété intellectuelle",
    "mesure de sécurité": "Il convient de mettre en œuvre des procédures appropriées pour garantir le respect des droits de propriété intellectuelle et des licences, en particulier lors de l'utilisation de logiciels et d'informations protégés par le droit d'auteur ou d'autres droits.",
    "capability": [
      "Réglementation_et_conformité"
    ]
  },
  {
    "numero": "5.33",
    "titre": "Protection des enregistrements",
    "mesure de sécurité": "Il convient de protéger les enregistrements (données archivées, journaux, documents officiels, etc.) contre la perte, la destruction, la falsification, l'accès non autorisé et la divulgation non autorisée, pendant toute leur durée de conservation.",
    "capability": [
      "Réglementation_et_conformité"
    ]
  },
  {
    "numero": "5.34",
    "titre": "Protection de la vie privée et des DCP",
    "mesure de sécurité": "Il convient d'assurer la confidentialité et la protection des données à caractère personnel (DCP) conformément aux lois et réglementations en vigueur, notamment en appliquant les principes de privacy by design et en mettant en œuvre des mesures techniques et organisationnelles appropriées.",
    "capability": [
      "Réglementation_et_conformité"
    ]
  },
  {
    "numero": "5.35",
    "titre": "Révision indépendante de la sécurité de l'information",
    "mesure de sécurité": "Il convient de faire procéder à des révisions indépendantes de la sécurité de l'information à intervalles planifiés, ou à la suite de changements significatifs, afin d'évaluer l'efficacité et la conformité du système de management de la sécurité de l'information de l'organisation.",
    "capability": [
      "Gouvernance",
      "Assurance_de_sécurité_de_l_information"
    ]
  },
  {
    "numero": "5.36",
    "titre": "Conformité aux politiques, règles et normes de sécurité de l'information",
    "mesure de sécurité": "Il convient de s'assurer de la conformité aux politiques, règles et normes de sécurité de l'information de l'organisation, en réalisant des vérifications régulières et des audits internes, et en prenant des mesures correctives en cas de non-conformité.",
    "capability": [
      "Gouvernance",
      "Assurance_de_sécurité_de_l_information"
    ]
  },
  {
    "numero": "5.37",
    "titre": "Procédures d'exploitation documentées",
    "mesure de sécurité": "Il convient de documenter les procédures d'exploitation et de les tenir à jour. Ces procédures d'exploitation documentées doivent être examinées régulièrement et mises à jour lorsque nécessaire. Les modifications apportées aux procédures doivent être autorisées et, lorsque cela est techniquement possible, les systèmes d'information doivent être gérés de manière cohérente en utilisant les mêmes procédures, outils et utilitaires.",
    "capability": [
      "Configuration_sécurisée"
    ]
  },
  {
    "numero": "6.1",
    "titre": "Sélection des candidats",
    "mesure de sécurité": "Il convient d'effectuer des vérifications d'antécédents pour tous les candidats avant leur entrée en fonction, et de façon périodique par la suite, en tenant compte des lois, réglementations et principes éthiques applicables, et en proportionnant le niveau de vérification aux exigences métier, à la sensibilité des informations qui seront accessibles et aux risques perçus.",
    "capability": [
      "Sécurité_des_ressources_humaines"
    ]
  },
  {
    "numero": "6.2",
    "titre": "Termes et conditions du contrat de travail",
    "mesure de sécurité": "Il convient que le contrat de travail définisse les responsabilités de l'employé et de l'organisation en matière de sécurité de l'information, notamment l'obligation de respecter les politiques et procédures de sécurité de l'information de l'organisation.",
    "capability": [
      "Sécurité_des_ressources_humaines"
    ]
  },
  {
    "numero": "6.3",
    "titre": "Sensibilisation, enseignement et formation en sécurité de l'information",
    "mesure de sécurité": "Il convient de mettre en œuvre des programmes de sensibilisation, d'enseignement et de formation à la sécurité de l'information pour tout le personnel de l'organisation, avec des mises à jour régulières, de sorte que chacun dispose des connaissances et compétences nécessaires adaptées à son rôle pour appliquer correctement les mesures de sécurité.",
    "capability": [
      "Sécurité_des_ressources_humaines"
    ]
  },
  {
    "numero": "6.4",
    "titre": "Processus disciplinaire",
    "mesure de sécurité": "Il convient de définir un processus disciplinaire permettant de prendre des sanctions appropriées en cas de manquement d'un employé aux politiques ou aux mesures de sécurité de l'information.",
    "capability": [
      "Sécurité_des_ressources_humaines"
    ]
  },
  {
    "numero": "6.5",
    "titre": "Responsabilités après la fin ou le changement d’un emploi",
    "mesure de sécurité": "Il convient de définir et de communiquer à la personne concernée les responsabilités en matière de sécurité de l'information qui continuent de s'appliquer après la fin ou le changement de son emploi (par exemple, la confidentialité à respecter, la restitution des actifs, etc.).",
    "capability": [
      "Sécurité_des_ressources_humaines"
    ]
  },
  {
    "numero": "6.6",
    "titre": "Accords de confidentialité ou de non-divulgation",
    "mesure de sécurité": "Il convient d'identifier les besoins de l'organisation en matière de confidentialité et de non-divulgation, et de mettre en place des accords de confidentialité ou de non-divulgation conformes à ces besoins. Ces accords doivent être revus régulièrement afin de s'assurer qu'ils demeurent adaptés aux exigences de sécurité de l'information.",
    "capability": [
      "Sécurité_des_ressources_humaines"
    ]
  },
  {
    "numero": "6.7",
    "titre": "Travail à distance",
    "mesure de sécurité": "Il convient de mettre en œuvre des mesures de sécurité pour le travail à distance, afin de protéger les informations consultées, traitées ou stockées en dehors des locaux de l'organisation (par exemple, utilisation de VPN, chiffrement des communications et des dispositifs, politiques de sécurité pour les postes de travail à distance, etc.).",
    "capability": [
      "Sécurité_des_ressources_humaines"
    ]
  },
  {
    "numero": "6.8",
    "titre": "Déclaration des événements de sécurité de l'information",
    "mesure de sécurité": "Il convient d'exiger de tous les employés et contractuels qu'ils déclarent rapidement, par les canaux appropriés, tout événement de sécurité de l'information observé ou suspecté, de manière à permettre une évaluation et une réponse efficaces à ces événements.",
    "capability": [
      "Sécurité_des_ressources_humaines",
      "Gestion_des_événements_de_sécurité_de_l_information"
    ]
  },
  {
    "numero": "7.1",
    "titre": "Périmètres de sécurité physique",
    "mesure de sécurité": "Il convient de définir des périmètres de sécurité physique autour des locaux et des zones critiques contenant des informations sensibles ou des actifs importants, et de mettre en place des protections appropriées à ces périmètres (p. ex. clôtures, murs, contrôles d'accès aux entrées) pour empêcher toute intrusion non autorisée.",
    "capability": [
      "Sécurité_physique"
    ]
  },
  {
    "numero": "7.2",
    "titre": "Les entrées physiques",
    "mesure de sécurité": "Il convient de contrôler les points d'entrée physiques des zones sécurisées au moyen de dispositifs de sécurité appropriés, de manière à ce que seules les personnes autorisées puissent pénétrer dans ces zones.",
    "capability": [
      "Sécurité_physique"
    ]
  },
  {
    "numero": "7.3",
    "titre": "Sécurisation des bureaux, des salles et des installations",
    "mesure de sécurité": "Il convient de mettre en œuvre des mesures de sécurité physique pour protéger les bureaux, les salles et les installations de l'organisation contenant des informations ou des actifs sensibles, par exemple en verrouillant les locaux, en utilisant des systèmes d'alarme et en limitant l'accès aux seules personnes autorisées.",
    "capability": [
      "Sécurité_physique"
    ]
  },
  {
    "numero": "7.4",
    "titre": "Surveillance de la sécurité physique",
    "mesure de sécurité": "Il convient de surveiller la sécurité physique des locaux de l'organisation (par exemple, par des systèmes de vidéosurveillance CCTV, des dispositifs de détection d'intrusion ou des rondes de sécurité) afin de détecter et dissuader toute tentative d'accès physique non autorisé ou tout incident de sécurité physique.",
    "capability": [
      "Sécurité_physique"
    ]
  },
  {
    "numero": "7.5",
    "titre": "Protection contre les menaces physiques et environnementales",
    "mesure de sécurité": "Il convient de mettre en place des mesures de protection contre les menaces physiques et environnementales – telles que les catastrophes naturelles, les incidents environnementaux (incendie, inondation) ou les actes malveillants – afin de réduire les risques pour les actifs de l'organisation. Par exemple, installer des détecteurs de fumée et des extincteurs, des systèmes anti-inondation, un contrôle de la température et de l'humidité, etc.",
    "capability": [
      "Sécurité_physique"
    ]
  },
  {
    "numero": "7.6",
    "titre": "Travail dans les zones sécurisées",
    "mesure de sécurité": "Il convient d'établir des procédures pour le travail dans les zones sécurisées, de façon à ce que seules les personnes autorisées y travaillent et qu'elles respectent des protocoles de sécurité spécifiques (par exemple, surveillance accrue, enregistrement des entrées/sorties, interdiction d'introduire des dispositifs non autorisés).",
    "capability": [
      "Sécurité_physique"
    ]
  },
  {
    "numero": "7.7",
    "titre": "Bureau vide et écran vide",
    "mesure de sécurité": "Il convient d'adopter une politique de bureau vide et d'écran vide, afin de s'assurer qu'aucune information sensible n'est laissée sans protection sur les bureaux ou affichée sur les écrans en l'absence des utilisateurs. Concrètement, les documents papier et supports amovibles doivent être rangés en lieu sûr lorsque non utilisés, et les sessions utilisateur verrouillées ou terminées dès qu'un poste de travail est inoccupé.",
    "capability": [
      "Sécurité_physique"
    ]
  },
  {
    "numero": "7.8",
    "titre": "Emplacement et protection du matériel",
    "mesure de sécurité": "Il convient d'installer le matériel informatique et les autres équipements dans des emplacements appropriés et de les protéger pour réduire les risques liés à des accès non autorisés ou à des menaces environnementales. Par exemple, placer les serveurs dans des salles dédiées avec contrôle d'accès et conditions environnementales maîtrisées, afin de préserver la sécurité et le bon fonctionnement des équipements.",
    "capability": [
      "Sécurité_physique"
    ]
  },
  {
    "numero": "7.9",
    "titre": "Sécurité des actifs hors des locaux",
    "mesure de sécurité": "Il convient d'assurer la sécurité des actifs (informations et équipements) lorsqu'ils sont utilisés ou stockés en dehors des locaux de l'organisation. Des mesures appropriées doivent être appliquées, telles que le chiffrement des données sur les ordinateurs portables, l'utilisation de dispositifs sécurisés pour le transport de documents, et des politiques encadrant le stockage d'informations à l'extérieur.",
    "capability": [
      "Sécurité_physique"
    ]
  },
  {
    "numero": "7.10",
    "titre": "Supports de stockage",
    "mesure de sécurité": "Il convient de gérer les supports de stockage (tels que disques durs, clés USB, CD/DVD, bandes magnétiques) tout au long de leur cycle de vie de manière sécurisée. Cela inclut de contrôler leur accès, de les stocker dans des conditions sûres, de les chiffrer si nécessaire, et de procéder à leur élimination ou réutilisation de manière sécurisée (par exemple par destruction physique ou effacement sécurisé des données).",
    "capability": [
      "Gestion_des_actifs",
      "Sécurité_physique"
    ]
  },
  {
    "numero": "7.11",
    "titre": "Services supports",
    "mesure de sécurité": "Il convient de protéger les services supports essentiels (tels que l'alimentation électrique, le réseau électrique, le système de climatisation, l'alimentation en eau et autres utilités) qui permettent le fonctionnement des systèmes d'information. Des solutions de secours ou de redondance doivent être prévues (comme des onduleurs, générateurs, doubles alimentations, etc.) afin d'assurer la continuité des opérations en cas de défaillance de ces services.",
    "capability": [
      "Sécurité_physique"
    ]
  },
  {
    "numero": "7.12",
    "titre": "Sécurité du câblage",
    "mesure de sécurité": "Il convient de protéger le câblage des réseaux et télécommunications contre les écoutes, les interférences ou les détériorations. Par exemple, utiliser des conduits ou chemins de câbles sécurisés, réduire les émissions électromagnétiques ou mettre en place des mécanismes de détection de coupure ou d'interception, afin d'empêcher qu'un tiers ne puisse compromettre la confidentialité ou l'intégrité des données en manipulant les câbles.",
    "capability": [
      "Sécurité_physique"
    ]
  },
  {
    "numero": "7.13",
    "titre": "Maintenance du matériel",
    "mesure de sécurité": "Il convient d'assurer une maintenance régulière et adéquate du matériel informatique, conformément aux recommandations du fabricant ou aux besoins de sécurité. Cette maintenance doit être réalisée par du personnel autorisé, et tout équipement de remplacement temporaire doit offrir un niveau de sécurité équivalent. L'objectif est de garantir que les équipements fonctionnent correctement et ne compromettent pas la sécurité de l'information (par exemple à cause de pannes non corrigées).",
    "capability": [
      "Sécurité_physique"
    ]
  },
  {
    "numero": "7.14",
    "titre": "Élimination ou recyclage sécurisé(e) du matériel",
    "mesure de sécurité": "Il convient d'éliminer ou de recycler le matériel de manière sécurisée lorsqu'il arrive en fin de vie ou qu'il n'est plus utilisé. Cela implique de s'assurer que toutes les informations et logiciels sensibles qu'il contient sont définitivement supprimés ou rendus inaccessibles avant sa mise au rebut, son recyclage ou sa réutilisation (par exemple en effectuant un effacement sécurisé des données ou en détruisant physiquement le support de stockage).",
    "capability": [
      "Sécurité_physique"
    ]
  },
  {
    "numero": "8.1",
    "titre": "Terminaux finaux des utilisateurs",
    "mesure de sécurité": "Il convient de mettre en œuvre des mesures de sécurité pour protéger les terminaux finaux des utilisateurs (postes de travail, ordinateurs portables, smartphones, tablettes, etc.) et les informations auxquelles ils ont accès. Par exemple, installer des logiciels de sécurité (antivirus, pare-feu personnel), appliquer des correctifs à jour, chiffrer les données sensibles stockées sur ces appareils, et contrôler les accès aux ports et interfaces.",
    "capability": [
      "Sécurité_système_et_réseau"
    ]
  },
  {
    "numero": "8.2",
    "titre": "Droits d'accès privilégiés",
    "mesure de sécurité": "Il convient de restreindre et de gérer l'attribution et l'utilisation des droits d'accès privilégiés (administrateurs systèmes, comptes avec privilèges élevés). Des processus formels doivent encadrer l'octroi de ces droits, et un suivi régulier doit être effectué pour vérifier leur utilisation appropriée et retirer les privilèges non nécessaires.",
    "capability": [
      "Protection_des_informations"
    ]
  },
  {
    "numero": "8.3",
    "titre": "Restrictions d'accès aux informations",
    "mesure de sécurité": "Il convient de restreindre l'accès aux informations et aux fonctions des systèmes en fonction des besoins et des autorisations de chaque utilisateur, conformément à la politique de contrôle d'accès de l'organisation. Cela signifie que chaque utilisateur ou entité ne doit pouvoir accéder qu'aux informations et fonctionnalités qui lui sont explicitement autorisées.",
    "capability": [
      "Protection_des_informations"
    ]
  },
  {
    "numero": "8.4",
    "titre": "Accès aux codes source",
    "mesure de sécurité": "Il convient de restreindre et de contrôler strictement l'accès aux codes source des logiciels, de manière à ce que seuls les développeurs et personnels autorisés puissent y accéder. Des mécanismes de gestion du code source (dépôts sécurisés, contrôle de versions avec gestion des droits) doivent être en place pour prévenir toute modification non autorisée ou fuite de propriété intellectuelle.",
    "capability": [
      "Gestion_des_identités_et_des_accès"
    ]
  },
  {
    "numero": "8.5",
    "titre": "Authentification sécurisée",
    "mesure de sécurité": "Il convient de mettre en place des mécanismes d'authentification sécurisés pour contrôler l'accès aux systèmes et aux informations. Par exemple, utiliser des méthodes robustes telles que l'authentification multi-facteur (MFA) lorsque cela est approprié, gérer les informations d'authentification (mots de passe, certificats, tokens) de façon sécurisée, et s'assurer que les procédures d'authentification répondent aux exigences de sécurité de l'organisation.",
    "capability": [
      "Gestion_des_identités_et_des_accès"
    ]
  },
  {
    "numero": "8.6",
    "titre": "Dimensionnement",
    "mesure de sécurité": "Il convient de surveiller et de gérer la capacité et les performances des ressources informatiques (processeurs, mémoire, stockage, bande passante réseau, etc.) afin de répondre aux besoins de l'organisation. Des ajustements proactifs doivent être réalisés pour anticiper les évolutions de charge et éviter toute saturation ou dégradation des performances qui pourrait affecter la disponibilité des services.",
    "capability": [
      "Gestion_des_identités_et_des_accès"
    ]
  },
  {
    "numero": "8.7",
    "titre": "Protection contre les programmes malveillants (malware)",
    "mesure de sécurité": "Il convient de déployer des contrôles de sécurité pour prévenir et détecter les programmes malveillants (malwares) et pour s'en remettre. Cela inclut l'installation de logiciels anti-malware à jour, l'analyse régulière des systèmes, la sensibilisation des utilisateurs aux risques liés aux logiciels malveillants (phishing, pièces jointes suspectes, etc.), et la mise en place de mesures de confinement et de récupération en cas d'infection (par exemple, plans de reprise).",
    "capability": [
      "Sécurité_système_et_réseau"
    ]
  },
  {
    "numero": "8.8",
    "titre": "Gestion des vulnérabilités techniques",
    "mesure de sécurité": "Il convient de gérer les vulnérabilités techniques des systèmes d'information en se tenant informé des vulnérabilités connues affectant les matériels, logiciels et services utilisés, et en appliquant en temps utile des mesures correctives (comme l'installation de correctifs de sécurité ou l'application de contournements) pour atténuer les risques associés.",
    "capability": [
      "Configuration_sécurisée",
      "Gestion_des_menaces_et_des_vulnérabilités"
    ]
  },
  {
    "numero": "8.9",
    "titre": "Gestion des configurations",
    "mesure de sécurité": "Il convient de définir, de documenter, de contrôler et de tenir à jour la configuration de tous les composants critiques des systèmes d'information (équipements, logiciels, services). Un processus de gestion des configurations doit s'assurer que seules des configurations approuvées sont utilisées, que tout changement est évalué, approuvé et suivi, et que des configurations sécurisées par défaut sont appliquées pour réduire les vulnérabilités.",
    "capability": [
      "Configuration_sécurisée"
    ]
  },
  {
    "numero": "8.10",
    "titre": "Suppression des informations",
    "mesure de sécurité": "Il convient de supprimer des systèmes d'information les données et informations qui ne sont plus nécessaires, de manière sécurisée et irréversible, sauf si leur conservation est requise par la loi ou par les politiques internes. Des procédures d'effacement sécurisé ou de destruction doivent être utilisées en fonction de la sensibilité des informations, afin d'empêcher toute restitution ou reconstruction ultérieure des données supprimées.",
    "capability": [
      "Protection_des_informations"
    ]
  },
  {
    "numero": "8.11",
    "titre": "Masquage des données",
    "mesure de sécurité": "Il convient d'utiliser le masquage des données (techniques d'anonymisation ou de pseudonymisation) pour limiter l'exposition des données sensibles ou personnelles lorsque cela est pertinent, notamment dans les environnements de test, de développement ou lors de partages d'informations. Le masquage réduit le risque de divulgation non autorisée en remplaçant les données réelles par des données fictives mais cohérentes.",
    "capability": [
      "Protection_des_informations"
    ]
  },
  {
    "numero": "8.12",
    "titre": "Prévention de la fuite de données",
    "mesure de sécurité": "Il convient de mettre en place des mesures de prévention des fuites de données (Data Loss Prevention, DLP) sur les systèmes et les réseaux de l'organisation, afin de détecter et prévenir la divulgation non autorisée d'informations sensibles. Par exemple, déployer des outils DLP qui contrôlent les transferts de données (courriels, téléchargements, copies sur supports amovibles) et bloquent ou alertent en cas de tentative d'exfiltration de données confidentielles.",
    "capability": [
      "Protection_des_informations"
    ]
  },
  {
    "numero": "8.13",
    "titre": "Sauvegarde des informations",
    "mesure de sécurité": "Il convient de maintenir des copies de sauvegarde des informations, des logiciels et des systèmes, et de tester régulièrement ces sauvegardes, conformément à la politique de sauvegarde convenue.",
    "capability": [
      "Protection_des_informations"
    ]
  },
  {
    "numero": "8.14",
    "titre": "Redondance des moyens de traitement de l'information",
    "mesure de sécurité": "Il convient de prévoir des moyens redondants pour les installations de traitement de l'information critiques, afin d'assurer la continuité des services en cas de défaillance d'un composant. Par exemple, mettre en place des serveurs redondants, des liaisons réseau doubles ou des centres de données de secours, de sorte qu'une panne matérielle ou logicielle n'affecte pas la disponibilité globale des systèmes.",
    "capability": [
      "Sécurité_système_et_réseau",
      "Continuité"
    ]
  },
  {
    "numero": "8.15",
    "titre": "Journalisation",
    "mesure de sécurité": "Il convient d'activer la journalisation des événements de sécurité et des activités significatives sur les systèmes d'information, et de protéger les journaux ainsi produits contre toute altération ou accès non autorisé. De plus, ces journaux doivent être examinés régulièrement pour détecter des événements inhabituels ou des signes d'incidents de sécurité nécessitant une enquête.",
    "capability": [
      "Assurance_de_sécurité_de_l_information"
    ]
  },
  {
    "numero": "8.16",
    "titre": "Activités de surveillance",
    "mesure de sécurité": "Il convient de mettre en place des activités de surveillance continues (monitoring) des réseaux, systèmes et applications de l'organisation, afin de détecter en temps opportun les comportements anormaux, les tentatives d'accès non autorisé ou autres événements de sécurité. Ceci peut inclure l'utilisation de systèmes de détection d'intrusion (IDS/IPS), de systèmes de gestion des informations et des événements de sécurité (SIEM) et d'outils d'analyse des journaux.",
    "capability": [
      "Assurance_de_sécurité_de_l_information"
    ]
  },
  {
    "numero": "8.17",
    "titre": "Synchronisation des horloges",
    "mesure de sécurité": "Il convient de synchroniser les horloges de tous les systèmes informatiques pertinents de l'organisation sur une source de temps commune de référence, afin que les horodatages des événements consignés dans les journaux soient cohérents et précis. Cette synchronisation facilite la corrélation des événements de sécurité à travers différents systèmes et peut être réalisée, par exemple, via le protocole NTP.",
    "capability": [
      "Assurance_de_sécurité_de_l_information"
    ]
  },
  {
    "numero": "8.18",
    "titre": "Utilisation de programmes utilitaires à privilèges",
    "mesure de sécurité": "Il convient de restreindre et de contrôler l'utilisation des programmes utilitaires à privilèges (par exemple, outils d'administration système capables de contourner les contrôles de sécurité). Ces programmes ne doivent être accessibles qu'aux personnes autorisées et pour des besoins justifiés, et leur utilisation doit être surveillée et enregistrée afin d'éviter tout abus ou usage malveillant.",
    "capability": [
      "Configuration_sécurisée"
    ]
  },
  {
    "numero": "8.19",
    "titre": "Installation de logiciels sur des systèmes opérationnels",
    "mesure de sécurité": "Il convient de contrôler l'installation de logiciels sur les systèmes opérationnels par le biais de procédures formelles. Toute installation de logiciel doit être autorisée, testée et approuvée préalablement afin de garantir qu'elle n'introduit pas de vulnérabilités ou de perturbations, et de vérifier qu'elle respecte les politiques de l'organisation (notamment en matière de licences et de sécurité).",
    "capability": [
      "Configuration_sécurisée"
    ]
  },
  {
    "numero": "8.20",
    "titre": "Sécurité des réseaux",
    "mesure de sécurité": "Il convient de gérer et de protéger les réseaux de l'organisation afin de préserver la confidentialité, l'intégrité et la disponibilité des informations qui y transitent. Des mesures de sécurité appropriées doivent être mises en œuvre, telles que la segmentation du réseau, l'utilisation de pare-feux, le chiffrement des communications sensibles et la surveillance du trafic réseau pour détecter d'éventuelles activités malveillantes.",
    "capability": [
      "Sécurité_système_et_réseau"
    ]
  },
  {
    "numero": "8.21",
    "titre": "Sécurité des services réseau",
    "mesure de sécurité": "Il convient de définir les exigences de sécurité, les niveaux de service attendus et les responsabilités de gestion pour chaque service réseau utilisé par l'organisation (par exemple, service de télécommunication, hébergement internet, services cloud). Ces éléments doivent être formalisés dans les contrats ou accords de service correspondants, que les services réseau soient fournis en interne ou externalisés, afin d'assurer un niveau de sécurité conforme aux besoins de l'organisation.",
    "capability": [
      "Sécurité_système_et_réseau",
      "Gestion_des_identités_et_des_accès"
    ]
  },
  {
    "numero": "8.22",
    "titre": "Cloisonnement des réseaux",
    "mesure de sécurité": "Il convient de cloisonner les réseaux de l'organisation en zones ou segments, selon les besoins de sécurité, afin d'isoler les systèmes et informations sensibles. Ce cloisonnement (via des VLAN, des sous-réseaux séparés, des pare-feux entre zones, etc.) vise à limiter la propagation des incidents et à appliquer des règles de sécurité spécifiques par zone en fonction du niveau de confiance ou de criticité des actifs qui y résident.",
    "capability": [
      "Sécurité_système_et_réseau"
    ]
  },
  {
    "numero": "8.23",
    "titre": "Filtrage web",
    "mesure de sécurité": "Il convient de mettre en place un mécanisme de filtrage web pour contrôler et restreindre l'accès des utilisateurs aux sites web, en accord avec la politique de l'organisation. Par exemple, cela peut inclure le blocage des sites malveillants ou inappropriés, la catégorisation des sites autorisés/interdits, et la surveillance des activités web pour prévenir les menaces en ligne (comme l'accès à des sites de phishing ou de malware).",
    "capability": [
      "Sécurité_système_et_réseau"
    ]
  },
  {
    "numero": "8.24",
    "titre": "Utilisation de la cryptographie",
    "mesure de sécurité": "Il convient d'utiliser des techniques cryptographiques appropriées pour protéger la confidentialité, l'intégrité et, le cas échéant, l'authenticité des informations. Une politique d'utilisation de la cryptographie doit être définie, couvrant des aspects tels que les algorithmes autorisés, la gestion des clés cryptographiques, et les cas d'utilisation (chiffrement des données stockées, chiffrement des communications, signatures numériques, etc.).",
    "capability": [
      "Protection_des_informations"
    ]
  },
  {
    "numero": "8.25",
    "titre": "Cycle de vie de développement sécurisé",
    "mesure de sécurité": "Il convient d'intégrer des pratiques de sécurité tout au long du cycle de vie de développement des systèmes d'information. Cela implique d'établir des règles et contrôles de sécurité lors des phases de spécification, conception, développement, test, déploiement et maintenance, afin de s'assurer que les logiciels et systèmes sont conçus et développés en respectant les principes de sécurité (par exemple, en effectuant des revues de code, des tests de sécurité applicative, en suivant des guides de codage sécurisé, etc.).",
    "capability": [
      "Sécurité_des_applications"
    ]
  },
  {
    "numero": "8.26",
    "titre": "Exigences de sécurité des applications",
    "mesure de sécurité": "Il convient d'identifier les exigences de sécurité pertinentes pour les nouvelles applications ou évolutions d'applications dès les premières étapes de leur acquisition ou développement, et de s'assurer que ces exigences sont prises en compte dans les spécifications et mises en œuvre dans les solutions finales.",
    "capability": [
      "Sécurité_système_et_réseau",
      "Sécurité_des_applications"
    ]
  },
  {
    "numero": "8.27",
    "titre": "Principes d'ingénierie et d'architecture des systèmes sécurisés",
    "mesure de sécurité": "Il convient de définir, documenter et appliquer des principes d'architecture et d'ingénierie pour la conception de systèmes sécurisés. Ces principes (par exemple, le principe du moindre privilège, la défense en profondeur, la simplicité, la modularité, la résilience) doivent guider le développement ou l'acquisition de nouvelles technologies afin de s'assurer que la sécurité est intégrée de manière cohérente et efficace dans l'architecture des systèmes d'information.",
    "capability": [
      "Sécurité_des_applications"
    ]
  },
  {
    "numero": "8.28",
    "titre": "Codage sécurisé",
    "mesure de sécurité": "Il convient d'appliquer des pratiques de codage sécurisé lors du développement de logiciels, de façon à réduire les vulnérabilités dans le code. Cela peut inclure le respect de normes de codage, l'utilisation d'outils d'analyse de code statique pour détecter les failles courantes, la sensibilisation des développeurs aux erreurs de programmation susceptibles d'affecter la sécurité, et la réalisation de revues de code focalisées sur la sécurité.",
    "capability": [
      "Sécurité_des_applications"
    ]
  },
  {
    "numero": "8.29",
    "titre": "Tests de sécurité dans le développement et l'acceptation",
    "mesure de sécurité": "Il convient d'effectuer des tests de sécurité lors des phases de développement et d'acceptation des nouveaux systèmes ou des modifications système. Ces tests (tels que des tests d'intrusion, des analyses de vulnérabilités, des tests de charge pour vérifier la résilience) doivent permettre de vérifier que les contrôles de sécurité fonctionnent comme prévu et que le système répond aux critères de sécurité définis avant sa mise en production.",
    "capability": [
      "Sécurité_des_applications"
    ]
  },
  {
    "numero": "8.30",
    "titre": "Développement externalisé",
    "mesure de sécurité": "Il convient de prendre en compte les exigences de sécurité lorsque le développement de logiciels ou de systèmes est externalisé. L'organisation doit s'assurer que les prestataires respectent des critères de sécurité équivalents (via des clauses contractuelles, des contrôles réguliers, des droits d'audit) et doit suivre l'avancement du projet pour vérifier que les mesures de sécurité convenues sont bien appliquées pendant le développement externalisé.",
    "capability": [
      "Sécurité_des_applications"
    ]
  },
  {
    "numero": "8.31",
    "titre": "Séparation des environnements de développement, de test et opérationnels",
    "mesure de sécurité": "Il convient de séparer les environnements de développement, de test et d'exploitation (production) afin de réduire les risques de modifications non autorisées ou d'accès inappropriés aux données de production. Chaque environnement doit être isolé avec des jeux de données distincts (les tests ne devant pas se faire sur des données de production réelles sans précautions), et les droits d'accès des personnels doivent être limités à l'environnement dans lequel ils opèrent.",
    "capability": [
      "Sécurité_des_applications"
    ]
  },
  {
    "numero": "8.32",
    "titre": "Gestion des changements",
    "mesure de sécurité": "Il convient de contrôler et d'approuver formellement tous les changements apportés aux systèmes d'information, aux logiciels ou aux services, au moyen d'un processus de gestion des changements. Ce processus doit inclure l'évaluation préalable des impacts de sécurité des changements, l'autorisation par les instances appropriées, le test et la documentation des modifications, et un suivi pour s'assurer que les changements sont correctement implémentés et qu'ils n'ont pas d'effets indésirables sur la sécurité.",
    "capability": [
      "Sécurité_des_applications"
    ]
  },
  {
    "numero": "8.33",
    "titre": "Informations de test",
    "mesure de sécurité": "Il convient de sélectionner soigneusement les informations utilisées pour les tests (jeux de données de test) et de les protéger au même titre que des données réelles si elles sont sensibles. Idéalement, des données fictives ou anonymisées devraient être utilisées. Toute information de test doit être gérée et effacée une fois les tests terminés, afin de prévenir toute fuite ou utilisation abusive.",
    "capability": [
      "Protection_des_informations"
    ]
  },
  {
    "numero": "8.34",
    "titre": "Protection des systèmes d'information pendant les tests d'audit",
    "mesure de sécurité": "Il convient de contrôler l'accès aux systèmes d'information lors de la réalisation de tests d'audit (internes ou externes) afin de s'assurer que ces tests n'altèrent pas la sécurité ou la disponibilité des systèmes et des données. Par exemple, les auditeurs ne devraient accéder qu'aux informations et ressources nécessaires à leur audit, et leurs activités de test devraient être surveillées. Des mesures doivent être en place pour restaurer l'état normal du système après les tests d'audit et corriger toute faiblesse identifiée.",
    "capability": [
      "Sécurité_des_applications"
    ]
  }
];

var nis2_controls_data = [
  {
    fonction: "Identifier (ID)",
    categorie: "Gestion d'actifs (ID.AM)",
    description: "Les données, le personnel, les dispositifs, les systèmes et les installations qui permettent à l'organisation d'atteindre ses objectifs opérationnels sont identifiés et gérés en fonction de leur importance relative par rapport aux objectifs de l'organisation et à sa stratégie en matière de risques."
  },
  {
    fonction: "Identifier (ID)",
    categorie: "Environnement opérationnel (ID.BE)",
    description: "La mission, les objectifs, les parties prenantes et les activités de l'organisation sont compris et classés par ordre de priorité ; ces informations sont utilisées pour définir les rôles, les responsabilités et les décisions en matière de gestion des risques liés à la cybersécurité."
  },
  {
    fonction: "Identifier (ID)",
    categorie: "Gouvernance (ID.GV)",
    description: "Les politiques, procédures et processus de gestion et de suivi des exigences réglementaires, juridiques, environnementales et opérationnelles de l'organisation sont compris et contribuent à la gestion du risque de cybersécurité."
  },
  {
    fonction: "Identifier (ID)",
    categorie: "Évaluation des risques (ID.RA)",
    description: "L'organisation comprend le risque de cybersécurité pour les opérations de l'organisation (y compris la mission, les fonctions, l'image ou la réputation), ses actifs et les individus."
  },
  {
    fonction: "Identifier (ID)",
    categorie: "Stratégie de gestion des risques (ID.RM)",
    description: "Les priorités, contraintes, tolérances au risque et hypothèses de l'organisation sont établies et utilisées pour soutenir les décisions relatives au risque opérationnel."
  },
  {
    fonction: "Identifier (ID)",
    categorie: "Gestion des risques liés à la chaîne d'approvisionnement (ID.SC)",
    description: "Les priorités, les contraintes, les tolérances au risque et les hypothèses de l'organisation sont établies et utilisées pour soutenir les décisions liées à la gestion des risques de la chaîne d'approvisionnement. L'organisation a établi et mis en œuvre les processus pour identifier, évaluer et gérer les risques liés à la chaîne d'approvisionnement."
  },
  {
    fonction: "Protéger (PR)",
    categorie: "Gestion des identités, authentification et contrôle d'accès (PR.AC)",
    description: "L'accès aux actifs physiques et logiques et aux installations associées est limité aux utilisateurs, processus et dispositifs autorisés, et est géré en fonction du risque évalué d'accès non autorisé aux activités et transactions autorisées."
  },
  {
    fonction: "Protéger (PR)",
    categorie: "Sensibilisation et formation (PR.AT)",
    description: "Le personnel et les partenaires de l'organisation reçoivent une formation de sensibilisation à la cybersécurité et sont formés à l'exécution de leurs tâches et responsabilités liées à la cybersécurité, conformément aux politiques, procédures et accords connexes."
  },
  {
    fonction: "Protéger (PR)",
    categorie: "Sécurité des données (PR.DS)",
    description: "Les informations et les enregistrements (données) sont gérés conformément à la stratégie de l'organisation en matière de risques afin de protéger la confidentialité, l'intégrité et la disponibilité des informations."
  },
  {
    fonction: "Protéger (PR)",
    categorie: "Processus et procédures de protection de l'information (PR.IP)",
    description: "Les politiques de sécurité (qui traitent de l'objectif, de la portée, des rôles, des responsabilités, de l'engagement de la direction et de la coordination entre les entités organisationnelles), les processus et les procédures sont maintenus et utilisés pour gérer la protection des systèmes d'information et des actifs."
  },
  {
    fonction: "Protéger (PR)",
    categorie: "Maintenance (PR.MA)",
    description: "La maintenance et la réparation des composants des systèmes de contrôle et d'information industriels sont effectuées conformément aux politiques et procédures."
  },
  {
    fonction: "Protéger (PR)",
    categorie: "Technologie de protection (PR.PT)",
    description: "Les solutions de sécurité technique sont gérées de manière à garantir la sécurité et la résilience des systèmes et des actifs, conformément aux politiques, procédures et accords connexes."
  },
  {
    fonction: "Détecter (DE)",
    categorie: "Anomalies et événements (DE.AE)",
    description: "Les activités anormales sont détectées et l'impact potentiel des événements est compris."
  },
  {
    fonction: "Détecter (DE)",
    categorie: "Surveillance continue de la sécurité (DE.CM)",
    description: "Le système d'information et les actifs sont surveillés pour identifier les événements de cybersécurité et vérifier l'efficacité des mesures de protection."
  },
  {
    fonction: "Détecter (DE)",
    categorie: "Processus de détection (DE.DP)",
    description: "Les processus et procédures de détection sont maintenus et testés pour garantir la prise de conscience des événements anormaux."
  },
  {
    fonction: "Répondre (RS)",
    categorie: "Planification de la réponse (RS.RP)",
    description: "Les processus et procédures de réponse sont exécutés et maintenus, afin de garantir une réponse aux incidents de cybersécurité détectés."
  },
  {
    fonction: "Répondre (RS)",
    categorie: "Communications (RS.CO)",
    description: "Les activités de réponse sont coordonnées avec les parties prenantes internes et externes (par exemple, le soutien externe des forces de l'ordre)."
  },
  {
    fonction: "Répondre (RS)",
    categorie: "Analyse (RS.AN)",
    description: "Une analyse est effectuée pour garantir une réponse efficace et soutenir les activités de rétablissement."
  },
  {
    fonction: "Répondre (RS)",
    categorie: "Atténuation (RS.MI)",
    description: "Des activités sont effectuées pour empêcher l'expansion d'un événement, atténuer ses effets et résoudre l'incident."
  },
  {
    fonction: "Répondre (RS)",
    categorie: "Améliorations (RS.IM)",
    description: "Les activités de réponse de l'organisation sont améliorées par l'intégration des enseignements tirés des activités de détection/réponse actuelles et précédentes."
  },
  {
    fonction: "Rétablir (RC)",
    categorie: "Planification du rétablissement (RC.RP)",
    description: "Des processus et des procédures de rétablissement sont exécutés et maintenus pour assurer la restauration des systèmes ou des actifs touchés par des incidents de cybersécurité."
  },
  {
    fonction: "Rétablir (RC)",
    categorie: "Améliorations (RC.IM)",
    description: "Les plans de rétablissement et les processus de reprise sont améliorés en intégrant les leçons apprises dans les activités futures."
  },
  {
    fonction: "Rétablir (RC)",
    categorie: "Communications (RC.CO)",
    description: "Les activités de restauration sont coordonnées avec des parties internes et externes (par exemple, les centres de coordination, les fournisseurs de services Internet, les propriétaires des systèmes attaquants, les victimes, d'autres CSIRT et les vendeurs)."
  }
];

const CATEGORY_LABELS = nis2_controls_data.reduce((acc, item) => {
  const match = item.categorie.match(/\(([^)]+)\)/);
  if (match) acc[match[1]] = item.categorie;
  return acc;
}, {});

const demoControls = (() => {
  const statuses = ["non_implemente", "en_cours", "implemente", "verifie"];
  const capabilityBaseline = {
    gouvernance: 2.8,
    gestion_des_actifs: 2.5,
    protection_des_informations: 2.4,
    securite_des_ressources_humaines: 2.0,
    securite_physique: 2.1,
    securite_systeme_et_reseau: 2.3,
    securite_des_applications: 2.0,
    configuration_securisee: 1.9,
    gestion_des_identites_et_des_acces: 2.4,
    gestion_des_menaces_et_des_vulnerabilites: 1.8,
    continuite: 2.2,
    securite_des_relations_fournisseurs: 1.7,
    reglementation_et_conformite: 2.6,
    gestion_des_evenements_de_securite_de_l_information: 2.0,
    assurance_de_securite_de_l_information: 1.8
  };
  const toCmmLevel = (score) => {
    if (score >= 4.5) return "Optimis\xE9";
    if (score >= 3.6) return "G\xE9r\xE9";
    if (score >= 2.8) return "D\xE9fini";
    if (score >= 2.0) return "R\xE9p\xE9table";
    return "Initial";
  };
  const randStatus = () => statuses[Math.floor(Math.random() * statuses.length)];
  const jitter = () => (Math.random() * 0.8) - 0.4; // +/-0.4 pour éviter l'uniformité
  return iso27002_data_default.map((item, index) => {
    const subs = Array.isArray(item.capability)
      ? item.capability.map((c) => normalizeCapabilityKey(c))
      : [normalizeCapabilityKey(item.capability)];
    const base = capabilityBaseline[subs[0]] ?? 2.1;
    const score = Math.max(1, Math.min(5, base + jitter()));
    return {
      id: `CTRL-DEMO-${String(index + 1).padStart(3, "0")}`,
      reference: item.numero ? `${item.numero} ${item.titre}` : item.titre,
      numero: item.numero || String(index + 1),
      title: item.titre,
      measure: item["mesure de s\xE9curit\xE9"],
      description: item["mesure de s\xE9curit\xE9"],
      recommendations: [],
      category: DOMAIN_BY_PREFIX[item.numero?.split('.')[0]] || 'Autre',
      subcategory: subs,
      status: randStatus(),
      cmm: toCmmLevel(score),
      comments: "",
      evidence: "",
      justificationControl: ""
    };
  });
})();
// IDs des risques de démo (définis plus bas dans DEMO_DATA)
const demoRiskIds = [
  "risk-demo-1", "risk-demo-2", "risk-demo-3", "risk-demo-4", "risk-demo-5",
  "risk-demo-6", "risk-demo-7", "risk-demo-8", "risk-demo-9", "risk-demo-10"
];

const demoSoa = demoControls.map((ctrl, index) => {
  const isApplicable = Math.random() < 0.85;
  const nonApplicableJustifications = [
    "Non applicable - Contexte organisationnel ne nécessite pas ce contrôle",
    "Hors périmètre du SMSI actuel",
    "Non pertinent pour notre secteur d'activité",
    "Couvert par un contrôle compensatoire",
    "Infrastructure externalisée - responsabilité du fournisseur",
    "Activité non réalisée dans l'organisation"
  ];

  // Assigner 0-3 risques aléatoires si applicable
  let linkedRiskIds = [];
  if (isApplicable && Math.random() > 0.4) {
    const numRisks = Math.floor(Math.random() * 3) + 1;
    const shuffled = [...demoRiskIds].sort(() => Math.random() - 0.5);
    linkedRiskIds = shuffled.slice(0, numRisks);
  }

  return {
    controlId: ctrl.id,
    applicable: isApplicable,
    justification: isApplicable ? "" : nonApplicableJustifications[index % nonApplicableJustifications.length],
    linkedRiskIds: linkedRiskIds
  };
});
const demoDocumentReview = createDefaultDocumentReview().map((item, index) => ({
  reference: item.reference,
  justification: `Ex: Justification ${index + 1}`,
  tool: ""
}));
// Jeu de maturité NIS2 de démonstration (utilisé pour remplir les radars)
const demoNis2Maturity = {
  "ID.AM": 1.6,
  "ID.BE": 2.4,
  "ID.GV": 2.1,
  "ID.RA": 1.9,
  "ID.RM": 2.2,
  "ID.SC": 1.4,
  "PR.AC": 2.7,
  "PR.AT": 1.8,
  "PR.DS": 2.5,
  "PR.IP": 2.0,
  "PR.MA": 1.6,
  "PR.PT": 2.3,
  "DE.AE": 1.7,
  "DE.CM": 2.6,
  "DE.DP": 2.0,
  "RS.RP": 1.8,
  "RS.CO": 2.4,
  "RS.AN": 1.6,
  "RS.MI": 2.2,
  "RS.IM": 1.7,
  "RC.RP": 2.1,
  "RC.IM": 1.8,
  "RC.CO": 2.3
};
const demoNis2Controls = loadNis2Controls().map((ctrl, idx) => {
  const statusCycle = ["non_implemente", "en_cours", "implemente", "verifie"];
  const cmmCycle = ["Initial", "R\xE9p\xE9table", "D\xE9fini", "G\xE9r\xE9", "Optimis\xE9"];
  return Object.assign({}, ctrl, {
    status: statusCycle[idx % statusCycle.length],
    cmm: cmmCycle[(idx + 1) % cmmCycle.length],
    justification: ctrl.justification || ""
  });
});
const demoNis2Plan = (() => {
  const plan = cloneDefaultNis2Plan({ includeActions: true });
  plan.statuses = [
    { id: "nis2-status-1", text: "Cartographie des actifs critiques consolidée", status: "done" },
    { id: "nis2-status-2", text: "Comité de pilotage NIS2 opérationnel", status: "in_progress" },
    { id: "nis2-status-3", text: "Circuit de notification 24/7 défini", status: "in_progress" },
    { id: "nis2-status-4", text: "Plan d'exercices de crise validé", status: "todo" },
    { id: "nis2-status-5", text: "Catalogue fournisseurs priorisés", status: "todo" },
    { id: "nis2-status-6", text: "Tableau de bord NIS2 publié", status: "done" }
  ];
  plan.budgetBlocks = [
    { id: "nis2-budget-1", title: "Budget de mise en conformité 2026-2028" },
    { id: "nis2-budget-2", title: "Budget 2026" },
    { id: "nis2-budget-3", title: "Budget infra (3 ans)" },
    { id: "nis2-budget-4", title: "Budget SSI (3 ans)" }
  ];
  plan.customBlocks = [
    { id: "nis2-highlight-1", title: "Budget engagé 2026", value: 540000, format: "currency" },
    { id: "nis2-highlight-2", title: "Couverture MFA comptes sensibles", value: 68, format: "percent" },
    { id: "nis2-highlight-3", title: "Fournisseurs critiques évalués", value: 41, format: "percent" }
  ];
  const demoActionTemplates = {
    "nis2-domain-1": [
      { text: "Mettre à jour la politique NIS2 et la valider en comité.", phase: 1, status: "in_progress", workType: "programme" },
      { text: "Formaliser la matrice RACI SSI.", phase: 1, status: "done" },
      { text: "Définir les indicateurs de suivi trimestriels.", phase: 2, status: "todo", workType: "service" },
      { text: "Mettre en place la revue annuelle de conformité.", phase: 3, status: "todo", workType: "initiative" }
    ],
    "nis2-domain-2": [
      { text: "Recalibrer l'appétence au risque et les critères d'acceptation.", phase: 1, status: "done" },
      { text: "Réaliser l'analyse des risques sur les processus critiques.", phase: 2, status: "in_progress", workType: "projet" },
      { text: "Définir le plan de traitement priorisé.", phase: 3, status: "todo" }
    ],
    "nis2-domain-3": [
      { text: "Actualiser le playbook d'escalade et les contacts 24/7.", phase: 1, status: "in_progress", workType: "service" },
      { text: "Mettre en place le journal de notification NIS2.", phase: 2, status: "todo" },
      { text: "Exécuter un exercice de notification en 24h.", phase: 3, status: "todo" }
    ],
    "nis2-domain-4": [
      { text: "Revoir la BIA sur les services essentiels.", phase: 1, status: "done" },
      { text: "Définir les RTO/RPO cibles par service.", phase: 1, status: "in_progress" },
      { text: "Tester un scénario de reprise multi-sites.", phase: 2, status: "todo", workType: "projet" },
      { text: "Mettre à jour le kit de communication de crise.", phase: 3, status: "suspended", workType: "initiative" }
    ],
    "nis2-domain-5": [
      { text: "Segmenter les fournisseurs selon la criticité.", phase: 1, status: "done" },
      { text: "Lancer la campagne d'évaluation des tiers.", phase: 2, status: "in_progress" },
      { text: "Intégrer des clauses de notification d'incident.", phase: 3, status: "todo", workType: "service" }
    ],
    "nis2-domain-6": [
      { text: "Intégrer des exigences sécurité dans les achats.", phase: 1, status: "in_progress", workType: "service" },
      { text: "Mettre en place des revues de code sécurité.", phase: 2, status: "todo", workType: "projet" },
      { text: "Mettre à jour le processus de patching applicatif.", phase: 2, status: "todo" }
    ],
    "nis2-domain-7": [
      { text: "Centraliser les journaux critiques (SIEM).", phase: 1, status: "in_progress", workType: "programme" },
      { text: "Définir les règles de rétention et d'horodatage.", phase: 2, status: "todo" },
      { text: "Planifier des audits de traçabilité semestriels.", phase: 3, status: "todo", workType: "initiative" }
    ],
    "nis2-domain-8": [
      { text: "Mettre en place un calendrier de scans.", phase: 1, status: "done" },
      { text: "Définir la SLA de remédiation par criticité.", phase: 2, status: "in_progress" },
      { text: "Suivre les vulnérabilités des dépendances logicielles.", phase: 3, status: "todo" }
    ],
    "nis2-domain-9": [
      { text: "Lancer le parcours de formation cybersécurité.", phase: 1, status: "in_progress", workType: "programme" },
      { text: "Mettre en place des campagnes de phishing interne.", phase: 2, status: "todo" },
      { text: "Mesurer le taux de couverture par population.", phase: 3, status: "todo" }
    ],
    "nis2-domain-10": [
      { text: "Inventorier les usages de chiffrement.", phase: 1, status: "done" },
      { text: "Standardiser les algorithmes et tailles de clés.", phase: 2, status: "in_progress" },
      { text: "Planifier la rotation des certificats.", phase: 3, status: "todo" }
    ],
    "nis2-domain-11": [
      { text: "Cartographier les zones sensibles et accès.", phase: 1, status: "in_progress" },
      { text: "Mettre en place un contrôle d'accès renforcé.", phase: 2, status: "todo" },
      { text: "Tester la procédure d'accès d'urgence.", phase: 3, status: "todo" }
    ],
    "nis2-domain-12": [
      { text: "Réviser les profils à privilèges.", phase: 1, status: "in_progress" },
      { text: "Déployer MFA sur les comptes critiques.", phase: 2, status: "done" },
      { text: "Automatiser les revues trimestrielles d'habilitations.", phase: 3, status: "todo", workType: "service" }
    ]
  };
  const budgetProfiles = [
    { budget2026: 170000, budget2027: 130000, budget2028: 95000, budgetCategory: "ssi", owner: "Camille Renard", workType: "programme", progress: 64 },
    { budget2026: 98000, budget2027: 76000, budget2028: 56000, budgetCategory: "infra", owner: "Louis Perret", workType: "projet", progress: 36 },
    { budget2026: 82000, budget2027: 61000, budget2028: 45000, budgetCategory: "divers", owner: "Sara Nguyen", workType: "initiative", progress: 22 },
    { budget2026: 145000, budget2027: 112000, budget2028: 82000, budgetCategory: "infra", owner: "Julien Moreau", workType: "service", progress: 48 },
    { budget2026: 115000, budget2027: 90000, budget2028: 70000, budgetCategory: "ssi", owner: "Nora Ben Amar", workType: "programme", progress: 57 },
    { budget: 230000, budgetCategory: "infra", owner: "Alexis Bernard", workType: "projet", progress: 31 },
    { budget: 155000, budgetCategory: "ssi", owner: "Marine Dupont", workType: "service", progress: 43 },
    { budget2026: 72000, budget2027: 52000, budget2028: 39000, budgetCategory: "divers", owner: "Omar El Khoury", workType: "initiative", progress: 19 },
    { budget2026: 135000, budget2027: 100000, budget2028: 75000, budgetCategory: "ssi", owner: "Helene Caron", workType: "projet", progress: 61 },
    { budget2026: 92000, budget2027: 68000, budget2028: 51000, budgetCategory: "infra", owner: "Renaud Leclerc", workType: "programme", progress: 52 },
    { budget2026: 64000, budget2027: 47000, budget2028: 35000, budgetCategory: "divers", owner: "Claire Martin", workType: "initiative", progress: 26 },
    { budget: 195000, budgetCategory: "ssi", owner: "Noah Fontaine", workType: "service", progress: 39 }
  ];
  let budgetIndex = 0;
  plan.domains = plan.domains.map((domain, domainIndex) => {
    const templates = demoActionTemplates[domain.id] || domain.actions || [];
    const actions = templates.map((template, actionIndex) => {
      const base = (domain.actions || [])[actionIndex] || {};
      const profile = budgetProfiles[budgetIndex % budgetProfiles.length];
      budgetIndex += 1;
      const status = template.status || base.status || "todo";
      const progress = status === "done"
        ? 100
        : status === "in_progress"
          ? (typeof profile.progress === "number" ? profile.progress : 55)
          : status === "suspended"
            ? 20
            : 0;
      return Object.assign({}, base, template, {
        id: base.id || `nis2-act-${domainIndex + 1}-${actionIndex + 1}`,
        status,
        owner: template.owner || base.owner || profile.owner || "",
        workType: template.workType || base.workType || profile.workType || "initiative",
        budgetCategory: template.budgetCategory || profile.budgetCategory || base.budgetCategory || "divers",
        budget: typeof profile.budget !== "undefined" ? profile.budget : base.budget,
        budget2026: typeof profile.budget2026 !== "undefined" ? profile.budget2026 : base.budget2026,
        budget2027: typeof profile.budget2027 !== "undefined" ? profile.budget2027 : base.budget2027,
        budget2028: typeof profile.budget2028 !== "undefined" ? profile.budget2028 : base.budget2028,
        progress
      });
    });
    return Object.assign({}, domain, { actions });
  });
  return plan;
})();
const demoCriticalAssetsMeta = {
  sectionTitle: "Actifs critiques",
  tableTitle: "Actifs critiques",
  subtitle: "Cartographie des actifs critiques et exigences de continuité.",
  metaLine: "Version 1.2 | 14 octobre 2025"
};
const demoCriticalAssets = [
  {
    id: "asset-demo-1",
    priority: 1,
    beneficiary: "Direction Transports",
    productCode: "NAVIG",
    productName: "Portail de gestion des horaires",
    productTags: ["Cloud", "Externe"],
    rto: "2 h",
    rpo: "30 min",
    availability: "Niveau 2 - Reprise rapide",
    availabilityNote: "Heures de pointe matin/soir",
    confidentiality: "Sensible",
    impact: "Opérationnel",
    dacpSensitive: "Oui",
    mfaStatus: "Partiel",
    mfaDetail: "MFA appli mobile"
  },
  {
    id: "asset-demo-2",
    priority: 2,
    beneficiary: "Santé Publique",
    productCode: "MEDIX",
    productName: "Plateforme de dossiers médicaux partagés",
    productTags: [],
    rto: "1 h",
    rpo: "15 min",
    availability: "Niveau 1 - Haute disponibilité",
    availabilityNote: "24h/7",
    confidentiality: "Très sensible",
    impact: "Réglementaire",
    dacpSensitive: "Oui",
    mfaStatus: "Implémenté",
    mfaDetail: "MFA fort (FIDO2)"
  },
  {
    id: "asset-demo-3",
    priority: 3,
    beneficiary: "Ressources Humaines",
    productCode: "PAYRH",
    productName: "Système de paie et avantages",
    productTags: ["Processus critique"],
    rto: "4 h",
    rpo: "1 h",
    availability: "Niveau 2 - Reprise rapide",
    availabilityNote: "Clôture mensuelle",
    confidentiality: "Sensible",
    impact: "Financier",
    dacpSensitive: "Non",
    mfaStatus: "Non implémenté",
    mfaDetail: "À planifier"
  },
  {
    id: "asset-demo-4",
    priority: 4,
    beneficiary: "Gestion des secours",
    productCode: "ALERT",
    productName: "Plateforme d'alerte multicanal",
    productTags: ["Haute disponibilité"],
    rto: "30 min",
    rpo: "0 h",
    availability: "Niveau 1 - Haute disponibilité",
    availabilityNote: "Campagnes d'alerte urgentes",
    confidentiality: "Restreint",
    impact: "Sociétal",
    dacpSensitive: "Oui",
    mfaStatus: "Partiel",
    mfaDetail: "MFA opérateurs"
  },
  {
    id: "asset-demo-5",
    priority: 5,
    beneficiary: "Finances & Achats",
    productCode: "ARCAD",
    productName: "Portail fournisseurs et facturation",
    productTags: ["SaaS"],
    rto: "8 h",
    rpo: "4 h",
    availability: "Niveau 3 - Reprise standard",
    availabilityNote: "Clôtures trimestrielles",
    confidentiality: "Sensible",
    impact: "Réputation",
    dacpSensitive: "Non",
    mfaStatus: "Implémenté",
    mfaDetail: "MFA via application"
  }
];
const demoProjectRisks = [
  {
    id: "proj-risk-demo-1",
    projectName: "Migration ERP Finance",
    beneficiary: "Direction Financière",
    reportNumber: "RISK-ERP-2024-01",
    revisionNumber: 2,
    description: "Migration de l'ERP avec dépendances critiques et forte exposition aux données sensibles.",
    analysisLink: "https://exemple.com/analyses/erp-finance.pdf",
    completionDate: "2024-05-12",
    status: "red",
    recommendations: [
      { id: "proj-reco-1", title: "Plan de continuité", text: "Mettre à jour le plan de secours avant bascule.", implemented: false },
      { id: "proj-reco-2", title: "Tests de reprise", text: "Planifier un test de reprise complet sur l'environnement cible.", implemented: true }
    ]
  },
  {
    id: "proj-risk-demo-2",
    projectName: "Portail fournisseurs",
    beneficiary: "Achats",
    reportNumber: "RISK-SUP-2024-03",
    revisionNumber: 1,
    description: "Portail exposé à des données contractuelles et flux fournisseurs critiques.",
    analysisLink: "https://exemple.com/analyses/portail-fournisseurs.pdf",
    completionDate: "2024-04-18",
    status: "yellow",
    recommendations: [
      { id: "proj-reco-3", title: "Revue des accès", text: "Auditer les comptes externes et activer MFA.", implemented: false }
    ]
  },
  {
    id: "proj-risk-demo-3",
    projectName: "Automatisation RH",
    beneficiary: "Ressources Humaines",
    reportNumber: "RISK-HR-2024-02",
    revisionNumber: 3,
    description: "Automatisation des workflows RH avec dépendances multiples aux SI internes.",
    analysisLink: "",
    completionDate: "2024-03-25",
    status: "green",
    recommendations: [
      { id: "proj-reco-4", title: "Journalisation", text: "Mettre en place une journalisation renforcée des actions sensibles.", implemented: true }
    ]
  }
];
const demoData = {
  title: "Jeu de d\xE9monstration SMSI",
  controls: demoControls,
  soa: demoSoa,
  documentReview: demoDocumentReview,
  actions: [
    { id: "action-demo-1", linkType: "control", linkId: "CTRL-DEMO-001", controlId: "CTRL-DEMO-001", ncIds: [], title: "R\xE9diger la politique", description: "", priority: "high", progress: 20, dueDate: "2024-06-15", assignedTo: "Alice Martin", status: "en_cours", comments: [] },
    { id: "action-demo-2", linkType: "risk", linkId: ["risk-demo-1", "risk-demo-3"], controlId: "", ncIds: [], title: "Traiter les risques \xE9lectrique et intrusion", description: "", priority: "critical", progress: 50, dueDate: "2024-05-30", assignedTo: "Bernard Dupond", status: "en_cours", comments: [], criticalAssetIds: ["asset-demo-2", "asset-demo-4"] },
    { id: "action-demo-3", linkType: "nc", linkId: "nc-demo-1", controlId: "", ncIds: ["nc-demo-1"], title: "Corriger la non-conformit\xE9", description: "", priority: "medium", progress: 10, dueDate: "2024-07-01", assignedTo: "Claire Durand", status: "planifie", comments: [] },
    { id: "action-demo-4", linkType: "audit", linkId: "audit-demo-1", controlId: "", ncIds: [], title: "Pr\xE9parer l'audit interne", description: "", priority: "high", progress: 0, dueDate: "2024-08-10", assignedTo: "Elise Leroy", status: "planifie", comments: [] },
    { id: "action-demo-5", linkType: "threat", linkId: "threat-demo-1", controlId: "", ncIds: [], title: "Suivre la menace externe", description: "", priority: "medium", progress: 40, dueDate: "2024-06-20", assignedTo: "David Rousseau", status: "en_cours", comments: [], criticalAssetIds: ["asset-demo-3"] },
    { id: "action-demo-6", linkType: "control", linkId: "CTRL-DEMO-005", controlId: "CTRL-DEMO-005", ncIds: [], title: "V\xE9rifier les acc\xE8s", description: "", priority: "low", progress: 100, dueDate: "2024-04-30", assignedTo: "Fran\xE7ois Petit", status: "termine", comments: [] },
    { id: "action-demo-7", linkType: "risk", linkId: "risk-demo-3", controlId: "", ncIds: [], title: "Mettre en \xE6uvre le plan de r\xE9ponse", description: "", priority: "high", progress: 60, dueDate: "2024-07-15", assignedTo: "Gabrielle Marchand", status: "en_cours", comments: [], criticalAssetIds: ["asset-demo-4"] },
    { id: "action-demo-8", linkType: "nc", linkId: "nc-demo-3", controlId: "", ncIds: ["nc-demo-3"], title: "Documenter la proc\xE9dure", description: "", priority: "medium", progress: 30, dueDate: "2024-06-05", assignedTo: "Hugo Charpentier", status: "planifie", comments: [] },
    { id: "action-demo-9", linkType: "audit", linkId: "audit-demo-4", controlId: "", ncIds: [], title: "Suivi d'audit externe", description: "", priority: "high", progress: 15, dueDate: "2024-09-01", assignedTo: "Isabelle Moreau", status: "planifie", comments: [], criticalAssetIds: ["asset-demo-5"] },
    { id: "action-demo-10", linkType: "threat", linkId: "threat-demo-5", controlId: "", ncIds: [], title: "Cl\xF4turer la menace", description: "", priority: "low", progress: 80, dueDate: "2024-05-15", assignedTo: "Jean-Luc Robert", status: "en_cours", comments: [] },
    { id: "action-demo-11", linkType: "continuous", linkId: "continuous_improvement", controlId: "", ncIds: [], title: "Boucle d'am\xE9lioration continue trimestrielle", description: "", priority: "low", progress: 15, dueDate: "2024-10-01", assignedTo: "", status: "planifie", comments: [] },
    { id: "action-demo-12", linkType: "control", linkId: "CTRL-DEMO-010", controlId: "CTRL-DEMO-010", ncIds: [], title: "Revoir la proc\xE9dure d'acc\xE8s", description: "", priority: "medium", progress: 0, dueDate: "2024-03-31", assignedTo: "David Rousseau", status: "annule", comments: [] }
  ],
  actionsHistory: {
    critical: [4, 3, 3, 2, 2, 1],
    high: [9, 8, 7, 6, 5, 4],
    medium: [11, 10, 9, 7, 5, 4],
    low: [8, 7, 6, 5, 4, 3]
  },
  criticalAssets: demoCriticalAssets,
  criticalAssetsMeta: demoCriticalAssetsMeta,
  projectRisks: demoProjectRisks,
  risks: [
    { id: "risk-demo-1", title: "Panne \xE9lectrique", description: "Coupure prolong\xE9e", impact: 4, probability: 3, score: 12, level: "Significatif", treatment: "G\xE9n\xE9rer un plan de secours", comments: [], criticalAssetIds: ["asset-demo-2", "asset-demo-5"] },
    { id: "risk-demo-2", title: "Perte de donn\xE9es", description: "Suppression accidentelle", impact: 5, probability: 2, score: 10, level: "Significatif", treatment: "Sauvegardes r\xE9guli\xE8res", comments: [] },
    { id: "risk-demo-3", title: "Intrusion r\xE9seau", description: "Acc\xE8s non autoris\xE9", impact: 5, probability: 4, score: 20, level: "Critique", treatment: "Renforcer le pare-feu", comments: [], criticalAssetIds: ["asset-demo-1", "asset-demo-3"] },
    { id: "risk-demo-4", title: "D\xE9faillance mat\xE9rielle", description: "Panne serveur", impact: 3, probability: 3, score: 9, level: "Mineur", treatment: "Maintenance pr\xE9ventive", comments: [] },
    { id: "risk-demo-5", title: "Erreur humaine", description: "Saisie incorrecte", impact: 2, probability: 4, score: 8, level: "Mineur", treatment: "Formation", comments: [] },
    { id: "risk-demo-6", title: "Virus informatique", description: "Contamination des postes", impact: 4, probability: 4, score: 16, level: "Grave", treatment: "Antivirus", comments: [] },
    { id: "risk-demo-7", title: "Fuite d'information", description: "Divulgation non autoris\xE9e", impact: 5, probability: 3, score: 15, level: "Grave", treatment: "Sensibilisation", comments: [] },
    { id: "risk-demo-8", title: "Retard projet", description: "D\xE9lai de livraison", impact: 3, probability: 2, score: 6, level: "Mineur", treatment: "Suivi du planning", comments: [] },
    { id: "risk-demo-9", title: "Indisponibilit\xE9 internet", description: "Panne fournisseur", impact: 3, probability: 4, score: 12, level: "Significatif", treatment: "Liaison de secours", comments: [] },
    { id: "risk-demo-10", title: "Non-conformit\xE9 r\xE9glementaire", description: "Nouvelle loi", impact: 4, probability: 2, score: 8, level: "Mineur", treatment: "Veille juridique", comments: [] }
  ],
  threats: [
    { id: "threat-demo-1", title: "Phishing cibl\xE9", context: "Emails", description: "Campagne de hame\xE7onnage", likelihood: 4, impact: 4, score: 16, level: "Grave", status: "open", updatedAt: "2024-04-01", createdAt: "2024-04-01", criticalAssetIds: ["asset-demo-3"] },
    { id: "threat-demo-2", title: "Logiciel malveillant", context: "Postes", description: "Propagation de malware", likelihood: 5, impact: 5, score: 25, level: "Critique", status: "open", updatedAt: "2024-04-02", createdAt: "2024-04-02", criticalAssetIds: ["asset-demo-1", "asset-demo-4"] },
    { id: "threat-demo-3", title: "Vol de mat\xE9riel", context: "Bureaux", description: "Disparition d'\xE9quipements", likelihood: 3, impact: 2, score: 6, level: "Mineur", status: "mitigated", updatedAt: "2024-04-03", createdAt: "2024-04-03" },
    { id: "threat-demo-4", title: "Inondation", context: "Datacenter", description: "Risque naturel", likelihood: 3, impact: 5, score: 15, level: "Grave", status: "open", updatedAt: "2024-04-04", createdAt: "2024-04-04" },
    { id: "threat-demo-5", title: "D\xE9ni de service", context: "Serveurs web", description: "Attaque DDoS", likelihood: 4, impact: 4, score: 16, level: "Grave", status: "open", updatedAt: "2024-04-05", createdAt: "2024-04-05" },
    { id: "threat-demo-6", title: "Espionnage", context: "R\xE9seau", description: "Interception de trafic", likelihood: 5, impact: 4, score: 20, level: "Critique", status: "mitigated", updatedAt: "2024-04-06", createdAt: "2024-04-06" },
    { id: "threat-demo-7", title: "Perte de cl\xE9 USB", context: "Utilisateur", description: "Donn\xE9es sensibles perdues", likelihood: 4, impact: 2, score: 8, level: "Mineur", status: "closed", updatedAt: "2024-04-07", createdAt: "2024-04-07" },
    { id: "threat-demo-8", title: "Intrusion physique", context: "Locaux", description: "Acc\xE8s non autoris\xE9", likelihood: 3, impact: 4, score: 12, level: "Significatif", status: "open", updatedAt: "2024-04-08", createdAt: "2024-04-08" },
    { id: "threat-demo-9", title: "Bris de glace", context: "Bureaux", description: "Vandalisme", likelihood: 3, impact: 2, score: 6, level: "Mineur", status: "closed", updatedAt: "2024-04-09", createdAt: "2024-04-09" },
    { id: "threat-demo-10", title: "Erreur de configuration", context: "Syst\xE8mes", description: "Mauvais param\xE9trage", likelihood: 4, impact: 4, score: 16, level: "Grave", status: "mitigated", updatedAt: "2024-04-10", createdAt: "2024-04-10" }
  ],
  nonconformities: [
    { id: "nc-demo-1", title: "Proc\xE9dure absente", description: "Absence de proc\xE9dure document\xE9e", type: "Majeure", status: "detection", detectionDate: "2024-03-15", assignedTo: "Alice Martin", auditId: "audit-demo-1" },
    { id: "nc-demo-2", title: "Formulaire incomplet", description: "Champs obligatoires manquants", type: "Mineure", status: "analyse", detectionDate: "2024-03-20", assignedTo: "Bernard Dupond", auditId: "audit-demo-2" },
    { id: "nc-demo-3", title: "Sauvegarde non test\xE9e", description: "Tests de restauration non r\xE9alis\xE9s", type: "Majeure", status: "action", detectionDate: "2024-03-22", assignedTo: "Claire Durand", auditId: "audit-demo-1", criticalAssetIds: ["asset-demo-4"] },
    { id: "nc-demo-4", title: "Contrat fournisseur manquant", description: "Absence de clause s\xE9curit\xE9", type: "Mineure", status: "detection", detectionDate: "2024-03-25", assignedTo: "David Rousseau", auditId: "" },
    { id: "nc-demo-5", title: "Journalisation insuffisante", description: "Pas de traces compl\xE8tes", type: "Majeure", status: "analyse", detectionDate: "2024-03-27", assignedTo: "", auditId: "audit-demo-3" },
    { id: "nc-demo-6", title: "Absence de revue", description: "Aucune preuve de revue", type: "Mineure", status: "action", detectionDate: "2024-03-29", assignedTo: "Fran\xE7ois Petit", auditId: "audit-demo-5" },
    { id: "nc-demo-7", title: "Mise \xE0 jour retard\xE9e", description: "Correctifs non appliqu\xE9s", type: "Mineure", status: "cloture", detectionDate: "2024-04-01", assignedTo: "Gabrielle Marchand", auditId: "" },
    { id: "nc-demo-8", title: "Plan de continuit\xE9 obsol\xE8te", description: "Plan non mis \xE0 jour", type: "Majeure", status: "action", detectionDate: "2024-04-03", assignedTo: "Hugo Charpentier", auditId: "audit-demo-4", criticalAssetIds: ["asset-demo-2", "asset-demo-5"] },
    { id: "nc-demo-9", title: "Sensibilisation insuffisante", description: "Formations non suivies", type: "Mineure", status: "analyse", detectionDate: "2024-04-05", assignedTo: "Isabelle Moreau", auditId: "" },
    { id: "nc-demo-10", title: "Mauvaise gestion des acc\xE8s", description: "Droits trop larges", type: "Majeure", status: "detection", detectionDate: "2024-04-08", assignedTo: "Jean-Luc Robert", auditId: "audit-demo-2" }
  ],
  audits: [
    { id: "audit-demo-1", title: "Audit interne r\xE9seau", description: "", type: "interne", scope: "R\xE9seau", plannedDate: "2024-07-10", auditor: "Elise Leroy", status: "planifie" },
    { id: "audit-demo-2", title: "Audit RGPD", description: "", type: "externe", scope: "Conformit\xE9", plannedDate: "2024-05-20", auditor: "Bernard Dupond", status: "realise" },
    { id: "audit-demo-3", title: "Audit s\xE9curit\xE9 physique", description: "", type: "interne", scope: "Locaux", plannedDate: "2024-06-05", auditor: "Claire Durand", status: "planifie" },
    { id: "audit-demo-4", title: "Audit fournisseurs", description: "", type: "externe", scope: "Prestataires", plannedDate: "2024-09-15", auditor: "Hugo Charpentier", status: "planifie" },
    { id: "audit-demo-5", title: "Audit interne processus", description: "", type: "interne", scope: "Processus", plannedDate: "2024-05-10", auditor: "Alice Martin", status: "realise" },
    { id: "audit-demo-6", title: "Audit ISO 27001", description: "", type: "externe", scope: "SMSI", plannedDate: "2024-08-01", auditor: "Gabrielle Marchand", status: "planifie" },
    { id: "audit-demo-7", title: "Audit sauvegardes", description: "", type: "interne", scope: "Informatique", plannedDate: "2024-04-30", auditor: "Fran\xE7ois Petit", status: "realise" },
    { id: "audit-demo-8", title: "Audit d\xE9veloppement", description: "", type: "interne", scope: "Applications", plannedDate: "2024-07-20", auditor: "David Rousseau", status: "planifie" },
    { id: "audit-demo-9", title: "Audit sensibilisation", description: "", type: "interne", scope: "RH", plannedDate: "2024-06-25", auditor: "Isabelle Moreau", status: "planifie" },
    { id: "audit-demo-10", title: "Audit externe annuel", description: "", type: "externe", scope: "Global", plannedDate: "2024-11-10", auditor: "Jean-Luc Robert", status: "planifie" }
  ],
  reviews: [
    { id: "review-demo-1", date: "2024-01-15", participants: "Direction", inputs: "Rapport pr\xE9c\xE9dent", decisions: "Actions correctives", comments: [] },
    { id: "review-demo-2", date: "2024-02-10", participants: "Comit\xE9 s\xE9curit\xE9", inputs: "Incidents", decisions: "Am\xE9liorations", comments: [] },
    { id: "review-demo-3", date: "2024-03-05", participants: "Direction, DSI", inputs: "Nouveaux risques", decisions: "Priorisation", comments: [] },
    { id: "review-demo-4", date: "2024-04-01", participants: "RH", inputs: "Formations", decisions: "Plan annuel", comments: [] },
    { id: "review-demo-5", date: "2024-04-30", participants: "DSI", inputs: "Indicateurs", decisions: "Budget", comments: [] },
    { id: "review-demo-6", date: "2024-05-20", participants: "Direction", inputs: "Audit interne", decisions: "Mesures correctives", comments: [] },
    { id: "review-demo-7", date: "2024-06-18", participants: "Comit\xE9 s\xE9curit\xE9", inputs: "Tableau de bord", decisions: "Nouvelles politiques", comments: [] },
    { id: "review-demo-8", date: "2024-07-10", participants: "DSI, RH", inputs: "Revue pr\xE9c\xE9dente", decisions: "Ajustements", comments: [] },
    { id: "review-demo-9", date: "2024-08-05", participants: "Direction", inputs: "Risque majeur", decisions: "Plan d'action", comments: [] },
    { id: "review-demo-10", date: "2024-09-01", participants: "Comit\xE9", inputs: "Performance", decisions: "Am\xE9liorations", comments: [] }
  ],
  stakeholders: [
    { id: "st-1", name: "Alice Martin", role: "Responsable s\xE9curit\xE9", contact: "alice@example.com", notes: "" },
    { id: "st-2", name: "Bernard Dupond", role: "DSI", contact: "bernard@example.com", notes: "" },
    { id: "st-3", name: "Claire Durand", role: "RH", contact: "claire@example.com", notes: "" },
    { id: "st-4", name: "David Rousseau", role: "Chef projet", contact: "david@example.com", notes: "" },
    { id: "st-5", name: "Elise Leroy", role: "Auditrice interne", contact: "elise@example.com", notes: "" },
    { id: "st-6", name: "Fran\xE7ois Petit", role: "Utilisateur cl\xE9", contact: "francois@example.com", notes: "" },
    { id: "st-7", name: "Gabrielle Marchand", role: "Consultante", contact: "gabrielle@example.com", notes: "" },
    { id: "st-8", name: "Hugo Charpentier", role: "D\xE9veloppeur", contact: "hugo@example.com", notes: "" },
    { id: "st-9", name: "Isabelle Moreau", role: "Responsable qualit\xE9", contact: "isabelle@example.com", notes: "" },
    { id: "st-10", name: "Jean-Luc Robert", role: "Direction", contact: "jeanluc@example.com", notes: "" }
  ],
  documents: [
    { id: "doc-demo-1", category: "Processus", type: "Procedure reunion", tool: "Teams", link: "https://exemple.com/teams1" },
    { id: "doc-demo-2", category: "Enregistrement", type: "Compte rendu reunion", tool: "Teams", link: "https://exemple.com/teams2" },
    { id: "doc-demo-3", category: "Politique", type: "Manuel qualite", tool: "SMQ", link: "https://exemple.com/smq1" },
    { id: "doc-demo-4", category: "Procédure", type: "Procedure qualite", tool: "SMQ", link: "https://exemple.com/smq2" },
    { id: "doc-demo-5", category: "Procédure", type: "Guide reunion", tool: "Webex", link: "https://exemple.com/webex1" },
    { id: "doc-demo-6", category: "Processus", type: "Support formation", tool: "Webex", link: "https://exemple.com/webex2" },
    { id: "doc-demo-7", category: "Politique", type: "Politique archivage", tool: "GED", link: "https://exemple.com/ged1" },
    { id: "doc-demo-8", category: "Procédure", type: "Procedure indexation", tool: "GED", link: "https://exemple.com/ged2" },
    { id: "doc-demo-9", category: "Enregistrement", type: "Registre securite", tool: "SMQ", link: "https://exemple.com/smq3" },
    { id: "doc-demo-10", category: "Enregistrement", type: "Checklist reunion", tool: "Teams", link: "https://exemple.com/teams3" }
  ],
  kpis: [
    { id: "kpi-demo-1", title: "Taux de conformit\xE9", description: "Conformit\xE9 des contr\xF4les", comments: "", progress: 60 },
    { id: "kpi-demo-2", title: "Actions ouvertes", description: "Nombre d'actions en cours", comments: "", progress: 40 },
    { id: "kpi-demo-3", title: "Incidents s\xE9curit\xE9", description: "Incidents enregistr\xE9s", comments: "", progress: 20 },
    { id: "kpi-demo-4", title: "Audits r\xE9alis\xE9s", description: "% d'audits termin\xE9s", comments: "", progress: 50 },
    { id: "kpi-demo-5", title: "Revu de direction", description: "Avancement des revues", comments: "", progress: 30 },
    { id: "kpi-demo-6", title: "Sensibilisation", description: "Taux de formation", comments: "", progress: 70 },
    { id: "kpi-demo-7", title: "Disponibilit\xE9 syst\xE8mes", description: "Uptime mensuel", comments: "", progress: 95 },
    { id: "kpi-demo-8", title: "Tests de sauvegarde", description: "R\xE9ussite des restaurations", comments: "", progress: 45 },
    { id: "kpi-demo-9", title: "Nombre de menaces", description: "Menaces identifi\xE9es", comments: "", progress: 25 },
    { id: "kpi-demo-10", title: "Mises \xE0 jour", description: "Syst\xE8mes \xE0 jour", comments: "", progress: 80 }
  ],
  nis2Controls: demoNis2Controls,
  nis2Plan: demoNis2Plan,
  targetMaturity: {
    iso27002: 4,
    nis2: demoNis2Maturity,
    nis2Target: 4
  }
};

  // modules/data.js
  var appData = {
    controls: [],
    actions: [],
    actionsHistory: {},
    criticalAssets: [],
    criticalAssetsMeta: getDefaultCriticalAssetsMeta(),
    projectRisks: [],
    risks: [],
    threats: [],
    nonconformities: [],
    audits: [],
    kpis: [],
    reviews: [],
    stakeholders: [],
    documents: [],
    documentReview: [],
    nis2Controls: [],
    nis2Plan: getEmptyNis2Plan(),
    soclePillars: cloneDefaultSoclePillars(),
    soa: [],
    nextActionId: 0,
    title: "Cyber-Assistant",
    targetMaturity: {
      iso27002: 3,
      nis2: Object.assign({}, DEFAULT_NIS2_MATURITY),
      nis2Target: 0
    }
  };
  window.appData = appData;
  var charts = {};
  window.charts = charts;
  async function loadData() {
    try {
      const savedData = localStorage.getItem("smsi_data");
      if (savedData) {
        appData = JSON.parse(savedData);
          window.appData = appData;
        if (!appData.actionsHistory) appData.actionsHistory = {};
        if (!appData.reviews) appData.reviews = [];
        if (!appData.stakeholders) appData.stakeholders = [];
        if (!appData.documents) appData.documents = [];
        if (!appData.documentReview) appData.documentReview = createDefaultDocumentReview();
        else appData.documentReview = createDefaultDocumentReview(appData.documentReview);
        if (!appData.projectRisks) appData.projectRisks = [];
        if (!appData.threats) appData.threats = [];
        if (!Array.isArray(appData.nis2Controls)) appData.nis2Controls = loadNis2Controls();
        if (!appData.soa) appData.soa = [];
        if (typeof appData.nextActionId !== "number") appData.nextActionId = 0;
        if (!appData.title) appData.title = "Cyber-Assistant";
        if (!appData.targetMaturity) {
          appData.targetMaturity = {
            iso27002: 3,
            nis2: Object.assign({}, DEFAULT_NIS2_MATURITY),
            nis2Target: 0
          };
        } else {
          const tm = appData.targetMaturity;
          if (typeof tm.iso27002 === "object") {
            const values = Object.values(tm.iso27002)
              .map(v => parseInt(v, 10))
              .filter(v => !isNaN(v));
            tm.iso27002 = values.length
              ? Math.round(values.reduce((a, b) => a + b, 0) / values.length)
              : 3;
          } else if (typeof tm.iso27002 !== "number") {
            const v = parseInt(tm.iso27002, 10);
            tm.iso27002 = isNaN(v) ? 3 : v;
          }
          tm.nis2 = buildNis2Maturity(tm.nis2 || {});
          const hasNonZeroNis2 = Object.values(tm.nis2 || {}).some(v => typeof v === "number" && v > 0);
          const rawNis2Target = tm.nis2Target;
          const parsedNis2Target = typeof rawNis2Target === "number"
            ? rawNis2Target
            : parseInt(rawNis2Target, 10);
          tm.nis2Target = Number.isFinite(parsedNis2Target) ? parsedNis2Target : (hasNonZeroNis2 ? 3 : 0);
          if (tm.nis2Target < 0) tm.nis2Target = 0;
          if (!tm.iso27002 || tm.iso27002 <= 0) tm.iso27002 = 3;
        }
        ensureNis2Plan();
        ensureCriticalAssets();
        ensureProjectRisks();
        appData.soclePillars = normalizeSoclePillars(appData.soclePillars);
        if (Array.isArray(appData.controls)) {
          const mapNumToLevel = { 1: "Initial", 2: "Répétable", 3: "Défini", 4: "Géré", 5: "Optimisé" };
          const legacyCategories = {
            "Organisationnelle": "Organisationnel",
            "Organisationnelles": "Organisationnel",
            "Organisationnels": "Organisationnel",
            "Humaines": "Humain",
            "Humaine": "Humain",
            "Humains": "Humain",
            "Physiques": "Physique",
            "Physique": "Physique",
            "Technologiques": "Technologique",
            "Technologique": "Technologique",
            "Techniques": "Technologique",
            "Technique": "Technologique",
            "Autre": "Non classé",
            "Autres": "Non classé"
          };
          const validCategories = new Set([...Object.values(DOMAIN_BY_PREFIX), "Autre", "Non classé"]);
          const needsSoaMigration = appData.soa.length === 0;
          appData.controls.forEach(c => {
            if (typeof c.cmm === "undefined") {
              c.cmm = "Initial";
            } else if (typeof c.cmm === "number") {
              c.cmm = mapNumToLevel[c.cmm] || "Initial";
            }
            if (needsSoaMigration && (c.selected !== undefined || c.justification !== undefined)) {
              appData.soa.push({
                controlId: c.id,
                applicable: c.selected !== undefined ? c.selected : true,
                justification: c.justification || ""
              });
            }
            delete c.selected;
            delete c.justification;
            if (typeof c.justificationControl === "undefined") c.justificationControl = "";
            delete c.maturity;
            if (typeof c.category === "string") {
              let cat = c.category.trim();
              if (cat === "ISO27002") {
                c.category = "Non classé";
              } else if (legacyCategories[cat]) {
                c.category = legacyCategories[cat];
              } else if (!validCategories.has(cat)) {
                c.category = "Non classé";
              } else {
                c.category = cat;
              }
            } else {
              c.category = "Non classé";
            }
            if (!c.category || c.category === "Non classé") {
              c.category = inferIso27002Category(c);
            }
            if (typeof c.subcategory === "string") {
              c.subcategory = [c.subcategory];
            }
            if (Array.isArray(c.subcategory)) {
              c.subcategory = c.subcategory.map(sc => normalizeCapabilityKey(sc));
            }
            if (!c.subcategory || (Array.isArray(c.subcategory) && c.subcategory.length === 0)) {
              const numero = String(c.numero || "").trim();
              const title = String(c.title || "").trim();
              const match = iso27002_data_default.find(
                item => item.numero === numero || item.titre === title
              );
              if (match) {
                if (Array.isArray(match.capability) && match.capability.length > 0) {
                  c.subcategory = match.capability.map(sc => normalizeCapabilityKey(sc));
                } else if (typeof match.capability === "string" && match.capability) {
                  c.subcategory = [normalizeCapabilityKey(match.capability)];
                }
              }
              if (!c.subcategory || (Array.isArray(c.subcategory) && c.subcategory.length === 0)) {
                c.subcategory = [];
              }
            }
          });
        }
        if (Array.isArray(appData.actions)) {
          appData.actions.forEach(a => {
            if (!Array.isArray(a.comments)) a.comments = [];
          });
        }
        if (Array.isArray(appData.risks)) {
          appData.risks.forEach(r => {
            if (!Array.isArray(r.comments)) r.comments = [];
            r.score = (r.impact || 1) * (r.probability || 1);
            r.level = calculateRiskLevel(r.score);
          });
        }
        ensureRiskIds();
        if (Array.isArray(appData.reviews)) {
          appData.reviews.forEach(r => {
            if (!Array.isArray(r.comments)) r.comments = [];
          });
        }
        if (Array.isArray(appData.audits)) {
          appData.audits.forEach(a => {
            if (!a.type) a.type = "interne";
          });
        }
        if (Array.isArray(appData.nonconformities)) {
          appData.nonconformities.forEach(nc => {
            if (!nc.auditId) nc.auditId = "";
          });
        }
        loadThreats();
        normalizeCriticalAssetLinks(appData.actions);
        normalizeCriticalAssetLinks(appData.risks);
        normalizeCriticalAssetLinks(appData.threats);
        normalizeCriticalAssetLinks(appData.nonconformities);

        // Normalisation complète de toutes les sections pour compatibilité ascendante
        if (typeof normalizeSoa === 'function') appData.soa = normalizeSoa(appData.soa || []);
        if (typeof normalizeActions === 'function') appData.actions = normalizeActions(appData.actions || []);
        if (typeof normalizeRisks === 'function') appData.risks = normalizeRisks(appData.risks || []);
        if (typeof normalizeThreats === 'function') appData.threats = normalizeThreats(appData.threats || []);
        if (typeof normalizeNonconformities === 'function') appData.nonconformities = normalizeNonconformities(appData.nonconformities || []);
        if (typeof normalizeAudits === 'function') appData.audits = normalizeAudits(appData.audits || []);
        if (typeof normalizeKpis === 'function') appData.kpis = normalizeKpis(appData.kpis || []);
        if (typeof normalizeReviews === 'function') appData.reviews = normalizeReviews(appData.reviews || []);
        if (typeof normalizeStakeholders === 'function') appData.stakeholders = normalizeStakeholders(appData.stakeholders || []);
        if (typeof normalizeDocuments === 'function') appData.documents = normalizeDocuments(appData.documents || []);
        if (typeof normalizeControls === 'function') appData.controls = normalizeControls(appData.controls || []);
      } else {
        await initializeDefaultData();
      }
      } catch (error) {
        console.error("Erreur lors du chargement des donn\xE9es:", error);
        await initializeDefaultData();
      }
      saveData();
      loadControlsTable();
      loadNis2ControlsTable();
      updateDashboard();
    }
  function ensureNis2Plan() {
    appData.nis2Plan = normalizeNis2Plan(appData.nis2Plan);
  }
  async function initializeDefaultData() {
    appData.controls = await loadISO27002Controls();
    appData.soa = appData.controls.map(c => ({ controlId: c.id, applicable: true, justification: "" }));
    appData.nis2Controls = loadNis2Controls();
    appData.nis2Plan = getEmptyNis2Plan();
    appData.soclePillars = cloneDefaultSoclePillars();
    appData.title = "Cyber-Assistant";
    appData.actions = [];
    appData.criticalAssets = [];
    appData.criticalAssetsMeta = getDefaultCriticalAssetsMeta();
    appData.projectRisks = [];
    appData.risks = [];
    appData.nonconformities = [];
    appData.audits = [];
    appData.kpis = [];
    appData.reviews = [];
      appData.stakeholders = [];
      appData.documents = [];
      appData.documentReview = createDefaultDocumentReview();
      appData.threats = [];
        appData.targetMaturity = {
          iso27002: 3.5,
          nis2: Object.assign({}, DEFAULT_NIS2_MATURITY),
          nis2Target: 0
        };
      appData.nextActionId = 0;
      saveData();
    }
async function loadISO27002Controls() {
  return iso27002_data_default.map((item, index) => ({
    id: `CTRL-${String(index + 1).padStart(3, "0")}`,
    reference: item.numero ? `${item.numero} ${item.titre}` : item.titre,
    numero: item.numero || String(index + 1),
    title: item.titre,
    measure: item["mesure de s\xE9curit\xE9"],
    description: item["mesure de s\xE9curit\xE9"],
    recommendations: item.recommandations || [],
    category: DOMAIN_BY_PREFIX[item.numero?.split('.')[0]] || "Non classé",
    subcategory: Array.isArray(item.capability)
      ? item.capability.map(c => normalizeCapabilityKey(c))
      : [normalizeCapabilityKey(item.capability)],
    status: "non_implemente",
    cmm: "Initial",
    comments: "",
    evidence: "",
    justificationControl: ""
  }));
}

function loadNis2Controls() {
  return nis2_controls_data.map((item, index) => ({
    id: `NIS2-${String(index + 1).padStart(3, "0")}`,
    fonction: item.fonction,
    categorie: item.categorie,
    description: item.description,
    status: "non_implemente",
    cmm: "Initial",
    justification: ""
  }));
}
  function getLocalStorageUsage() {
    let totalSize = 0;
    for (let key in localStorage) {
      if (localStorage.hasOwnProperty(key)) {
        totalSize += (localStorage[key].length + key.length) * 2; // UTF-16 = 2 bytes per char
      }
    }
    return totalSize;
  }

  function checkLocalStorageQuota(dataSize) {
    const QUOTA_WARNING_THRESHOLD = 4 * 1024 * 1024; // 4MB warning
    const QUOTA_LIMIT = 5 * 1024 * 1024; // 5MB typical limit
    const currentUsage = getLocalStorageUsage();
    const projectedUsage = currentUsage + dataSize;

    if (projectedUsage > QUOTA_LIMIT) {
      return { ok: false, usage: currentUsage, projected: projectedUsage, limit: QUOTA_LIMIT };
    }
    if (projectedUsage > QUOTA_WARNING_THRESHOLD) {
      return { ok: true, warning: true, usage: currentUsage, projected: projectedUsage, limit: QUOTA_LIMIT };
    }
    return { ok: true, warning: false, usage: currentUsage, projected: projectedUsage, limit: QUOTA_LIMIT };
  }

  function refreshStorageHealth() {
    const usageText = document.getElementById("storageUsageText");
    const usageBar = document.getElementById("storageUsageBar");
    const datasetSizeEl = document.getElementById("storageDatasetSize");
    const lastSaveEl = document.getElementById("storageLastSave");
    const warningEl = document.getElementById("storageHealthWarning");
    if (!usageText && !usageBar && !datasetSizeEl && !lastSaveEl && !warningEl) return;

    let usage = 0;
    let limit = 5 * 1024 * 1024;
    let warn = false;
    try {
      const quota = checkLocalStorageQuota(0);
      usage = quota.usage || 0;
      limit = quota.limit || limit;
      warn = !!quota.warning;
    } catch (e) {
      try {
        usage = getLocalStorageUsage();
      } catch (err) {
        usage = 0;
      }
    }

    const safeLimit = Number.isFinite(limit) && limit > 0 ? limit : 5 * 1024 * 1024;
    const percent = safeLimit ? Math.min(100, Math.round((usage / safeLimit) * 100)) : 0;
    if (usageText) {
      usageText.textContent = `${formatSnapshotSize(usage)} / ${formatSnapshotSize(safeLimit)} (${percent}%)`;
    }
    if (usageBar) {
      usageBar.style.width = `${percent}%`;
      usageBar.classList.toggle("storage-meter__bar--warn", percent >= 80);
      usageBar.classList.toggle("storage-meter__bar--danger", percent >= 95);
    }

    let dataSize = 0;
    try {
      const saved = localStorage.getItem("smsi_data") || "";
      dataSize = saved.length * 2;
    } catch (e) {
      dataSize = 0;
    }
    if (datasetSizeEl) datasetSizeEl.textContent = formatSnapshotSize(dataSize);

    let lastSaveRaw = "";
    try {
      lastSaveRaw = localStorage.getItem("smsi_last_save") || "";
    } catch (e) {
      lastSaveRaw = "";
    }
    let lastSaveText = "Jamais";
    if (lastSaveRaw) {
      const ts = Date.parse(lastSaveRaw);
      if (Number.isFinite(ts)) lastSaveText = new Date(ts).toLocaleString();
    }
    if (lastSaveEl) lastSaveEl.textContent = lastSaveText;

    if (warningEl) {
      const nearLimit = warn || percent >= 80;
      if (nearLimit) {
        warningEl.textContent = "Stockage presque plein : exportez vos données ou supprimez des snapshots.";
        warningEl.hidden = false;
      } else {
        warningEl.hidden = true;
      }
    }
  }

  function saveData() {
    // Protection anti-abus en mode démo
    if (DEMO_MODE) {
      if (!demoRateLimiter.checkAction('save')) {
        demoRateLimiter.showRateLimitMessage();
        return false;
      }
      if (!checkDemoStorageLimit()) {
        return false;
      }
    }

    // Validation optionnelle si le module est chargé
    if (typeof window.validateBeforeSave === 'function') {
      const validation = window.validateBeforeSave(appData);
      if (!validation.valid) {
        console.warn('[CSI] Erreurs de validation:', validation.errors);
      }
      if (validation.warnings && validation.warnings.length > 0) {
        console.debug('[CSI] Avertissements:', validation.warnings.slice(0, 5));
      }
    }

    warnMissingSubcategories();
    ensureNis2Plan();
    ensureCriticalAssets();
    const serialized = JSON.stringify(appData);
    const dataSize = serialized.length * 2; // UTF-16

    // Vérification du quota avant sauvegarde
    const quotaCheck = checkLocalStorageQuota(dataSize);
    if (!quotaCheck.ok) {
      const usageMB = (quotaCheck.usage / (1024 * 1024)).toFixed(2);
      const limitMB = (quotaCheck.limit / (1024 * 1024)).toFixed(2);
      showToast2(`Espace de stockage insuffisant (${usageMB}MB / ${limitMB}MB). Exportez vos données et supprimez les anciennes.`, "error");
      downloadJsonBackup(serialized, "smsi_backup_quota_exceeded.json");
      return false;
    }
    if (quotaCheck.warning) {
      const usageMB = (quotaCheck.projected / (1024 * 1024)).toFixed(2);
      showToast2(`Attention: stockage presque plein (${usageMB}MB). Pensez à exporter vos données.`, "warning");
    }

    try {
      localStorage.setItem("smsi_data", serialized);
      localStorage.setItem("smsi_last_save", (/* @__PURE__ */ new Date()).toISOString());
      const autoSaveStatus = document.getElementById("autoSaveStatus");
      if (autoSaveStatus) {
        autoSaveStatus.textContent = "Sauvegarde réalisée";
        autoSaveStatus.style.opacity = "1";
        setTimeout(() => {
          autoSaveStatus.style.opacity = "0";
        }, 2e3);
      }
      refreshStorageHealth();
      return true;
    } catch (error) {
      console.error("Erreur lors de la sauvegarde:", error);
      // Détection spécifique de l'erreur de quota
      const isQuotaError = error.name === 'QuotaExceededError' ||
                          error.code === 22 ||
                          error.code === 1014 ||
                          (error.name === 'NS_ERROR_DOM_QUOTA_REACHED');
      if (isQuotaError) {
        showToast2("Quota localStorage dépassé. Vos données ont été téléchargées en backup.", "error");
      } else {
        showToast2("La sauvegarde a échoué. Vos données sont conservées localement.", "error");
      }
      downloadJsonBackup(serialized, "smsi_backup_emergency.json");
      return false;
    }
  }
  function loadThreats() {
    const saved = localStorage.getItem("smsi_data");
    if (saved) {
      try {
        const data = JSON.parse(saved);
        appData.threats = (data.threats || []).map(t => {
          const prob = parseInt(t.likelihood, 10) ||
            (t.likelihood === "low" ? 3 : t.likelihood === "medium" ? 4 : t.likelihood === "high" ? 5 : 1);
          const imp = parseInt(t.impact, 10) ||
            (t.impact === "minor" ? 2 : t.impact === "major" ? 4 : t.impact === "critical" ? 5 : 1);
          const score = imp * prob;
          return { ...t, likelihood: prob, impact: imp, score, level: calculateRiskLevel(score) };
        });
      } catch (e) {
        console.error("Erreur chargement menaces", e);
      }
    }
  }
  function saveThreats() {
    try {
      const saved = localStorage.getItem("smsi_data");
      const data = saved ? JSON.parse(saved) : {};
      data.threats = appData.threats;
      localStorage.setItem("smsi_data", JSON.stringify({ ...data, ...appData, threats: appData.threats }));
      localStorage.setItem("smsi_last_save", (/* @__PURE__ */ new Date()).toISOString());
      refreshStorageHealth();
    } catch (e) {
      console.error("Erreur sauvegarde menaces", e);
    }
  }

  // modules/controls.js
var CONTROL_STATUS_OPTIONS = [
  { value: "non_implemente", label: "Non impl\xE9ment\xE9" },
  { value: "en_cours", label: "En cours" },
  { value: "implemente", label: "Impl\xE9ment\xE9" },
  { value: "verifie", label: "V\xE9rifi\xE9" }
];
const DOMAIN_OPTIONS = ["Organisationnel", "Humain", "Physique", "Technologique", "Non classé"];
var CMM_LEVEL_OPTIONS = ["Initial", "Répétable", "Défini", "Géré", "Optimisé"].map(l => ({ value: l, label: l }));

  // Vérifie si un contrôle est déclaré non-applicable dans la SoA
  function isControlNonApplicable(controlId) {
    const soaEntry = appData.soa.find(e => e.controlId === controlId);
    return soaEntry && soaEntry.applicable === false;
  }

  // Récupère la justification SoA pour un contrôle
  function getSoaJustification(controlId) {
    const soaEntry = appData.soa.find(e => e.controlId === controlId);
    return soaEntry ? soaEntry.justification || "" : "";
  }

  // Navigue vers l'onglet SoA et met en évidence le contrôle
  function navigateToSoaControl(controlId) {
    // Naviguer vers l'onglet SoA
    const soaNavBtn = document.querySelector('.nav__item[data-tab="soa"]');
    if (soaNavBtn) {
      soaNavBtn.click();
      // Après navigation, mettre en évidence la ligne du contrôle
      setTimeout(() => {
        const soaRow = document.querySelector(`.soa-decision[data-id="${controlId}"]`);
        if (soaRow) {
          const row = soaRow.closest("tr");
          if (row) {
            row.scrollIntoView({ behavior: "smooth", block: "center" });
            row.classList.add("highlight-row");
            setTimeout(() => row.classList.remove("highlight-row"), 2500);
          }
        }
      }, 150);
    }
  }

  function loadControlsTable() {
    const tbody = document.getElementById("controlsTableBody");
    if (!tbody) return;
    tbody.innerHTML = "";

    const catFilter = document.getElementById("controlsCategoryFilter")?.value || "";
    const statusFilter = document.getElementById("controlsStatusFilter")?.value || "";
    const cmmFilter = document.getElementById("controlsCmmFilter")?.value || "";
    let capabilityFilter = document.getElementById("controlsCapabilityFilter")?.value || "";
    capabilityFilter = capabilityFilter ? normalizeCapabilityKey(capabilityFilter) : "";

    const filtered = appData.controls.filter(c => {
      const capabilityMatch = !capabilityFilter || (Array.isArray(c.subcategory) ? c.subcategory.includes(capabilityFilter) : c.subcategory === capabilityFilter);
      return (!catFilter || c.category === catFilter) &&
             (!statusFilter || c.status === statusFilter) &&
             (!cmmFilter || c.cmm === cmmFilter) &&
             capabilityMatch;
    });

    const sortedControls = [...filtered].sort((a, b) => a.reference.localeCompare(b.reference, void 0, { numeric: true }));
    sortedControls.forEach((control) => {
      const row = document.createElement("tr");
      const actionIds = appData.actions
        .filter((a) => a.linkType === "control" && a.linkId === control.id)
        .map((a) => a.id)
        .join(", ");
      const statusClasses = {
        "non_implemente": "status-badge--non_implemente",
        "en_cours": "status-badge--en_cours",
        "implemente": "status-badge--implemente",
        "verifie": "status-badge--verifie"
      };
      const statusLabels = {
        "non_implemente": "Non impl\xE9ment\xE9",
        "en_cours": "En cours",
        "implemente": "Impl\xE9ment\xE9",
        "verifie": "V\xE9rifi\xE9"
      };
      const capabilityDisplay = Array.isArray(control.subcategory)
        ? control.subcategory.map(sc => {
            const cap = OPERATIONAL_CAPABILITIES.find((c) => c.key === sc);
            return cap ? cap.label : sc.replace(/_/g, " ");
          }).join(", ")
        : (() => {
            const cap = OPERATIONAL_CAPABILITIES.find((c) => c.key === control.subcategory);
            return cap ? cap.label : (control.subcategory || "").replace(/_/g, " ");
          })();

      // Vérifier si le contrôle est déclaré non-applicable dans la SoA
      const isNonApplicable = isControlNonApplicable(control.id);
      const nonApplicableBadge = isNonApplicable
        ? `<span class="status-badge status-badge--non-applicable" title="Déclaré non applicable dans la DdA">
             <i class="fas fa-ban"></i> Non applicable
           </span>
           <a href="#" class="soa-link" data-control-id="${control.id}" title="Voir dans la Déclaration d'applicabilité">
             <i class="fas fa-external-link-alt"></i>
           </a>`
        : "";

      // Attributs ISO 27002:2022
      const domain = getControlDomain(control.reference);
      const domainBadgeClass = domain.toLowerCase().replace(/\s+/g, '-');

      // Efficacité
      const effectivenessLabels = {
        effective: "✓ Efficace",
        partially: "◐ Partiel",
        ineffective: "✗ Non efficace",
        not_tested: "○ Non testé"
      };
      const effectivenessClasses = {
        effective: "effectiveness--good",
        partially: "effectiveness--medium",
        ineffective: "effectiveness--bad",
        not_tested: "effectiveness--neutral"
      };

      row.innerHTML = `
            <td><strong>${escapeHtml(control.reference)}</strong></td>
            <td>
              <div class="control-title">${escapeHtml(control.title)}</div>
              ${nonApplicableBadge ? `<div class="control-soa-indicator">${nonApplicableBadge}</div>` : ""}
              <div class="control-attrs">
                <span class="badge badge--domain badge--${domainBadgeClass}">${domain}</span>
                ${capabilityDisplay ? `<span class="control-capability" title="${escapeHtml(capabilityDisplay)}">${escapeHtml(capabilityDisplay.substring(0, 25))}${capabilityDisplay.length > 25 ? '...' : ''}</span>` : ''}
              </div>
            </td>
            <td class="control-attrs-col">
              <div class="control-attr-row">
                <span class="attr-label">Type:</span>
                <span class="attr-value">${control.isPreventive ? 'Préventif' : control.isDetective ? 'Détectif' : control.isCorrective ? 'Correctif' : 'Non défini'}</span>
              </div>
              <div class="control-attr-row">
                <span class="attr-label">CMM:</span>
                <span class="attr-value cmm-badge cmm-badge--${(control.cmm || 'initial').toLowerCase()}">${control.cmm || 'Initial'}</span>
              </div>
            </td>
            <td>${escapeHtml(control.owner || '-')}</td>
            <td><span class="status-badge ${statusClasses[control.status]}">${statusLabels[control.status]}</span></td>
            <td class="cmm-cell">
              <span class="cmm-badge cmm-badge--${(control.cmm || 'initial').toLowerCase().replace('é', 'e')}">${control.cmm || 'Initial'}</span>
              ${control.effectiveness ? `<div class="effectiveness ${effectivenessClasses[control.effectiveness] || ''}">${effectivenessLabels[control.effectiveness] || '-'}</div>` : ''}
            </td>
            <td>${control.lastEvaluation || '-'}</td>
            <td class="evidence-cell">
              ${control.evidence ? `<span class="evidence-indicator" title="${escapeHtml(control.evidence)}"><i class="fas fa-file-alt"></i> Preuves</span>` : '<span class="no-evidence">-</span>'}
              ${control.kpiRef ? `<span class="kpi-ref" title="KPI: ${escapeHtml(control.kpiRef)}"><i class="fas fa-chart-line"></i></span>` : ''}
            </td>
        `;

      // Gestionnaire pour le lien vers la SoA
      const soaLink = row.querySelector(".soa-link");
      if (soaLink) {
        soaLink.addEventListener("click", (e) => {
          e.preventDefault();
          e.stopPropagation();
          navigateToSoaControl(control.id);
        });
      }

      row.addEventListener("click", () => {
        editControl(control.id);
      });
      tbody.appendChild(row);
    });
  }
  // Génère le HTML des badges de risques liés pour la SoA
  function renderSoaRiskBadges(linkedRiskIds) {
    if (!linkedRiskIds || linkedRiskIds.length === 0) {
      return '<span class="soa-risks__empty">Aucun</span>';
    }
    return linkedRiskIds.map(riskId => {
      const risk = appData.risks.find(r => r.id === riskId);
      if (!risk) return '';
      const levelClass = (risk.level || 'insignifiant').toLowerCase().normalize("NFD").replace(/[\u0300-\u036f]/g, "");
      return `<span class="soa-risks__badge soa-risks__badge--${levelClass}" title="${escapeHtml(risk.title)}">${escapeHtml(risk.id)}</span>`;
    }).join('');
  }

  // Ouvre la modale pour éditer les risques associés à un contrôle dans la SoA
  function editSoaRisks(controlId) {
    const entry = appData.soa.find(e => e.controlId === controlId);
    const control = appData.controls.find(c => c.id === controlId);
    if (!entry || !control) return;

    // Options des risques avec niveau et titre
    const riskOptions = appData.risks.map(r => ({
      value: r.id,
      label: `[${r.level}] ${r.title}`
    }));

    if (riskOptions.length === 0) {
      showToast2("Aucun risque défini. Créez d'abord des risques dans le module Risques SMSI.", "warning");
      return;
    }

    openModal(`Risques associés - ${control.reference}`, [
      {
        name: "linkedRiskIds",
        label: "Sélectionner les risques justifiant l'applicabilité de ce contrôle",
        type: "multiselect",
        value: entry.linkedRiskIds || [],
        options: riskOptions,
        help: "Maintenez Ctrl (ou Cmd) pour sélectionner plusieurs risques"
      }
    ], (data) => {
      entry.linkedRiskIds = Array.isArray(data.linkedRiskIds) ? data.linkedRiskIds : [];
      saveData();
      loadSoaTable();
      showToast2("Risques associés mis à jour", "success");
    });
  }

  // ============================================================
  // Normalisation des entrées SOA pour compatibilité ascendante
  // ============================================================

  /**
   * Normalise une entrée SOA pour s'assurer que tous les champs requis existent
   * Gère la compatibilité avec les anciennes versions qui n'avaient que:
   * - controlId, applicable, justification
   */
  function normalizeSoaEntry(entry) {
    if (!entry || typeof entry !== 'object') {
      return {
        controlId: '',
        applicable: true,
        justification: '',
        linkedRiskIds: [],
        source: '',
        implStatus: '',
        implDate: '',
        exclusionReason: '',
        owner: '',
        evidenceRef: '',
        lastReviewDate: ''
      };
    }
    return {
      controlId: entry.controlId || '',
      applicable: typeof entry.applicable === 'boolean' ? entry.applicable : true,
      justification: entry.justification || '',
      // Nouveaux champs avec valeurs par défaut
      linkedRiskIds: Array.isArray(entry.linkedRiskIds) ? entry.linkedRiskIds : [],
      source: entry.source || '',
      implStatus: entry.implStatus || '',
      implDate: entry.implDate || '',
      exclusionReason: entry.exclusionReason || '',
      owner: entry.owner || '',
      evidenceRef: entry.evidenceRef || '',
      lastReviewDate: entry.lastReviewDate || ''
    };
  }

  /**
   * Normalise un tableau d'entrées SOA
   */
  function normalizeSoa(soaArray) {
    if (!Array.isArray(soaArray)) return [];
    return soaArray.map(entry => normalizeSoaEntry(entry));
  }

  /**
   * S'assure que toutes les entrées SOA dans appData sont normalisées
   */
  function ensureSoaNormalized() {
    if (Array.isArray(appData.soa)) {
      appData.soa = normalizeSoa(appData.soa);
    } else {
      appData.soa = [];
    }
  }

  // Export des fonctions de normalisation SOA
  window.normalizeSoaEntry = normalizeSoaEntry;
  window.normalizeSoa = normalizeSoa;
  window.ensureSoaNormalized = ensureSoaNormalized;

  // ============================================================
  // Normalisation pour TOUTES les autres sections de l'application
  // ============================================================

  /**
   * Normalise une entrée Action
   */
  function normalizeActionEntry(entry) {
    if (!entry || typeof entry !== 'object') return null;
    const result = {
      id: entry.id || '',
      linkType: entry.linkType || 'control',
      linkId: entry.linkId || entry.controlId || '',
      controlId: entry.controlId || '',
      ncIds: Array.isArray(entry.ncIds) ? entry.ncIds : [],
      title: entry.title || '',
      description: entry.description || '',
      priority: entry.priority || 'medium',
      progress: typeof entry.progress === 'number' ? entry.progress : (parseInt(entry.progress, 10) || 0),
      dueDate: entry.dueDate || '',
      assignedTo: entry.assignedTo || '',
      status: entry.status || 'todo',
      comments: Array.isArray(entry.comments) ? entry.comments : [],
      criticalAssetIds: Array.isArray(entry.criticalAssetIds) ? entry.criticalAssetIds : []
    };
    // Préserver les champs budget si présents (compatibilité NIS2 Plan actions)
    if (entry.budget2026 !== undefined) result.budget2026 = entry.budget2026;
    if (entry.budget2027 !== undefined) result.budget2027 = entry.budget2027;
    if (entry.budget2028 !== undefined) result.budget2028 = entry.budget2028;
    if (entry.budgetCategory) result.budgetCategory = entry.budgetCategory;
    if (entry.workType) result.workType = entry.workType;
    if (entry.owner) result.owner = entry.owner;
    if (entry.phase !== undefined) result.phase = entry.phase;
    return result;
  }

  /**
   * Normalise un tableau d'Actions
   */
  function normalizeActions(list) {
    if (!Array.isArray(list)) return [];
    return list.map(e => normalizeActionEntry(e)).filter(Boolean);
  }

  /**
   * Normalise une entrée Risque
   */
  function normalizeRiskEntry(entry) {
    if (!entry || typeof entry !== 'object') return null;
    const impact = parseInt(entry.impact, 10) || 1;
    const probability = parseInt(entry.probability, 10) || 1;
    const score = impact * probability;
    return {
      id: entry.id || '',
      title: entry.title || '',
      description: entry.description || '',
      impact: impact,
      probability: probability,
      score: score,
      level: entry.level || calculateRiskLevel(score),
      treatment: entry.treatment || '',
      comments: Array.isArray(entry.comments) ? entry.comments : [],
      codirDate: entry.codirDate || '',
      criticalAssetIds: Array.isArray(entry.criticalAssetIds) ? entry.criticalAssetIds : []
    };
  }

  /**
   * Normalise un tableau de Risques
   */
  function normalizeRisks(list) {
    if (!Array.isArray(list)) return [];
    return list.map(e => normalizeRiskEntry(e)).filter(Boolean);
  }

  /**
   * Normalise une entrée Menace
   */
  function normalizeThreatEntry(entry) {
    if (!entry || typeof entry !== 'object') return null;
    return {
      id: entry.id || '',
      title: entry.title || '',
      description: entry.description || '',
      likelihood: entry.likelihood || 1,
      impact: entry.impact || 1,
      score: entry.score || 1,
      level: entry.level || '',
      category: entry.category || '',
      comments: Array.isArray(entry.comments) ? entry.comments : [],
      criticalAssetIds: Array.isArray(entry.criticalAssetIds) ? entry.criticalAssetIds : []
    };
  }

  /**
   * Normalise un tableau de Menaces
   */
  function normalizeThreats(list) {
    if (!Array.isArray(list)) return [];
    return list.map(e => normalizeThreatEntry(e)).filter(Boolean);
  }

  /**
   * Normalise une entrée Non-conformité
   */
  function normalizeNcEntry(entry) {
    if (!entry || typeof entry !== 'object') return null;
    return {
      id: entry.id || '',
      title: entry.title || '',
      description: entry.description || '',
      type: entry.type || 'minor',
      status: entry.status || 'open',
      detectionDate: entry.detectionDate || '',
      assignedTo: entry.assignedTo || '',
      auditId: entry.auditId || '',
      rootCause: entry.rootCause || '',
      correctiveAction: entry.correctiveAction || '',
      closureDate: entry.closureDate || '',
      criticalAssetIds: Array.isArray(entry.criticalAssetIds) ? entry.criticalAssetIds : []
    };
  }

  /**
   * Normalise un tableau de Non-conformités
   */
  function normalizeNonconformities(list) {
    if (!Array.isArray(list)) return [];
    return list.map(e => normalizeNcEntry(e)).filter(Boolean);
  }

  /**
   * Normalise une entrée Audit
   */
  function normalizeAuditEntry(entry) {
    if (!entry || typeof entry !== 'object') return null;
    return {
      id: entry.id || '',
      title: entry.title || '',
      type: entry.type || 'internal',
      description: entry.description || '',
      scope: entry.scope || '',
      plannedDate: entry.plannedDate || '',
      executionDate: entry.executionDate || '',
      auditor: entry.auditor || '',
      status: entry.status || 'planned',
      findings: entry.findings || '',
      recommendations: entry.recommendations || ''
    };
  }

  /**
   * Normalise un tableau d'Audits
   */
  function normalizeAudits(list) {
    if (!Array.isArray(list)) return [];
    return list.map(e => normalizeAuditEntry(e)).filter(Boolean);
  }

  /**
   * Normalise une entrée KPI
   */
  function normalizeKpiEntry(entry) {
    if (!entry || typeof entry !== 'object') return null;
    return {
      id: entry.id || '',
      title: entry.title || '',
      description: entry.description || '',
      target: entry.target || '',
      current: entry.current || '',
      unit: entry.unit || '',
      frequency: entry.frequency || 'monthly',
      lastUpdate: entry.lastUpdate || '',
      trend: entry.trend || 'stable',
      history: Array.isArray(entry.history) ? entry.history : []
    };
  }

  /**
   * Normalise un tableau de KPIs
   */
  function normalizeKpis(list) {
    if (!Array.isArray(list)) return [];
    return list.map(e => normalizeKpiEntry(e)).filter(Boolean);
  }

  /**
   * Normalise une entrée Revue de direction
   */
  function normalizeReviewEntry(entry) {
    if (!entry || typeof entry !== 'object') return null;
    return {
      id: entry.id || '',
      date: entry.date || '',
      participants: entry.participants || '',
      inputs: entry.inputs || '',
      decisions: entry.decisions || '',
      comments: entry.comments || '',
      nextReviewDate: entry.nextReviewDate || ''
    };
  }

  /**
   * Normalise un tableau de Revues
   */
  function normalizeReviews(list) {
    if (!Array.isArray(list)) return [];
    return list.map(e => normalizeReviewEntry(e)).filter(Boolean);
  }

  /**
   * Normalise une entrée Partie prenante
   */
  function normalizeStakeholderEntry(entry) {
    if (!entry || typeof entry !== 'object') return null;
    return {
      id: entry.id || '',
      name: entry.name || '',
      role: entry.role || '',
      contact: entry.contact || '',
      notes: entry.notes || '',
      category: entry.category || '',
      influence: entry.influence || '',
      expectations: entry.expectations || ''
    };
  }

  /**
   * Normalise un tableau de Parties prenantes
   */
  function normalizeStakeholders(list) {
    if (!Array.isArray(list)) return [];
    return list.map(e => normalizeStakeholderEntry(e)).filter(Boolean);
  }

  /**
   * Normalise une entrée Document
   */
  function normalizeDocumentEntry(entry) {
    if (!entry || typeof entry !== 'object') return null;
    return {
      id: entry.id || '',
      type: entry.type || '',
      title: entry.title || '',
      reference: entry.reference || '',
      version: entry.version || '',
      status: entry.status || 'draft',
      owner: entry.owner || '',
      lastReview: entry.lastReview || '',
      nextReview: entry.nextReview || '',
      location: entry.location || '',
      notes: entry.notes || ''
    };
  }

  /**
   * Normalise un tableau de Documents
   */
  function normalizeDocuments(list) {
    if (!Array.isArray(list)) return [];
    return list.map(e => normalizeDocumentEntry(e)).filter(Boolean);
  }

  /**
   * Normalise une entrée Control ISO 27002
   */
  function normalizeControlEntry(entry) {
    if (!entry || typeof entry !== 'object') return null;
    return {
      id: entry.id || '',
      reference: entry.reference || '',
      numero: entry.numero || '',
      title: entry.title || '',
      measure: entry.measure || '',
      description: entry.description || '',
      recommendations: Array.isArray(entry.recommendations) ? entry.recommendations : [],
      category: entry.category || 'Non classé',
      status: entry.status || 'non_implemente',
      cmm: entry.cmm || 'Initial',
      comments: entry.comments || '',
      evidence: entry.evidence || '',
      justificationControl: entry.justificationControl || '',
      subcategory: Array.isArray(entry.subcategory) ? entry.subcategory : []
    };
  }

  /**
   * Normalise un tableau de Controls
   */
  function normalizeControls(list) {
    if (!Array.isArray(list)) return [];
    return list.map(e => normalizeControlEntry(e)).filter(Boolean);
  }

  /**
   * Fonction globale de normalisation de toutes les données importées
   */
  function normalizeAllImportedData(data) {
    return {
      controls: normalizeControls(data.controls),
      soa: normalizeSoa(data.soa),
      actions: normalizeActions(data.actions),
      risks: normalizeRisks(data.risks),
      threats: normalizeThreats(data.threats),
      nonconformities: normalizeNonconformities(data.nonconformities),
      audits: normalizeAudits(data.audits),
      kpis: normalizeKpis(data.kpis),
      reviews: normalizeReviews(data.reviews),
      stakeholders: normalizeStakeholders(data.stakeholders),
      documents: normalizeDocuments(data.documents),
      // Ces sections ont déjà leurs propres normalisations
      criticalAssets: data.criticalAssets,
      projectRisks: data.projectRisks,
      nis2Controls: data.nis2Controls,
      documentReview: data.documentReview,
      nis2Plan: data.nis2Plan,
      soclePillars: data.soclePillars,
      targetMaturity: data.targetMaturity,
      criticalAssetsMeta: data.criticalAssetsMeta,
      nextActionId: data.nextActionId,
      title: data.title,
      actionsHistory: data.actionsHistory
    };
  }

  // Export des fonctions de normalisation globales
  window.normalizeActions = normalizeActions;
  window.normalizeRisks = normalizeRisks;
  window.normalizeThreats = normalizeThreats;
  window.normalizeNonconformities = normalizeNonconformities;
  window.normalizeAudits = normalizeAudits;
  window.normalizeKpis = normalizeKpis;
  window.normalizeReviews = normalizeReviews;
  window.normalizeStakeholders = normalizeStakeholders;
  window.normalizeDocuments = normalizeDocuments;
  window.normalizeControls = normalizeControls;
  window.normalizeAllImportedData = normalizeAllImportedData;

  // Options pour les sources de sélection SOA (ISO 27001:2022 clause 6.1.3)
  const SOA_SOURCE_OPTIONS = [
    { value: "", label: "— Sélectionner —" },
    { value: "risk", label: "Risque identifié" },
    { value: "legal", label: "Exigence légale" },
    { value: "contractual", label: "Exigence contractuelle" },
    { value: "best_practice", label: "Bonne pratique" },
    { value: "regulatory", label: "Exigence réglementaire" },
    { value: "business", label: "Exigence métier" }
  ];

  // Options pour l'état d'implémentation SOA
  const SOA_IMPL_STATUS_OPTIONS = [
    { value: "", label: "— État —" },
    { value: "not_started", label: "Non démarré" },
    { value: "planned", label: "Planifié" },
    { value: "in_progress", label: "En cours" },
    { value: "implemented", label: "Implémenté" },
    { value: "verified", label: "Vérifié" }
  ];

  // Export des constantes SoA
  window.SOA_SOURCE_OPTIONS = SOA_SOURCE_OPTIONS;
  window.SOA_IMPL_STATUS_OPTIONS = SOA_IMPL_STATUS_OPTIONS;

  // Fonction pour obtenir le domaine d'un contrôle
  function getControlDomain(reference) {
    if (!reference) return "Non classé";
    const prefix = reference.split('.')[0];
    return DOMAIN_BY_PREFIX[prefix] || "Non classé";
  }

  function loadSoaTable() {
    const tbody = document.getElementById("soaTableBody");
    if (!tbody) return;

    // Protection : s'assurer que le module SOA reste actif pendant le rechargement
    const soaModule = document.getElementById("soa");
    const wasActive = soaModule && soaModule.classList.contains("module--active");

    tbody.innerHTML = "";

    const applicabilityFilter = document.getElementById("soaApplicabilityFilter")?.value || "";
    const domainFilter = document.getElementById("soaDomainFilter")?.value || "";

    const sortedControls = [...appData.controls].sort((a, b) => a.reference.localeCompare(b.reference, void 0, { numeric: true }));

    // Stats pour les indicateurs de couverture
    let applicableCount = 0;
    let notApplicableCount = 0;
    let implementedCount = 0;
    const domainStats = {};

    sortedControls.forEach((control) => {
      let entry = appData.soa.find(e => e.controlId === control.id);
      if (!entry) {
        entry = {
          controlId: control.id,
          applicable: true,
          justification: "",
          linkedRiskIds: [],
          source: "",
          implStatus: "",
          implDate: "",
          exclusionReason: "",
          owner: "",
          evidenceRef: "",
          lastReviewDate: ""
        };
        appData.soa.push(entry);
      }
      // S'assurer que les nouveaux champs existent
      if (!entry.linkedRiskIds) entry.linkedRiskIds = [];
      if (!entry.source) entry.source = "";
      if (!entry.implStatus) entry.implStatus = "";
      if (!entry.implDate) entry.implDate = "";
      if (!entry.exclusionReason) entry.exclusionReason = "";
      if (!entry.owner) entry.owner = "";
      if (!entry.evidenceRef) entry.evidenceRef = "";
      if (!entry.lastReviewDate) entry.lastReviewDate = "";

      const domain = getControlDomain(control.reference);

      // Stats
      if (entry.applicable) {
        applicableCount++;
        if (entry.implStatus === "implemented" || entry.implStatus === "verified") {
          implementedCount++;
        }
      } else {
        notApplicableCount++;
      }

      // Stats par domaine
      if (!domainStats[domain]) {
        domainStats[domain] = { total: 0, applicable: 0, implemented: 0 };
      }
      domainStats[domain].total++;
      if (entry.applicable) {
        domainStats[domain].applicable++;
        if (entry.implStatus === "implemented" || entry.implStatus === "verified") {
          domainStats[domain].implemented++;
        }
      }

      // Filtres
      if (applicabilityFilter === "selected" && !entry.applicable) return;
      if (applicabilityFilter === "not_selected" && entry.applicable) return;
      if (domainFilter && domain !== domainFilter) return;

      const riskBadgesHtml = renderSoaRiskBadges(entry.linkedRiskIds);

      // Options source
      const sourceOptions = SOA_SOURCE_OPTIONS.map(opt =>
        `<option value="${opt.value}" ${entry.source === opt.value ? "selected" : ""}>${opt.label}</option>`
      ).join("");

      // Options état implémentation
      const implStatusOptions = SOA_IMPL_STATUS_OPTIONS.map(opt =>
        `<option value="${opt.value}" ${entry.implStatus === opt.value ? "selected" : ""}>${opt.label}</option>`
      ).join("");

      const row = document.createElement("tr");
      row.className = entry.applicable ? "soa-row--applicable" : "soa-row--not-applicable";
      row.dataset.controlId = control.id;

      // Structure à 6 colonnes pour une meilleure lisibilité
      row.innerHTML = `
            <td class="soa-cell-control">
              <div class="soa-control-header">
                <strong class="soa-control-ref">${escapeHtml(control.reference)}</strong>
                <span class="badge badge--domain badge--${domain.toLowerCase()}">${domain}</span>
              </div>
              <div class="soa-control-title">${escapeHtml(control.title)}</div>
              <div class="soa-control-desc">${escapeHtml((control.description || "").substring(0, 120))}${control.description && control.description.length > 120 ? "..." : ""}</div>
            </td>
            <td class="soa-cell-applicability">
              <select class="form-control form-control--sm soa-decision" data-id="${control.id}">
                <option value="selected" ${entry.applicable ? "selected" : ""}>✓ Applicable</option>
                <option value="not_selected" ${!entry.applicable ? "selected" : ""}>✗ Non applicable</option>
              </select>
              <select class="form-control form-control--sm soa-source mt-1" data-id="${control.id}" ${!entry.applicable ? "disabled" : ""} title="Source de l'exigence">
                ${sourceOptions}
              </select>
              ${!entry.applicable ? `<textarea class="form-control form-control--sm soa-exclusion mt-1" data-id="${control.id}" placeholder="Motif d'exclusion" rows="2">${escapeHtml(entry.exclusionReason || "")}</textarea>` : ""}
            </td>
            <td class="soa-cell-impl">
              <select class="form-control form-control--sm soa-impl-status" data-id="${control.id}" ${!entry.applicable ? "disabled" : ""}>
                ${implStatusOptions}
              </select>
              <input type="text" class="form-control form-control--sm soa-owner mt-1" data-id="${control.id}" value="${escapeHtml(entry.owner || "")}" placeholder="Responsable" ${!entry.applicable ? "disabled" : ""}>
              ${entry.applicable ? `<input type="date" class="form-control form-control--sm soa-impl-date mt-1" data-id="${control.id}" value="${entry.implDate || ""}" title="Date d'implémentation">` : ""}
            </td>
            <td class="soa-cell-risks">
              <div class="soa-risks">
                ${riskBadgesHtml}
                <button type="button" class="btn btn--outline btn--xs soa-edit-risks" data-id="${control.id}" title="Lier des risques">
                  <i class="fas fa-link"></i>
                </button>
              </div>
            </td>
            <td class="soa-cell-docs">
              <input type="text" class="form-control form-control--sm soa-evidence" data-id="${control.id}" value="${escapeHtml(entry.evidenceRef || "")}" placeholder="Réf. preuves" ${!entry.applicable ? "disabled" : ""}>
              <input type="date" class="form-control form-control--sm soa-review-date mt-1" data-id="${control.id}" value="${entry.lastReviewDate || ""}" title="Dernière revue" ${!entry.applicable ? "disabled" : ""}>
            </td>
            <td class="soa-cell-justification">
              <textarea class="form-control soa-justification" data-id="${control.id}" placeholder="Justification" rows="3">${escapeHtml(entry.justification || "")}</textarea>
            </td>`;
      tbody.appendChild(row);
    });

    // Mettre à jour les indicateurs de couverture
    updateSoaCoverageStats(applicableCount, notApplicableCount, implementedCount, domainStats);

    // Gestionnaires pour le select d'applicabilité
    tbody.querySelectorAll(".soa-decision").forEach((select) => {
      select.dataset.readonlyLock = "true";
      if (isReadOnlyMode()) select.disabled = true;
      select.addEventListener("change", function(e) {
        e.preventDefault();
        e.stopPropagation();
        if (!ensureEditMode("mettre à jour la DdA")) {
          refreshReadOnlyLocks();
          loadSoaTable(); // Recharger pour annuler
          return;
        }
        const id = this.getAttribute("data-id");
        const entry = appData.soa.find(e => e.controlId === id);
        if (entry) {
          entry.applicable = this.value === "selected";
          if (entry.applicable) {
            entry.exclusionReason = "";
          }
          saveData();
          loadSoaTable(); // Recharger pour afficher/masquer les champs conditionnels
        }
      });
    });

    // Gestionnaires pour la source
    tbody.querySelectorAll(".soa-source").forEach((select) => {
      select.dataset.readonlyLock = "true";
      if (isReadOnlyMode()) select.disabled = true;
      select.addEventListener("change", function() {
        if (!ensureEditMode("mettre à jour la DdA")) {
          refreshReadOnlyLocks();
          return;
        }
        const id = this.getAttribute("data-id");
        const entry = appData.soa.find(e => e.controlId === id);
        if (entry) {
          entry.source = this.value;
          saveData();
        }
      });
    });

    // Gestionnaires pour l'état d'implémentation
    tbody.querySelectorAll(".soa-impl-status").forEach((select) => {
      select.dataset.readonlyLock = "true";
      if (isReadOnlyMode()) select.disabled = true;
      select.addEventListener("change", function() {
        if (!ensureEditMode("mettre à jour la DdA")) {
          refreshReadOnlyLocks();
          return;
        }
        const id = this.getAttribute("data-id");
        const entry = appData.soa.find(e => e.controlId === id);
        if (entry) {
          entry.implStatus = this.value;
          saveData();
          loadSoaTable(); // Recharger pour mettre à jour les stats
        }
      });
    });

    // Gestionnaires pour la date d'implémentation
    tbody.querySelectorAll(".soa-impl-date").forEach((input) => {
      input.dataset.readonlyLock = "true";
      if (isReadOnlyMode()) input.disabled = true;
      input.addEventListener("change", function() {
        if (!ensureEditMode("mettre à jour la DdA")) {
          refreshReadOnlyLocks();
          return;
        }
        const id = this.getAttribute("data-id");
        const entry = appData.soa.find(e => e.controlId === id);
        if (entry) {
          entry.implDate = this.value;
          saveData();
        }
      });
    });

    // Gestionnaires pour le propriétaire
    tbody.querySelectorAll(".soa-owner").forEach((input) => {
      input.dataset.readonlyLock = "true";
      if (isReadOnlyMode()) input.disabled = true;
      input.addEventListener("change", function(e) {
        e.preventDefault();
        e.stopPropagation();
        if (!ensureEditMode("mettre à jour la DdA")) {
          refreshReadOnlyLocks();
          return;
        }
        const id = this.getAttribute("data-id");
        const entry = appData.soa.find(e => e.controlId === id);
        if (entry) {
          entry.owner = this.value;
          saveData();
        }
      });
    });

    // Gestionnaires pour les preuves/références documentaires
    tbody.querySelectorAll(".soa-evidence").forEach((input) => {
      input.dataset.readonlyLock = "true";
      if (isReadOnlyMode()) input.disabled = true;
      input.addEventListener("change", function() {
        if (!ensureEditMode("mettre à jour la DdA")) {
          refreshReadOnlyLocks();
          return;
        }
        const id = this.getAttribute("data-id");
        const entry = appData.soa.find(e => e.controlId === id);
        if (entry) {
          entry.evidenceRef = this.value;
          saveData();
        }
      });
    });

    // Gestionnaires pour la date de dernière revue
    tbody.querySelectorAll(".soa-review-date").forEach((input) => {
      input.dataset.readonlyLock = "true";
      if (isReadOnlyMode()) input.disabled = true;
      input.addEventListener("change", function() {
        if (!ensureEditMode("mettre à jour la DdA")) {
          refreshReadOnlyLocks();
          return;
        }
        const id = this.getAttribute("data-id");
        const entry = appData.soa.find(e => e.controlId === id);
        if (entry) {
          entry.lastReviewDate = this.value;
          saveData();
        }
      });
    });

    // Gestionnaires pour le motif d'exclusion
    tbody.querySelectorAll(".soa-exclusion").forEach((input) => {
      input.dataset.readonlyLock = "true";
      if (isReadOnlyMode()) input.disabled = true;
      input.addEventListener("change", function() {
        if (!ensureEditMode("mettre à jour la DdA")) {
          refreshReadOnlyLocks();
          return;
        }
        const id = this.getAttribute("data-id");
        const entry = appData.soa.find(e => e.controlId === id);
        if (entry) {
          entry.exclusionReason = this.value;
          saveData();
        }
      });
    });

    // Gestionnaires pour la justification
    tbody.querySelectorAll(".soa-justification").forEach((input) => {
      input.dataset.readonlyLock = "true";
      if (isReadOnlyMode()) input.disabled = true;
      input.addEventListener("change", function() {
        if (!ensureEditMode("mettre à jour la DdA")) {
          refreshReadOnlyLocks();
          return;
        }
        const id = this.getAttribute("data-id");
        const entry = appData.soa.find(e => e.controlId === id);
        if (entry) {
          entry.justification = this.value;
          saveData();
        }
      });
    });

    // Gestionnaires pour le bouton d'édition des risques
    tbody.querySelectorAll(".soa-edit-risks").forEach((btn) => {
      btn.dataset.readonlyLock = "true";
      if (isReadOnlyMode()) btn.disabled = true;
      btn.addEventListener("click", function(e) {
        e.preventDefault();
        e.stopPropagation();
        if (!ensureEditMode("modifier les risques associés")) {
          return;
        }
        const id = this.getAttribute("data-id");
        editSoaRisks(id);
      });
    });

    // Protection : restaurer l'état actif du module SOA si nécessaire
    if (wasActive && soaModule && !soaModule.classList.contains("module--active")) {
      console.warn("[SOA] Module SOA a perdu son état actif pendant loadSoaTable - restauration");
      document.querySelectorAll(".module").forEach(m => m.classList.remove("module--active"));
      soaModule.classList.add("module--active");
      const soaNav = document.querySelector('.nav__item[data-tab="soa"]');
      if (soaNav) {
        document.querySelectorAll(".nav__item").forEach(n => n.classList.remove("nav__item--active"));
        soaNav.classList.add("nav__item--active");
      }
    }
  }

  // Mettre à jour les statistiques de couverture SOA
  function updateSoaCoverageStats(applicable, notApplicable, implemented, domainStats) {
    safeSetText("soaApplicableCount", applicable);
    safeSetText("soaNotApplicableCount", notApplicable);

    const implPercent = applicable > 0 ? Math.round((implemented / applicable) * 100) : 0;
    safeSetText("soaImplementedCount", implPercent + "%");

    const implBar = document.getElementById("soaImplementedBar");
    if (implBar) {
      implBar.style.width = implPercent + "%";
      implBar.className = "soa-coverage__fill";
      if (implPercent >= 80) implBar.classList.add("soa-coverage__fill--good");
      else if (implPercent >= 50) implBar.classList.add("soa-coverage__fill--medium");
      else implBar.classList.add("soa-coverage__fill--low");
    }

    // Couverture par domaine
    const domainsContainer = document.getElementById("soaDomainsCoverage");
    if (domainsContainer) {
      domainsContainer.innerHTML = Object.entries(domainStats).map(([domain, stats]) => {
        const pct = stats.applicable > 0 ? Math.round((stats.implemented / stats.applicable) * 100) : 0;
        return `<div class="soa-domain-stat">
          <span class="soa-domain-stat__label">${domain}</span>
          <span class="soa-domain-stat__value">${pct}%</span>
        </div>`;
      }).join("");
    }
  }

  // Options pour le statut de conformité Doc Review
  const DOC_REVIEW_STATUS_OPTIONS = [
    { value: "", label: "— Statut —" },
    { value: "conforme", label: "Conforme", icon: "check-circle", class: "success" },
    { value: "partiel", label: "Partiellement conforme", icon: "exclamation-circle", class: "warning" },
    { value: "non_conforme", label: "Non conforme", icon: "times-circle", class: "danger" },
    { value: "na", label: "Non applicable", icon: "minus-circle", class: "muted" }
  ];

  function loadDocReviewTable() {
    const tbody = document.getElementById("docReviewTableBody");
    if (!tbody) return;
    tbody.innerHTML = "";

    const statusFilter = document.getElementById("docReviewStatusFilter")?.value || "";

    // Stats pour les indicateurs
    let conformeCount = 0;
    let partielCount = 0;
    let nonConformeCount = 0;
    let totalItems = 0;

    DOCUMENT_REVIEW_ITEMS.forEach((item) => {
      if (item.type === "chapter") {
        const row = document.createElement("tr");
        row.className = "table-section";
        row.innerHTML = `<td colspan="7">${escapeHtml(item.title)}</td>`;
        tbody.appendChild(row);
      } else {
        let entry = appData.documentReview.find(e => e.reference === item.reference);
        if (!entry) {
          entry = {
            reference: item.reference,
            justification: "",
            tool: "",
            status: "",
            responsible: "",
            lastReview: "",
            nextReview: "",
            documentLink: ""
          };
          appData.documentReview.push(entry);
        }
        // S'assurer que les nouveaux champs existent
        if (!entry.status) entry.status = "";
        if (!entry.responsible) entry.responsible = "";
        if (!entry.lastReview) entry.lastReview = "";
        if (!entry.nextReview) entry.nextReview = "";
        if (!entry.documentLink) entry.documentLink = "";

        // Stats
        totalItems++;
        if (entry.status === "conforme") conformeCount++;
        else if (entry.status === "partiel") partielCount++;
        else if (entry.status === "non_conforme") nonConformeCount++;

        // Filtre par statut
        if (statusFilter && entry.status !== statusFilter) return;

        // Options statut
        const statusOptions = DOC_REVIEW_STATUS_OPTIONS.map(opt =>
          `<option value="${opt.value}" ${entry.status === opt.value ? "selected" : ""}>${opt.label}</option>`
        ).join("");

        // Classe de ligne selon statut
        let rowClass = "";
        if (entry.status === "conforme") rowClass = "doc-review-row--conforme";
        else if (entry.status === "partiel") rowClass = "doc-review-row--partiel";
        else if (entry.status === "non_conforme") rowClass = "doc-review-row--non-conforme";
        else if (entry.status === "na") rowClass = "doc-review-row--na";

        const row = document.createElement("tr");
        row.className = rowClass;
        row.innerHTML = `
            <td><strong>${escapeHtml(item.reference)}</strong></td>
            <td class="doc-review-desc">${escapeHtml(item.description)}</td>
            <td>
              <select class="form-control form-control--sm doc-review-status" data-ref="${escapeHtml(item.reference)}">
                ${statusOptions}
              </select>
            </td>
            <td>
              <input type="text" class="form-control form-control--sm doc-review-responsible" data-ref="${escapeHtml(item.reference)}" value="${escapeHtml(entry.responsible || "")}" placeholder="Responsable">
            </td>
            <td><textarea class="form-control doc-review-justification" data-ref="${escapeHtml(item.reference)}" rows="2" placeholder="Justification de conformité">${escapeHtml(entry.justification || "")}</textarea></td>
            <td>
              <textarea class="form-control doc-review-tool" data-ref="${escapeHtml(item.reference)}" rows="2" placeholder="Document/preuve">${escapeHtml(entry.tool || "")}</textarea>
              ${entry.documentLink ? `<a href="${escapeHtml(entry.documentLink)}" target="_blank" class="doc-review-link"><i class="fas fa-external-link-alt"></i></a>` : ""}
              <input type="text" class="form-control form-control--sm doc-review-doclink mt-1" data-ref="${escapeHtml(item.reference)}" value="${escapeHtml(entry.documentLink || "")}" placeholder="Lien document">
            </td>
            <td>
              <input type="date" class="form-control form-control--sm doc-review-last-review" data-ref="${escapeHtml(item.reference)}" value="${entry.lastReview || ""}" title="Dernière revue">
              <input type="date" class="form-control form-control--sm doc-review-next-review mt-1" data-ref="${escapeHtml(item.reference)}" value="${entry.nextReview || ""}" title="Prochaine revue">
            </td>
        `;
        tbody.appendChild(row);
      }
    });

    // Mettre à jour les statistiques
    updateDocReviewStats(conformeCount, partielCount, nonConformeCount, totalItems);

    // Gestionnaires pour le statut
    tbody.querySelectorAll(".doc-review-status").forEach((select) => {
      select.dataset.readonlyLock = "true";
      if (isReadOnlyMode()) select.disabled = true;
      select.addEventListener("change", function() {
        if (!ensureEditMode("mettre à jour la revue documentaire")) {
          refreshReadOnlyLocks();
          loadDocReviewTable();
          return;
        }
        const ref = this.getAttribute("data-ref");
        const entry = appData.documentReview.find(e => e.reference === ref);
        if (entry) {
          entry.status = this.value;
          saveData();
          loadDocReviewTable(); // Recharger pour mettre à jour les stats et styles
        }
      });
    });

    // Gestionnaires pour le responsable
    tbody.querySelectorAll(".doc-review-responsible").forEach((input) => {
      input.dataset.readonlyLock = "true";
      if (isReadOnlyMode()) input.disabled = true;
      input.addEventListener("change", function() {
        if (!ensureEditMode("mettre à jour la revue documentaire")) {
          refreshReadOnlyLocks();
          return;
        }
        const ref = this.getAttribute("data-ref");
        const entry = appData.documentReview.find(e => e.reference === ref);
        if (entry) {
          entry.responsible = this.value;
          saveData();
        }
      });
    });

    // Gestionnaires pour la justification
    tbody.querySelectorAll(".doc-review-justification").forEach((input) => {
      input.dataset.readonlyLock = "true";
      if (isReadOnlyMode()) input.disabled = true;
      input.addEventListener("change", function() {
        if (!ensureEditMode("mettre à jour la revue documentaire")) {
          refreshReadOnlyLocks();
          return;
        }
        const ref = this.getAttribute("data-ref");
        const entry = appData.documentReview.find(e => e.reference === ref);
        if (entry) {
          entry.justification = this.value;
          saveData();
        }
      });
    });

    // Gestionnaires pour l'outil/preuve
    tbody.querySelectorAll(".doc-review-tool").forEach((input) => {
      input.dataset.readonlyLock = "true";
      if (isReadOnlyMode()) input.disabled = true;
      input.addEventListener("change", function() {
        if (!ensureEditMode("mettre à jour la revue documentaire")) {
          refreshReadOnlyLocks();
          return;
        }
        const ref = this.getAttribute("data-ref");
        const entry = appData.documentReview.find(e => e.reference === ref);
        if (entry) {
          entry.tool = this.value;
          saveData();
        }
      });
    });

    // Gestionnaires pour le lien document
    tbody.querySelectorAll(".doc-review-doclink").forEach((input) => {
      input.dataset.readonlyLock = "true";
      if (isReadOnlyMode()) input.disabled = true;
      input.addEventListener("change", function() {
        if (!ensureEditMode("mettre à jour la revue documentaire")) {
          refreshReadOnlyLocks();
          return;
        }
        const ref = this.getAttribute("data-ref");
        const entry = appData.documentReview.find(e => e.reference === ref);
        if (entry) {
          entry.documentLink = this.value;
          saveData();
          loadDocReviewTable();
        }
      });
    });

    // Gestionnaires pour les dates
    tbody.querySelectorAll(".doc-review-last-review, .doc-review-next-review").forEach((input) => {
      input.dataset.readonlyLock = "true";
      if (isReadOnlyMode()) input.disabled = true;
      input.addEventListener("change", function() {
        if (!ensureEditMode("mettre à jour la revue documentaire")) {
          refreshReadOnlyLocks();
          return;
        }
        const ref = this.getAttribute("data-ref");
        const entry = appData.documentReview.find(e => e.reference === ref);
        if (entry) {
          if (this.classList.contains("doc-review-last-review")) {
            entry.lastReview = this.value;
          } else {
            entry.nextReview = this.value;
          }
          saveData();
        }
      });
    });
  }

  // Mettre à jour les statistiques Doc Review
  function updateDocReviewStats(conforme, partiel, nonConforme, total) {
    safeSetText("docReviewConformeCount", conforme);
    safeSetText("docReviewPartielCount", partiel);
    safeSetText("docReviewNonConformeCount", nonConforme);

    const conformePercent = total > 0 ? Math.round((conforme / total) * 100) : 0;
    safeSetText("docReviewProgressValue", conformePercent + "%");

    const progressBar = document.getElementById("docReviewProgressBar");
    if (progressBar) {
      progressBar.style.width = conformePercent + "%";
      progressBar.className = "doc-review-stats__fill";
      if (conformePercent >= 80) progressBar.classList.add("doc-review-stats__fill--good");
      else if (conformePercent >= 50) progressBar.classList.add("doc-review-stats__fill--medium");
      else progressBar.classList.add("doc-review-stats__fill--low");
    }
  }

  window.loadDocReviewTable = loadDocReviewTable;

  function loadNis2ControlsTable() {
    const tbody = document.getElementById("nis2ControlsTableBody");
    if (!tbody) return;
    const functionFilter = document.getElementById("nis2FunctionFilter");
    const categoryFilter = document.getElementById("nis2CategoryFilter");
    const statusFilter = document.getElementById("nis2StatusFilter");
    const cmmFilter = document.getElementById("nis2CmmFilter");

    if (functionFilter && functionFilter.options.length === 1) {
      const funcs = [...new Set(appData.nis2Controls.map(c => c.fonction))];
      funcs.forEach(f => {
        const opt = document.createElement("option");
        opt.value = f;
        opt.textContent = f;
        functionFilter.appendChild(opt);
      });
    }
    if (categoryFilter && categoryFilter.options.length === 1) {
      const cats = [...new Set(appData.nis2Controls.map(c => c.categorie))];
      cats.forEach(c => {
        const opt = document.createElement("option");
        opt.value = c;
        opt.textContent = c;
        categoryFilter.appendChild(opt);
      });
    }

    const fVal = functionFilter?.value || "";
    const cVal = categoryFilter?.value || "";
    const sVal = statusFilter?.value || "";
    const cmmVal = cmmFilter?.value || "";
    const filtered = appData.nis2Controls.filter(c =>
      (!fVal || c.fonction === fVal) &&
      (!cVal || c.categorie === cVal) &&
      (!sVal || c.status === sVal) &&
      (!cmmVal || c.cmm === cmmVal)
    );

    tbody.innerHTML = "";
    filtered.forEach(c => {
      const row = document.createElement("tr");
      const statusClasses = {
        "non_implemente": "status-badge--non_implemente",
        "en_cours": "status-badge--en_cours",
        "implemente": "status-badge--implemente",
        "verifie": "status-badge--verifie"
      };
      const statusLabels = {
        "non_implemente": "Non impl\xE9ment\xE9",
        "en_cours": "En cours",
        "implemente": "Impl\xE9ment\xE9",
        "verifie": "V\xE9rifi\xE9"
      };
      row.innerHTML = `
        <td>${escapeHtml(c.fonction)}</td>
        <td>${escapeHtml(c.categorie)}</td>
        <td>${escapeHtml(c.description)}</td>
        <td><span class="status-badge ${statusClasses[c.status]}">${statusLabels[c.status]}</span></td>
        <td>${c.cmm || ""}</td>
        <td>${escapeHtml(c.justification || "")}</td>
      `;
      row.addEventListener("click", () => {
        editNis2Control(c.id);
      });
      tbody.appendChild(row);
    });
  }

  function editNis2Control(id) {
    const control = appData.nis2Controls.find(c => c.id === id);
    if (!control) return;
    openModal("Suivi du contr\xF4le NIS2", [
      { name: "fonction", label: "Fonction", value: control.fonction, readOnly: true },
      { name: "categorie", label: "Cat\xE9gorie", value: control.categorie, readOnly: true },
      { name: "description", label: "Description", type: "textarea", value: control.description, readOnly: true },
      { name: "status", label: "Statut", type: "select", value: control.status, options: CONTROL_STATUS_OPTIONS },
      { name: "cmm", label: "Niveau de maturit\xE9 CMM", type: "select", value: control.cmm, options: CMM_LEVEL_OPTIONS },
      { name: "justification", label: "Justification", type: "textarea", value: control.justification || "" }
    ], (data) => {
      control.status = data.status;
      control.cmm = data.cmm;
      control.justification = data.justification;
      saveData();
      loadNis2ControlsTable();
      const cmmMap = { "Initial": 1, "R\xE9p\xE9table": 2, "D\xE9fini": 3, "G\xE9r\xE9": 4, "Optimis\xE9": 5 };
      const sums = Object.fromEntries(NIS2_CATEGORIES.map(c => [c, 0]));
      const counts = Object.fromEntries(NIS2_CATEGORIES.map(c => [c, 0]));
      appData.nis2Controls.forEach(c => {
        const match = c.categorie.match(/\(([^)]+)\)/);
        const cat = match ? match[1] : c.categorie;
        const val = typeof c.cmm === "number" ? c.cmm : cmmMap[c.cmm];
        if (cat && val) {
          sums[cat] += val;
          counts[cat] += 1;
        }
      });
      const averages = Object.fromEntries(NIS2_CATEGORIES.map(c => [c, counts[c] ? +(sums[c] / counts[c]).toFixed(2) : 0]));
      appData.targetMaturity.nis2 = averages;
      updateDashboard();
      renderNis2ProgramRadar();
    });
  }
  // Options pour la fréquence de revue des contrôles
  const CONTROL_REVIEW_FREQ_OPTIONS = [
    { value: "", label: "— Sélectionner —" },
    { value: "monthly", label: "Mensuelle" },
    { value: "quarterly", label: "Trimestrielle" },
    { value: "biannual", label: "Semestrielle" },
    { value: "annual", label: "Annuelle" },
    { value: "on_change", label: "Sur changement" }
  ];

  // Options pour l'efficacité mesurée
  const CONTROL_EFFECTIVENESS_OPTIONS = [
    { value: "", label: "— Non évaluée —" },
    { value: "effective", label: "✓ Efficace" },
    { value: "partially", label: "◐ Partiellement efficace" },
    { value: "ineffective", label: "✗ Non efficace" },
    { value: "not_tested", label: "○ Non testé" }
  ];

  function editControl(controlId) {
    const control = appData.controls.find((c) => c.id === controlId);
    if (!control) return;

    // Vérifier le statut SoA
    const isNonApplicable = isControlNonApplicable(control.id);
    const soaJustification = getSoaJustification(control.id);

    // S'assurer que les nouveaux champs existent
    if (!control.owner) control.owner = "";
    if (!control.lastEvaluation) control.lastEvaluation = "";
    if (!control.reviewFrequency) control.reviewFrequency = "";
    if (!control.effectiveness) control.effectiveness = "";
    if (!control.evidence) control.evidence = "";
    if (!control.kpiRef) control.kpiRef = "";

    // Construire les champs de base
    const fields = [
      { name: "numero", label: "Numérotation", value: control.numero, readOnly: true },
      { name: "reference", label: "Référence", value: control.reference, readOnly: true },
      { name: "title", label: "Titre", value: control.title, readOnly: true },
      { name: "category", label: "Catégorie", type: "select", value: control.category, options: DOMAIN_OPTIONS },
      { name: "status", label: "Statut", type: "select", value: control.status, options: CONTROL_STATUS_OPTIONS },
      { name: "cmm", label: "Niveau de maturité CMM", type: "select", value: control.cmm, options: CMM_LEVEL_OPTIONS },
      { name: "owner", label: "Propriétaire du contrôle", value: control.owner || "", placeholder: "Nom du responsable" },
      { name: "lastEvaluation", label: "Dernière évaluation", type: "date", value: control.lastEvaluation || "" },
      { name: "reviewFrequency", label: "Fréquence de revue", type: "select", value: control.reviewFrequency, options: CONTROL_REVIEW_FREQ_OPTIONS },
      { name: "effectiveness", label: "Efficacité mesurée", type: "select", value: control.effectiveness, options: CONTROL_EFFECTIVENESS_OPTIONS },
      { name: "evidence", label: "Preuves / Artefacts", type: "textarea", value: control.evidence || "", placeholder: "Documents, logs, captures..." },
      { name: "kpiRef", label: "KPI associé", value: control.kpiRef || "", placeholder: "Référence du KPI mesurant l'efficacité" },
      { name: "justificationControl", label: "Commentaires", type: "textarea", value: control.justificationControl || "" }
    ];

    openModal("Suivi du contrôle", fields, (data) => {
      control.category = data.category;
      control.status = data.status;
      control.cmm = data.cmm || control.cmm;
      control.owner = data.owner || "";
      control.lastEvaluation = data.lastEvaluation || "";
      control.reviewFrequency = data.reviewFrequency || "";
      control.effectiveness = data.effectiveness || "";
      control.evidence = data.evidence || "";
      control.kpiRef = data.kpiRef || "";
      control.justificationControl = data.justificationControl || control.justificationControl;
      saveData();
      loadControlsTable();
      loadSoaTable();
      updateDashboard();
    }, null, {
      // Callback pour ajouter du contenu personnalisé après ouverture de la modale
      afterOpen: () => {
        if (isNonApplicable) {
          const modalBody = document.getElementById("modalBody");
          if (modalBody) {
            const alertDiv = document.createElement("div");
            alertDiv.className = "soa-alert soa-alert--warning";
            alertDiv.innerHTML = `
              <div class="soa-alert__icon"><i class="fas fa-exclamation-triangle"></i></div>
              <div class="soa-alert__content">
                <strong>Ce contrôle est déclaré non applicable</strong>
                ${soaJustification ? `<p class="soa-alert__justification">Justification : ${escapeHtml(soaJustification)}</p>` : ""}
                <a href="#" class="soa-alert__link" id="goToSoaLink">
                  <i class="fas fa-external-link-alt"></i> Voir dans la Déclaration d'applicabilité
                </a>
              </div>
            `;
            modalBody.insertBefore(alertDiv, modalBody.firstChild);

            // Gestionnaire pour le lien
            document.getElementById("goToSoaLink")?.addEventListener("click", (e) => {
              e.preventDefault();
              closeModal();
              navigateToSoaControl(control.id);
            });
          }
        }
      }
    });
  }
  function viewControlDetails(controlId) {
    const control = appData.controls.find((c) => c.id === controlId);
    if (!control) return;
    const actionList = appData.actions
      .filter((a) => a.linkType === "control" && a.linkId === control.id)
      .map((a) => a.id)
      .join(", ");
    const recommendations = Array.isArray(control.recommendations) ? control.recommendations.join("\n- ") : "";
    alert(`R\xE9f\xE9rence: ${control.reference}
Titre: ${control.title}
Mesure: ${control.measure || control.description}
Recommandations:
- ${recommendations}
Cat\xE9gorie: ${control.category}
Statut: ${control.status}
Niveau CMM: ${control.cmm}
Actions li\xE9es: ${actionList || "Aucune"}`);
  }
  function addControl() {
    openModal("Ajouter un contr\xF4le", [
      { name: "numero", label: "Num\xE9rotation" },
      { name: "reference", label: "R\xE9f\xE9rence" },
      { name: "title", label: "Titre" },
      { name: "category", label: "Cat\xE9gorie", type: "select", value: "Organisationnel", options: DOMAIN_OPTIONS },
      { name: "status", label: "Statut", type: "select", value: "non_implemente", options: CONTROL_STATUS_OPTIONS },
      { name: "cmm", label: "Niveau de maturit\xE9 CMM", type: "select", value: "Initial", options: CMM_LEVEL_OPTIONS }
    ], (data) => {
      if (!data.reference || !data.title) return false;
      appData.controls.push({
        id: data.reference,
        numero: data.numero || "",
        reference: data.reference,
        title: data.title,
        description: "",
        category: data.category,
        status: data.status || "non_implemente",
        cmm: data.cmm || "Initial",
        comments: "",
        evidence: "",
        justificationControl: ""
      });
      appData.soa.push({ controlId: data.reference, applicable: true, justification: "" });
      saveData();
      loadControlsTable();
      loadSoaTable();
      updateDashboard();
    });
  }
  function deleteControl(controlId) {
    if (!confirm("Supprimer ce contr\xF4le ?")) return;
    appData.controls = appData.controls.filter((c) => c.id !== controlId);
    appData.actions = appData.actions.filter((a) => !(a.linkType === "control" && a.linkId === controlId));
    saveData();
    loadControlsTable();
    loadSoaTable();
    updateDashboard();
  }
  window.loadControlsTable = loadControlsTable;
  window.loadNis2ControlsTable = loadNis2ControlsTable;
  window.loadSoaTable = loadSoaTable;
  window.editNis2Control = editNis2Control;
  window.editControl = editControl;
  window.viewControlDetails = viewControlDetails;
  window.addControl = addControl;
  window.deleteControl = deleteControl;
  // Fonctions SoA helper pour indicateur non-applicable et risques liés
  window.isControlNonApplicable = isControlNonApplicable;
  window.getSoaJustification = getSoaJustification;
  window.navigateToSoaControl = navigateToSoaControl;
  window.editSoaRisks = editSoaRisks;
  window.renderSoaRiskBadges = renderSoaRiskBadges;

  // modules/actions.js
  var ACTION_PRIORITY_OPTIONS = [
    { value: "critical", label: "Critique" },
    { value: "high", label: "Haute" },
    { value: "medium", label: "Moyenne" },
    { value: "low", label: "Faible" }
  ];
  var ACTION_STATUS_OPTIONS = [
    { value: "planifie", label: "Planifi\xE9" },
    { value: "en_cours", label: "En cours" },
    { value: "termine", label: "Termin\xE9" },
    { value: "annule", label: "Annul\xE9" }
  ];
  function loadActionsTable() {
    const tbody = document.getElementById("actionsTableBody");
    if (!tbody) return;
    tbody.innerHTML = "";
    const sort = document.getElementById("actionsSort")?.value || "priority";
    updateActionsFilterOptions();
    const filterType = document.getElementById("actionsFilterType")?.value || "all";
    const filterValue = document.getElementById("actionsFilterValue")?.value || "all";
    const priorityOrder = { critical: 0, high: 1, medium: 2, low: 3 };
    const getActionIdNumber = (value) => {
      const text = value ? String(value) : "";
      const match = text.match(/(\d+)(?!.*\d)/);
      return match ? parseInt(match[1], 10) : NaN;
    };
    const getActionLinkSortKey = (action) => {
      const type = action.linkType || (action.controlId ? "control" : "");
      const ids = getActionLinkIds(action);
      const byId = (items, id) => items.find((item) => item.id === id);
      if (type === "risk") {
        const labels = ids
          .map((id) => {
            const r = byId(appData.risks, id);
            return r ? r.title : id;
          })
          .filter(Boolean);
        const idKey = ids.slice().sort().join(",");
        return `risk:${idKey}:${labels.join(" | ")}`;
      }
      if (type === "audit") {
        const audit = byId(appData.audits, ids[0]);
        const label = audit ? audit.title : ids[0];
        const key = ids[0] ? `audit:${ids[0]}` : "audit:missing";
        return `${key}:${label || ""}`;
      }
      if (type === "nc") {
        const nc = byId(appData.nonconformities, ids[0]);
        const auditId = nc && nc.auditId ? nc.auditId : "";
        if (auditId) {
          const audit = byId(appData.audits, auditId);
          const label = audit ? audit.title : auditId;
          return `audit:${auditId}:${label || ""}`;
        }
        const label = nc ? nc.title : ids[0];
        return `nc:${ids[0] || ""}:${label || ""}`;
      }
      if (type === "control") {
        const ctrl = byId(appData.controls, ids[0]);
        const label = ctrl ? ctrl.reference : ids[0];
        return `control:${ids[0] || ""}:${label || ""}`;
      }
      if (type === "threat") {
        const th = byId(appData.threats, ids[0]);
        const label = th ? th.title : ids[0];
        return `threat:${ids[0] || ""}:${label || ""}`;
      }
      if (type === "continuous") {
        const label = "Am\xE9lioration continue";
        return `continuous:${ids[0] || ""}:${label}`;
      }
      return `other:${type || ""}`;
    };
    const sourceActions = filterType !== "all" && filterValue !== "all"
      ? appData.actions.filter((action) => matchesActionFilter(action, filterType, filterValue))
      : appData.actions;
    const sortedActions = [...sourceActions].sort((a, b) => {
      switch (sort) {
        case "id": {
          const aNum = getActionIdNumber(a.id);
          const bNum = getActionIdNumber(b.id);
          if (Number.isFinite(aNum) && Number.isFinite(bNum)) return aNum - bNum;
          return String(a.id || "").localeCompare(String(b.id || ""), void 0, { numeric: true, sensitivity: "base" });
        }
        case "link": {
          const keyA = getActionLinkSortKey(a);
          const keyB = getActionLinkSortKey(b);
          const cmp = keyA.localeCompare(keyB, void 0, { numeric: true, sensitivity: "base" });
          if (cmp !== 0) return cmp;
          const fallbackA = getActionIdNumber(a.id);
          const fallbackB = getActionIdNumber(b.id);
          if (Number.isFinite(fallbackA) && Number.isFinite(fallbackB)) return fallbackA - fallbackB;
          return String(a.id || "").localeCompare(String(b.id || ""), void 0, { numeric: true, sensitivity: "base" });
        }
        case "dueDate":
          return (new Date(a.dueDate || "9999-12-31")) - (new Date(b.dueDate || "9999-12-31"));
        case "progress":
          return (a.progress || 0) - (b.progress || 0);
        default:
          return priorityOrder[a.priority] - priorityOrder[b.priority];
      }
    });
    sortedActions.forEach((action) => {
      const row = document.createElement("tr");
      let linkLabels = [];
      let linkClass = "";
      const type = action.linkType || (action.controlId ? "control" : "");
      const ids = getActionLinkIds(action);
      switch (type) {
        case "control": {
          const id = ids[0];
          if (id) {
            const ctrl = appData.controls.find((c) => c.id === id);
            linkLabels = [ctrl ? ctrl.reference : id];
            linkClass = "link-badge--control";
          }
          break;
        }
        case "risk": {
          linkLabels = ids
            .map((id) => {
              const r = appData.risks.find((rr) => rr.id === id);
              return r ? r.title : id;
            })
            .filter(Boolean);
          if (linkLabels.length) linkClass = "link-badge--risk";
          break;
        }
        case "nc": {
          const id = ids[0];
          if (id) {
            const nc = appData.nonconformities.find((n) => n.id === id);
            linkLabels = [nc ? nc.title : id];
            linkClass = "link-badge--nc";
          }
          break;
        }
        case "audit": {
          const id = ids[0];
          if (id) {
            const au = appData.audits.find((a) => a.id === id);
            linkLabels = [au ? au.title : id];
            linkClass = "link-badge--audit";
          }
          break;
        }
        case "threat": {
          const id = ids[0];
          if (id) {
            const th = appData.threats.find((t) => t.id === id);
            linkLabels = [th ? th.title : id];
            linkClass = "link-badge--threat";
          }
          break;
        }
        case "continuous": {
          linkLabels = ids.length ? ids.map(() => "Am\xE9lioration continue") : [];
          if (linkLabels.length) linkClass = "link-badge--continuous";
          break;
        }
      }
      const linkHtml = linkLabels.length
        ? linkLabels.map((label) => `<span class="link-badge ${linkClass}">${escapeHtml(label)}</span>`).join(" ")
        : "N/A";
      const assetBadges = buildCriticalAssetBadges(action.criticalAssetIds);
      const priorityClasses = { critical: "priority--critical", high: "priority--high", medium: "priority--medium", low: "priority--low" };
      const priorityLabels = { critical: "Critique", high: "Haute", medium: "Moyenne", low: "Faible" };
      row.innerHTML = `
            <td>${escapeHtml(action.id)}</td>
            <td>${escapeHtml(action.title)}</td>
            <td>${linkHtml}</td>
            <td>${assetBadges}</td>
            <td><span class="status-badge ${priorityClasses[action.priority]}">${priorityLabels[action.priority]}</span></td>
            <td><div class="progress-bar"><div class="progress-bar__fill" style="width: ${action.progress}%"></div></div><span>${action.progress}%</span></td>
            <td>${action.dueDate ? new Date(action.dueDate).toLocaleDateString() : "N/A"}</td>
            <td>${action.assignedTo ? escapeHtml(action.assignedTo) : "N/A"}</td>
            <td>${(action.comments || []).length}</td>
        `;
      row.addEventListener("click", (e) => {
        if (e.target.closest(".icon-btn--danger") || e.target.closest(".critical-asset-link")) return;
        editAction(action.id);
      });
      attachCriticalAssetLinkHandlers(row);
      tbody.appendChild(row);
    });
  }

  function getLinkOptions(type) {
    switch (type) {
      case "control":
        return appData.controls.map(c => ({ value: c.id, label: c.reference }));
      case "risk":
        return appData.risks.map(r => ({ value: r.id, label: r.title }));
      case "nc":
        return appData.nonconformities.map(nc => ({ value: nc.id, label: nc.title }));
      case "audit":
        return appData.audits.map(a => ({ value: a.id, label: a.title }));
      case "threat":
        return appData.threats.map(t => ({ value: t.id, label: t.title }));
      case "continuous":
        return [
          { value: "continuous_improvement", label: "Am\xE9lioration continue" }
        ];
      default:
        return [];
    }
  }
  function getActionFilterOptions(type) {
    switch (type) {
      case "risk":
        return appData.risks.map(r => ({ value: r.id, label: r.title }));
      case "nc":
        return appData.nonconformities.map(nc => ({ value: nc.id, label: nc.title }));
      case "audit":
        return appData.audits.map(a => ({ value: a.id, label: a.title }));
      case "threat":
        return appData.threats.map(t => ({ value: t.id, label: t.title }));
      case "owner": {
        const names = new Set();
        const addName = (value) => {
          const cleaned = typeof value === "string" ? value.trim() : "";
          if (cleaned) names.add(cleaned);
        };
        appData.actions.forEach((action) => addName(action.assignedTo));
        appData.stakeholders.forEach((st) => addName(st.name));
        const options = Array.from(names)
          .sort((a, b) => a.localeCompare(b, void 0, { sensitivity: "base" }))
          .map((name) => ({ value: name, label: name }));
        if (appData.actions.some((action) => !action.assignedTo)) {
          options.unshift({ value: "unassigned", label: "Non assigné" });
        }
        return options;
      }
      case "continuous":
        return [
          { value: "continuous_improvement", label: "Am\xE9lioration continue" }
        ];
      default:
        return [];
    }
  }
  function updateActionsFilterOptions() {
    const typeSelect = document.getElementById("actionsFilterType");
    const valueSelect = document.getElementById("actionsFilterValue");
    if (!typeSelect || !valueSelect) return;
    const type = typeSelect.value || "all";
    const options = getActionFilterOptions(type);
    const current = valueSelect.value;
    valueSelect.innerHTML = "";
    const allOption = document.createElement("option");
    allOption.value = "all";
    allOption.textContent = "Tous";
    valueSelect.appendChild(allOption);
    options.forEach((opt) => {
      const option = document.createElement("option");
      option.value = opt.value;
      option.textContent = opt.label;
      valueSelect.appendChild(option);
    });
    const hasCurrent = options.some((opt) => opt.value === current);
    valueSelect.value = hasCurrent ? current : "all";
    valueSelect.disabled = type === "all" || options.length === 0;
  }
  function matchesActionFilter(action, filterType, filterValue) {
    if (!action || !filterType || !filterValue || filterType === "all" || filterValue === "all") return true;
    if (filterType === "risk") {
      return action.linkType === "risk" && getActionLinkIds(action).includes(filterValue);
    }
    if (filterType === "nc") {
      if (action.linkType === "nc" && getActionLinkIds(action).includes(filterValue)) return true;
      return (action.ncIds || []).includes(filterValue);
    }
    if (filterType === "audit") {
      if (action.linkType === "audit" && getActionLinkIds(action).includes(filterValue)) return true;
      const ncIds = [];
      if (action.linkType === "nc") ncIds.push(...getActionLinkIds(action));
      if (Array.isArray(action.ncIds)) ncIds.push(...action.ncIds);
      if (!ncIds.length) return false;
      return ncIds.some((ncId) => {
        const nc = appData.nonconformities.find((n) => n.id === ncId);
        return nc && nc.auditId === filterValue;
      });
    }
    if (filterType === "threat") {
      return action.linkType === "threat" && getActionLinkIds(action).includes(filterValue);
    }
    if (filterType === "owner") {
      const assignedTo = (action.assignedTo || "").trim();
      if (filterValue === "unassigned") return !assignedTo;
      return assignedTo === filterValue;
    }
    if (filterType === "continuous") {
      return action.linkType === "continuous" && getActionLinkIds(action).includes(filterValue);
    }
    return true;
  }
  function normalizeActionLinkValue(type, value) {
    if (type === "risk") {
      if (Array.isArray(value)) return value.filter(Boolean);
      if (value) return [value];
      return [];
    }
    if (Array.isArray(value)) return value[0] || "";
    return value || "";
  }
  function getActionLinkIds(action) {
    if (!action) return [];
    const type = action.linkType || (action.controlId ? "control" : "");
    if (type === "risk") return normalizeActionLinkValue("risk", action.linkId);
    const id = action.linkId || action.controlId;
    return id ? [id] : [];
  }
  function actionLinksToRisk(action, riskId) {
    if (!action || action.linkType !== "risk") return false;
    return getActionLinkIds(action).includes(riskId);
  }
  function normalizeActionProgressStatus(rawProgress, rawStatus) {
    const parsed = parseInt(rawProgress, 10);
    const progress = Number.isFinite(parsed) ? Math.max(0, Math.min(100, parsed)) : 0;
    let status = rawStatus || "planifie";
    if (progress >= 100 && status !== "annule") status = "termine";
    if (status === "termine" && progress < 100) return { progress: 100, status };
    return { progress, status };
  }
  function bindActionStatusProgressSync() {
    const progressInput = document.getElementById("modal-input-progress");
    const statusSelect = document.getElementById("modal-input-status");
    if (!progressInput || !statusSelect) return;
    const getProgress = () => {
      const raw = parseInt(progressInput.value, 10);
      if (!Number.isFinite(raw)) return null;
      const value = Math.max(0, Math.min(100, raw));
      if (String(value) !== String(progressInput.value)) progressInput.value = String(value);
      return value;
    };
    const syncFromProgress = () => {
      const value = getProgress();
      if (value === null) return;
      if (value >= 100 && statusSelect.value !== "termine" && statusSelect.value !== "annule") {
        statusSelect.value = "termine";
      }
    };
    const syncFromStatus = () => {
      if (statusSelect.value === "termine") {
        progressInput.value = "100";
      }
    };
    progressInput.addEventListener("input", syncFromProgress);
    progressInput.addEventListener("change", syncFromProgress);
    statusSelect.addEventListener("change", syncFromStatus);
    syncFromStatus();
    syncFromProgress();
  }
  function editAction(actionId) {
    const action = appData.actions.find((a) => a.id === actionId);
    if (!action) return;
    const linkType = action.linkType || "control";
    const linkOptions = getLinkOptions(linkType);
    const currentLinkId = action.linkId || action.controlId || "";
    const normalizedValue = normalizeActionLinkValue(linkType, currentLinkId);
    const linkValue = linkType === "risk"
      ? normalizedValue.filter((id) => linkOptions.some((opt) => opt.value === id))
      : (linkOptions.some((opt) => opt.value === normalizedValue)
        ? normalizedValue
        : (linkOptions[0]?.value || ""));
    const criticalAssetIds = Array.isArray(action.criticalAssetIds) ? action.criticalAssetIds : [];
    const criticalAssetOptions = getCriticalAssetOptions(criticalAssetIds);
    openModal("Modifier une action", [
      { name: "title", label: "Titre de l'action", value: action.title },
      { name: "linkType", label: "Type de lien", type: "select", value: linkType, options: [
          { value: "control", label: "Contr\xF4le" },
          { value: "risk", label: "Risque" },
          { value: "nc", label: "Non-conformit\xE9" },
          { value: "audit", label: "Audit" },
          { value: "threat", label: "Menace" },
          { value: "continuous", label: "Am\xE9lioration continue" }
        ] },
      { name: "linkId", label: "\xC9l\xE9ment li\xE9", type: linkType === "risk" ? "multiselect" : "select", value: linkValue, options: linkOptions },
      { name: "criticalAssetIds", label: "Actifs critiques", type: "multiselect", value: criticalAssetIds, options: criticalAssetOptions },
      {
        name: "assignedTo",
        label: "Responsable",
        type: "select",
        value: action.assignedTo,
        options: appData.stakeholders.map((s) => ({ value: s.name, label: s.name }))
      },
      { name: "priority", label: "Priorit\xE9", type: "select", value: action.priority, options: ACTION_PRIORITY_OPTIONS },
      { name: "dueDate", label: "\xC9ch\xE9ance", type: "date", value: action.dueDate },
      { name: "status", label: "Statut", type: "select", value: action.status, options: ACTION_STATUS_OPTIONS },
      { name: "progress", label: "Progression (%)", value: action.progress, type: "number" },
      { name: "existingComments", label: "Commentaires existants", type: "textarea", value: (action.comments || []).map(c => `[${new Date(c.date).toLocaleString()}] ${c.text}`).join("\n"), readOnly: true },
      { name: "newComment", label: "Nouveau commentaire", type: "textarea" }
    ], (data) => {
      const resolvedLinkType = data.linkType || "control";
      const resolvedLinkValue = normalizeActionLinkValue(resolvedLinkType, data.linkId);
      const normalized = normalizeActionProgressStatus(data.progress, data.status);
      action.title = data.title;
      action.linkType = resolvedLinkType;
      action.linkId = resolvedLinkType === "risk" ? resolvedLinkValue : (resolvedLinkValue || "");
      action.controlId = resolvedLinkType === "control" ? (resolvedLinkValue || "") : "";
      action.progress = normalized.progress;
      action.priority = data.priority;
      action.assignedTo = data.assignedTo;
      action.dueDate = data.dueDate;
      action.status = normalized.status;
      action.criticalAssetIds = Array.isArray(data.criticalAssetIds) ? data.criticalAssetIds : [];
      if (data.newComment) {
        if (!Array.isArray(action.comments)) action.comments = [];
        action.comments.push({ text: data.newComment, date: new Date().toISOString() });
      }
      saveData();
      loadActionsTable();
      updateDashboard();
      loadNonConformitiesTable();
      updateActionsChart();
    }, () => deleteAction(actionId));
    const typeSelect = document.getElementById("modal-input-linkType");
    const idSelect = document.getElementById("modal-input-linkId");
    if (typeSelect && idSelect) {
      const applyLinkOptions = (selected = []) => {
        const options = getLinkOptions(typeSelect.value);
        idSelect.innerHTML = "";
        idSelect.multiple = typeSelect.value === "risk";
        if (idSelect.multiple) {
          idSelect.size = Math.min(6, Math.max(3, options.length));
        } else {
          idSelect.removeAttribute("size");
        }
        options.forEach(opt => {
          const o = document.createElement("option");
          o.value = opt.value;
          o.textContent = opt.label;
          if (Array.isArray(selected) && selected.includes(opt.value)) o.selected = true;
          idSelect.appendChild(o);
        });
        if (!idSelect.multiple && options.length && !selected.length) {
          idSelect.value = options[0].value;
        }
      };
      applyLinkOptions(Array.isArray(linkValue) ? linkValue : [linkValue].filter(Boolean));
      typeSelect.addEventListener("change", () => applyLinkOptions());
    }
    bindActionStatusProgressSync();
  }
  function deleteAction(actionId) {
    if (!confirm("Supprimer cette action ?")) return;
    appData.actions = appData.actions.filter((a) => a.id !== actionId);
    saveData();
    loadActionsTable();
    updateDashboard();
    loadNonConformitiesTable();
    updateActionsChart();
  }
  function addAction(defaultNcIds = []) {
    if (!Array.isArray(defaultNcIds)) defaultNcIds = [];
    const criticalAssetOptions = getCriticalAssetOptions();
    openModal("Ajouter une action", [
      { name: "title", label: "Titre de l'action" },
      { name: "linkType", label: "Type de lien", type: "select", value: "control", options: [
          { value: "control", label: "Contr\xF4le" },
          { value: "risk", label: "Risque" },
          { value: "nc", label: "Non-conformit\xE9" },
          { value: "audit", label: "Audit" },
          { value: "threat", label: "Menace" },
          { value: "continuous", label: "Am\xE9lioration continue" }
        ] },
      { name: "linkId", label: "\xC9l\xE9ment li\xE9", type: "select", options: getLinkOptions("control") },
      { name: "criticalAssetIds", label: "Actifs critiques", type: "multiselect", value: [], options: criticalAssetOptions },
      {
        name: "assignedTo",
        label: "Responsable",
        type: "select",
        options: appData.stakeholders.map((s) => ({ value: s.name, label: s.name }))
      },
      { name: "priority", label: "Priorit\xE9", type: "select", value: "medium", options: ACTION_PRIORITY_OPTIONS },
      { name: "dueDate", label: "\xC9ch\xE9ance", type: "date" },
      { name: "status", label: "Statut", type: "select", value: "planifie", options: ACTION_STATUS_OPTIONS },
      { name: "progress", label: "Progression (%)", type: "number", value: "0" },
      { name: "comment", label: "Commentaire", type: "textarea" }
    ], (data) => {
      const resolvedLinkType = data.linkType || "control";
      const resolvedLinkValue = normalizeActionLinkValue(resolvedLinkType, data.linkId);
      const normalized = normalizeActionProgressStatus(data.progress, data.status);
      if (!data.title) return false;
      const id = "action-" + appData.nextActionId;
      appData.nextActionId++;
      const ncIds = Array.isArray(defaultNcIds) ? defaultNcIds.slice() : [];
      const comments = [];
      if (data.comment) {
        comments.push({ text: data.comment, date: new Date().toISOString() });
      }
      appData.actions.push({
        id,
        linkType: resolvedLinkType,
        linkId: resolvedLinkType === "risk" ? resolvedLinkValue : (resolvedLinkValue || ""),
        controlId: resolvedLinkType === "control" ? (resolvedLinkValue || "") : "",
        ncIds,
        title: data.title,
        description: "",
        priority: data.priority || "medium",
        progress: normalized.progress,
        dueDate: data.dueDate || "",
        assignedTo: data.assignedTo || "",
        status: normalized.status,
        criticalAssetIds: Array.isArray(data.criticalAssetIds) ? data.criticalAssetIds : [],
        comments
      });
      saveData();
      loadActionsTable();
      updateDashboard();
      loadNonConformitiesTable();
      updateActionsChart();
    });
    const typeSelect = document.getElementById("modal-input-linkType");
    const idSelect = document.getElementById("modal-input-linkId");
    if (typeSelect && idSelect) {
      const applyLinkOptions = () => {
        const options = getLinkOptions(typeSelect.value);
        idSelect.innerHTML = "";
        idSelect.multiple = typeSelect.value === "risk";
        if (idSelect.multiple) {
          idSelect.size = Math.min(6, Math.max(3, options.length));
        } else {
          idSelect.removeAttribute("size");
        }
        options.forEach(opt => {
          const o = document.createElement("option");
          o.value = opt.value;
          o.textContent = opt.label;
          idSelect.appendChild(o);
        });
        if (!idSelect.multiple && options.length) {
          idSelect.value = options[0].value;
        }
      };
      typeSelect.addEventListener("change", applyLinkOptions);
    }
    bindActionStatusProgressSync();
  }

  function manageActionComments(actionId) {
    if (!ensureEditMode("gérer les commentaires de l'action")) return;
    const action = appData.actions.find((a) => a.id === actionId);
    if (!action) return;
    if (!Array.isArray(action.comments)) action.comments = [];
    let continueLoop = true;
    while (continueLoop) {
      const list = action.comments.map((c, i) => `${i + 1}. [${new Date(c.date).toLocaleString()}] ${c.text}`).join("\n");
      const choice = prompt(`${list || "Aucun commentaire"}\n\n(A)jouter, (E)diter, (S)upprimer, (Q)uitter`);
      if (!choice) break;
      const ch = choice.toLowerCase();
      if (ch === "a") {
        if (action.comments.length >= 10) {
          alert("Maximum 10 commentaires atteint");
          continue;
        }
        const text = prompt("Nouveau commentaire:");
        if (text) {
          action.comments.push({ text, date: new Date().toISOString() });
          saveData();
        }
      } else if (ch === "e") {
        const idx = parseInt(prompt("Num\xE9ro du commentaire \xE0 \xE9diter:"), 10) - 1;
        if (idx >= 0 && idx < action.comments.length) {
          const text = prompt("Modifier le commentaire:", action.comments[idx].text);
          if (text !== null) {
            action.comments[idx].text = text;
            action.comments[idx].date = new Date().toISOString();
            saveData();
          }
        }
      } else if (ch === "s") {
        const idx = parseInt(prompt("Num\xE9ro du commentaire \xE0 supprimer:"), 10) - 1;
        if (idx >= 0 && idx < action.comments.length) {
          if (confirm("Supprimer ce commentaire ?")) {
            action.comments.splice(idx, 1);
            saveData();
          }
        }
      } else if (ch === "q") {
        continueLoop = false;
      }
    }
    loadActionsTable();
  }
  window.loadActionsTable = loadActionsTable;
  window.editAction = editAction;
  window.deleteAction = deleteAction;
  window.manageActionComments = manageActionComments;
  window.addAction = addAction;

  // modules/nis2-program.js
  const NIS2_STATUS_META = {
    todo: { label: "À réaliser", className: "status-badge status-badge--non_implemente" },
    in_progress: { label: "En cours", className: "status-badge status-badge--en_cours" },
    suspended: { label: "Suspendu", className: "status-badge status-badge--en_cours" },
    done: { label: "Réalisé", className: "status-badge status-badge--verifie" }
  };
  const NIS2_URGENCY_LABELS = { high: "Urgence haute", medium: "Priorité moyenne", standard: "Priorité standard" };
  const NIS2_ACTION_TYPE_OPTIONS = [
    { value: "service", label: "Service" },
    { value: "projet", label: "Projet" },
    { value: "initiative", label: "Initiative" },
    { value: "programme", label: "Programme" }
  ];
  const NIS2_ACTION_TYPE_LABELS = NIS2_ACTION_TYPE_OPTIONS.reduce((acc, opt) => {
    acc[opt.value] = opt.label;
    return acc;
  }, {});
  const NIS2_DOMAIN_ICONS = {
    [NORMALIZE("Gouvernance & responsabilité")]: "fa-user-shield",
    [NORMALIZE("Gestion des risques (SMSI)")]: "fa-scale-balanced",
    [NORMALIZE("Gestion des incidents & notification")]: "fa-bell",
    [NORMALIZE("Gestion des incidents et notification")]: "fa-bell",
    [NORMALIZE("Continuité d’activité & gestion de crise (BCP/DRP)")]: "fa-life-ring",
    [NORMALIZE("Sécurité de la chaîne d’approvisionnement")]: "fa-truck-fast",
    [NORMALIZE("Sécurité dans l’acquisition, le développement & la maintenance")]: "fa-code-branch",
    [NORMALIZE("Traçabilité et auditabilité")]: "fa-magnifying-glass-chart",
    [NORMALIZE("Gestion des vulnérabilités")]: "fa-bug-slash",
    [NORMALIZE("Sensibilisation & formation")]: "fa-chalkboard-user",
    [NORMALIZE("Cryptographie")]: "fa-lock",
    [NORMALIZE("Sécurité physique")]: "fa-building-lock",
    [NORMALIZE("Gestion des accès au SI")]: "fa-key"
  };
  let nis2PlanViewMode = "detail";
  let nis2PlanShowOpenOnly = false;
  const getSortedNis2Phases = () => {
    const phases = Array.isArray(appData?.nis2Plan?.phases) ? [...appData.nis2Plan.phases] : [...DEFAULT_NIS2_PHASES];
    return phases.sort((a, b) => (a.order || 0) - (b.order || 0));
  };
  const getNis2PhaseLabel = (phaseId) => {
    if (!phaseId) return "Hors phase";
    const found = getSortedNis2Phases().find((p) => p.id === phaseId);
    return found ? found.label : `Phase ${phaseId}`;
  };
  const ensureNis2Phases = () => {
    if (!appData.nis2Plan) appData.nis2Plan = {};
    if (!Array.isArray(appData.nis2Plan.phases) || appData.nis2Plan.phases.length === 0) {
      appData.nis2Plan.phases = [...DEFAULT_NIS2_PHASES];
    }
    const map = new Map();
    appData.nis2Plan.phases.forEach((p) => {
      const id = p.id || generateId("phase");
      if (!map.has(id)) map.set(id, { id, label: p.label || id, order: typeof p.order === "number" ? p.order : 99, fallback: !!p.fallback });
    });
    const hasFallback = [...map.values()].some((p) => p.fallback);
    if (!hasFallback) {
      map.set("phase-unplanned", { id: "phase-unplanned", label: "Hors phase", order: 99, fallback: true });
    }
    appData.nis2Plan.phases = [...map.values()].sort((a, b) => (a.order || 0) - (b.order || 0));
    return appData.nis2Plan.phases;
  };
  function normalizePhaseId(value) {
    const phases = ensureNis2Phases();
    const fallback = phases.find((p) => p.fallback) || phases[phases.length - 1];
    const str = value === undefined || value === null ? "" : value.toString();
    if (phases.some((p) => p.id === str)) return str;
    const asNum = parseInt(str, 10);
    if (!isNaN(asNum)) {
      const numericMatch = phases.find((p) => parseInt(p.order, 10) === asNum || p.id === `phase-${asNum}`);
      if (numericMatch) return numericMatch.id;
    }
    return fallback?.id || "phase-unplanned";
  }
  function parseBudgetValue(value) {
    if (typeof value === "number") return value;
    if (!value || typeof value !== "string") return NaN;
    const sanitized = value.replace(/\s/g, "").replace(",", ".").replace(/[^0-9.\-]/g, "");
    const parsed = parseFloat(sanitized);
    return isNaN(parsed) ? NaN : parsed;
  }
  const formatCurrency = (val) => {
    if (!Number.isFinite(val)) return "—";
    const optsStandard = { maximumFractionDigits: 0 };
    const optsCompact = { notation: "compact", maximumFractionDigits: 1 };
    try {
      if (Math.abs(val) >= 1_000_000) {
        return new Intl.NumberFormat("fr-FR", optsCompact).format(val) + " €";
      }
      return new Intl.NumberFormat("fr-FR", optsStandard).format(val) + " €";
    } catch (e) {
      return val.toLocaleString("fr-FR") + " €";
    }
  };
  const formatBudgetValue = (val) => {
    const num = parseBudgetValue(val);
    return Number.isFinite(num) ? formatCurrency(num) : "—";
  };
  const getActionBudgetTotal = (action = {}) => {
    const yearlyKeys = ["budget2026", "budget2027", "budget2028"];
    let total = 0;
    let hasValue = false;
    yearlyKeys.forEach((k) => {
      const num = parseBudgetValue(action[k]);
      if (Number.isFinite(num)) {
        total += num;
        hasValue = true;
      }
    });
    if (!hasValue) {
      const legacy = parseBudgetValue(action.budget);
      if (Number.isFinite(legacy)) return legacy;
      return NaN;
    }
    return total;
  };
  function ensureNis2PlanToolbar() {
    const container = document.getElementById("nis2PlanContainer");
    if (!container) return;
    const infoBox = container.previousElementSibling?.classList?.contains("info-box") ? container.previousElementSibling : null;
    const anchor = infoBox || container;
    let toolbar = document.getElementById("nis2PlanToolbar");
    if (!toolbar) {
      toolbar = document.createElement("div");
      toolbar.id = "nis2PlanToolbar";
      toolbar.className = "nis2-plan-toolbar";
      toolbar.innerHTML = `
        <div class="nis2-plan-toolbar__group">
          <button class="chip-btn" type="button" data-view-mode="detail">Vue détaillée</button>
          <button class="chip-btn" type="button" data-view-mode="executive">Vue exécutive</button>
        </div>
        <div class="nis2-plan-toolbar__group">
          <button class="chip-btn" type="button" id="nis2PlanFocusBtn" data-focus="open">Mettre en avant les actions ouvertes</button>
        </div>
      `;
      anchor.insertAdjacentElement("afterend", toolbar);
    }
    if (!toolbar.dataset.bound) {
      toolbar.dataset.bound = "true";
      toolbar.addEventListener("click", (event) => {
        const viewBtn = event.target.closest("[data-view-mode]");
        if (viewBtn) {
          nis2PlanViewMode = viewBtn.dataset.viewMode || "detail";
          ensureNis2PlanToolbar();
          return renderNis2Plan();
        }
        const focusBtn = event.target.closest("#nis2PlanFocusBtn");
        if (focusBtn) {
          nis2PlanShowOpenOnly = !nis2PlanShowOpenOnly;
          ensureNis2PlanToolbar();
          return renderNis2Plan();
        }
      });
    }
    toolbar.querySelectorAll("[data-view-mode]").forEach((btn) => {
      const isActive = btn.dataset.viewMode === nis2PlanViewMode;
      btn.classList.toggle("is-active", isActive);
      btn.setAttribute("aria-pressed", isActive ? "true" : "false");
    });
    const focusBtn = toolbar.querySelector("#nis2PlanFocusBtn");
    if (focusBtn) {
      focusBtn.classList.toggle("is-active", !!nis2PlanShowOpenOnly);
      focusBtn.setAttribute("aria-pressed", nis2PlanShowOpenOnly ? "true" : "false");
    }
  }
  function refreshNis2PhaseFilter() {
    const select = document.getElementById("nis2PhaseFilter");
    if (!select) return;
    const current = select.value || "all";
    const phases = getSortedNis2Phases();
    select.innerHTML = `<option value="all">Toutes les phases</option>` + phases.map((p) => `<option value="${p.id}">${escapeHtml(p.label)}</option>`).join("");
    const exists = phases.some((p) => p.id === current);
    select.value = exists ? current : "all";
  }
  function renderNis2PhasesPanel() {
    if (isReadOnlyMode && typeof isReadOnlyMode === "function" && isReadOnlyMode()) return;
    ensureNis2Phases();
    const container = document.getElementById("nis2PlanContainer");
    if (!container) return;
    const anchor = document.getElementById("nis2PlanToolbar") || container;
    let panel = document.getElementById("nis2PhasesPanel");
    if (!panel) {
      panel = document.createElement("div");
      panel.id = "nis2PhasesPanel";
      panel.className = "nis2-phases-panel";
      anchor.insertAdjacentElement("afterend", panel);
    }
    const phases = getSortedNis2Phases();
    const chips = phases.map((p) => `
      <div class="phase-chip" data-phase-id="${p.id}">
        <span class="phase-chip__label">${escapeHtml(p.label)}</span>
        <span class="phase-chip__order">#${p.order || 0}</span>
        <div class="phase-chip__actions">
          <button class="icon-btn" type="button" data-phase-action="edit" data-phase-id="${p.id}" title="Modifier la phase"><i class="fas fa-pen"></i></button>
          ${p.fallback ? "" : `<button class="icon-btn icon-btn--danger" type="button" data-phase-action="delete" data-phase-id="${p.id}" title="Supprimer la phase"><i class="fas fa-trash"></i></button>`}
        </div>
      </div>
    `).join("");
    panel.innerHTML = `
      <div class="nis2-phases-panel__header">
        <div class="nis2-phases-panel__title"><i class="fas fa-timeline"></i> Phases du plan</div>
        <button class="btn btn--outline btn--sm" type="button" data-phase-action="add"><i class="fas fa-plus"></i> Ajouter une phase</button>
      </div>
      <div class="nis2-phases-panel__list">${chips}</div>
    `;
    if (!panel.dataset.bound) {
      panel.dataset.bound = "true";
      panel.addEventListener("click", (event) => {
        const addBtn = event.target.closest("[data-phase-action=\"add\"]");
        if (addBtn) return openNis2PhaseModal();
        const editBtn = event.target.closest("[data-phase-action=\"edit\"]");
        if (editBtn) return openNis2PhaseModal(editBtn.dataset.phaseId);
        const delBtn = event.target.closest("[data-phase-action=\"delete\"]");
        if (delBtn) return deleteNis2Phase(delBtn.dataset.phaseId);
      });
    }
  }
  function renderSocleGrid(containerId, limit = null) {
    const container = document.getElementById(containerId);
    if (!container) return;
    const pillars = normalizeSoclePillars(appData.soclePillars);
    appData.soclePillars = pillars;
    const items = typeof limit === "number" ? pillars.slice(0, limit) : pillars;
    container.innerHTML = "";
    items.forEach((pillar, idx) => {
      const pillarIndex = idx;
      const card = document.createElement("div");
      card.className = `socle-card socle-card--pillar${pillarIndex + 1}`;
      const listItems = (pillar.items || []).map((item) => `<li>${escapeHtml(item)}</li>`).join("");
      card.innerHTML = `
        <div class="socle-card__header">
          <div class="socle-card__icon"><i class="fas ${escapeHtml(pillar.icon || "")}"></i></div>
          <div class="socle-card__meta">
            <p class="socle-card__eyebrow">Pilier ${pillarIndex + 1}</p>
            <h3 class="socle-card__title">${escapeHtml(pillar.title)}</h3>
            <p class="socle-card__subtitle">${escapeHtml(pillar.subtitle)}</p>
          </div>
          <button class="icon-btn socle-card__edit" type="button" data-readonly-lock="true" data-pillar-index="${pillarIndex}" title="Modifier le pilier">
            <i class="fas fa-pen"></i>
          </button>
        </div>
        <div class="socle-card__body">
          <h4>Éléments clefs</h4>
          <ul>${listItems}</ul>
          <div class="socle-card__footer"><i class="fas fa-user-tie"></i> ${escapeHtml(pillar.owner)}</div>
        </div>
      `;
      container.appendChild(card);
    });
    container.querySelectorAll(".socle-card__edit").forEach((btn) => {
      btn.addEventListener("click", () => {
        const index = parseInt(btn.dataset.pillarIndex, 10);
        if (Number.isInteger(index)) openSoclePillarEditor(index);
      });
    });
  }
  function renderSocleGrids() {
    renderSocleGrid("nis2SocleInline", 3);
    renderSocleGrid("nis2SocleFull", 5);
    refreshReadOnlyLocks();
  }
  function loadNis2Program() {
    ensureNis2Plan();
    renderNis2ProgramRadar();
    renderNis2StatusList();
    renderNis2ActionsChart();
    renderNis2ActionsByDomainChart();
    renderNis2Plan();
    renderSocleGrids();
  }
  function openSoclePillarEditor(index) {
    if (!ensureEditMode("modifier un pilier")) {
      refreshReadOnlyLocks();
      return;
    }
    const pillars = normalizeSoclePillars(appData.soclePillars);
    appData.soclePillars = pillars;
    const pillar = pillars[index];
    if (!pillar) return;
    openModal(`Modifier Pilier ${index + 1}`, [
      { name: "title", label: "Titre", value: pillar.title },
      { name: "subtitle", label: "Sous-titre", value: pillar.subtitle },
      { name: "items", label: "Éléments clefs (un par ligne)", type: "textarea", value: (pillar.items || []).join("\n") },
      { name: "owner", label: "Chef de projet", value: pillar.owner }
    ], (data) => {
      if (!data.title) return false;
      pillar.title = data.title;
      pillar.subtitle = data.subtitle || "";
      pillar.items = data.items
        ? data.items.split(/\r?\n/).map((item) => item.trim()).filter(Boolean)
        : [];
      pillar.owner = data.owner || "";
      saveData();
      renderSocleGrids();
    });
  }
  function renderNis2ProgramRadar() {
    const canvas = document.getElementById("nis2ProgramRadar");
    if (!canvas) return;
    if (charts.nis2ProgramRadar) charts.nis2ProgramRadar.destroy();
    // Recalcule un radar cohérent (mêmes couleurs et échelle que le dashboard)
    const { textColor, textSecondary, borderColor, secondary, warning } = getThemeVars();
    const labels = NIS2_CATEGORIES.map((k) => {
      const full = CATEGORY_LABELS[k] || k;
      return full.split(" (")[0].trim();
    });
    const averages = appData.targetMaturity?.nis2 || {};
    const categoryData = NIS2_CATEGORIES.map((k) => Number(averages[k]) || 0);
    const globalTarget = Number(appData.targetMaturity?.nis2Target) || 0;
    const globalData = NIS2_CATEGORIES.map(() => globalTarget);
    const renderLegend = typeof renderRadarLegend === "function" ? renderRadarLegend : () => {};

    charts.nis2ProgramRadar = new Chart(canvas, {
      type: "radar",
      data: {
        labels,
        datasets: [
          {
            label: "Cible globale",
            data: globalData,
            backgroundColor: withAlpha(warning, 0.08),
            borderColor: withAlpha(warning, 0.55),
            borderWidth: 3,
            pointRadius: 2,
            borderDash: [4, 4],
            fill: true
          },
          {
            label: "Maturité moyenne",
            data: categoryData,
            backgroundColor: withAlpha(secondary, 0.2),
            borderColor: withAlpha(secondary, 1),
            borderWidth: 3,
            pointBackgroundColor: withAlpha(secondary, 1),
            pointRadius: 3,
            pointHoverRadius: 5,
            fill: true
          }
        ]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        scales: {
          r: {
            beginAtZero: true,
            suggestedMin: 0,
            suggestedMax: 5,
            ticks: {
              stepSize: 1,
              color: textColor,
              backdropColor: "transparent",
              font: { size: 12 }
            },
            grid: { color: withAlpha(borderColor, 0.7) },
            angleLines: { color: withAlpha(borderColor, 0.7) },
            pointLabels: {
              display: true,
              color: textSecondary || borderColor,
              font: { size: 14, weight: "400" },
              callback: (_label, index) => `${index + 1}`
            }
          }
        },
        plugins: { legend: { display: false } }
      }
    });
    renderLegend("nis2ProgramRadarLegend", labels);
    attachLegendTooltip("nis2ProgramRadarLegend");
  }
  function renderNis2ActionsChart() {
    const canvas = document.getElementById("nis2ActionsChart");
    if (!canvas) return;
    if (charts.nis2ActionsChart) charts.nis2ActionsChart.destroy();
    ensureNis2Plan();
    const themeVars = getThemeVars();
    const actions = (appData.nis2Plan.domains || []).flatMap((d) => d.actions || []);
    const statuses = ["done", "in_progress", "suspended", "todo"];
    const labels = ["Réalisées", "En cours", "Suspendues", "À réaliser"];
    const palette = [
      withAlpha(themeVars.success, 0.85),
      withAlpha(themeVars.warning, 0.85),
      withAlpha(themeVars.secondary, 0.85),
      withAlpha(themeVars.primary, 0.85)
    ];
    const borderPalette = [
      withAlpha(themeVars.success, 1),
      withAlpha(themeVars.warning, 1),
      withAlpha(themeVars.secondary, 1),
      withAlpha(themeVars.primary, 1)
    ];
    const data = statuses.map((s) => actions.filter((a) => (a.status || "todo") === s).length);
    const total = data.reduce((a, b) => a + b, 0);
    charts.nis2ActionsChart = new Chart(canvas, {
      type: "doughnut",
      data: {
        labels,
        datasets: [
          {
            data,
            backgroundColor: palette,
            borderColor: borderPalette,
            borderWidth: 2,
            hoverOffset: 10
          }
        ]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        cutout: "58%",
        plugins: {
          legend: {
            position: "bottom",
            labels: { usePointStyle: true, padding: 16 }
          },
          tooltip: {
            callbacks: {
              label: (context) => {
                const value = context.raw || 0;
                const suffix = value > 1 ? "actions" : "action";
                const percent = total ? Math.round(value / total * 100) : 0;
                return `${context.label}: ${value} ${suffix} (${percent}%)`;
              }
            }
          }
        }
      }
    });
    renderNis2ActionsLegend(labels, data, borderPalette);
  }
  function renderNis2ActionsByDomainChart() {
    const canvas = document.getElementById("nis2ActionsDomainChart");
    if (!canvas) return;
    if (charts.nis2ActionsByDomainChart) charts.nis2ActionsByDomainChart.destroy();
    ensureNis2Plan();
    const themeVars = getThemeVars();
    const domains = appData.nis2Plan.domains || [];
    const labels = domains.map((domain, index) => {
      const title = (domain.title || "").trim();
      return title || `Domaine ${index + 1}`;
    });
    const data = domains.map((domain) => (domain.actions || []).length);
    const total = data.reduce((sum, value) => sum + value, 0);
    const palette = [
      themeVars.primary,
      themeVars.secondary,
      themeVars.warning,
      themeVars.success,
      themeVars.info,
      themeVars.danger
    ];
    const colors = labels.map((_, idx) => withAlpha(palette[idx % palette.length], 0.8));
    const borderColors = labels.map((_, idx) => withAlpha(palette[idx % palette.length], 1));
    charts.nis2ActionsByDomainChart = new Chart(canvas, {
      type: "doughnut",
      data: {
        labels,
        datasets: [
          {
            data,
            backgroundColor: colors,
            borderColor: borderColors,
            borderWidth: 2,
            hoverOffset: 10
          }
        ]
      },
      options: {
        responsive: true,
        maintainAspectRatio: false,
        cutout: "58%",
        plugins: {
          legend: {
            display: labels.length <= 6,
            position: "bottom",
            labels: { usePointStyle: true, padding: 14 }
          },
          tooltip: {
            callbacks: {
              label: (context) => {
                const value = context.raw || 0;
                const suffix = value > 1 ? "actions" : "action";
                const percent = total ? Math.round(value / total * 100) : 0;
                return `${context.label}: ${value} ${suffix} (${percent}%)`;
              }
            }
          }
        }
      }
    });
  }
  function renderNis2ActionsLegend(labels, data, colors) {
    const legend = document.getElementById("nis2ActionsLegend");
    if (!legend) return;
    legend.innerHTML = "";
    legend.style.display = "none";
  }
  function renderNis2StatusList() {
    const list = document.getElementById("nis2StatusList");
    if (!list) return;
    list.innerHTML = "";
    const shortenNis2StatusText = (text) => {
      const normalized = NORMALIZE(text);
      const map = {
        "entite essentielle nis2 enregistree": "Entité essentielle NIS2 enregistrée",
        "analyse de maturite nis2 realisee": "Analyse NIS2 réalisée",
        "systeme de management de la securite implemente (smsi)": "SMSI implémenté",
        "obtention d'une certification iso27001 (perimetre iam)": "Certification ISO27001 (IAM)",
        "atteindre un niveau de maturite superieur a 3": "Maturité >3 atteinte",
        "obtenir le label cyberfun du ccb belgique": "Label CyberFun (CCB)"
      };
      if (map[normalized]) return map[normalized];
      if (normalized.includes("enregistree comme entite essentielle nis2 au ccb")) {
        return "Entité essentielle NIS2 enregistrée";
      }
      return text.length > 70 ? text.slice(0, 67) + "…" : text;
    };
    (appData.nis2Plan.statuses || []).forEach((item, idx) => {
      const normalizedText = NORMALIZE(item.text || "");
      if (normalizedText.includes("trajectoire") && normalizedText.includes("nis2 2027")) return;
      const displayText = shortenNis2StatusText(item.text || `Statut ${idx + 1}`);
      const statusKey = item.status || "todo";
      const dotClass = statusKey === "done" ? "nis2-status-dot--done" : statusKey === "in_progress" ? "nis2-status-dot--progress" : "nis2-status-dot--todo";
      const meta = NIS2_STATUS_META[statusKey] || NIS2_STATUS_META.todo;
      const iconClass = statusKey === "done" ? "fa-circle-check" : statusKey === "in_progress" ? "fa-hourglass-half" : "fa-flag";
      const isCritical = normalizedText.includes("cyberfun");
      const li = document.createElement("li");
      li.className = `nis2-status-item nis2-status-item--${statusKey}${isCritical ? " nis2-status-item--critical" : ""}`;
      li.innerHTML = `
        <div class="nis2-status-icon"><i class="fas ${iconClass}"></i></div>
        <div class="nis2-status-content">
          <div class="nis2-status-chip">
            <span class="nis2-status-dot ${dotClass}" aria-hidden="true"></span>
            <span class="nis2-status-chip__label">${meta.label}</span>
          </div>
          <div class="nis2-status-text">${escapeHtml(displayText)}</div>
        </div>
        <button class="icon-btn" type="button" data-nis2-status-id="${item.id}" title="Modifier le statut clé"><i class="fas fa-pen"></i></button>
      `;
      list.appendChild(li);
    });
  }
  function renderNis2Plan() {
    const container = document.getElementById("nis2PlanContainer");
    if (!container) return;
    ensureNis2Plan();
    ensureNis2PlanToolbar();
    refreshNis2PhaseFilter();
    if (!isReadOnlyMode || !isReadOnlyMode()) renderNis2PhasesPanel();
    const phaseFilter = (document.getElementById("nis2PhaseFilter")?.value || "all").toString();
    const pillarFilter = (document.getElementById("nis2PillarFilter")?.value || "all").toString();
    const viewMode = nis2PlanViewMode || "detail";
    const focusOpen = !!nis2PlanShowOpenOnly;
    const allowEdit = !(isReadOnlyMode && typeof isReadOnlyMode === "function" ? isReadOnlyMode() : false);
    container.classList.toggle("nis2-plan--executive", viewMode === "executive");
    container.classList.toggle("nis2-plan--focus-open", focusOpen);
    container.classList.add("nis2-plan-list");
    container.classList.remove("nis2-plan-grid");
    container.innerHTML = "";
    let displayed = 0;
    let totalActions = 0;
    let doneActions = 0;
    let openActions = 0;
    let progressSum = 0;
    let budgetSum = 0;
    let budgetInfra = 0;
    let budgetSsi = 0;
    let budgetAll = 0;
    let budget2026Total = 0;
    const customBlocks = appData.nis2Plan.customBlocks || [];
    const summaryEl = document.createElement("div");
    summaryEl.className = "nis2-plan-summary-strip";
    const frag = document.createDocumentFragment();
    (appData.nis2Plan.domains || []).forEach((domain, dIndex) => {
      const pillarMatches = pillarFilter === "all" || String(domain.pillar || "") === pillarFilter;
      if (!pillarMatches) return;
      const phases = getSortedNis2Phases();
      const iconClass = NIS2_DOMAIN_ICONS[NORMALIZE(domain.title || "")] || "fa-circle-nodes";
      let actions = (domain.actions || []).map((a) => {
        const phaseId = normalizePhaseId(a.phase);
        if (a.phase !== phaseId) a.phase = phaseId;
        return a;
      });
      const hadActions = actions.length > 0;
      if (phaseFilter !== "all") actions = actions.filter(a => String(a.phase || "") === phaseFilter);
      if (focusOpen) actions = actions.filter((a) => a.status !== "done");
      if (!actions.length && hadActions) return;
      displayed++;
      const urgencyLabel = NIS2_URGENCY_LABELS[domain.urgency] || NIS2_URGENCY_LABELS.standard;
      const objectiveHtml = domain.objective ? `<p class="info-box__text">${escapeHtml(domain.objective)}</p>` : "";
      const actionsByPhase = {};
      phases.forEach((p) => { actionsByPhase[p.id] = []; });
      actions.forEach((a) => {
        const key = a.phase || normalizePhaseId(a.phase);
        if (!actionsByPhase[key]) actionsByPhase[key] = [];
        actionsByPhase[key].push(a);
      });
      const totalDomainActions = actions.length;
      const doneDomain = actions.filter((a) => a.status === "done").length;
      const openDomain = actions.filter((a) => a.status !== "done").length;
      const progressValues = actions.map((a) => {
        const progress = Number.isFinite(a.progress) ? a.progress : a.status === "done" ? 100 : 0;
        return Math.max(0, Math.min(100, Math.round(progress)));
      });
      const avgProgress = progressValues.length ? Math.round(progressValues.reduce((a, b) => a + b, 0) / progressValues.length) : 0;
      const domainBudgetSum = actions.reduce((sum, a) => {
        const val = getActionBudgetTotal(a);
        return Number.isFinite(val) ? sum + val : sum;
      }, 0);
      actions.forEach((a) => {
        const val = getActionBudgetTotal(a);
        if (!Number.isFinite(val)) return;
        budgetAll += val;
        const cat = (a.budgetCategory || "").toLowerCase();
        if (cat === "infra") budgetInfra += val;
        else if (cat === "ssi" || cat === "spot") budgetSsi += val;
        const b26 = parseBudgetValue(a.budget2026);
        if (Number.isFinite(b26)) budget2026Total += b26;
      });
      const domainBudgetDisplay = Number.isFinite(domainBudgetSum) && domainBudgetSum >= 0 ? formatCurrency(domainBudgetSum) : "—";
      totalActions += totalDomainActions;
      doneActions += doneDomain;
      openActions += openDomain;
      progressSum += progressValues.reduce((a, b) => a + b, 0);
      budgetSum += domainBudgetSum;
      const tableColspan = allowEdit ? 10 : 9;
      let actionsHtml = phases.map((phase) => {
        const rows = actionsByPhase[phase.id] || [];
        if (!rows.length) return "";
        const label = getNis2PhaseLabel(phase.id);
        const rowsHtml = rows.map((action, aIndex) => {
          const meta = NIS2_STATUS_META[action.status] || NIS2_STATUS_META.todo;
          const dotClass = action.status === "done" ? "nis2-status-dot--done" : action.status === "in_progress" ? "nis2-status-dot--progress" : "nis2-status-dot--todo";
          const phaseLabel = getNis2PhaseLabel(action.phase);
          const typeLabel = NIS2_ACTION_TYPE_LABELS[action.workType] || NIS2_ACTION_TYPE_LABELS.initiative;
          const ownerLabel = action.owner ? escapeHtml(action.owner) : "—";
          const budget26 = formatBudgetValue(action.budget2026);
          const budget27 = formatBudgetValue(action.budget2027);
          const budget28 = formatBudgetValue(action.budget2028);
          const totalBudget = getActionBudgetTotal(action);
          const budgetTotalLabel = Number.isFinite(totalBudget) ? formatCurrency(totalBudget) : "—";
          const progress = Number.isFinite(action.progress) ? Math.max(0, Math.min(100, Math.round(action.progress))) : (action.status === "done" ? 100 : 0);
          return `
          <tr>
            <td>
              <div class="nis2-action-main">
                <span class="nis2-status-dot ${dotClass}"></span>
                <div>
                  <div class="nis2-action-text">${escapeHtml(action.text || `Action ${dIndex + 1}-${aIndex + 1}`)}</div>
                  <div class="nis2-action-meta">
                    <span class="chip chip--phase">${escapeHtml(phaseLabel)}</span>
                    <span class="chip chip--type">${escapeHtml(typeLabel)}</span>
                    <span class="chip chip--owner">${ownerLabel}</span>
                  </div>
                </div>
              </div>
            </td>
            <td class="text-right">${budget26}</td>
            <td class="text-right">${budget27}</td>
            <td class="text-right">${budget28}</td>
            <td class="text-right">${budgetTotalLabel}</td>
            <td class="text-center">${ownerLabel}</td>
            <td class="text-center">${escapeHtml(phaseLabel)}</td>
            <td class="text-center nis2-progress-cell">
              <div class="progress-bar"><div class="progress-bar__fill" style="width: ${progress}%"></div></div>
              <span class="progress-label">${progress}%</span>
            </td>
            <td><span class="${meta.className}">${meta.label}</span></td>
            ${allowEdit ? `<td class="text-right">
              <button class="icon-btn" type="button" data-nis2-action="edit-action" data-domain-id="${domain.id}" data-action-id="${action.id}" title="Modifier l'action"><i class="fas fa-pen"></i></button>
              <button class="icon-btn icon-btn--danger" type="button" data-nis2-action="delete-action" data-domain-id="${domain.id}" data-action-id="${action.id}" title="Supprimer l'action"><i class="fas fa-trash"></i></button>
            </td>` : ""}
          </tr>
        `;
        }).join("");
        return `
          <tr class="nis2-phase-group"><td colspan="${tableColspan}">${escapeHtml(label)}</td></tr>
          ${rowsHtml}
        `;
      }).join("");
      if (!actionsHtml) {
        actionsHtml = `
          <tr>
            <td colspan="${tableColspan}">
              <div class="nis2-empty">Aucune action pour ce domaine.</div>
            </td>
          </tr>
        `;
      }
      const card = document.createElement("div");
      card.className = `nis2-plan-card nis2-plan-card--pillar${domain.pillar || 1}`;
      card.innerHTML = `
        <div class="nis2-plan-card__header">
          <div class="nis2-plan-card__header-left">
            <div class="nis2-plan-card__icon"><i class="fas ${iconClass}"></i></div>
            <div>
              <p class="nis2-plan-card__eyebrow">Pilier ${escapeHtml(domain.pillar || "")}</p>
              <h4 class="module__title">${escapeHtml(domain.title || `Domaine ${dIndex + 1}`)}</h4>
              ${objectiveHtml}
              <div class="nis2-plan-meta">
                <span class="nis2-tag">Pilier ${escapeHtml(domain.pillar || "")}</span>
                <span class="nis2-tag">${escapeHtml(urgencyLabel)}</span>
                <span class="nis2-tag"><i class="fas fa-list-check"></i> ${doneDomain}/${totalDomainActions} livrées</span>
              </div>
            </div>
          </div>
          ${allowEdit ? `<div>
            <button class="icon-btn" type="button" data-nis2-action="edit-domain" data-domain-id="${domain.id}" title="Modifier le domaine"><i class="fas fa-pen"></i></button>
          </div>` : ""}
        </div>
        <div class="nis2-plan-stats">
          <span class="stat-chip"><i class="fas fa-gauge-high"></i> Avancement ${avgProgress}%</span>
          <span class="stat-chip"><i class="fas fa-hourglass-half"></i> ${openDomain} en cours / à faire</span>
          <span class="stat-chip"><i class="fas fa-check-circle"></i> ${doneDomain} réalisés</span>
          <span class="stat-chip"><i class="fas fa-coins"></i> Budget ${domainBudgetDisplay}</span>
        </div>
        <div class="nis2-actions">
          <table class="nis2-plan-table ${viewMode === "executive" ? "nis2-plan-table--executive" : ""}">
            <thead>
              <tr>
                <th>Action</th>
                <th>Budget 2026</th>
                <th>Budget 2027</th>
                <th>Budget 2028</th>
                <th>Budget total</th>
                <th>Responsable</th>
                <th>Phase</th>
                <th>Avancement</th>
                <th>Statut</th>
                ${allowEdit ? `<th class="text-right">Actions</th>` : ""}
              </tr>
            </thead>
            <tbody>
              ${actionsHtml}
            </tbody>
          </table>
        </div>
        ${allowEdit ? `<div>
          <button class="btn btn--outline btn--sm" type="button" data-nis2-action="add-action" data-domain-id="${domain.id}">
            <i class="fas fa-plus"></i> Ajouter une action
          </button>
        </div>` : ""}
      `;
      frag.appendChild(card);
    });
    if (totalActions > 0) {
      const avg = totalActions ? Math.round(progressSum / totalActions) : 0;
      const budgetDisplay = Number.isFinite(budgetSum) ? formatCurrency(budgetSum) : "—";
      summaryEl.innerHTML = `
        <div class="stat-chip"><i class="fas fa-chart-line"></i> ${totalActions} actions (filtre appliqué)</div>
        <div class="stat-chip"><i class="fas fa-check-double"></i> ${doneActions} réalisées</div>
        <div class="stat-chip"><i class="fas fa-circle-notch"></i> ${openActions} en cours / à faire</div>
        <div class="stat-chip"><i class="fas fa-gauge"></i> Avancement moyen ${avg}%</div>
        <div class="stat-chip"><i class="fas fa-sack-dollar"></i> Budget cumulé ${budgetDisplay}</div>
      `;
      container.appendChild(summaryEl);
    }
    renderNis2BudgetGrid({ budgetAll, budget2026Total, budgetInfra, budgetSsi, customBlocks });
    container.appendChild(frag);
    if (displayed === 0) {
      container.innerHTML = `<div class="nis2-empty">Aucune action ne correspond à vos filtres actuels.</div>`;
    }
    renderNis2ActionsChart();
    renderNis2ActionsByDomainChart();
  }
  function renderNis2BudgetGrid({ budgetAll, budget2026Total, budgetInfra, budgetSsi, customBlocks = [] }) {
    const module = document.getElementById("nis2-program");
    const anchor = module ? module.querySelector(".nis2-top-grid") : null;
    if (!module || !anchor) return;
    const customGrid = renderNis2CustomBlocks(customBlocks);
    const budgetBlocks = Array.isArray(appData?.nis2Plan?.budgetBlocks) && appData.nis2Plan.budgetBlocks.length
      ? appData.nis2Plan.budgetBlocks
      : cloneDefaultNis2Plan().budgetBlocks || [];
    const hasBudgetBlocks = budgetBlocks.length > 0;
    const hasBudgetValues = [budgetAll, budget2026Total, budgetInfra, budgetSsi].some((val) => Number.isFinite(val) && val > 0);
    let grid = document.getElementById("nis2BudgetGrid");
    if (!hasBudgetBlocks && !hasBudgetValues) {
      if (grid) grid.remove();
      return;
    }
    if (!grid) {
      grid = document.createElement("div");
      grid.id = "nis2BudgetGrid";
      grid.className = "nis2-budget-grid";
    }
    if (customGrid && customGrid.parentNode) {
      if (customGrid.nextSibling) customGrid.parentNode.insertBefore(grid, customGrid.nextSibling);
      else customGrid.parentNode.appendChild(grid);
    } else {
      module.insertBefore(grid, anchor);
    }
    const safe = (idx, fallback) => budgetBlocks[idx]?.title || fallback;
    const readonly = isReadOnlyMode && typeof isReadOnlyMode === "function" ? isReadOnlyMode() : false;
    grid.innerHTML = `
      <div class="kpi-card kpi-card--primary" data-nis2-budget-idx="0" role="button" tabindex="0" aria-label="Modifier le bloc budget 1">
        <div class="kpi-card__icon"><i class="fas fa-coins"></i></div>
        <div class="kpi-card__content">
          <p class="kpi-card__eyebrow">${escapeHtml(safe(0, "Budget 3 ans"))}</p>
          <div class="kpi-card__value">${formatCurrency(budgetAll)}</div>
          ${readonly ? "" : `<div class="nis2-budget-card__hint"><i class="fas fa-pen"></i> Modifier</div>`}
        </div>
      </div>
      <div class="kpi-card kpi-card--success" data-nis2-budget-idx="1" role="button" tabindex="0" aria-label="Modifier le bloc budget 2">
        <div class="kpi-card__icon"><i class="fas fa-calendar-alt"></i></div>
        <div class="kpi-card__content">
          <p class="kpi-card__eyebrow">${escapeHtml(safe(1, "Budget 2026"))}</p>
          <div class="kpi-card__value">${formatCurrency(budget2026Total)}</div>
          ${readonly ? "" : `<div class="nis2-budget-card__hint"><i class="fas fa-pen"></i> Modifier</div>`}
        </div>
      </div>
      <div class="kpi-card kpi-card--warning" data-nis2-budget-idx="2" role="presentation" tabindex="-1" aria-label="Bloc budget 3 (titre fixe)">
        <div class="kpi-card__icon"><i class="fas fa-server"></i></div>
        <div class="kpi-card__content">
          <p class="kpi-card__eyebrow">${escapeHtml(safe(2, "Budget Total Infra 3 ans"))}</p>
          <div class="kpi-card__value">${formatCurrency(budgetInfra)}</div>
        </div>
      </div>
      <div class="kpi-card kpi-card--info" data-nis2-budget-idx="3" role="presentation" tabindex="-1" aria-label="Bloc budget 4 (titre fixe)">
        <div class="kpi-card__icon"><i class="fas fa-shield-alt"></i></div>
        <div class="kpi-card__content">
          <p class="kpi-card__eyebrow">${escapeHtml(safe(3, "Budget SSI & SPOT 3 ans"))}</p>
          <div class="kpi-card__value">${formatCurrency(budgetSsi)}</div>
        </div>
      </div>
    `;
    if (!grid.dataset.bound) {
      grid.dataset.bound = "true";
      const handler = (event) => {
        const card = event.target.closest("[data-nis2-budget-idx]");
        if (!card) return;
        const idx = parseInt(card.dataset.nis2BudgetIdx, 10);
        openNis2BudgetBlockModal(Number.isFinite(idx) ? idx : 0);
      };
      grid.addEventListener("click", handler);
      grid.addEventListener("keydown", (event) => {
        if (event.key === "Enter" || event.key === " ") {
          event.preventDefault();
          handler(event);
        }
      });
    }
  }
  function renderNis2CustomBlocks(blocks = []) {
    const module = document.getElementById("nis2-program");
    const anchor = module ? module.querySelector(".nis2-top-grid") : null;
    if (!module || !anchor) return null;
    const fallbackBlocks = cloneDefaultNis2Plan().customBlocks || [];
    const hasBlocks = Array.isArray(blocks) && blocks.length > 0;
    if (!hasBlocks && fallbackBlocks.length === 0) {
      const existing = document.getElementById("nis2CustomBlocks");
      if (existing) existing.remove();
      return null;
    }
    const source = Array.isArray(blocks) && blocks.length ? blocks : fallbackBlocks;
    const minLength = Math.max(3, fallbackBlocks.length || 0);
    const safeBlocks = source.slice(0, Math.max(source.length, minLength));
    while (safeBlocks.length < minLength) {
      const template = fallbackBlocks[safeBlocks.length % fallbackBlocks.length] || { title: `Bloc ${safeBlocks.length + 1}`, value: 0, format: safeBlocks.length > 0 ? "currency" : "percent" };
      const format = (template.format || (safeBlocks.length > 0 ? "currency" : "percent")).toLowerCase();
      const fallbackValue = format === "currency" ? parseLooseNumber(template.value) : clampPercent(template.value, 0);
      safeBlocks.push({
        id: template.id || `nis2-highlight-${safeBlocks.length + 1}`,
        title: template.title,
        value: Number.isFinite(fallbackValue) ? fallbackValue : 0,
        format
      });
    }
    let grid = document.getElementById("nis2CustomBlocks");
    if (!grid) {
      grid = document.createElement("div");
      grid.id = "nis2CustomBlocks";
      grid.className = "nis2-custom-grid";
      module.insertBefore(grid, anchor);
    } else {
      module.insertBefore(grid, anchor);
    }
    const accentClasses = ["kpi-card--primary", "kpi-card--warning", "kpi-card--success"];
    const iconSet = ["fa-bullseye", "fa-rocket", "fa-flag-checkered"];
    grid.innerHTML = safeBlocks.slice(0, 3).map((block, idx) => {
      const format = (block.format || (idx > 0 ? "currency" : "percent")).toLowerCase();
      const rawValue = typeof block.value !== "undefined" ? block.value : block.percent;
      const value = format === "currency"
        ? (() => {
            const num = parseLooseNumber(rawValue);
            return Number.isFinite(num) ? num : 0;
          })()
        : clampPercent(rawValue, 0);
      const accent = accentClasses[idx % accentClasses.length];
      const icon = iconSet[idx % iconSet.length];
      const eyebrow = escapeHtml(block.title || `Bloc ${idx + 1}`);
      const valueDisplay = format === "currency" ? formatCurrency(value) : `${value}%`;
      return `
        <div class="kpi-card nis2-custom-card ${accent}" role="button" tabindex="0" data-nis2-block-id="${block.id}" data-nis2-block-index="${idx}" aria-label="Modifier le bloc ${escapeHtml(block.title || `Bloc ${idx + 1}`)}">
          <div class="kpi-card__icon"><i class="fas ${icon}"></i></div>
          <div class="kpi-card__content">
            <p class="kpi-card__eyebrow">${eyebrow}</p>
            <div class="kpi-card__value">${valueDisplay}</div>
            ${isReadOnlyMode && typeof isReadOnlyMode === "function" && isReadOnlyMode() ? "" : `<div class="nis2-custom-card__hint"><i class="fas fa-pen"></i> Modifier</div>`}
          </div>
        </div>
      `;
    }).join("");
    if (!grid.dataset.bound) {
      grid.dataset.bound = "true";
      const openFromEvent = (event) => {
        const card = event.target.closest("[data-nis2-block-id]");
        if (!card) return;
        const index = parseInt(card.dataset.nis2BlockIndex, 10);
        openNis2CustomBlockModal(card.dataset.nis2BlockId, Number.isFinite(index) ? index : 0);
      };
      grid.addEventListener("click", openFromEvent);
      grid.addEventListener("keydown", (event) => {
        if (event.key === "Enter" || event.key === " ") {
          event.preventDefault();
          openFromEvent(event);
        }
      });
    }
    return grid;
  }
  function openNis2CustomBlockModal(blockId = "", fallbackIndex = 0) {
    if (!ensureEditMode("modifier un bloc d'objectif NIS2")) return;
    ensureNis2Plan();
    const blocks = appData.nis2Plan.customBlocks || [];
    const defaults = cloneDefaultNis2Plan().customBlocks || [];
    const foundIndex = blocks.findIndex((b) => b.id === blockId);
    const index = foundIndex >= 0 ? foundIndex : (Number.isFinite(fallbackIndex) ? fallbackIndex : 0);
    const fallback = defaults[index % defaults.length] || { title: `Bloc ${index + 1}`, value: 0, format: index > 0 ? "currency" : "percent" };
    const block = blocks[index] || fallback;
    const format = (block.format || fallback.format || (index > 0 ? "currency" : "percent")).toLowerCase();
    const currentValue = typeof block.value !== "undefined" ? block.value : (typeof block.percent !== "undefined" ? block.percent : fallback.value);
    const valueFieldLabel = format === "currency" ? "Montant (€)" : "Progression (%)";
    const valueFieldHelp = format === "currency" ? "Coût budgétaire en euros" : "Entre 0 et 100";
    openModal(`Bloc NIS2 ${index + 1}`, [
      { name: "title", label: "Titre du bloc", value: block.title || "" },
      { name: "value", label: valueFieldLabel, type: "number", value: currentValue ?? 0, help: valueFieldHelp }
    ], (data) => {
      const cleanValue = format === "currency"
        ? (() => {
            const num = parseLooseNumber(data.value ?? currentValue ?? 0);
            return Number.isFinite(num) ? num : 0;
          })()
        : clampPercent(data.value ?? currentValue ?? 0, clampPercent(currentValue ?? 0, 0));
      const updated = {
        id: block.id || blockId || `nis2-highlight-${index + 1}`,
        title: data.title || block.title || `Bloc ${index + 1}`,
        value: cleanValue,
        format
      };
      while (blocks.length < Math.max(index + 1, 3)) {
        const template = defaults[blocks.length % defaults.length] || { title: `Bloc ${blocks.length + 1}`, value: 0, format: blocks.length > 0 ? "currency" : "percent" };
        blocks.push({
          id: template.id || `nis2-highlight-${blocks.length + 1}`,
          title: template.title,
          value: Number.isFinite(template.value) ? template.value : 0,
          format: template.format || (blocks.length > 0 ? "currency" : "percent")
        });
      }
      blocks[index] = updated;
      appData.nis2Plan.customBlocks = blocks;
      saveData();
      renderNis2Plan();
      return true;
    });
  }
  function openNis2BudgetBlockModal(index = 0) {
    if (!ensureEditMode("modifier un bloc budget NIS2")) return;
    ensureNis2Plan();
    const defaults = cloneDefaultNis2Plan().budgetBlocks || [];
    const blocks = Array.isArray(appData.nis2Plan.budgetBlocks) ? [...appData.nis2Plan.budgetBlocks] : [];
    const fallback = defaults[index % (defaults.length || 1)] || { id: `nis2-budget-${index + 1}`, title: `Budget ${index + 1}` };
    const block = blocks[index] || fallback;
    openModal(`Bloc budget ${index + 1}`, [
      { name: "title", label: "Titre du bloc budget", value: block.title || fallback.title || `Budget ${index + 1}` }
    ], (data) => {
      while (blocks.length < Math.max(index + 1, defaults.length)) {
        const tpl = defaults[blocks.length % (defaults.length || 1)] || { id: `nis2-budget-${blocks.length + 1}`, title: `Budget ${blocks.length + 1}` };
        blocks.push({ id: tpl.id, title: tpl.title });
      }
      blocks[index] = {
        id: block.id || fallback.id || `nis2-budget-${index + 1}`,
        title: data.title || block.title || fallback.title || `Budget ${index + 1}`
      };
      appData.nis2Plan.budgetBlocks = blocks;
      saveData();
      renderNis2Plan();
      return true;
    });
  }
  function openNis2DomainModal(domainId) {
    if (!ensureEditMode("modifier un domaine NIS2")) return;
    const domain = (appData.nis2Plan.domains || []).find(d => d.id === domainId) || { title: "", objective: "", pillar: 1, urgency: "standard" };
    const isEdit = !!domainId;
    openModal(isEdit ? "Modifier un domaine NIS2" : "Nouveau domaine NIS2", [
      { name: "title", label: "Titre du domaine", value: domain.title },
      { name: "objective", label: "Objectif", type: "textarea", value: domain.objective },
      { name: "pillar", label: "Pilier", type: "number", value: domain.pillar || 1 },
      { name: "urgency", label: "Priorité", type: "select", value: domain.urgency || "standard", options: [
        { value: "high", label: "Urgence haute" },
        { value: "medium", label: "Priorité moyenne" },
        { value: "standard", label: "Priorité standard" }
      ] }
    ], (data) => {
      if (!data.title) return false;
      const clean = {
        title: data.title,
        objective: data.objective || "",
        pillar: parseInt(data.pillar, 10) || 1,
        urgency: data.urgency || "standard"
      };
      if (isEdit) {
        const idx = appData.nis2Plan.domains.findIndex(d => d.id === domainId);
        if (idx >= 0) {
          appData.nis2Plan.domains[idx] = Object.assign({}, appData.nis2Plan.domains[idx], clean);
        }
      } else {
        const newId = "nis2-domain-" + (appData.nis2Plan.domains.length + 1);
        appData.nis2Plan.domains.push(Object.assign({ id: newId, actions: [] }, clean));
      }
      saveData();
      renderNis2Plan();
      return true;
    }, isEdit ? () => deleteNis2Domain(domainId) : null);
  }
  function openNis2ActionModal(domainId = "", actionId = "") {
    if (!ensureEditMode("modifier une action NIS2")) return;
    ensureNis2Plan();
    ensureNis2Phases();
    const domains = appData.nis2Plan.domains || [];
    if (domains.length === 0) {
      showToast("Ajoutez d'abord un domaine NIS2 avant de créer une action.", "warning");
      return;
    }
    const domain = domains.find(d => d.id === domainId) || domains[0];
    const action = (domain.actions || []).find(a => a.id === actionId) || {
      text: "",
      phase: normalizePhaseId(1),
      status: "todo",
      owner: "",
      workType: "initiative",
      progress: 0,
      budget: "",
      budget2026: "",
      budget2027: "",
      budget2028: "",
      budgetCategory: "divers"
    };
    const phases = getSortedNis2Phases();
    const defaultPhase = phases[0]?.id || normalizePhaseId(action.phase);
    const phaseOptions = phases.map((p) => ({ value: p.id, label: p.label }));
    const isEdit = !!actionId;
    const actionTypeOptions = typeof NIS2_ACTION_TYPE_OPTIONS !== "undefined" ? NIS2_ACTION_TYPE_OPTIONS : [
      { value: "service", label: "Service" },
      { value: "projet", label: "Projet" },
      { value: "initiative", label: "Initiative" },
      { value: "programme", label: "Programme" }
    ];
    const budgetCategoryOptions = [
      { value: "infra", label: "Infra" },
      { value: "ssi", label: "SSI & SPOT" },
      { value: "divers", label: "Divers" }
    ];
    openModal(isEdit ? "Modifier une action NIS2" : "Nouvelle action NIS2", [
      { name: "domainId", label: "Domaine", type: "select", value: domain.id, options: domains.map(d => ({ value: d.id, label: d.title })) },
      { name: "text", label: "Description de l'action", type: "textarea", value: action.text },
      { name: "phase", label: "Phase", type: "select", value: action.phase || defaultPhase, options: phaseOptions },
      { name: "workType", label: "Type (Service / Projet / Initiative / Programme)", type: "select", value: action.workType || "initiative", options: actionTypeOptions },
      { name: "owner", label: "Responsable de l'action", value: action.owner || "" },
      { name: "budget2026", label: "Budget 2026", type: "number", value: action.budget2026 || "" },
      { name: "budget2027", label: "Budget 2027", type: "number", value: action.budget2027 || "" },
      { name: "budget2028", label: "Budget 2028", type: "number", value: action.budget2028 || "" },
      { name: "budget", label: "Budget global (optionnel)", value: action.budget || "" },
      { name: "budgetCategory", label: "Catégorie budget", type: "select", value: action.budgetCategory || "divers", options: budgetCategoryOptions },
      { name: "progress", label: "Avancement (%)", type: "number", value: action.progress ?? (action.status === "done" ? 100 : 0) },
      { name: "status", label: "Statut", type: "select", value: action.status || "todo", options: [
        { value: "todo", label: "À réaliser" },
        { value: "in_progress", label: "En cours" },
        { value: "suspended", label: "Suspendu" },
        { value: "done", label: "Réalisé" }
      ] }
    ], (data) => {
      if (!data.text) return false;
      const targetDomain = domains.find(d => d.id === data.domainId) || domain;
      if (!Array.isArray(targetDomain.actions)) targetDomain.actions = [];
      if (isEdit) {
        const idx = targetDomain.actions.findIndex(a => a.id === actionId);
        if (idx >= 0) {
          targetDomain.actions[idx] = Object.assign({}, targetDomain.actions[idx], {
            text: data.text,
            phase: normalizePhaseId(data.phase),
            status: data.status || "todo",
            owner: data.owner || "",
            workType: data.workType || "initiative",
            budget: data.budget || "",
            budget2026: data.budget2026 || "",
            budget2027: data.budget2027 || "",
            budget2028: data.budget2028 || "",
            budgetCategory: data.budgetCategory || "divers",
            progress: Math.max(0, Math.min(100, parseInt(data.progress, 10) || 0))
          });
        }
      } else {
        const newActionId = generateId("nis2-act");
        targetDomain.actions.push({
          id: newActionId,
          text: data.text,
          phase: normalizePhaseId(data.phase),
          status: data.status || "todo",
          owner: data.owner || "",
          workType: data.workType || "initiative",
          budget: data.budget || "",
          budget2026: data.budget2026 || "",
          budget2027: data.budget2027 || "",
          budget2028: data.budget2028 || "",
          budgetCategory: data.budgetCategory || "divers",
          progress: Math.max(0, Math.min(100, parseInt(data.progress, 10) || 0))
        });
      }
      saveData();
      renderNis2Plan();
      return true;
    }, isEdit ? () => deleteNis2Action(domainId, actionId) : null);
  }
  function deleteNis2Domain(domainId) {
    if (!ensureEditMode("supprimer un domaine NIS2")) return;
    if (!confirm("Supprimer ce domaine et ses actions ?")) return;
    appData.nis2Plan.domains = (appData.nis2Plan.domains || []).filter(d => d.id !== domainId);
    saveData();
    renderNis2Plan();
  }
  function deleteNis2Action(domainId, actionId) {
    if (!ensureEditMode("supprimer une action NIS2")) return;
    const domain = (appData.nis2Plan.domains || []).find(d => d.id === domainId);
    if (!domain) return;
    domain.actions = (domain.actions || []).filter(a => a.id !== actionId);
    saveData();
    renderNis2Plan();
  }
  function openNis2PhaseModal(phaseId = "") {
    if (!ensureEditMode("modifier les phases NIS2")) return;
    ensureNis2Phases();
    const phases = getSortedNis2Phases();
    const phase = phases.find((p) => p.id === phaseId) || { label: "", order: phases.length + 1 };
    const isEdit = !!phaseId;
    openModal(isEdit ? "Modifier une phase NIS2" : "Nouvelle phase NIS2", [
      { name: "label", label: "Nom de la phase", value: phase.label },
      { name: "order", label: "Ordre d'affichage", type: "number", value: phase.order || phases.length + 1 }
    ], (data) => {
      if (!data.label) return false;
      const order = parseInt(data.order, 10);
      if (isEdit) {
        const idx = appData.nis2Plan.phases.findIndex((p) => p.id === phaseId);
        if (idx >= 0) {
          appData.nis2Plan.phases[idx] = Object.assign({}, appData.nis2Plan.phases[idx], {
            label: data.label,
            order: isNaN(order) ? appData.nis2Plan.phases[idx].order : order
          });
        }
      } else {
        const newId = generateId("phase");
        appData.nis2Plan.phases.push({
          id: newId,
          label: data.label,
          order: isNaN(order) ? phases.length + 1 : order,
          fallback: false
        });
      }
      appData.nis2Plan.phases = getSortedNis2Phases();
      saveData();
      renderNis2Plan();
      return true;
    }, isEdit && !phase.fallback ? () => deleteNis2Phase(phaseId) : null);
  }
  function deleteNis2Phase(phaseId) {
    if (!ensureEditMode("supprimer une phase NIS2")) return;
    ensureNis2Phases();
    const phases = getSortedNis2Phases();
    const target = phases.find((p) => p.id === phaseId);
    if (!target || target.fallback) return;
    const fallback = phases.find((p) => p.fallback && p.id !== phaseId) || phases.find((p) => p.id !== phaseId);
    if (!fallback) return;
    if (!confirm("Supprimer cette phase ? Les actions seront déplacées vers la phase de secours.")) return;
    appData.nis2Plan.phases = phases.filter((p) => p.id !== phaseId);
    (appData.nis2Plan.domains || []).forEach((d) => {
      d.actions = (d.actions || []).map((a) => {
        if (a.phase === phaseId) a.phase = fallback.id;
        return a;
      });
    });
    saveData();
    renderNis2Plan();
  }
  function setupNis2PlanInteractions() {
    const container = document.getElementById("nis2PlanContainer");
    if (container) {
      container.addEventListener("click", (event) => {
        const btn = event.target.closest("[data-nis2-action]");
        if (!btn) return;
        const action = btn.dataset.nis2Action;
        const domainId = btn.dataset.domainId;
        const actionId = btn.dataset.actionId;
        if (isReadOnlyMode && typeof isReadOnlyMode === "function" && isReadOnlyMode()) return;
        if (action === "edit-domain") return openNis2DomainModal(domainId);
        if (action === "add-action") return openNis2ActionModal(domainId);
        if (action === "edit-action") return openNis2ActionModal(domainId, actionId);
        if (action === "delete-action") return deleteNis2Action(domainId, actionId);
      });
    }
    const phasesPanel = document.getElementById("nis2PhasesPanel");
    if (phasesPanel && !phasesPanel.dataset.bound) {
      phasesPanel.dataset.bound = "true";
      phasesPanel.addEventListener("click", (event) => {
        const addBtn = event.target.closest("[data-phase-action=\"add\"]");
        if (addBtn) return openNis2PhaseModal();
        const editBtn = event.target.closest("[data-phase-action=\"edit\"]");
        if (editBtn) return openNis2PhaseModal(editBtn.dataset.phaseId);
        const delBtn = event.target.closest("[data-phase-action=\"delete\"]");
        if (delBtn) return deleteNis2Phase(delBtn.dataset.phaseId);
      });
    }
    const toolbar = document.getElementById("nis2PlanToolbar");
    if (toolbar && !toolbar.dataset.bound) {
      toolbar.dataset.bound = "true";
      toolbar.addEventListener("click", (event) => {
        const viewBtn = event.target.closest("[data-view-mode]");
        if (viewBtn) {
          nis2PlanViewMode = viewBtn.dataset.viewMode || "detail";
          ensureNis2PlanToolbar();
          return renderNis2Plan();
        }
        const focusBtn = event.target.closest("#nis2PlanFocusBtn");
        if (focusBtn) {
          nis2PlanShowOpenOnly = !nis2PlanShowOpenOnly;
          ensureNis2PlanToolbar();
          return renderNis2Plan();
        }
      });
    }
    const statusList = document.getElementById("nis2StatusList");
    if (statusList) {
      statusList.addEventListener("click", (event) => {
        const btn = event.target.closest("[data-nis2-status-id]");
        if (!btn) return;
        openNis2StatusModal(btn.dataset.nis2StatusId);
      });
    }
  }
  window.renderNis2Plan = renderNis2Plan;
  window.loadNis2Program = loadNis2Program;
  window.openNis2DomainModal = openNis2DomainModal;
  window.openNis2ActionModal = openNis2ActionModal;
  window.openNis2BudgetBlockModal = openNis2BudgetBlockModal;
  function openNis2StatusModal(statusId = "") {
    ensureNis2Plan();
    const statuses = appData.nis2Plan.statuses || [];
    const status = statuses.find(s => s.id === statusId) || { text: "", status: "todo" };
    const isEdit = !!statusId;
    openModal(isEdit ? "Modifier un statut clé" : "Nouveau statut clé", [
      { name: "text", label: "Description", type: "textarea", value: status.text || "" },
      { name: "status", label: "Statut", type: "select", value: status.status || "todo", options: [
        { value: "todo", label: "À réaliser" },
        { value: "in_progress", label: "En cours" },
        { value: "suspended", label: "Suspendu" },
        { value: "done", label: "Réalisé" }
      ] }
    ], (data) => {
      if (!data.text) return false;
      if (isEdit) {
        const idx = statuses.findIndex(s => s.id === statusId);
        if (idx >= 0) statuses[idx] = Object.assign({}, statuses[idx], { text: data.text, status: data.status });
      } else {
        statuses.push({
          id: generateId("nis2-status"),
          text: data.text,
          status: data.status || "todo"
        });
      }
      appData.nis2Plan.statuses = statuses;
      saveData();
      renderNis2StatusList();
      return true;
    }, isEdit ? () => {
      const idx = statuses.findIndex(s => s.id === statusId);
      if (idx >= 0 && confirm("Supprimer ce statut clé ?")) {
        statuses.splice(idx, 1);
        saveData();
        renderNis2StatusList();
        return true;
      }
      return false;
    } : null);
  }

  // modules/critical-assets.js
  const CRITICAL_ASSET_AVAILABILITY_OPTIONS = [
    { value: "Niveau 1 - Haute disponibilité", label: "Niveau 1 - Haute disponibilité" },
    { value: "Niveau 2 - Reprise rapide", label: "Niveau 2 - Reprise rapide" },
    { value: "Niveau 3 - Reprise standard", label: "Niveau 3 - Reprise standard" },
    { value: "Niveau 4 - Reprise prolongée", label: "Niveau 4 - Reprise prolongée" }
  ];
  const CRITICAL_ASSET_CONF_OPTIONS = [
    { value: "Sensible", label: "Sensible" },
    { value: "Très sensible", label: "Très sensible" },
    { value: "Restreint", label: "Restreint" },
    { value: "Public", label: "Public" }
  ];
  const CRITICAL_ASSET_IMPACT_OPTIONS = [
    { value: "Sociétal", label: "Sociétal" },
    { value: "Opérationnel", label: "Opérationnel" },
    { value: "Financier", label: "Financier" },
    { value: "Réputation", label: "Réputation" },
    { value: "Réglementaire", label: "Réglementaire" }
  ];
  const CRITICAL_ASSET_MFA_STATUS_OPTIONS = [
    { value: "Non implémenté", label: "Non implémenté" },
    { value: "Partiel", label: "Partiel" },
    { value: "Implémenté", label: "Implémenté" },
    { value: "Externe", label: "Externe" }
  ];
  const CRITICAL_ASSET_DACP_OPTIONS = [
    { value: "Oui", label: "Oui" },
    { value: "Non", label: "Non" }
  ];
  function getDefaultCriticalAssetsMeta() {
    return {
      sectionTitle: "Actifs critiques",
      tableTitle: "Actifs critiques",
      subtitle: "Inventaire des actifs critiques et exigences de continuité.",
      metaLine: ""
    };
  }
  function normalizeCriticalAssetsMeta(meta) {
    const defaults = getDefaultCriticalAssetsMeta();
    const clean = { ...defaults };
    if (meta && typeof meta === "object") {
      ["sectionTitle", "tableTitle", "subtitle", "metaLine"].forEach((key) => {
        if (typeof meta[key] === "string") {
          const trimmed = meta[key].trim();
          if (key === "sectionTitle" || key === "tableTitle") {
            clean[key] = trimmed || defaults[key];
          } else {
            clean[key] = trimmed;
          }
        }
      });
    }
    return clean;
  }
  function normalizeCriticalAssetEntry(entry = {}, index = 0) {
    const clean = typeof entry === "object" && entry ? { ...entry } : {};
    if (!clean.id) clean.id = generateId("asset");
    const priority = parseInt(clean.priority, 10);
    clean.priority = Number.isFinite(priority) && priority > 0 ? priority : index + 1;
    const toText = (value) => typeof value === "string" ? value.trim() : "";
    clean.beneficiary = toText(clean.beneficiary);
    clean.productCode = toText(clean.productCode);
    clean.productName = toText(clean.productName);
    if (Array.isArray(clean.productTags)) {
      clean.productTags = clean.productTags.map((tag) => String(tag).trim()).filter(Boolean);
    } else if (typeof clean.productTags === "string") {
      clean.productTags = clean.productTags.split(",").map((tag) => tag.trim()).filter(Boolean);
    } else {
      clean.productTags = [];
    }
    clean.rto = toText(clean.rto);
    clean.rpo = toText(clean.rpo);
    clean.availability = toText(clean.availability);
    clean.availabilityNote = toText(clean.availabilityNote);
    clean.confidentiality = toText(clean.confidentiality);
    clean.impact = toText(clean.impact);
    const dacpRaw = clean.dacpSensitive;
    if (typeof dacpRaw === "boolean") {
      clean.dacpSensitive = dacpRaw ? "Oui" : "Non";
    } else {
      const dacpText = toText(dacpRaw);
      clean.dacpSensitive = dacpText || "Non";
    }
    clean.mfaStatus = toText(clean.mfaStatus);
    clean.mfaDetail = toText(clean.mfaDetail);
    return clean;
  }
  function normalizeCriticalAssets(list) {
    if (!Array.isArray(list)) return [];
    return list.map((entry, index) => normalizeCriticalAssetEntry(entry, index));
  }
  function normalizeCriticalAssetLinks(list) {
    if (!Array.isArray(list)) return;
    list.forEach((item) => {
      if (!item) return;
      if (Array.isArray(item.criticalAssetIds)) {
        item.criticalAssetIds = item.criticalAssetIds.map((id) => String(id || "").trim()).filter(Boolean);
      } else if (typeof item.criticalAssetIds === "string") {
        item.criticalAssetIds = item.criticalAssetIds.split(",").map((id) => id.trim()).filter(Boolean);
      } else {
        item.criticalAssetIds = [];
      }
    });
  }
  function ensureCriticalAssets() {
    appData.criticalAssets = normalizeCriticalAssets(appData.criticalAssets || []);
    appData.criticalAssetsMeta = normalizeCriticalAssetsMeta(appData.criticalAssetsMeta);
  }
  function getCriticalAssetLabel(asset) {
    if (!asset) return "";
    const code = asset.productCode ? asset.productCode.trim() : "";
    const name = asset.productName ? asset.productName.trim() : "";
    const beneficiary = asset.beneficiary ? asset.beneficiary.trim() : "";
    const parts = [];
    if (code) parts.push(code);
    if (name) parts.push(name);
    if (parts.length) return parts.join(" — ");
    if (beneficiary) return beneficiary;
    return asset.id || "";
  }
  function getCriticalAssetLookup() {
    return new Map((appData.criticalAssets || []).map((asset) => [asset.id, asset]));
  }
  function getCriticalAssetOptions(selectedIds = []) {
    const options = (appData.criticalAssets || []).map((asset) => ({
      value: asset.id,
      label: getCriticalAssetLabel(asset)
    }));
    const existing = new Set(options.map((opt) => opt.value));
    (Array.isArray(selectedIds) ? selectedIds : []).forEach((id) => {
      const clean = String(id || "").trim();
      if (clean && !existing.has(clean)) {
        options.push({ value: clean, label: clean });
        existing.add(clean);
      }
    });
    return options;
  }
  function buildCriticalAssetBadges(assetIds) {
    const ids = Array.isArray(assetIds) ? assetIds.filter(Boolean) : [];
    if (!ids.length) return "Aucun";
    const lookup = getCriticalAssetLookup();
    return ids.map((id) => {
      const asset = lookup.get(id);
      const label = asset ? getCriticalAssetLabel(asset) : id;
      return `<a href="#" class="link-badge link-badge--asset critical-asset-link" data-asset-id="${escapeHtml(id)}" title="Ouvrir l'actif critique">${escapeHtml(label || id)}</a>`;
    }).join(" ");
  }
  function attachCriticalAssetLinkHandlers(container) {
    if (!container) return;
    container.querySelectorAll(".critical-asset-link").forEach((link) => {
      link.addEventListener("click", (event) => {
        event.preventDefault();
        event.stopPropagation();
        const assetId = link.dataset.assetId;
        switchModule("critical-assets");
        setTimeout(() => {
          openCriticalAssetModal(assetId);
        }, 200);
      });
    });
  }
  function renderCriticalAssetsMeta() {
    ensureCriticalAssets();
    const meta = appData.criticalAssetsMeta || getDefaultCriticalAssetsMeta();
    safeSetText("criticalAssetsSectionTitle", meta.sectionTitle || "Actifs critiques");
    safeSetText("criticalAssetsTableTitle", meta.tableTitle || "Actifs critiques");
    safeSetText("criticalAssetsSectionSubtitle", meta.subtitle || "");
    const metaLine = document.getElementById("criticalAssetsMetaLine");
    if (metaLine) metaLine.textContent = meta.metaLine || "";
  }
  function updateCriticalAssetsFilterOptions() {
    const beneficiarySelect = document.getElementById("criticalAssetsBeneficiaryFilter");
    const availabilitySelect = document.getElementById("criticalAssetsAvailabilityFilter");
    const impactSelect = document.getElementById("criticalAssetsImpactFilter");
    if (!beneficiarySelect || !availabilitySelect || !impactSelect) return;
    const collator = new Intl.Collator("fr", { sensitivity: "base", numeric: true });
    const beneficiaries = new Set();
    const availability = new Set();
    const impacts = new Set();
    (appData.criticalAssets || []).forEach((asset) => {
      if (asset.beneficiary) beneficiaries.add(asset.beneficiary);
      if (asset.availability) availability.add(asset.availability);
      if (asset.impact) impacts.add(asset.impact);
    });
    const buildOptions = (select, values, label) => {
      const current = select.value || "all";
      const sorted = Array.from(values).sort((a, b) => collator.compare(a, b));
      select.innerHTML = "";
      const allOpt = document.createElement("option");
      allOpt.value = "all";
      allOpt.textContent = label;
      select.appendChild(allOpt);
      sorted.forEach((value) => {
        const opt = document.createElement("option");
        opt.value = value;
        opt.textContent = value;
        select.appendChild(opt);
      });
      const hasCurrent = sorted.includes(current);
      select.value = hasCurrent ? current : "all";
    };
    buildOptions(beneficiarySelect, beneficiaries, "Tous les bénéficiaires");
    buildOptions(availabilitySelect, availability, "Tous les niveaux");
    buildOptions(impactSelect, impacts, "Tous les impacts");
  }
  function resetCriticalAssetsFilters() {
    const beneficiarySelect = document.getElementById("criticalAssetsBeneficiaryFilter");
    const availabilitySelect = document.getElementById("criticalAssetsAvailabilityFilter");
    const impactSelect = document.getElementById("criticalAssetsImpactFilter");
    if (beneficiarySelect) beneficiarySelect.value = "all";
    if (availabilitySelect) availabilitySelect.value = "all";
    if (impactSelect) impactSelect.value = "all";
    loadCriticalAssetsTable();
  }
  function getAssetTagClass(tag) {
    const lower = String(tag || "").toLowerCase();
    if (lower.includes("obsol")) return "asset-tag asset-tag--alert";
    return "asset-tag";
  }
  function loadCriticalAssetsTable() {
    const tbody = document.getElementById("criticalAssetsTableBody");
    if (!tbody) return;
    ensureCriticalAssets();
    renderCriticalAssetsMeta();
    updateCriticalAssetsFilterOptions();
    const beneficiaryFilter = document.getElementById("criticalAssetsBeneficiaryFilter")?.value || "all";
    const availabilityFilter = document.getElementById("criticalAssetsAvailabilityFilter")?.value || "all";
    const impactFilter = document.getElementById("criticalAssetsImpactFilter")?.value || "all";
    const filtered = (appData.criticalAssets || []).filter((asset) => {
      if (beneficiaryFilter !== "all" && asset.beneficiary !== beneficiaryFilter) return false;
      if (availabilityFilter !== "all" && asset.availability !== availabilityFilter) return false;
      if (impactFilter !== "all" && asset.impact !== impactFilter) return false;
      return true;
    });
    const sorted = filtered.slice().sort((a, b) => (a.priority || 0) - (b.priority || 0));
    tbody.innerHTML = "";
    if (!sorted.length) {
      const row = document.createElement("tr");
      row.innerHTML = `<td colspan="10"><div class="nis2-empty">Aucun actif critique.</div></td>`;
      tbody.appendChild(row);
      return;
    }
    sorted.forEach((asset, index) => {
      const row = document.createElement("tr");
      const tagsHtml = (asset.productTags || []).map((tag) => `<span class="${getAssetTagClass(tag)}">${escapeHtml(tag)}</span>`).join(" ");
      const productHtml = `
        <div class="asset-product">
          ${tagsHtml ? `<div class="asset-product__tags">${tagsHtml}</div>` : ""}
          <div class="asset-product__code">${escapeHtml(asset.productCode || "")}</div>
          <div class="asset-product__name">${escapeHtml(asset.productName || "")}</div>
        </div>
      `;
      const availabilityNote = asset.availabilityNote ? `<div class="asset-availability__note">${escapeHtml(asset.availabilityNote)}</div>` : "";
      const availabilityHtml = `
        <div>${escapeHtml(asset.availability || "")}</div>
        ${availabilityNote}
      `;
      const confHtml = asset.confidentiality ? `<span class="asset-tag asset-tag--conf">${escapeHtml(asset.confidentiality)}</span>` : "—";
      const impactHtml = asset.impact ? `<span class="asset-tag asset-tag--impact">${escapeHtml(asset.impact)}</span>` : "—";
      const dacpValue = asset.dacpSensitive || "";
      const dacpClass = dacpValue === "Oui" ? "asset-tag asset-tag--yes" : "asset-tag asset-tag--no";
      const dacpHtml = dacpValue ? `<span class="${dacpClass}">${escapeHtml(dacpValue)}</span>` : "—";
      const mfaStatus = asset.mfaStatus || "";
      const mfaDetail = asset.mfaDetail || "";
      let mfaStatusClass = "asset-tag";
      if (mfaStatus) {
        const lower = mfaStatus.toLowerCase();
        mfaStatusClass = lower.includes("non") || lower.includes("partiel") ? "asset-tag asset-tag--mfa" : "asset-tag asset-tag--yes";
      }
      const mfaStatusHtml = mfaStatus ? `<span class="${mfaStatusClass}">${escapeHtml(mfaStatus)}</span>` : "";
      const mfaDetailHtml = mfaDetail ? `<span class="asset-tag">${escapeHtml(mfaDetail)}</span>` : "";
      const mfaHtml = (mfaStatusHtml || mfaDetailHtml)
        ? `<div class="asset-mfa">${mfaStatusHtml}${mfaDetailHtml}</div>`
        : "—";
      row.innerHTML = `
        <td><span class="asset-priority">${escapeHtml(asset.priority || index + 1)}</span></td>
        <td>${escapeHtml(asset.beneficiary || "")}</td>
        <td>${productHtml}</td>
        <td>${asset.rto ? `<span class="asset-pill">${escapeHtml(asset.rto)}</span>` : "—"}</td>
        <td>${asset.rpo ? `<span class="asset-pill">${escapeHtml(asset.rpo)}</span>` : "—"}</td>
        <td>${availabilityHtml}</td>
        <td>${confHtml}</td>
        <td>${impactHtml}</td>
        <td>${dacpHtml}</td>
        <td>${mfaHtml}</td>
      `;
      row.addEventListener("click", () => {
        openCriticalAssetModal(asset.id);
      });
      tbody.appendChild(row);
    });
  }
  function openCriticalAssetsTitlesModal() {
    ensureCriticalAssets();
    const meta = appData.criticalAssetsMeta || getDefaultCriticalAssetsMeta();
    openModal("Renommer la section", [
      { name: "sectionTitle", label: "Titre de la section", value: meta.sectionTitle },
      { name: "tableTitle", label: "Titre du tableau", value: meta.tableTitle },
      { name: "subtitle", label: "Sous-titre", value: meta.subtitle },
      { name: "metaLine", label: "Ligne d'information", value: meta.metaLine }
    ], (data) => {
      appData.criticalAssetsMeta = normalizeCriticalAssetsMeta(data);
      saveData();
      renderCriticalAssetsMeta();
      return true;
    });
  }
  function ensureSelectOptions(options, value) {
    if (!value) return options;
    return options.some((opt) => opt.value === value)
      ? options
      : [...options, { value, label: value }];
  }
  function openCriticalAssetModal(assetId = "") {
    ensureCriticalAssets();
    const asset = (appData.criticalAssets || []).find((a) => a.id === assetId);
    const isEdit = !!asset;
    const current = asset || normalizeCriticalAssetEntry({}, (appData.criticalAssets || []).length);
    const availabilityOptions = ensureSelectOptions(CRITICAL_ASSET_AVAILABILITY_OPTIONS, current.availability);
    const confOptions = ensureSelectOptions(CRITICAL_ASSET_CONF_OPTIONS, current.confidentiality);
    const impactOptions = ensureSelectOptions(CRITICAL_ASSET_IMPACT_OPTIONS, current.impact);
    const mfaOptions = ensureSelectOptions(CRITICAL_ASSET_MFA_STATUS_OPTIONS, current.mfaStatus);
    const dacpOptions = ensureSelectOptions(CRITICAL_ASSET_DACP_OPTIONS, current.dacpSensitive);
    openModal(isEdit ? "Modifier un actif critique" : "Nouvel actif critique", [
      { name: "priority", label: "Priorité", type: "number", value: current.priority },
      { name: "beneficiary", label: "Bénéficiaire", value: current.beneficiary },
      { name: "productCode", label: "Produit (code)", value: current.productCode },
      { name: "productName", label: "Produit (nom)", value: current.productName },
      { name: "productTags", label: "Tags produit (séparés par des virgules)", value: (current.productTags || []).join(", ") },
      { name: "rto", label: "RTO", value: current.rto },
      { name: "rpo", label: "RPO", value: current.rpo },
      { name: "availability", label: "Disponibilité", type: "select", value: current.availability, options: availabilityOptions },
      { name: "availabilityNote", label: "Note disponibilité", value: current.availabilityNote },
      { name: "confidentiality", label: "Confidentialité", type: "select", value: current.confidentiality, options: confOptions },
      { name: "impact", label: "Impact", type: "select", value: current.impact, options: impactOptions },
      { name: "dacpSensitive", label: "DACP sensible", type: "select", value: current.dacpSensitive, options: dacpOptions },
      { name: "mfaStatus", label: "Obligation MFA", type: "select", value: current.mfaStatus, options: mfaOptions },
      { name: "mfaDetail", label: "Détail MFA", value: current.mfaDetail }
    ], (data) => {
      const clean = normalizeCriticalAssetEntry({
        id: current.id,
        priority: data.priority || current.priority,
        beneficiary: data.beneficiary,
        productCode: data.productCode,
        productName: data.productName,
        productTags: data.productTags,
        rto: data.rto,
        rpo: data.rpo,
        availability: data.availability,
        availabilityNote: data.availabilityNote,
        confidentiality: data.confidentiality,
        impact: data.impact,
        dacpSensitive: data.dacpSensitive,
        mfaStatus: data.mfaStatus,
        mfaDetail: data.mfaDetail
      }, 0);
      if (isEdit) {
        const idx = appData.criticalAssets.findIndex((a) => a.id === assetId);
        if (idx >= 0) appData.criticalAssets[idx] = { ...appData.criticalAssets[idx], ...clean };
      } else {
        appData.criticalAssets.push(clean);
      }
      ensureCriticalAssets();
      saveData();
      loadCriticalAssetsTable();
      loadActionsTable();
      loadRisksTable();
      loadThreatsTable();
      loadNonConformitiesTable();
      return true;
    }, isEdit ? () => deleteCriticalAsset(assetId) : null);
  }
  function deleteCriticalAsset(assetId) {
    if (!confirm("Supprimer cet actif critique ?")) return false;
    appData.criticalAssets = (appData.criticalAssets || []).filter((asset) => asset.id !== assetId);
    if (Array.isArray(appData.actions)) {
      appData.actions.forEach((action) => {
        action.criticalAssetIds = (action.criticalAssetIds || []).filter((id) => id !== assetId);
      });
    }
    if (Array.isArray(appData.risks)) {
      appData.risks.forEach((risk) => {
        risk.criticalAssetIds = (risk.criticalAssetIds || []).filter((id) => id !== assetId);
      });
    }
    if (Array.isArray(appData.threats)) {
      appData.threats.forEach((threat) => {
        threat.criticalAssetIds = (threat.criticalAssetIds || []).filter((id) => id !== assetId);
      });
    }
    if (Array.isArray(appData.nonconformities)) {
      appData.nonconformities.forEach((nc) => {
        nc.criticalAssetIds = (nc.criticalAssetIds || []).filter((id) => id !== assetId);
      });
    }
    ensureCriticalAssets();
    saveData();
    loadCriticalAssetsTable();
    loadActionsTable();
    loadRisksTable();
    loadThreatsTable();
    loadNonConformitiesTable();
    return true;
  }
  window.loadCriticalAssetsTable = loadCriticalAssetsTable;
  window.openCriticalAssetModal = openCriticalAssetModal;

  // modules/project-risks.js
  const PROJECT_RISK_STATUS_OPTIONS = [
    { value: "red", label: "Rouge" },
    { value: "yellow", label: "Jaune" },
    { value: "green", label: "Vert" },
    { value: "todo", label: "À réaliser" },
    { value: "suspended", label: "Suspendue" }
  ];
  const PROJECT_RISK_STATUS_LABELS = {
    red: "Rouge",
    yellow: "Jaune",
    green: "Vert",
    todo: "À réaliser",
    suspended: "Suspendue"
  };
  const PROJECT_RISK_STATUS_BADGES = {
    red: "status-badge--rouge",
    yellow: "status-badge--jaune",
    green: "status-badge--vert",
    todo: "status-badge--a_realiser",
    suspended: "status-badge--suspendue"
  };
  function normalizeProjectRiskStatus(status) {
    const raw = NORMALIZE(String(status || "").trim()).replace(/[_-]+/g, " ");
    if (["rouge", "red"].includes(raw)) return "red";
    if (["jaune", "yellow"].includes(raw)) return "yellow";
    if (["vert", "green"].includes(raw)) return "green";
    if (["a realiser", "todo"].includes(raw)) return "todo";
    if (["suspendue", "suspendu", "suspended"].includes(raw)) return "suspended";
    return "yellow";
  }
  function normalizeProjectRiskRecommendations(list) {
    if (!Array.isArray(list)) return [];
    return list.map((rec) => {
      const clean = typeof rec === "object" && rec ? { ...rec } : {};
      if (!clean.id) clean.id = generateId("proj-reco");
      clean.title = typeof clean.title === "string" ? clean.title.trim() : "";
      clean.text = typeof clean.text === "string" ? clean.text.trim() : "";
      clean.implemented = !!clean.implemented;
      return clean;
    }).filter((rec) => rec.title || rec.text);
  }
  function normalizeProjectRiskEntry(entry = {}, index = 0) {
    const clean = typeof entry === "object" && entry ? { ...entry } : {};
    if (!clean.id) clean.id = generateId("proj-risk");
    const toText = (value) => typeof value === "string" ? value.trim() : "";
    clean.projectName = toText(clean.projectName);
    clean.beneficiary = toText(clean.beneficiary);
    clean.reportNumber = toText(clean.reportNumber);
    const revision = parseInt(clean.revisionNumber, 10);
    clean.revisionNumber = Number.isFinite(revision) ? revision : 0;
    clean.description = toText(clean.description);
    clean.analysisLink = toText(clean.analysisLink);
    clean.completionDate = toText(clean.completionDate);
    clean.status = normalizeProjectRiskStatus(clean.status);
    clean.recommendations = normalizeProjectRiskRecommendations(clean.recommendations);
    return clean;
  }
  function normalizeProjectRisks(list) {
    if (!Array.isArray(list)) return [];
    return list.map((entry, index) => normalizeProjectRiskEntry(entry, index));
  }
  function ensureProjectRisks() {
    appData.projectRisks = normalizeProjectRisks(appData.projectRisks || []);
  }
  function updateProjectRiskFilterOptions() {
    const beneficiarySelect = document.getElementById("projectRisksBeneficiaryFilter");
    const statusSelect = document.getElementById("projectRisksStatusFilter");
    if (!beneficiarySelect || !statusSelect) return;
    const collator = new Intl.Collator("fr", { sensitivity: "base", numeric: true });
    const beneficiaries = new Set();
    (appData.projectRisks || []).forEach((risk) => {
      if (risk.beneficiary) beneficiaries.add(risk.beneficiary);
    });
    const buildOptions = (select, values, label) => {
      const current = select.value || "all";
      const sorted = Array.from(values).sort((a, b) => collator.compare(a, b));
      select.innerHTML = "";
      const allOpt = document.createElement("option");
      allOpt.value = "all";
      allOpt.textContent = label;
      select.appendChild(allOpt);
      sorted.forEach((value) => {
        const opt = document.createElement("option");
        opt.value = value;
        opt.textContent = value;
        select.appendChild(opt);
      });
      select.value = sorted.includes(current) ? current : "all";
    };
    buildOptions(beneficiarySelect, beneficiaries, "Tous les bénéficiaires");
    const currentStatus = statusSelect.value || "all";
    statusSelect.innerHTML = "";
    const allStatus = document.createElement("option");
    allStatus.value = "all";
    allStatus.textContent = "Tous les statuts";
    statusSelect.appendChild(allStatus);
    PROJECT_RISK_STATUS_OPTIONS.forEach((opt) => {
      const option = document.createElement("option");
      option.value = opt.value;
      option.textContent = opt.label;
      statusSelect.appendChild(option);
    });
    const statusValues = PROJECT_RISK_STATUS_OPTIONS.map((opt) => opt.value);
    statusSelect.value = statusValues.includes(currentStatus) ? currentStatus : "all";
  }
  function resetProjectRiskFilters() {
    const searchInput = document.getElementById("projectRisksSearch");
    const beneficiarySelect = document.getElementById("projectRisksBeneficiaryFilter");
    const statusSelect = document.getElementById("projectRisksStatusFilter");
    const pendingFilter = document.getElementById("projectRisksPendingFilter");
    if (searchInput) searchInput.value = "";
    if (beneficiarySelect) beneficiarySelect.value = "all";
    if (statusSelect) statusSelect.value = "all";
    if (pendingFilter) pendingFilter.checked = false;
    loadProjectRisksTable();
  }
  function getProjectRiskStatusMeta(status) {
    const key = normalizeProjectRiskStatus(status);
    return {
      key,
      label: PROJECT_RISK_STATUS_LABELS[key] || key,
      badge: PROJECT_RISK_STATUS_BADGES[key] || "status-badge"
    };
  }
  function buildProjectRiskRecommendationRow(rec = {}, options = {}) {
    const row = document.createElement("div");
    row.className = "project-risk-reco";
    row.dataset.recoId = rec.id || generateId("proj-reco");
    row.innerHTML = `
      <div class="project-risk-reco__row">
        <label class="modal__label">Titre de recommandation</label>
        <input type="text" class="form-control project-risk-reco__title" value="${escapeHtml(rec.title || "")}" data-readonly-lock="true">
      </div>
      <div class="project-risk-reco__row">
        <label class="modal__label">Recommandation</label>
        <textarea class="form-control project-risk-reco__text" data-readonly-lock="true">${escapeHtml(rec.text || "")}</textarea>
      </div>
      <div class="project-risk-reco__meta">
        <label class="project-risk-reco__check">
          <input type="checkbox" class="project-risk-reco__implemented" ${rec.implemented ? "checked" : ""} data-readonly-lock="true">
          Implémenté
        </label>
        <button type="button" class="btn btn--outline btn--sm project-risk-reco__remove" data-readonly-lock="true">
          <i class="fas fa-trash"></i> Supprimer
        </button>
      </div>
    `;
    const removeBtn = row.querySelector(".project-risk-reco__remove");
    if (options.readOnly && removeBtn) {
      removeBtn.disabled = true;
    } else if (removeBtn) {
      removeBtn.addEventListener("click", () => {
        row.remove();
        if (typeof options.onRemove === "function") options.onRemove();
      });
    }
    return row;
  }
  function renderProjectRiskRecommendations(container, recommendations = []) {
    const section = document.createElement("div");
    section.className = "project-risk-recos";
    section.innerHTML = `
      <div class="project-risk-recos__header">
        <h4 class="project-risk-recos__title">Recommandations</h4>
        <button type="button" class="btn btn--outline btn--sm project-risk-recos__add" data-readonly-lock="true">
          <i class="fas fa-plus"></i> Ajouter
        </button>
      </div>
      <div class="project-risk-recos__list"></div>
      <p class="project-risk-recos__empty">Aucune recommandation ajoutée.</p>
    `;
    const list = section.querySelector(".project-risk-recos__list");
    const empty = section.querySelector(".project-risk-recos__empty");
    const addBtn = section.querySelector(".project-risk-recos__add");
    const readOnly = typeof isReadOnlyMode === "function" && isReadOnlyMode();
    const updateEmpty = () => {
      if (empty) empty.hidden = list.children.length > 0;
    };
    const addRow = (rec) => {
      list.appendChild(buildProjectRiskRecommendationRow(rec, { readOnly, onRemove: updateEmpty }));
      updateEmpty();
    };
    recommendations.forEach(addRow);
    updateEmpty();
    if (addBtn) {
      if (readOnly) {
        addBtn.disabled = true;
      } else {
        addBtn.addEventListener("click", () => addRow({}));
      }
    }
    container.appendChild(section);
    refreshReadOnlyLocks();
    return section;
  }
  function collectProjectRiskRecommendations(section) {
    if (!section) return [];
    const rows = section.querySelectorAll(".project-risk-reco");
    const recommendations = [];
    rows.forEach((row) => {
      const title = row.querySelector(".project-risk-reco__title")?.value.trim() || "";
      const text = row.querySelector(".project-risk-reco__text")?.value.trim() || "";
      const implemented = !!row.querySelector(".project-risk-reco__implemented")?.checked;
      if (!title && !text) return;
      recommendations.push({
        id: row.dataset.recoId || generateId("proj-reco"),
        title,
        text,
        implemented
      });
    });
    return recommendations;
  }
  function loadProjectRisksTable() {
    const tbody = document.getElementById("projectRisksTableBody");
    if (!tbody) return;
    ensureProjectRisks();
    updateProjectRiskFilterOptions();
    tbody.innerHTML = "";
    const searchQuery = NORMALIZE(document.getElementById("projectRisksSearch")?.value.trim() || "");
    const beneficiaryFilter = document.getElementById("projectRisksBeneficiaryFilter")?.value || "all";
    const statusFilter = document.getElementById("projectRisksStatusFilter")?.value || "all";
    const pendingOnly = !!document.getElementById("projectRisksPendingFilter")?.checked;
    const filtered = (appData.projectRisks || []).filter((risk) => {
      if (beneficiaryFilter !== "all" && risk.beneficiary !== beneficiaryFilter) return false;
      if (statusFilter !== "all" && normalizeProjectRiskStatus(risk.status) !== statusFilter) return false;
      if (pendingOnly) {
        const recs = risk.recommendations || [];
        if (!recs.some((rec) => !rec.implemented)) return false;
      }
      if (searchQuery) {
        const haystack = NORMALIZE([
          risk.projectName,
          risk.beneficiary,
          risk.reportNumber,
          risk.description,
          risk.analysisLink,
          risk.completionDate
        ].filter(Boolean).join(" "));
        if (!haystack.includes(searchQuery)) return false;
      }
      return true;
    });
    if (!filtered.length) {
      const row = document.createElement("tr");
      const message = appData.projectRisks.length
        ? "Aucun résultat."
        : "Aucune analyse de risque projet.";
      row.innerHTML = `<td colspan="8"><div class="nis2-empty">${message}</div></td>`;
      tbody.appendChild(row);
      return;
    }
    filtered.forEach((risk) => {
      const row = document.createElement("tr");
      const statusMeta = getProjectRiskStatusMeta(risk.status);
      row.classList.add("project-risk-row", `project-risk-row--${statusMeta.key}`);
      const analysisLink = risk.analysisLink ? escapeHtml(risk.analysisLink) : "";
      const linkHtml = analysisLink
        ? `<a class="project-risk-link" href="${analysisLink}" target="_blank" rel="noopener">${analysisLink}</a>`
        : "—";
      const rawDate = risk.completionDate || "";
      let dateLabel = "—";
      if (rawDate) {
        const dateValue = new Date(rawDate);
        dateLabel = !Number.isNaN(dateValue.getTime()) ? dateValue.toLocaleDateString() : rawDate;
      }
      row.innerHTML = `
        <td>
          <div class="project-risk-name">
            <span class="project-risk-indicator project-risk-indicator--${statusMeta.key}" aria-hidden="true"></span>
            <strong>${escapeHtml(risk.projectName || "")}</strong>
          </div>
        </td>
        <td>${escapeHtml(risk.beneficiary || "")}</td>
        <td>${escapeHtml(risk.reportNumber || "")}</td>
        <td>${escapeHtml(risk.revisionNumber || "")}</td>
        <td>${escapeHtml(risk.description || "")}</td>
        <td>${linkHtml}</td>
        <td>${escapeHtml(dateLabel)}</td>
        <td><span class="status-badge ${statusMeta.badge}">${escapeHtml(statusMeta.label)}</span></td>
      `;
      row.addEventListener("click", (e) => {
        if (e.target && e.target.closest(".project-risk-link")) return;
        openProjectRiskModal(risk.id);
      });
      tbody.appendChild(row);
    });
  }
  function openProjectRiskModal(riskId = "") {
    ensureProjectRisks();
    const risk = (appData.projectRisks || []).find((r) => r.id === riskId);
    const isEdit = !!risk;
    const current = risk || normalizeProjectRiskEntry({}, (appData.projectRisks || []).length);
    let recommendationsSection = null;
    openModal(isEdit ? "Modifier une analyse de risque projet" : "Nouvelle analyse de risque projet", [
      { name: "projectName", label: "Nom du Projet", value: current.projectName },
      { name: "beneficiary", label: "Bénéficiaire", value: current.beneficiary },
      { name: "reportNumber", label: "Numéro du rapport", value: current.reportNumber },
      { name: "revisionNumber", label: "Nombre de révision", type: "number", value: current.revisionNumber || 0 },
      { name: "description", label: "Description projet", type: "textarea", value: current.description },
      { name: "analysisLink", label: "Lien vers l'analyse des risques", type: "url", value: current.analysisLink },
      { name: "completionDate", label: "Date de réalisation", type: "date", value: current.completionDate },
      { name: "status", label: "Statut", type: "select", value: current.status, options: PROJECT_RISK_STATUS_OPTIONS }
    ], (data) => {
      if (!data.projectName) return false;
      const recommendations = collectProjectRiskRecommendations(recommendationsSection);
      const clean = normalizeProjectRiskEntry({
        id: current.id,
        projectName: data.projectName,
        beneficiary: data.beneficiary,
        reportNumber: data.reportNumber,
        revisionNumber: data.revisionNumber,
        description: data.description,
        analysisLink: data.analysisLink,
        completionDate: data.completionDate,
        status: data.status,
        recommendations
      }, 0);
      if (isEdit) {
        const idx = appData.projectRisks.findIndex((r) => r.id === riskId);
        if (idx >= 0) appData.projectRisks[idx] = { ...appData.projectRisks[idx], ...clean };
      } else {
        appData.projectRisks.push(clean);
      }
      saveData();
      loadProjectRisksTable();
    }, isEdit ? () => deleteProjectRisk(riskId) : null);
    const modalBody = document.getElementById("modalBody");
    if (modalBody) {
      recommendationsSection = renderProjectRiskRecommendations(modalBody, current.recommendations || []);
    }
  }
  function addProjectRisk() {
    openProjectRiskModal();
  }
  function deleteProjectRisk(riskId) {
    if (!confirm("Supprimer cette analyse de risque projet ?")) return false;
    appData.projectRisks = (appData.projectRisks || []).filter((risk) => risk.id !== riskId);
    saveData();
    loadProjectRisksTable();
    return true;
  }
  window.loadProjectRisksTable = loadProjectRisksTable;
  window.openProjectRiskModal = openProjectRiskModal;
  window.addProjectRisk = addProjectRisk;
  window.deleteProjectRisk = deleteProjectRisk;

  // modules/risks.js
  var IMPACT_LEVEL_OPTIONS = [
    { value: 1, label: "N\xE9gligeable" },
    { value: 2, label: "Mineure" },
    { value: 3, label: "Mod\xE9r\xE9e" },
    { value: 4, label: "Majeure" },
    { value: 5, label: "Grave" }
  ];
  var IMPACT_LEVEL_LABELS = {
    1: "N\xE9gligeable",
    2: "Mineure",
    3: "Mod\xE9r\xE9e",
    4: "Majeure",
    5: "Grave"
  };
  var PROBABILITY_LEVEL_OPTIONS = [
    { value: 1, label: "Tr\xE8s improbable" },
    { value: 2, label: "Improbable" },
    { value: 3, label: "Possible" },
    { value: 4, label: "Probable" },
    { value: 5, label: "Tr\xE8s probable" }
  ];
  var PROBABILITY_LEVEL_LABELS = {
    1: "Tr\xE8s improbable",
    2: "Improbable",
    3: "Possible",
    4: "Probable",
    5: "Tr\xE8s probable"
  };
  var TREATMENT_OPTIONS = [
    "R\xE9duction",
    "Acceptation",
    "Transfert",
    "\xC9vitement"
  ];
  function calculateRiskLevel(score) {
    if (score >= 20) return "Critique";
    if (score >= 15) return "Grave";
    if (score >= 10) return "Significatif";
    if (score >= 5) return "Mineur";
    return "Insignifiant";
  }
  function generateRiskId() {
    return generateId('risk');
  }
  function ensureRiskIds() {
    if (!Array.isArray(appData.risks)) appData.risks = [];
    appData.risks.forEach(risk => {
      if (!risk.id) risk.id = generateRiskId();
    });
  }
  function loadRisksTable() {
    const tbody = document.getElementById("risksTableBody");
    if (!tbody) return;
    tbody.innerHTML = "";
    const levelOrder = {
      "Critique": 0,
      "Grave": 1,
      "Significatif": 2,
      "Mineur": 3,
      "Insignifiant": 4
    };
    const sortedRisks = [...appData.risks].sort((a, b) => levelOrder[a.level] - levelOrder[b.level]);
    sortedRisks.forEach((risk) => {
      const row = document.createElement("tr");
      row.classList.add('risk-row--' + risk.level.toLowerCase());
      const linkedActions = appData.actions.filter((a) => actionLinksToRisk(a, risk.id));
      const linkedActionsHtml = linkedActions.length
        ? linkedActions.map((action) => `
          <a href="#" class="link-badge link-badge--risk risk-action-link" data-action-id="${escapeHtml(action.id)}" title="Ouvrir l'action">
            ${escapeHtml(action.title || action.id)}
          </a>
        `).join(" ")
        : "Aucune";
      const assetBadges = buildCriticalAssetBadges(risk.criticalAssetIds);
        const levelClasses = {
          "Critique": {
            badge: "status-badge--critique",
            icon: "fas fa-bolt"
          },
          "Grave": {
            badge: "status-badge--grave",
            icon: "fas fa-exclamation-triangle"
          },
          "Significatif": {
            badge: "status-badge--significatif",
            icon: "fas fa-exclamation-circle"
          },
          "Mineur": {
            badge: "status-badge--mineur",
            icon: "fas fa-info-circle"
          },
          "Insignifiant": {
            badge: "status-badge--insignifiant",
            icon: "fas fa-check"
          }
        };
      row.innerHTML = `
            <td>${escapeHtml(risk.id)}</td>
            <td><strong>${escapeHtml(risk.title)}</strong></td>
            <td>${escapeHtml(risk.description || "")}</td>
            <td>${IMPACT_LEVEL_LABELS[risk.impact] || risk.impact}</td>
            <td>${PROBABILITY_LEVEL_LABELS[risk.probability] || risk.probability}</td>
            <td>${risk.score}</td>
              <td><span class="status-badge ${levelClasses[risk.level].badge}"><i class="${levelClasses[risk.level].icon}"></i>${escapeHtml(risk.level)}</span></td>
            <td>${linkedActionsHtml}</td>
            <td>${assetBadges}</td>
            <td>${escapeHtml(risk.treatment || "")}</td>
            <td><input type="date" class="risk-codir-date" data-risk-id="${risk.id}" value="${risk.codirDate || ''}"></td>
            <td>${(risk.comments || []).length}</td>
        `;
      row.querySelectorAll(".risk-action-link").forEach((link) => {
        link.addEventListener("click", (event) => {
          event.preventDefault();
          event.stopPropagation();
          const actionId = link.dataset.actionId;
          if (!actionId) return;
          switchModule("actions");
          setTimeout(() => {
            editAction(actionId);
          }, 200);
        });
      });
      row.addEventListener("click", (e) => {
        if (e.target.closest(".icon-btn--danger") || e.target.closest('.risk-codir-date') || e.target.closest(".critical-asset-link")) return;
        editRisk(risk.id);
      });
      attachCriticalAssetLinkHandlers(row);
      tbody.appendChild(row);
    });
    tbody.querySelectorAll('.risk-codir-date').forEach((input) => {
      input.dataset.readonlyLock = "true";
      if (isReadOnlyMode()) input.disabled = true;
      input.addEventListener('input', (e) => {
        if (isReadOnlyMode()) {
          showReadOnlyNotice("mettre à jour la date CODIR");
          refreshReadOnlyLocks();
          return;
        }
        const r = appData.risks.find((rr) => rr.id === input.dataset.riskId);
        if (r) {
          r.codirDate = input.value;
          saveData();
        }
      });
    });
  }
  function editRisk(riskId) {
    const risk = appData.risks.find((r) => r.id === riskId);
    if (!risk) return;
    const criticalAssetIds = Array.isArray(risk.criticalAssetIds) ? risk.criticalAssetIds : [];
    const criticalAssetOptions = getCriticalAssetOptions(criticalAssetIds);
    openModal("Modifier un risque", [
      { name: "title", label: "Titre du risque", value: risk.title },
      { name: "description", label: "Description", type: "textarea", value: risk.description },
      { name: "impact", label: "Impact", type: "select", value: String(risk.impact), options: IMPACT_LEVEL_OPTIONS },
      { name: "probability", label: "Probabilit\xE9", type: "select", value: String(risk.probability), options: PROBABILITY_LEVEL_OPTIONS },
      { name: "level", label: "Niveau", value: calculateRiskLevel(risk.impact * risk.probability), readOnly: true },
      { name: "treatment", label: "Traitement", type: "select", value: risk.treatment, options: TREATMENT_OPTIONS },
      { name: "criticalAssetIds", label: "Actifs critiques", type: "multiselect", value: criticalAssetIds, options: criticalAssetOptions },
      { name: "existingComments", label: "Commentaires existants", type: "textarea", value: (risk.comments || []).map(c => `[${new Date(c.date).toLocaleString()}] ${c.text}`).join("\n"), readOnly: true },
      { name: "newComment", label: "Nouveau commentaire", type: "textarea" }
    ], (data) => {
      risk.title = data.title;
      risk.description = data.description;
      risk.impact = parseInt(data.impact, 10) || 1;
      risk.probability = parseInt(data.probability, 10) || 1;
      risk.score = risk.impact * risk.probability;
      risk.level = calculateRiskLevel(risk.score);
      risk.treatment = data.treatment || "";
      risk.criticalAssetIds = Array.isArray(data.criticalAssetIds) ? data.criticalAssetIds : [];
      if (data.newComment) {
        if (!Array.isArray(risk.comments)) risk.comments = [];
        risk.comments.push({ text: data.newComment, date: new Date().toISOString() });
      }
      saveData();
      loadRisksTable();
      updateDashboard();
    }, () => deleteRisk(riskId));
    const impactInput = document.getElementById("modal-input-impact");
    const probInput = document.getElementById("modal-input-probability");
    const levelInput = document.getElementById("modal-input-level");
    const updateLvl = () => {
      const score = (parseInt(impactInput.value, 10) || 1) * (parseInt(probInput.value, 10) || 1);
      levelInput.value = calculateRiskLevel(score);
    };
    if (impactInput && probInput && levelInput) {
      impactInput.addEventListener("input", updateLvl);
      probInput.addEventListener("input", updateLvl);
    }
  }
  function viewRiskDetails(riskId) {
    const risk = appData.risks.find((r) => r.id === riskId);
    if (!risk) return;
    const actionList = appData.actions
      .filter((a) => actionLinksToRisk(a, risk.id))
      .map((a) => a.id)
      .join(", ");
    alert(`Risque: ${risk.title}
Description: ${risk.description}
Impact: ${IMPACT_LEVEL_LABELS[risk.impact] || risk.impact}
Probabilit\xE9: ${PROBABILITY_LEVEL_LABELS[risk.probability] || risk.probability}
Score: ${risk.score}
Niveau: ${risk.level}
Actions li\xE9es: ${actionList || "Aucune"}
Traitement: ${risk.treatment || ""}`);
  }
  function addRisk() {
    const criticalAssetOptions = getCriticalAssetOptions();
    openModal("Ajouter un risque", [
      { name: "title", label: "Titre du risque" },
      { name: "description", label: "Description", type: "textarea" },
      { name: "impact", label: "Impact", type: "select", value: "1", options: IMPACT_LEVEL_OPTIONS },
      { name: "probability", label: "Probabilit\xE9", type: "select", value: "1", options: PROBABILITY_LEVEL_OPTIONS },
      { name: "level", label: "Niveau", value: calculateRiskLevel(1), readOnly: true },
      { name: "treatment", label: "Traitement", type: "select", options: TREATMENT_OPTIONS },
      { name: "criticalAssetIds", label: "Actifs critiques", type: "multiselect", value: [], options: criticalAssetOptions },
      { name: "comment", label: "Commentaire", type: "textarea" }
    ], (data) => {
      if (!data.title) return false;
      const id = generateRiskId();
      const impact = parseInt(data.impact, 10) || 1;
      const probability = parseInt(data.probability, 10) || 1;
      const score = impact * probability;
      appData.risks.push({
        id,
        title: data.title,
        description: data.description || "",
        impact,
        probability,
        score,
        level: calculateRiskLevel(score),
        treatment: data.treatment || "",
        codirDate: "",
        criticalAssetIds: Array.isArray(data.criticalAssetIds) ? data.criticalAssetIds : [],
        comments: data.comment ? [{ text: data.comment, date: new Date().toISOString() }] : []
      });
      saveData();
      loadRisksTable();
      updateRiskMatrix();
      updateDashboard();
    });
    const impactInput = document.getElementById("modal-input-impact");
    const probInput = document.getElementById("modal-input-probability");
    const levelInput = document.getElementById("modal-input-level");
    const updateLvl = () => {
      const score = (parseInt(impactInput.value, 10) || 1) * (parseInt(probInput.value, 10) || 1);
      levelInput.value = calculateRiskLevel(score);
    };
    if (impactInput && probInput && levelInput) {
      impactInput.addEventListener("input", updateLvl);
      probInput.addEventListener("input", updateLvl);
    }
  }
  function deleteRisk(riskId) {
    if (!confirm("Supprimer ce risque ?")) return;
    appData.risks = appData.risks.filter((r) => r.id !== riskId);
    saveData();
    loadRisksTable();
    updateRiskMatrix();
    updateDashboard();
  }

  function manageRiskComments(riskId) {
    if (!ensureEditMode("gérer les commentaires du risque")) return;
    const risk = appData.risks.find((r) => r.id === riskId);
    if (!risk) return;
    if (!Array.isArray(risk.comments)) risk.comments = [];
    let continueLoop = true;
    while (continueLoop) {
      const list = risk.comments.map((c, i) => `${i + 1}. [${new Date(c.date).toLocaleString()}] ${c.text}`).join("\n");
      const choice = prompt(`${list || "Aucun commentaire"}\n\n(A)jouter, (E)diter, (S)upprimer, (Q)uitter`);
      if (!choice) break;
      const ch = choice.toLowerCase();
      if (ch === "a") {
        if (risk.comments.length >= 10) {
          alert("Maximum 10 commentaires atteint");
          continue;
        }
        const text = prompt("Nouveau commentaire:");
        if (text) {
          risk.comments.push({ text, date: new Date().toISOString() });
          saveData();
        }
      } else if (ch === "e") {
        const idx = parseInt(prompt("Numéro du commentaire à éditer:"), 10) - 1;
        if (idx >= 0 && idx < risk.comments.length) {
          const text = prompt("Modifier le commentaire:", risk.comments[idx].text);
          if (text !== null) {
            risk.comments[idx].text = text;
            risk.comments[idx].date = new Date().toISOString();
            saveData();
          }
        }
      } else if (ch === "s") {
        const idx = parseInt(prompt("Numéro du commentaire à supprimer:"), 10) - 1;
        if (idx >= 0 && idx < risk.comments.length) {
          if (confirm("Supprimer ce commentaire ?")) {
            risk.comments.splice(idx, 1);
            saveData();
          }
        }
      } else if (ch === "q") {
        continueLoop = false;
      }
    }
    loadRisksTable();
  }
  window.loadRisksTable = loadRisksTable;
  window.editRisk = editRisk;
  window.viewRiskDetails = viewRiskDetails;
  window.addRisk = addRisk;
  window.deleteRisk = deleteRisk;
  window.manageRiskComments = manageRiskComments;

  // modules/threats.js
var THREAT_STATUS_LABELS = {
    reduction: "Réduction",
    acceptation: "Acceptation",
    transfert: "Transfert",
    evitement: "Évitement"
};
var THREAT_STATUS_OPTIONS = [
  { value: "reduction", label: "Réduction" },
  { value: "acceptation", label: "Acceptation" },
  { value: "transfert", label: "Transfert" },
  { value: "evitement", label: "Évitement" }
];
  function loadThreatsTable() {
    const tbody = document.getElementById("threatsTableBody");
    if (!tbody) return;
    tbody.innerHTML = "";
    const levelOrder = {
      "Critique": 0,
      "Grave": 1,
      "Significatif": 2,
      "Mineur": 3,
      "Insignifiant": 4
    };
    const sortedThreats = [...appData.threats].sort((a, b) => levelOrder[a.level] - levelOrder[b.level]);
    sortedThreats.forEach((threat) => {
      const row = document.createElement("tr");
      const linkedActionIds = appData.actions
        .filter((a) => a.linkType === "threat" && a.linkId === threat.id)
        .map((a) => a.id)
        .join(", ");
      const assetBadges = buildCriticalAssetBadges(threat.criticalAssetIds);
      const levelClasses = {
        "Critique": "status-badge--critique",
        "Grave": "status-badge--grave",
        "Significatif": "status-badge--significatif",
        "Mineur": "status-badge--mineur",
        "Insignifiant": "status-badge--insignifiant"
      };
      row.innerHTML = `
            <td>${escapeHtml(threat.title)}</td>
            <td>${escapeHtml(threat.description || "")}</td>
            <td>${IMPACT_LEVEL_LABELS[threat.impact] || threat.impact}</td>
            <td>${PROBABILITY_LEVEL_LABELS[threat.likelihood] || threat.likelihood}</td>
            <td>${threat.score}</td>
            <td><span class="status-badge ${levelClasses[threat.level]}">${escapeHtml(threat.level)}</span></td>
            <td>${escapeHtml(THREAT_STATUS_LABELS[threat.status] || threat.status || "")}</td>
            <td>${linkedActionIds || "Aucune"}</td>
            <td>${assetBadges}</td>
        `;
      row.addEventListener("click", (e) => {
        if (e.target.closest(".icon-btn--danger") || e.target.closest(".critical-asset-link")) return;
        editThreat(threat.id);
      });
      attachCriticalAssetLinkHandlers(row);
      tbody.appendChild(row);
    });
  }
  function showThreatForm(threatId = null) {
    const threat = threatId ? appData.threats.find((t) => t.id === threatId) : null;
    const isEdit = !!threat;
    const likelihood = threat ? threat.likelihood : 1;
    const impact = threat ? threat.impact : 1;
    const criticalAssetIds = threat && Array.isArray(threat.criticalAssetIds) ? threat.criticalAssetIds : [];
    const criticalAssetOptions = getCriticalAssetOptions(criticalAssetIds);
    openModal(isEdit ? "Modifier une menace" : "Ajouter une menace", [
      { name: "title", label: "Titre", value: threat ? threat.title : "" },
      { name: "context", label: "Contexte", value: threat ? threat.context : "" },
      { name: "description", label: "Description", type: "textarea", value: threat ? threat.description : "" },
      { name: "likelihood", label: "Probabilité", type: "select", value: String(likelihood), options: PROBABILITY_LEVEL_OPTIONS },
      { name: "impact", label: "Impact", type: "select", value: String(impact), options: IMPACT_LEVEL_OPTIONS },
      { name: "level", label: "Niveau", value: calculateRiskLevel(likelihood * impact), readOnly: true },
      { name: "status", label: "Statut", type: "select", value: threat ? threat.status : "reduction", options: THREAT_STATUS_OPTIONS },
      { name: "criticalAssetIds", label: "Actifs critiques", type: "multiselect", value: criticalAssetIds, options: criticalAssetOptions }
    ], (data) => {
      const li = parseInt(data.likelihood, 10) || 1;
      const im = parseInt(data.impact, 10) || 1;
      const score = li * im;
      if (isEdit) {
        threat.title = data.title;
        threat.context = data.context;
        threat.description = data.description;
        threat.likelihood = li;
        threat.impact = im;
        threat.score = score;
        threat.level = calculateRiskLevel(score);
        threat.status = data.status;
        threat.criticalAssetIds = Array.isArray(data.criticalAssetIds) ? data.criticalAssetIds : [];
        threat.updatedAt = new Date().toISOString();
      } else {
        const newThreat = {
          id: generateId("threat"),
          title: data.title,
          context: data.context || "",
          description: data.description || "",
          likelihood: li,
          impact: im,
          score,
          level: calculateRiskLevel(score),
          status: data.status || "reduction",
          criticalAssetIds: Array.isArray(data.criticalAssetIds) ? data.criticalAssetIds : [],
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString()
        };
        appData.threats.push(newThreat);
      }
      saveThreats();
      loadThreatsTable();
      updateDashboard();
    }, threat ? () => deleteThreat(threatId) : null);
    const likInput = document.getElementById("modal-input-likelihood");
    const impInput = document.getElementById("modal-input-impact");
    const lvlInput = document.getElementById("modal-input-level");
    const updateLvl = () => {
      const sc = (parseInt(impInput.value, 10) || 1) * (parseInt(likInput.value, 10) || 1);
      lvlInput.value = calculateRiskLevel(sc);
    };
    if (likInput && impInput && lvlInput) {
      likInput.addEventListener("input", updateLvl);
      impInput.addEventListener("input", updateLvl);
    }
  }
  function editThreat(threatId) {
    showThreatForm(threatId);
  }
  function viewThreatDetails(threatId) {
    const threat = appData.threats.find((t) => t.id === threatId);
    if (!threat) return;
    const actionList = appData.actions
      .filter((a) => a.linkType === "threat" && a.linkId === threat.id)
      .map((a) => a.id)
      .join(", ");
    alert(`Titre: ${threat.title}
Contexte: ${threat.context || ""}
Description: ${threat.description || ""}
Probabilit\xE9: ${PROBABILITY_LEVEL_LABELS[threat.likelihood] || threat.likelihood}
Impact: ${IMPACT_LEVEL_LABELS[threat.impact] || threat.impact}
Score: ${threat.score}
Niveau: ${threat.level}
Statut: ${THREAT_STATUS_LABELS[threat.status] || threat.status}
Actions li\xE9es: ${actionList || "Aucune"}`);
  }
  function deleteThreat(threatId) {
    if (!confirm("Supprimer cette menace ?")) return;
    appData.threats = appData.threats.filter((t) => t.id !== threatId);
    saveThreats();
    loadThreatsTable();
  }
  window.loadThreatsTable = loadThreatsTable;
  window.showThreatForm = showThreatForm;
  window.editThreat = editThreat;
  window.viewThreatDetails = viewThreatDetails;
  window.deleteThreat = deleteThreat;

  // modules/nonconformities.js
var STATUS_OPTIONS = [
  { value: "detection", label: "D\xE9tection" },
  { value: "analyse", label: "Analyse" },
  { value: "action", label: "Action" },
  { value: "cloture", label: "Cl\xF4ture" }
];
var NC_TYPE_OPTIONS = [
  { value: "Majeure", label: "Majeure" },
  { value: "Mineure", label: "Mineure" }
];
  function loadNonConformitiesTable() {
    const tbody = document.getElementById("ncTableBody");
    if (!tbody) return;
    tbody.innerHTML = "";
    updateNcAuditFilterOptions();
    updateNcOwnerFilterOptions();
    const auditFilter = document.getElementById("ncAuditFilter")?.value || "all";
    const ownerFilter = document.getElementById("ncOwnerFilter")?.value || "all";
    const sortBy = document.getElementById("ncSort")?.value || "type";
    const typeOrder = { Majeure: 0, Mineure: 1 };
    const statusOrder = STATUS_OPTIONS.reduce((acc, option, index) => {
      acc[option.value] = index;
      return acc;
    }, {});
    const auditMap = new Map((appData.audits || []).map((audit) => [audit.id, audit.title || audit.id]));
    const collator = new Intl.Collator("fr", { sensitivity: "base", numeric: true });
    const getDateValue = (date) => {
      const time = date ? new Date(date).getTime() : NaN;
      return Number.isFinite(time) ? time : 0;
    };
    const getAuditTitle = (nc) => auditMap.get(nc.auditId) || "";
    const sortedNCs = [...appData.nonconformities].sort((a, b) => {
      let diff = 0;
      switch (sortBy) {
        case "status":
          diff = (statusOrder[a.status] ?? 99) - (statusOrder[b.status] ?? 99);
          break;
        case "date":
          diff = getDateValue(b.detectionDate) - getDateValue(a.detectionDate);
          break;
        case "audit":
          diff = collator.compare(getAuditTitle(a), getAuditTitle(b));
          break;
        case "owner":
          diff = collator.compare(a.assignedTo || "", b.assignedTo || "");
          break;
        case "type":
        default:
          diff = (typeOrder[a.type] ?? 99) - (typeOrder[b.type] ?? 99);
          break;
      }
      return diff !== 0 ? diff : collator.compare(a.title || "", b.title || "");
    });
    sortedNCs.forEach((nc) => {
      if (auditFilter !== "all" && nc.auditId !== auditFilter) return;
      if (ownerFilter !== "all") {
        const assignedTo = (nc.assignedTo || "").trim();
        if (ownerFilter === "unassigned") {
          if (assignedTo) return;
        } else if (assignedTo !== ownerFilter) {
          return;
        }
      }
      const row = document.createElement("tr");
      const typeClass = nc.type === "Majeure" ? "status-badge--eleve" : "status-badge--moyen";
      const relatedActions = appData.actions.filter((a) => {
        return (a.ncIds || []).includes(nc.id) || a.linkType === "nc" && a.linkId === nc.id;
      });
      const actionsText = relatedActions.length ? relatedActions.map((a) => a.title).join(", ") : "Aucune";
      const assetBadges = buildCriticalAssetBadges(nc.criticalAssetIds);
      const audit = appData.audits.find(a => a.id === nc.auditId);
      const auditText = audit ? audit.title : "Aucun";
      row.innerHTML = `
            <td>${escapeHtml(nc.title)}</td>
            <td><span class="status-badge ${typeClass}">${escapeHtml(nc.type)}</span></td>
            <td>${escapeHtml(nc.status)}</td>
            <td>${nc.detectionDate ? new Date(nc.detectionDate).toLocaleDateString() : "N/A"}</td>
            <td>${nc.assignedTo ? escapeHtml(nc.assignedTo) : "N/A"}</td>
            <td>${escapeHtml(auditText)}</td>
            <td>${escapeHtml(actionsText)}</td>
            <td>${assetBadges}</td>
            <td>${escapeHtml(nc.description || "")}</td>
        `;
      row.addEventListener("click", (e) => {
        if (e.target.closest(".icon-btn--danger") || e.target.closest(".critical-asset-link")) return;
        editNC(nc.id);
      });
      attachCriticalAssetLinkHandlers(row);
      tbody.appendChild(row);
    });
  }
  function editNC(ncId) {
    const nc = appData.nonconformities.find((n) => n.id === ncId);
    if (!nc) return;
    const linkedActions = appData.actions.filter((a) => (a.ncIds || []).includes(nc.id)).map((a) => a.id);
    const criticalAssetIds = Array.isArray(nc.criticalAssetIds) ? nc.criticalAssetIds : [];
    const criticalAssetOptions = getCriticalAssetOptions(criticalAssetIds);
    openModal("Modifier une non-conformit\xE9", [
      { name: "title", label: "Titre", value: nc.title },
      { name: "type", label: "Type", type: "select", value: nc.type, options: NC_TYPE_OPTIONS },
      { name: "status", label: "Statut", type: "select", value: nc.status, options: STATUS_OPTIONS },
      { name: "detectionDate", label: "Date de d\xE9tection", type: "date", value: nc.detectionDate },
      {
        name: "assignedTo",
        label: "Responsable",
        type: "select",
        value: nc.assignedTo,
        options: appData.stakeholders.map((s) => ({ value: s.name, label: s.name }))
      },
      { name: "auditId", label: "Audit li\xE9", type: "select", value: nc.auditId, options: appData.audits.map(a => ({ value: a.id, label: a.title })) },
      { name: "criticalAssetIds", label: "Actifs critiques", type: "multiselect", value: criticalAssetIds, options: criticalAssetOptions },
      { name: "description", label: "Op\xE9rations", type: "textarea", value: nc.description },
    ], (data) => {
      nc.title = data.title;
      nc.type = data.type;
      nc.status = data.status;
      nc.detectionDate = data.detectionDate;
      nc.assignedTo = data.assignedTo;
      nc.auditId = data.auditId;
      nc.description = data.description;
      nc.criticalAssetIds = Array.isArray(data.criticalAssetIds) ? data.criticalAssetIds : [];
      saveData();
      loadNonConformitiesTable();
      updateDashboard();
    }, () => deleteNC(ncId));
  }
  function updateNcAuditFilterOptions() {
    const select = document.getElementById("ncAuditFilter");
    if (!select) return;
    const current = select.value || "all";
    const audits = Array.isArray(appData.audits) ? appData.audits : [];
    select.innerHTML = '<option value="all">Filtrer : tous les audits</option>'
      + audits.map((audit) => `<option value="${audit.id}">${escapeHtml(audit.title || audit.id)}</option>`).join("");
    const hasCurrent = audits.some((audit) => audit.id === current);
    select.value = hasCurrent ? current : "all";
  }
  function updateNcOwnerFilterOptions() {
    const select = document.getElementById("ncOwnerFilter");
    if (!select) return;
    const current = select.value || "all";
    const names = new Set();
    const addName = (value) => {
      const cleaned = typeof value === "string" ? value.trim() : "";
      if (cleaned) names.add(cleaned);
    };
    (Array.isArray(appData.stakeholders) ? appData.stakeholders : []).forEach((st) => addName(st.name));
    (Array.isArray(appData.nonconformities) ? appData.nonconformities : []).forEach((nc) => addName(nc.assignedTo));
    const sorted = Array.from(names).sort((a, b) => a.localeCompare(b, void 0, { sensitivity: "base" }));
    const hasUnassigned = (Array.isArray(appData.nonconformities) ? appData.nonconformities : []).some((nc) => !nc.assignedTo);
    select.innerHTML = '<option value="all">Filtrer : tous les responsables</option>'
      + (hasUnassigned ? '<option value="unassigned">Filtrer : non assigné</option>' : "")
      + sorted.map((name) => `<option value="${escapeHtml(name)}">${escapeHtml(name)}</option>`).join("");
    const hasCurrent = current === "unassigned" ? hasUnassigned : names.has(current);
    select.value = hasCurrent ? current : "all";
  }
  function deleteNC(ncId) {
    if (!confirm("Supprimer cette non-conformit\xE9 ?")) return;
    appData.nonconformities = appData.nonconformities.filter((n) => n.id !== ncId);
    saveData();
    loadNonConformitiesTable();
    updateDashboard();
  }
  function addNC() {
    const criticalAssetOptions = getCriticalAssetOptions();
    openModal("Ajouter une non-conformit\xE9", [
      { name: "title", label: "Titre" },
      { name: "type", label: "Type", type: "select", value: "Mineure", options: NC_TYPE_OPTIONS },
      { name: "status", label: "Statut", type: "select", value: "detection", options: STATUS_OPTIONS },
      { name: "detectionDate", label: "Date de d\xE9tection", type: "date", value: (/* @__PURE__ */ new Date()).toISOString().slice(0, 10) },
      {
        name: "assignedTo",
        label: "Responsable",
        type: "select",
        options: appData.stakeholders.map((s) => ({ value: s.name, label: s.name }))
      },
      { name: "auditId", label: "Audit li\xE9", type: "select", options: appData.audits.map(a => ({ value: a.id, label: a.title })) },
      { name: "criticalAssetIds", label: "Actifs critiques", type: "multiselect", value: [], options: criticalAssetOptions },
      { name: "description", label: "Op\xE9rations", type: "textarea" }
    ], (data) => {
      if (!data.title) return false;
      const id = generateId("nc");
      appData.nonconformities.push({
        id,
        title: data.title,
        description: data.description || "",
        type: data.type || "Mineure",
        status: data.status || "detection",
        detectionDate: data.detectionDate || (/* @__PURE__ */ new Date()).toISOString().slice(0, 10),
        assignedTo: data.assignedTo || "",
        auditId: data.auditId || "",
        criticalAssetIds: Array.isArray(data.criticalAssetIds) ? data.criticalAssetIds : []
      });
      saveData();
      loadNonConformitiesTable();
      updateDashboard();
    });
  }
  window.loadNonConformitiesTable = loadNonConformitiesTable;
  window.editNC = editNC;
  window.deleteNC = deleteNC;
  window.addNC = addNC;

  // modules/audits.js
var AUDIT_STATUS_OPTIONS = [
    { value: "planifie", label: "Planifi\xE9" },
    { value: "realise", label: "R\xE9alis\xE9" }
];
var AUDIT_TYPE_OPTIONS = [
    { value: "interne", label: "Audit interne" },
    { value: "externe", label: "Audit externe" }
];
  function loadAuditsTable() {
    const tbody = document.getElementById("auditsTableBody");
    if (!tbody) return;
    tbody.innerHTML = "";
    appData.audits.forEach((audit) => {
      const row = document.createElement("tr");
      const linkedActionIds = appData.actions
        .filter((a) => a.linkType === "audit" && a.linkId === audit.id)
        .map((a) => a.id)
        .join(", ");
      row.innerHTML = `
            <td>${escapeHtml(audit.title)}</td>
            <td>${escapeHtml(audit.type || "")}</td>
            <td>${escapeHtml(audit.scope)}</td>
            <td>${audit.plannedDate ? new Date(audit.plannedDate).toLocaleDateString() : "N/A"}</td>
            <td>${audit.auditor ? escapeHtml(audit.auditor) : "N/A"}</td>
            <td>${escapeHtml(audit.status)}</td>
            <td>${linkedActionIds || "Aucune"}</td>
        `;
      row.addEventListener("click", (e) => {
        if (e.target.closest(".icon-btn--danger")) return;
        editAudit(audit.id);
      });
      tbody.appendChild(row);
    });
  }
  function editAudit(auditId) {
    const audit = appData.audits.find((a) => a.id === auditId);
    if (!audit) return;
    openModal("Modifier un audit", [
      { name: "title", label: "Titre de l'audit", value: audit.title },
      { name: "type", label: "Type", type: "select", value: audit.type, options: AUDIT_TYPE_OPTIONS },
      { name: "scope", label: "P\xE9rim\xE8tre", value: audit.scope },
      { name: "plannedDate", label: "Date planifi\xE9e", value: audit.plannedDate, type: "date" },
      { name: "auditor", label: "Auditeur", value: audit.auditor },
      { name: "status", label: "Statut", type: "select", value: audit.status, options: AUDIT_STATUS_OPTIONS }
    ], (data) => {
      audit.title = data.title;
      audit.type = data.type;
      audit.scope = data.scope;
      audit.plannedDate = data.plannedDate;
      audit.auditor = data.auditor;
      audit.status = data.status;
      saveData();
      loadAuditsTable();
      updateDashboard();
    }, () => deleteAudit(auditId));
  }
  function deleteAudit(auditId) {
    if (!confirm("Supprimer cet audit ?")) return;
    appData.audits = appData.audits.filter((a) => a.id !== auditId);
    saveData();
    loadAuditsTable();
    updateDashboard();
  }
  function addAudit() {
    openModal("Ajouter un audit", [
      { name: "title", label: "Titre de l'audit" },
      { name: "type", label: "Type", type: "select", value: "interne", options: AUDIT_TYPE_OPTIONS },
      { name: "scope", label: "P\xE9rim\xE8tre" },
      { name: "plannedDate", label: "Date planifi\xE9e", type: "date" },
      { name: "auditor", label: "Auditeur" },
      { name: "status", label: "Statut", type: "select", value: "planifie", options: AUDIT_STATUS_OPTIONS }
    ], (data) => {
      if (!data.title) return false;
      const id = generateId("audit");
      appData.audits.push({
        id,
        title: data.title,
        type: data.type || "interne",
        description: "",
        scope: data.scope,
        plannedDate: data.plannedDate,
        auditor: data.auditor,
        status: data.status || "planifie"
      });
      saveData();
      loadAuditsTable();
      updateDashboard();
    });
  }
  window.loadAuditsTable = loadAuditsTable;
  window.editAudit = editAudit;
  window.deleteAudit = deleteAudit;
  window.addAudit = addAudit;

  // modules/reviews.js
  function loadReviewsTable() {
    const tbody = document.getElementById("reviewsTableBody");
    if (!tbody) return;
    tbody.innerHTML = "";
    appData.reviews.forEach((review) => {
      const row = document.createElement("tr");
      row.innerHTML = `
            <td>${review.date ? new Date(review.date).toLocaleDateString() : "N/A"}</td>
            <td>${escapeHtml(review.participants || "")}</td>
            <td>${escapeHtml(review.inputs || "")}</td>
            <td>${escapeHtml(review.decisions || "")}</td>
            <td>${(review.comments || []).length}</td>`;
      row.addEventListener("click", () => {
        editReview(review.id);
      });
      tbody.appendChild(row);
    });
  }
  function addReview() {
    openModal("Ajouter une revue de direction", [
      { name: "date", label: "Date", type: "date", value: (/* @__PURE__ */ new Date()).toISOString().slice(0, 10) },
      { name: "participants", label: "Participants" },
      { name: "inputs", label: "Entr\xE9es de la revue" },
      { name: "decisions", label: "D\xE9cisions et actions" },
      { name: "comment", label: "Commentaire", type: "textarea" }
    ], (data) => {
      if (!data.date) return false;
      const id = generateId("review");
      appData.reviews.push({
        id,
        date: data.date,
        participants: data.participants,
        inputs: data.inputs,
        decisions: data.decisions,
        comments: data.comment ? [{ text: data.comment, date: new Date().toISOString() }] : []
      });
      saveData();
      loadReviewsTable();
    });
  }
  function editReview(reviewId) {
    const review = appData.reviews.find((r) => r.id === reviewId);
    if (!review) return;
    openModal("Modifier une revue de direction", [
      { name: "date", label: "Date", type: "date", value: review.date },
      { name: "participants", label: "Participants", value: review.participants },
      { name: "inputs", label: "Entr\xE9es de la revue", value: review.inputs },
      { name: "decisions", label: "D\xE9cisions et actions", value: review.decisions },
      { name: "existingComments", label: "Commentaires existants", type: "textarea", value: (review.comments || []).map(c => `[${new Date(c.date).toLocaleString()}] ${c.text}`).join("\n"), readOnly: true },
      { name: "newComment", label: "Nouveau commentaire", type: "textarea" }
    ], (data) => {
      review.date = data.date;
      review.participants = data.participants;
      review.inputs = data.inputs;
      review.decisions = data.decisions;
      if (data.newComment) {
        if (!Array.isArray(review.comments)) review.comments = [];
        review.comments.push({ text: data.newComment, date: new Date().toISOString() });
      }
      saveData();
      loadReviewsTable();
    }, () => deleteReview(reviewId));
  }
  function deleteReview(reviewId) {
    if (!confirm("Supprimer cette revue ?")) return;
    appData.reviews = appData.reviews.filter((r) => r.id !== reviewId);
    saveData();
    loadReviewsTable();
  }
  function manageReviewComments(reviewId) {
    if (!ensureEditMode("gérer les commentaires de revue")) return;
    const review = appData.reviews.find((r) => r.id === reviewId);
    if (!review) return;
    if (!Array.isArray(review.comments)) review.comments = [];
    let continueLoop = true;
    while (continueLoop) {
      const list = review.comments.map((c, i) => `${i + 1}. [${new Date(c.date).toLocaleString()}] ${c.text}`).join("\n");
      const choice = prompt(`${list || "Aucun commentaire"}\n\n(A)jouter, (E)diter, (S)upprimer, (Q)uitter`);
      if (!choice) break;
      const ch = choice.toLowerCase();
      if (ch === "a") {
        if (review.comments.length >= 10) {
          alert("Maximum 10 commentaires atteint");
          continue;
        }
        const text = prompt("Nouveau commentaire:");
        if (text) {
          review.comments.push({ text, date: new Date().toISOString() });
          saveData();
        }
      } else if (ch === "e") {
        const idx = parseInt(prompt("Numéro du commentaire à éditer:"), 10) - 1;
        if (idx >= 0 && idx < review.comments.length) {
          const text = prompt("Modifier le commentaire:", review.comments[idx].text);
          if (text !== null) {
            review.comments[idx].text = text;
            review.comments[idx].date = new Date().toISOString();
            saveData();
          }
        }
      } else if (ch === "s") {
        const idx = parseInt(prompt("Numéro du commentaire à supprimer:"), 10) - 1;
        if (idx >= 0 && idx < review.comments.length) {
          if (confirm("Supprimer ce commentaire ?")) {
            review.comments.splice(idx, 1);
            saveData();
          }
        }
      } else if (ch === "q") {
        continueLoop = false;
      }
    }
    loadReviewsTable();
  }
  window.loadReviewsTable = loadReviewsTable;
  window.addReview = addReview;
  window.editReview = editReview;
  window.deleteReview = deleteReview;
  window.manageReviewComments = manageReviewComments;

  // modules/stakeholders.js
  function loadStakeholdersTable() {
    const tbody = document.getElementById("stakeholdersTableBody");
    if (!tbody) return;
    tbody.innerHTML = "";
    appData.stakeholders.forEach((st) => {
      const row = document.createElement("tr");
      row.innerHTML = `
            <td>${escapeHtml(st.name || "")}</td>
            <td>${escapeHtml(st.role || "")}</td>
            <td>${escapeHtml(st.contact || "")}</td>
            <td>${escapeHtml(st.notes || "")}</td>
        `;
      row.addEventListener("click", (e) => {
        if (e.target.closest(".icon-btn--danger")) return;
        editStakeholder(st.id);
      });
      tbody.appendChild(row);
    });
  }
  function addStakeholder() {
    openModal("Ajouter une partie prenante", [
      { name: "name", label: "Nom de la partie prenante" },
      { name: "role", label: "R\xF4le" },
      { name: "contact", label: "Contact" },
      { name: "notes", label: "Commentaires" }
    ], (data) => {
      if (!data.name) return false;
      const id = generateId("stakeholder");
      appData.stakeholders.push({
        id,
        name: data.name,
        role: data.role,
        contact: data.contact,
        notes: data.notes
      });
      saveData();
      loadStakeholdersTable();
    });
  }
  function editStakeholder(stakeholderId) {
    const st = appData.stakeholders.find((s) => s.id === stakeholderId);
    if (!st) return;
    const oldName = st.name;
    openModal("Modifier une partie prenante", [
      { name: "name", label: "Nom", value: st.name },
      { name: "role", label: "R\xF4le", value: st.role },
      { name: "contact", label: "Contact", value: st.contact },
      { name: "notes", label: "Commentaires", value: st.notes }
    ], (data) => {
      st.name = data.name;
      st.role = data.role;
      st.contact = data.contact;
      st.notes = data.notes;
      if (oldName !== data.name) {
        appData.actions.forEach(a => {
          if (a.assignedTo === oldName) a.assignedTo = data.name;
        });
        appData.nonconformities.forEach(nc => {
          if (nc.assignedTo === oldName) nc.assignedTo = data.name;
        });
      }
      saveData();
      loadStakeholdersTable();
      loadActionsTable();
      loadNonConformitiesTable();
    }, () => deleteStakeholder(stakeholderId));
  }
  function deleteStakeholder(stakeholderId) {
    if (!confirm("Supprimer cette partie prenante ?")) return;
    const st = appData.stakeholders.find((s) => s.id === stakeholderId);
    appData.stakeholders = appData.stakeholders.filter((s) => s.id !== stakeholderId);
    if (st) {
      appData.actions.forEach(a => {
        if (a.assignedTo === st.name) a.assignedTo = "";
      });
      appData.nonconformities.forEach(nc => {
        if (nc.assignedTo === st.name) nc.assignedTo = "";
      });
    }
    saveData();
    loadStakeholdersTable();
    loadActionsTable();
    loadNonConformitiesTable();
  }
  window.loadStakeholdersTable = loadStakeholdersTable;
  window.addStakeholder = addStakeholder;
  window.editStakeholder = editStakeholder;
function loadDocumentsTable() {
    const tbody = document.getElementById("documentsTableBody");
    if (!tbody) return;
    tbody.innerHTML = "";
    appData.documents.forEach((doc) => {
      const row = document.createElement("tr");
      row.innerHTML = `
            <td>${escapeHtml(doc.category || "")}</td>
            <td>${escapeHtml(doc.type || "")}</td>
            <td>${escapeHtml(doc.tool || "")}</td>
            <td>${doc.link ? `<a href="${escapeHtml(doc.link)}" target="_blank">${escapeHtml(doc.link)}</a>` : ""}</td>
        `;
      row.addEventListener("click", (e) => {
        if (e.target.closest(".icon-btn--danger")) return;
        editDocument(doc.id);
      });
      tbody.appendChild(row);
    });
}
function addDocument() {
    openModal("Ajouter un document", [
      { name: "category", label: "Niveau", type: "select", options: ["Politique", "Processus", "Procédure", "Enregistrement"] },
      { name: "type", label: "Type de livrable" },
      { name: "tool", label: "Outil utilisé" },
      { name: "link", label: "Lien d'accès" }
    ], (data) => {
      if (!data.type) return false;
      const id = generateId("doc");
      appData.documents.push({ id, category: data.category, type: data.type, tool: data.tool, link: data.link });
      saveData();
      loadDocumentsTable();
    });
}
function editDocument(id) {
    const doc = appData.documents.find((d) => d.id === id);
    if (!doc) return;
    openModal("Modifier un document", [
      { name: "category", label: "Niveau", type: "select", options: ["Politique", "Processus", "Procédure", "Enregistrement"], value: doc.category },
      { name: "type", label: "Type de livrable", value: doc.type },
      { name: "tool", label: "Outil utilisé", value: doc.tool },
      { name: "link", label: "Lien d'accès", value: doc.link }
    ], (data) => {
      doc.type = data.type;
      doc.category = data.category;
      doc.tool = data.tool;
      doc.link = data.link;
      saveData();
      loadDocumentsTable();
    }, () => deleteDocument(id));
}
function deleteDocument(id) {
    if (!confirm("Supprimer ce document ?")) return;
    appData.documents = appData.documents.filter((d) => d.id !== id);
    saveData();
    loadDocumentsTable();
}
window.loadDocumentsTable = loadDocumentsTable;
window.addDocument = addDocument;
window.editDocument = editDocument;
window.deleteDocument = deleteDocument;
  window.deleteStakeholder = deleteStakeholder;

  // modules/setup.js
  function setupNavigation() {
    const navItems = document.querySelectorAll(".nav__item");
    const modules = document.querySelectorAll(".module");
    navItems.forEach((item) => {
      item.addEventListener("click", function() {
        const tabId = this.getAttribute("data-tab");
        navItems.forEach((nav) => nav.classList.remove("nav__item--active"));
        this.classList.add("nav__item--active");
        modules.forEach((module) => module.classList.remove("module--active"));
        const targetModule = document.getElementById(tabId);
        if (targetModule) {
          targetModule.classList.add("module--active");
          loadModuleContent(tabId);
        } else {
          console.warn(`Module non trouvé: ${tabId}`);
        }
      });
    });
    const title = document.getElementById("appTitle");
    if (title) {
      title.addEventListener("click", () => {
        const dashboardBtn = document.querySelector('.nav__item[data-tab="dashboard"]');
        if (dashboardBtn) dashboardBtn.click();
      });
      title.addEventListener("keypress", (e) => {
        if (e.key === "Enter" || e.key === " ") {
          e.preventDefault();
          const dashboardBtn = document.querySelector('.nav__item[data-tab="dashboard"]');
          if (dashboardBtn) dashboardBtn.click();
        }
      });
    }
  }
  function setupKpiCardNavigation() {
    const cards = document.querySelectorAll(".kpi-card[data-nav-target]");
    const goTo = (target) => {
      if (!target) return;
      if (typeof switchModule === "function") {
        switchModule(target);
        return;
      }
      const navBtn = document.querySelector(`.nav__item[data-tab="${target}"]`);
      if (navBtn) navBtn.click();
      else loadModuleContent(target);
    };
    cards.forEach((card) => {
      const target = card.dataset.navTarget;
      const handler = () => goTo(target);
      card.addEventListener("click", handler);
      card.addEventListener("keypress", (e) => {
        if (e.key === "Enter" || e.key === " ") {
          e.preventDefault();
          handler();
        }
      });
    });
  }
  function setupEventListeners() {
    const mapping = [
      { id: "addControlBtn", handler: addControl },
      { id: "addActionBtn", handler: addAction },
      { id: "addCriticalAssetBtn", handler: () => openCriticalAssetModal() },
      { id: "editCriticalAssetsTitlesBtn", handler: openCriticalAssetsTitlesModal },
      { id: "addProjectRiskBtn", handler: addProjectRisk },
      { id: "addNis2DomainBtn", handler: () => openNis2DomainModal() },
      { id: "addNis2ActionBtn", handler: () => openNis2ActionModal() },
      { id: "addNis2StatusBtn", handler: () => openNis2StatusModal() },
      { id: "addRiskBtn", handler: addRisk },
      { id: "addThreatBtn", handler: () => showThreatForm() },
      { id: "addNcBtn", handler: addNC },
      { id: "addAuditBtn", handler: addAudit },
      { id: "addReviewBtn", handler: addReview },
      { id: "addStakeholderBtn", handler: addStakeholder },
      { id: "addDocumentBtn", handler: addDocument },
      { id: "addKpiBtn", handler: addKpi },
      { id: "manualBtn", handler: openManual },
      { id: "bugReportBtn", handler: openBugReport },
      { id: "settingsBtn", handler: openSettings },
      { id: "serverSyncBtn", handler: handleServerSyncButton },
      { id: "editModeBtn", handler: handleEditModeButton },
      { id: "themeToggleBtn", handler: toggleTheme },
      { id: "exportSoaPdfPublic", handler: () => exportSoaPdf("public") },
      { id: "exportSoaPdfAudit", handler: () => exportSoaPdf("audit") },
      { id: "exportDocReviewPdfAudit", handler: exportDocReviewPdf }
    ];
    mapping.forEach(({ id, handler }) => {
      const el = document.getElementById(id);
      if (el) el.addEventListener("click", handler);
    });
    const catFilter = document.getElementById("controlsCategoryFilter");
    const statusFilter = document.getElementById("controlsStatusFilter");
    const cmmFilter = document.getElementById("controlsCmmFilter");
    const capabilityFilter = document.getElementById("controlsCapabilityFilter");
    const nis2FunctionFilter = document.getElementById("nis2FunctionFilter");
    const nis2CategoryFilter = document.getElementById("nis2CategoryFilter");
    const nis2StatusFilter = document.getElementById("nis2StatusFilter");
    const nis2CmmFilter = document.getElementById("nis2CmmFilter");
    const nis2PhaseFilter = document.getElementById("nis2PhaseFilter");
    const nis2PillarFilter = document.getElementById("nis2PillarFilter");
    const soaFilter = document.getElementById("soaApplicabilityFilter");
    const soaDomainFilter = document.getElementById("soaDomainFilter");
    const docReviewStatusFilter = document.getElementById("docReviewStatusFilter");
    const actionsSort = document.getElementById("actionsSort");
    const actionsFilterType = document.getElementById("actionsFilterType");
    const actionsFilterValue = document.getElementById("actionsFilterValue");
    const criticalAssetsBeneficiaryFilter = document.getElementById("criticalAssetsBeneficiaryFilter");
    const criticalAssetsAvailabilityFilter = document.getElementById("criticalAssetsAvailabilityFilter");
    const criticalAssetsImpactFilter = document.getElementById("criticalAssetsImpactFilter");
    const criticalAssetsResetFilters = document.getElementById("criticalAssetsResetFilters");
    const projectRisksSearch = document.getElementById("projectRisksSearch");
    const projectRisksBeneficiaryFilter = document.getElementById("projectRisksBeneficiaryFilter");
    const projectRisksStatusFilter = document.getElementById("projectRisksStatusFilter");
    const projectRisksPendingFilter = document.getElementById("projectRisksPendingFilter");
    const projectRisksResetFilters = document.getElementById("projectRisksResetFilters");
    const ncSort = document.getElementById("ncSort");
    const ncAuditFilter = document.getElementById("ncAuditFilter");
    const ncOwnerFilter = document.getElementById("ncOwnerFilter");
    [catFilter, statusFilter, cmmFilter, capabilityFilter].forEach((el) => {
      if (el) el.addEventListener("change", loadControlsTable);
    });
    [nis2FunctionFilter, nis2CategoryFilter, nis2StatusFilter, nis2CmmFilter].forEach((el) => {
      if (el) el.addEventListener("change", loadNis2ControlsTable);
    });
    [nis2PhaseFilter, nis2PillarFilter].forEach((el) => {
      if (el) el.addEventListener("change", renderNis2Plan);
    });
    if (soaFilter) soaFilter.addEventListener("change", loadSoaTable);
    if (soaDomainFilter) soaDomainFilter.addEventListener("change", loadSoaTable);
    if (docReviewStatusFilter) docReviewStatusFilter.addEventListener("change", loadDocReviewTable);
    if (actionsSort) actionsSort.addEventListener("change", loadActionsTable);
    if (actionsFilterType) actionsFilterType.addEventListener("change", () => {
      updateActionsFilterOptions();
      loadActionsTable();
    });
    if (actionsFilterValue) actionsFilterValue.addEventListener("change", loadActionsTable);
    [criticalAssetsBeneficiaryFilter, criticalAssetsAvailabilityFilter, criticalAssetsImpactFilter].forEach((el) => {
      if (el) el.addEventListener("change", loadCriticalAssetsTable);
    });
    if (criticalAssetsResetFilters) criticalAssetsResetFilters.addEventListener("click", resetCriticalAssetsFilters);
    if (projectRisksSearch) projectRisksSearch.addEventListener("input", loadProjectRisksTable);
    [projectRisksBeneficiaryFilter, projectRisksStatusFilter].forEach((el) => {
      if (el) el.addEventListener("change", loadProjectRisksTable);
    });
    if (projectRisksPendingFilter) projectRisksPendingFilter.addEventListener("change", loadProjectRisksTable);
    if (projectRisksResetFilters) projectRisksResetFilters.addEventListener("click", resetProjectRiskFilters);
    if (ncSort) ncSort.addEventListener("change", loadNonConformitiesTable);
    if (ncAuditFilter) ncAuditFilter.addEventListener("change", loadNonConformitiesTable);
    if (ncOwnerFilter) ncOwnerFilter.addEventListener("change", loadNonConformitiesTable);
    setupExcelExportButtons();
    setupKpiCardNavigation();
    setupNis2PlanInteractions();
    setupDashboardInteractions();
    window.addEventListener("beforeunload", saveData);
  }

  // Interactions Dashboard (filtres, export, refresh)
  function setupDashboardInteractions() {
    // Filtres temporels
    const chartFilters = document.getElementById('chartFilters');
    if (chartFilters) {
      chartFilters.querySelectorAll('.chart-filter').forEach(btn => {
        btn.addEventListener('click', () => {
          chartFilters.querySelectorAll('.chart-filter').forEach(b => b.classList.remove('chart-filter--active'));
          btn.classList.add('chart-filter--active');
          // Pour l'instant, on garde juste l'état visuel
          // Une implémentation complète nécessiterait de stocker l'historique des données
          showToast('Filtre appliqué : ' + btn.textContent, 'info');
        });
      });
    }

    // Bouton refresh activité
    const refreshBtn = document.getElementById('refreshActivityBtn');
    if (refreshBtn) {
      refreshBtn.addEventListener('click', () => {
        updateActivityTimeline();
        showToast('Activité actualisée', 'success');
      });
    }

    // Bouton export PDF
    const exportBtn = document.getElementById('exportDashboardBtn');
    if (exportBtn) {
      exportBtn.addEventListener('click', exportDashboardPDF);
    }
  }

  // Export Dashboard en PDF
  function exportDashboardPDF() {
    showToast('Préparation du PDF...', 'info');

    // Utiliser html2canvas si disponible, sinon utiliser la fonction print
    if (typeof html2canvas !== 'undefined') {
      const dashboard = document.getElementById('dashboard');
      html2canvas(dashboard, {
        scale: 2,
        useCORS: true,
        logging: false
      }).then(canvas => {
        const link = document.createElement('a');
        link.download = 'dashboard-smsi-' + new Date().toISOString().split('T')[0] + '.png';
        link.href = canvas.toDataURL('image/png');
        link.click();
        showToast('Dashboard exporté en image', 'success');
      }).catch(err => {
        console.error('Erreur export:', err);
        fallbackPrintPDF();
      });
    } else {
      fallbackPrintPDF();
    }
  }

  function fallbackPrintPDF() {
    // Créer une version imprimable
    const printContent = document.getElementById('dashboard').cloneNode(true);

    // Créer une fenêtre d'impression
    const printWindow = window.open('', '_blank');
    printWindow.document.write(`
      <!DOCTYPE html>
      <html>
      <head>
        <title>Dashboard SMSI - Export</title>
        <link rel="stylesheet" href="style.css">
        <style>
          body { background: white; padding: 20px; }
          .dashboard-footer { display: none; }
          @media print {
            body { -webkit-print-color-adjust: exact; print-color-adjust: exact; }
          }
        </style>
      </head>
      <body class="${document.body.className}">
        ${printContent.outerHTML}
      </body>
      </html>
    `);
    printWindow.document.close();

    setTimeout(() => {
      printWindow.print();
      showToast('Utilisez "Enregistrer en PDF" dans la boîte de dialogue', 'info');
    }, 500);
  }

  // ============================================================
  // EXPORT PDF - SOA (Déclaration d'applicabilité) - ISO 27001:2022 Clause 6.1.3
  // ============================================================

  function exportSoaPdf(type = "public") {
    const isAudit = type === "audit";
    const orgName = appData.settings?.organizationName || "Organisation";
    const today = new Date().toLocaleDateString('fr-FR');
    const docVersion = appData.settings?.soaVersion || "1.0";
    const docDate = new Date().toISOString().split('T')[0];

    // Préparer les données
    const sortedControls = [...appData.controls].sort((a, b) =>
      a.reference.localeCompare(b.reference, void 0, { numeric: true })
    );

    // Stats avancées
    let applicable = 0, notApplicable = 0, implemented = 0, verified = 0;
    let withOwner = 0, withEvidence = 0, withRisks = 0, reviewedLast6Months = 0;
    const domainStats = {};
    const sourceStats = {};
    const sixMonthsAgo = new Date();
    sixMonthsAgo.setMonth(sixMonthsAgo.getMonth() - 6);

    sortedControls.forEach(control => {
      const entry = appData.soa.find(e => e.controlId === control.id) || {};
      const domain = getControlDomain(control.reference);
      if (!domainStats[domain]) domainStats[domain] = { total: 0, applicable: 0, implemented: 0, verified: 0 };
      domainStats[domain].total++;

      if (entry.applicable !== false) {
        applicable++;
        domainStats[domain].applicable++;
        if (entry.implStatus === "implemented" || entry.implStatus === "verified") {
          implemented++;
          domainStats[domain].implemented++;
          if (entry.implStatus === "verified") {
            verified++;
            domainStats[domain].verified++;
          }
        }
        if (entry.owner) withOwner++;
        if (entry.evidenceRef) withEvidence++;
        if (entry.linkedRiskIds && entry.linkedRiskIds.length > 0) withRisks++;
        if (entry.lastReviewDate && new Date(entry.lastReviewDate) >= sixMonthsAgo) reviewedLast6Months++;
        if (entry.source) {
          sourceStats[entry.source] = (sourceStats[entry.source] || 0) + 1;
        }
      } else {
        notApplicable++;
      }
    });

    const implPercent = applicable > 0 ? Math.round((implemented / applicable) * 100) : 0;
    const verifiedPercent = applicable > 0 ? Math.round((verified / applicable) * 100) : 0;
    const ownerPercent = applicable > 0 ? Math.round((withOwner / applicable) * 100) : 0;
    const evidencePercent = applicable > 0 ? Math.round((withEvidence / applicable) * 100) : 0;

    // Labels
    const sourceLabels = {
      risk: "Risque identifié", legal: "Exigence légale", contractual: "Exigence contractuelle",
      best_practice: "Bonne pratique", regulatory: "Exigence réglementaire", business: "Exigence métier"
    };
    const implLabels = {
      not_started: "Non démarré", planned: "Planifié", in_progress: "En cours",
      implemented: "Implémenté", verified: "Vérifié"
    };

    // Construire le HTML
    let html = `
      <!DOCTYPE html>
      <html lang="fr">
      <head>
        <meta charset="UTF-8">
        <title>Déclaration d'Applicabilité (SOA) - ${orgName}</title>
        <style>
          * { box-sizing: border-box; margin: 0; padding: 0; }
          body { font-family: 'Segoe UI', Arial, sans-serif; font-size: 9pt; line-height: 1.4; color: #333; padding: 15mm; }
          .header { text-align: center; margin-bottom: 15px; padding-bottom: 12px; border-bottom: 3px solid #0f9b7a; }
          .header h1 { font-size: 20pt; color: #0f9b7a; margin-bottom: 3px; }
          .header .subtitle { font-size: 11pt; color: #666; }
          .header .org { font-size: 16pt; font-weight: 700; margin: 8px 0; color: #1a1a1a; }
          .header .date { font-size: 9pt; color: #888; }
          .header .version { font-size: 10pt; color: #0f9b7a; font-weight: 600; margin-top: 5px; }

          /* Document Info Box */
          .doc-info { background: linear-gradient(135deg, #f8fffe 0%, #f0fdf4 100%); border: 1px solid #d1fae5; padding: 12px; border-radius: 8px; margin-bottom: 15px; display: flex; flex-wrap: wrap; gap: 15px; }
          .doc-info-item { flex: 1; min-width: 150px; }
          .doc-info-item .label { font-size: 7pt; color: #666; text-transform: uppercase; letter-spacing: 0.5px; }
          .doc-info-item .value { font-size: 10pt; font-weight: 600; color: #166534; }

          /* Executive Summary */
          .exec-summary { background: #1e3a5f; color: white; padding: 15px; border-radius: 8px; margin-bottom: 15px; }
          .exec-summary h2 { font-size: 12pt; margin-bottom: 10px; color: #93c5fd; }
          .exec-summary-grid { display: grid; grid-template-columns: repeat(4, 1fr); gap: 15px; }
          .exec-summary-item { text-align: center; }
          .exec-summary-item .number { font-size: 24pt; font-weight: 700; line-height: 1; }
          .exec-summary-item .number.good { color: #4ade80; }
          .exec-summary-item .number.medium { color: #fbbf24; }
          .exec-summary-item .number.low { color: #f87171; }
          .exec-summary-item .label { font-size: 8pt; color: #bfdbfe; margin-top: 3px; }

          /* Stats boxes */
          .stats-row { display: flex; gap: 10px; margin-bottom: 15px; }
          .stat-box { flex: 1; background: #f8f9fa; padding: 10px; border-radius: 6px; border-left: 3px solid #0f9b7a; }
          .stat-box h3 { font-size: 9pt; color: #666; margin-bottom: 8px; }
          .stat-box .items { display: flex; flex-wrap: wrap; gap: 5px; }
          .stat-item { background: white; padding: 4px 8px; border-radius: 4px; font-size: 8pt; }
          .stat-item strong { color: #0f9b7a; }

          /* Domain coverage matrix */
          .domain-matrix { margin-bottom: 15px; }
          .domain-matrix h3 { font-size: 10pt; margin-bottom: 8px; color: #1e3a5f; }
          .domain-matrix table { width: 100%; border-collapse: collapse; font-size: 8pt; }
          .domain-matrix th, .domain-matrix td { padding: 6px 8px; text-align: center; border: 1px solid #e5e7eb; }
          .domain-matrix th { background: #f3f4f6; font-weight: 600; }
          .domain-matrix .domain-name { text-align: left; font-weight: 600; }
          .progress-bar { width: 60px; height: 8px; background: #e5e7eb; border-radius: 4px; overflow: hidden; display: inline-block; }
          .progress-bar .fill { height: 100%; border-radius: 4px; }
          .progress-bar .fill.good { background: #22c55e; }
          .progress-bar .fill.medium { background: #f59e0b; }
          .progress-bar .fill.low { background: #ef4444; }

          /* Main table */
          table.soa-table { width: 100%; border-collapse: collapse; font-size: 8pt; margin-top: 10px; }
          table.soa-table th { background: #0f9b7a; color: white; padding: 6px 4px; text-align: left; font-weight: 600; font-size: 7pt; }
          table.soa-table td { padding: 5px 4px; border-bottom: 1px solid #e5e7eb; vertical-align: top; }
          table.soa-table tr:nth-child(even) { background: #fafafa; }
          table.soa-table tr.not-applicable { background: #f5f5f5; color: #888; }

          .badge { display: inline-block; padding: 1px 5px; border-radius: 3px; font-size: 7pt; font-weight: 600; }
          .badge-applicable { background: #dcfce7; color: #166534; }
          .badge-not-applicable { background: #f1f5f9; color: #64748b; }
          .badge-org { background: #dbeafe; color: #1e40af; }
          .badge-hum { background: #f3e8ff; color: #7c3aed; }
          .badge-phy { background: #dcfce7; color: #166534; }
          .badge-tech { background: #ffedd5; color: #c2410c; }

          .impl-status { font-size: 7pt; padding: 1px 4px; border-radius: 2px; }
          .impl-verified { background: #22c55e; color: white; }
          .impl-implemented { background: #3b82f6; color: white; }
          .impl-in_progress { background: #f59e0b; color: white; }
          .impl-planned { background: #8b5cf6; color: white; }
          .impl-not_started { background: #94a3b8; color: white; }

          .source-badge { font-size: 6pt; background: #e0e7ff; color: #3730a3; padding: 1px 3px; border-radius: 2px; }
          .risk-badge { display: inline-block; padding: 1px 3px; border-radius: 2px; margin: 1px; font-size: 6pt; }
          .risk-critique { background: #fee2e2; color: #991b1b; }
          .risk-grave { background: #ffedd5; color: #c2410c; }
          .risk-significatif { background: #fef9c3; color: #854d0e; }
          .risk-mineur { background: #dcfce7; color: #166534; }

          /* Signatures section */
          .signatures { margin-top: 25px; padding-top: 15px; border-top: 2px solid #e5e7eb; }
          .signatures h3 { font-size: 10pt; margin-bottom: 15px; color: #1e3a5f; }
          .signatures-grid { display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px; }
          .signature-box { border: 1px solid #d1d5db; padding: 15px; border-radius: 6px; min-height: 100px; }
          .signature-box .role { font-size: 9pt; font-weight: 600; color: #374151; margin-bottom: 5px; }
          .signature-box .line { border-bottom: 1px solid #9ca3af; margin: 30px 0 5px 0; }
          .signature-box .fields { font-size: 8pt; color: #6b7280; }

          /* Version history */
          .version-history { margin-top: 20px; padding: 12px; background: #f9fafb; border-radius: 6px; }
          .version-history h3 { font-size: 9pt; margin-bottom: 8px; color: #374151; }
          .version-history table { width: 100%; font-size: 8pt; border-collapse: collapse; }
          .version-history th, .version-history td { padding: 4px 8px; text-align: left; border-bottom: 1px solid #e5e7eb; }
          .version-history th { background: #e5e7eb; font-weight: 600; }

          .footer { margin-top: 20px; padding-top: 12px; border-top: 1px solid #ddd; font-size: 8pt; color: #888; text-align: center; }
          .footer .ref { font-weight: 600; color: #0f9b7a; }
          .footer .confidential { color: #dc2626; font-weight: 700; margin-top: 5px; }

          .page-break { page-break-after: always; }
          @media print {
            body { padding: 10mm; }
            .no-print { display: none; }
            .page-break { page-break-after: always; }
          }
        </style>
      </head>
      <body>
        <!-- Page 1: Header + Executive Summary -->
        <div class="header">
          <h1>Déclaration d'Applicabilité (SOA)</h1>
          <div class="subtitle">ISO 27001:2022 - Clause 6.1.3 - Annexe A</div>
          <div class="org">${escapeHtml(orgName)}</div>
          <div class="version">Version ${docVersion} | ${isAudit ? 'DOCUMENT AUDIT COMPLET' : 'VERSION PUBLIQUE'}</div>
          <div class="date">Date d'émission: ${today}</div>
        </div>

        <!-- Document Information -->
        <div class="doc-info">
          <div class="doc-info-item">
            <div class="label">Référence document</div>
            <div class="value">SOA-${docDate.replace(/-/g, '')}</div>
          </div>
          <div class="doc-info-item">
            <div class="label">Périmètre</div>
            <div class="value">SMSI ${escapeHtml(orgName)}</div>
          </div>
          <div class="doc-info-item">
            <div class="label">Norme de référence</div>
            <div class="value">ISO/IEC 27001:2022</div>
          </div>
          <div class="doc-info-item">
            <div class="label">Classification</div>
            <div class="value">${isAudit ? 'Confidentiel' : 'Interne'}</div>
          </div>
        </div>

        <!-- Executive Summary -->
        <div class="exec-summary">
          <h2>📊 Synthèse Exécutive</h2>
          <div class="exec-summary-grid">
            <div class="exec-summary-item">
              <div class="number">${applicable}/${sortedControls.length}</div>
              <div class="label">Contrôles applicables</div>
            </div>
            <div class="exec-summary-item">
              <div class="number ${implPercent >= 80 ? 'good' : implPercent >= 50 ? 'medium' : 'low'}">${implPercent}%</div>
              <div class="label">Taux d'implémentation</div>
            </div>
            <div class="exec-summary-item">
              <div class="number ${verifiedPercent >= 70 ? 'good' : verifiedPercent >= 40 ? 'medium' : 'low'}">${verifiedPercent}%</div>
              <div class="label">Contrôles vérifiés</div>
            </div>
            <div class="exec-summary-item">
              <div class="number">${notApplicable}</div>
              <div class="label">Non applicables</div>
            </div>
          </div>
        </div>

        ${isAudit ? `
        <!-- Detailed Stats for Audit -->
        <div class="stats-row">
          <div class="stat-box">
            <h3>📋 Qualité de la documentation</h3>
            <div class="items">
              <div class="stat-item"><strong>${ownerPercent}%</strong> avec propriétaire</div>
              <div class="stat-item"><strong>${evidencePercent}%</strong> avec preuves</div>
              <div class="stat-item"><strong>${withRisks}</strong> liés à des risques</div>
              <div class="stat-item"><strong>${reviewedLast6Months}</strong> revus &lt;6 mois</div>
            </div>
          </div>
          <div class="stat-box">
            <h3>🎯 Sources de sélection</h3>
            <div class="items">
              ${Object.entries(sourceStats).map(([src, count]) =>
                `<div class="stat-item"><strong>${count}</strong> ${sourceLabels[src] || src}</div>`
              ).join('')}
            </div>
          </div>
        </div>
        ` : ''}

        <!-- Domain Coverage Matrix -->
        <div class="domain-matrix">
          <h3>📈 Couverture par domaine ISO 27002:2022</h3>
          <table>
            <thead>
              <tr>
                <th class="domain-name">Domaine</th>
                <th>Total</th>
                <th>Applicables</th>
                <th>Implémentés</th>
                <th>Vérifiés</th>
                <th>Progression</th>
              </tr>
            </thead>
            <tbody>
              ${Object.entries(domainStats).map(([domain, stats]) => {
                const pct = stats.applicable > 0 ? Math.round((stats.implemented / stats.applicable) * 100) : 0;
                return `
                  <tr>
                    <td class="domain-name">${domain}</td>
                    <td>${stats.total}</td>
                    <td>${stats.applicable}</td>
                    <td>${stats.implemented}</td>
                    <td>${stats.verified || 0}</td>
                    <td>
                      <div class="progress-bar">
                        <div class="fill ${pct >= 80 ? 'good' : pct >= 50 ? 'medium' : 'low'}" style="width:${pct}%"></div>
                      </div>
                      ${pct}%
                    </td>
                  </tr>
                `;
              }).join('')}
            </tbody>
          </table>
        </div>

        <div class="page-break"></div>

        <!-- Main SOA Table -->
        <h3 style="font-size:11pt;color:#1e3a5f;margin-bottom:10px;">📝 Détail des contrôles ISO 27002:2022</h3>
        <table class="soa-table">
          <thead>
            <tr>
              <th style="width:40px;">Réf.</th>
              <th style="width:22%;">Contrôle</th>
              <th style="width:55px;">Domaine</th>
              <th style="width:50px;">Applic.</th>
              ${isAudit ? '<th style="width:60px;">Source</th>' : ''}
              <th style="width:60px;">État</th>
              ${isAudit ? '<th style="width:70px;">Propriétaire</th>' : ''}
              ${isAudit ? '<th style="width:70px;">Preuves</th>' : ''}
              ${isAudit ? '<th style="width:60px;">Risques</th>' : ''}
              <th>Justification</th>
            </tr>
          </thead>
          <tbody>
    `;

    sortedControls.forEach(control => {
      const entry = appData.soa.find(e => e.controlId === control.id) || {};
      const domain = getControlDomain(control.reference);
      const isApplicable = entry.applicable !== false;

      // Risques liés
      let risksHtml = "";
      if (isAudit && entry.linkedRiskIds && entry.linkedRiskIds.length > 0) {
        risksHtml = entry.linkedRiskIds.slice(0, 3).map(riskId => {
          const risk = appData.risks.find(r => r.id === riskId);
          if (!risk) return "";
          const levelClass = risk.level ? `risk-${risk.level.toLowerCase()}` : "";
          return `<span class="risk-badge ${levelClass}">${escapeHtml((risk.title || risk.id).substring(0, 15))}</span>`;
        }).join("");
        if (entry.linkedRiskIds.length > 3) risksHtml += `<span style="font-size:6pt;">+${entry.linkedRiskIds.length - 3}</span>`;
      }

      html += `
        <tr class="${isApplicable ? '' : 'not-applicable'}">
          <td><strong>${escapeHtml(control.reference)}</strong></td>
          <td>${escapeHtml(control.title)}</td>
          <td><span class="badge badge-${domain.substring(0,3).toLowerCase()}">${domain.substring(0,4)}</span></td>
          <td><span class="badge ${isApplicable ? 'badge-applicable' : 'badge-not-applicable'}">${isApplicable ? '✓' : '✗'}</span></td>
          ${isAudit ? `<td>${entry.source ? `<span class="source-badge">${(sourceLabels[entry.source] || entry.source).substring(0, 10)}</span>` : '-'}</td>` : ''}
          <td>${entry.implStatus ? `<span class="impl-status impl-${entry.implStatus}">${(implLabels[entry.implStatus] || '').substring(0, 8)}</span>` : '-'}</td>
          ${isAudit ? `<td style="font-size:7pt;">${entry.owner ? escapeHtml(entry.owner.substring(0, 15)) : '-'}</td>` : ''}
          ${isAudit ? `<td style="font-size:7pt;">${entry.evidenceRef ? escapeHtml(entry.evidenceRef.substring(0, 15)) : '-'}</td>` : ''}
          ${isAudit ? `<td>${risksHtml || '-'}</td>` : ''}
          <td style="font-size:7pt;">
            ${entry.justification ? escapeHtml(entry.justification.substring(0, 80)) + (entry.justification.length > 80 ? '...' : '') : '-'}
            ${!isApplicable && entry.exclusionReason ? `<div style="color:#dc2626;font-style:italic;">Exclu: ${escapeHtml(entry.exclusionReason.substring(0, 50))}</div>` : ''}
          </td>
        </tr>
      `;
    });

    html += `
          </tbody>
        </table>

        ${isAudit ? `
        <div class="page-break"></div>

        <!-- Signatures Section -->
        <div class="signatures">
          <h3>✍️ Approbations et signatures</h3>
          <div class="signatures-grid">
            <div class="signature-box">
              <div class="role">Responsable SMSI</div>
              <div class="line"></div>
              <div class="fields">
                Nom: ________________________________<br>
                Date: ________________________________<br>
                Signature:
              </div>
            </div>
            <div class="signature-box">
              <div class="role">Direction Générale</div>
              <div class="line"></div>
              <div class="fields">
                Nom: ________________________________<br>
                Date: ________________________________<br>
                Signature:
              </div>
            </div>
            <div class="signature-box">
              <div class="role">Responsable Sécurité</div>
              <div class="line"></div>
              <div class="fields">
                Nom: ________________________________<br>
                Date: ________________________________<br>
                Signature:
              </div>
            </div>
          </div>
        </div>

        <!-- Version History -->
        <div class="version-history">
          <h3>📜 Historique des versions</h3>
          <table>
            <thead>
              <tr>
                <th>Version</th>
                <th>Date</th>
                <th>Auteur</th>
                <th>Modifications</th>
              </tr>
            </thead>
            <tbody>
              <tr>
                <td>${docVersion}</td>
                <td>${today}</td>
                <td>RSSI</td>
                <td>Version actuelle - Génération automatique</td>
              </tr>
              <tr>
                <td>0.9</td>
                <td>-</td>
                <td>-</td>
                <td>Révision initiale</td>
              </tr>
            </tbody>
          </table>
        </div>
        ` : ''}

        <div class="footer">
          <div class="ref">Référence: SOA-${orgName.replace(/\s+/g, '-').toUpperCase()}-${docDate}</div>
          <div>Document généré par Cyber-Assistant | Conforme ISO 27001:2022 Clause 6.1.3</div>
          ${isAudit ? '<div class="confidential">⚠️ DOCUMENT CONFIDENTIEL - DIFFUSION RESTREINTE - USAGE AUDIT UNIQUEMENT</div>' : '<div>© ' + new Date().getFullYear() + ' ' + escapeHtml(orgName) + '</div>'}
        </div>
      </body>
      </html>
    `;

    // Ouvrir dans une nouvelle fenêtre pour impression/PDF
    const printWindow = window.open('', '_blank');
    if (!printWindow) {
      showToast('Popup bloqué - Autorisez les popups pour exporter en PDF', 'error');
      return;
    }
    printWindow.document.write(html);
    printWindow.document.close();

    setTimeout(() => {
      printWindow.print();
      showToast(`SOA ${isAudit ? 'Audit' : 'Public'} prête - Utilisez "Enregistrer en PDF"`, 'success');
    }, 300);
  }

  // ============================================================
  // EXPORT PDF - Revue Documentaire (Pack Audit ISO 27001)
  // ============================================================

  function exportDocReviewPdf() {
    const orgName = appData.settings?.organizationName || "Organisation";
    const today = new Date().toLocaleDateString('fr-FR');

    // Stats
    let conforme = 0, partiel = 0, nonConforme = 0, total = 0;
    DOCUMENT_REVIEW_ITEMS.filter(i => i.reference).forEach(item => {
      const entry = appData.documentReview.find(e => e.reference === item.reference) || {};
      total++;
      if (entry.status === "conforme") conforme++;
      else if (entry.status === "partiel") partiel++;
      else if (entry.status === "non_conforme") nonConforme++;
    });
    const conformePercent = total > 0 ? Math.round((conforme / total) * 100) : 0;

    let html = `
      <!DOCTYPE html>
      <html lang="fr">
      <head>
        <meta charset="UTF-8">
        <title>Revue Documentaire ISO 27001:2022 - ${orgName}</title>
        <style>
          * { box-sizing: border-box; margin: 0; padding: 0; }
          body { font-family: 'Segoe UI', Arial, sans-serif; font-size: 10pt; line-height: 1.4; color: #333; padding: 20mm; }
          .header { text-align: center; margin-bottom: 20px; padding-bottom: 15px; border-bottom: 2px solid #3b82f6; }
          .header h1 { font-size: 18pt; color: #3b82f6; margin-bottom: 5px; }
          .header .subtitle { font-size: 11pt; color: #666; }
          .header .org { font-size: 14pt; font-weight: 600; margin: 10px 0; }
          .header .date { font-size: 9pt; color: #888; }
          .header .audit-label { background: #dc2626; color: white; padding: 4px 12px; border-radius: 4px; display: inline-block; margin-top: 10px; font-weight: 600; }
          .meta-box { background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 20px; display: flex; flex-wrap: wrap; gap: 20px; }
          .meta-item { flex: 1; min-width: 100px; text-align: center; }
          .meta-item .label { font-size: 8pt; color: #666; text-transform: uppercase; }
          .meta-item .value { font-size: 18pt; font-weight: 700; }
          .meta-item .value.good { color: #22c55e; }
          .meta-item .value.warning { color: #f59e0b; }
          .meta-item .value.danger { color: #ef4444; }
          .chapter { background: #1e40af; color: white; padding: 8px 12px; margin: 15px 0 10px 0; font-weight: 600; font-size: 11pt; border-radius: 4px; }
          table { width: 100%; border-collapse: collapse; font-size: 9pt; }
          th { background: #3b82f6; color: white; padding: 8px 6px; text-align: left; font-weight: 600; }
          td { padding: 6px; border-bottom: 1px solid #e0e0e0; vertical-align: top; }
          tr:nth-child(even) { background: #fafafa; }
          tr.conforme { background: rgba(34, 197, 94, 0.08); }
          tr.partiel { background: rgba(245, 158, 11, 0.08); }
          tr.non-conforme { background: rgba(239, 68, 68, 0.08); }
          tr.na { background: #f5f5f5; opacity: 0.7; }
          .status { display: inline-block; padding: 3px 8px; border-radius: 4px; font-size: 8pt; font-weight: 600; }
          .status-conforme { background: #22c55e; color: white; }
          .status-partiel { background: #f59e0b; color: white; }
          .status-non-conforme { background: #ef4444; color: white; }
          .status-na { background: #94a3b8; color: white; }
          .status-empty { background: #e5e7eb; color: #6b7280; }
          .ref-col { width: 80px; font-weight: 600; }
          .desc-col { width: 30%; }
          .dates { font-size: 8pt; color: #666; }
          .footer { margin-top: 30px; padding-top: 15px; border-top: 1px solid #ddd; font-size: 8pt; color: #888; text-align: center; }
          .checklist { margin-top: 30px; page-break-before: always; }
          .checklist h2 { color: #1e40af; margin-bottom: 15px; }
          .checklist-item { display: flex; align-items: flex-start; padding: 8px 0; border-bottom: 1px solid #eee; }
          .checklist-box { width: 18px; height: 18px; border: 2px solid #3b82f6; border-radius: 3px; margin-right: 10px; flex-shrink: 0; }
          .checklist-text { flex: 1; }
          .checklist-status { font-size: 9pt; padding: 2px 6px; border-radius: 3px; }
          @media print {
            body { padding: 10mm; }
          }
        </style>
      </head>
      <body>
        <div class="header">
          <h1>Revue Documentaire SMSI</h1>
          <div class="subtitle">ISO 27001:2022 - Clauses 4 à 10</div>
          <div class="org">${escapeHtml(orgName)}</div>
          <div class="date">Généré le ${today}</div>
          <div class="audit-label">PACK AUDIT CERTIFICATION</div>
        </div>

        <div class="meta-box">
          <div class="meta-item">
            <div class="label">Conformes</div>
            <div class="value good">${conforme}</div>
          </div>
          <div class="meta-item">
            <div class="label">Partiels</div>
            <div class="value warning">${partiel}</div>
          </div>
          <div class="meta-item">
            <div class="label">Non conformes</div>
            <div class="value danger">${nonConforme}</div>
          </div>
          <div class="meta-item">
            <div class="label">Taux de conformité</div>
            <div class="value ${conformePercent >= 80 ? 'good' : conformePercent >= 50 ? 'warning' : 'danger'}">${conformePercent}%</div>
          </div>
        </div>

        <table>
          <thead>
            <tr>
              <th class="ref-col">Clause</th>
              <th class="desc-col">Exigence</th>
              <th style="width:80px;">Statut</th>
              <th style="width:80px;">Responsable</th>
              <th>Justification / Preuve</th>
              <th style="width:90px;">Revue</th>
            </tr>
          </thead>
          <tbody>
    `;

    let currentChapter = "";
    DOCUMENT_REVIEW_ITEMS.forEach(item => {
      if (item.type === "chapter") {
        currentChapter = item.title;
        html += `<tr><td colspan="6" class="chapter">${escapeHtml(item.title)}</td></tr>`;
      } else {
        const entry = appData.documentReview.find(e => e.reference === item.reference) || {};
        const statusLabels = {
          conforme: "Conforme", partiel: "Partiel", non_conforme: "Non conforme", na: "N/A"
        };
        const statusClass = entry.status || "empty";

        html += `
          <tr class="${entry.status || ''}">
            <td class="ref-col">${escapeHtml(item.reference)}</td>
            <td class="desc-col">${escapeHtml(item.description)}</td>
            <td><span class="status status-${statusClass}">${statusLabels[entry.status] || 'À évaluer'}</span></td>
            <td>${escapeHtml(entry.responsible || '-')}</td>
            <td>
              ${entry.justification ? escapeHtml(entry.justification) : '-'}
              ${entry.tool ? `<br><em style="color:#666;">Preuve: ${escapeHtml(entry.tool)}</em>` : ''}
              ${entry.documentLink ? `<br><a href="${escapeHtml(entry.documentLink)}" style="color:#3b82f6;font-size:8pt;">📎 Document</a>` : ''}
            </td>
            <td class="dates">
              ${entry.lastReview ? `Dernière: ${entry.lastReview}` : '-'}
              ${entry.nextReview ? `<br>Prochaine: ${entry.nextReview}` : ''}
            </td>
          </tr>
        `;
      }
    });

    html += `
          </tbody>
        </table>

        <!-- Checklist Auditeur -->
        <div class="checklist">
          <h2>📋 Checklist Préparation Audit ISO 27001:2022</h2>
          <p style="margin-bottom:15px;color:#666;">Points de contrôle pour l'auditeur externe</p>
    `;

    const auditChecklist = [
      { ref: "4", label: "Contexte de l'organisation", items: ["Enjeux internes/externes documentés", "Parties intéressées identifiées", "Domaine d'application défini"] },
      { ref: "5", label: "Leadership", items: ["Politique SSI signée et communiquée", "Rôles et responsabilités attribués", "Engagement de la direction documenté"] },
      { ref: "6", label: "Planification", items: ["Processus d'appréciation des risques", "Plan de traitement des risques", "Déclaration d'applicabilité (SOA)", "Objectifs SSI mesurables"] },
      { ref: "7", label: "Supports", items: ["Ressources allouées", "Compétences documentées", "Programme de sensibilisation", "Plan de communication", "Maîtrise documentaire"] },
      { ref: "8", label: "Fonctionnement", items: ["Processus opérationnels documentés", "Appréciations des risques à jour", "Traitement des risques en cours"] },
      { ref: "9", label: "Évaluation de la performance", items: ["Programme d'audit interne", "Indicateurs de performance (KPI)", "Revue de direction planifiée", "Procès-verbaux de revue"] },
      { ref: "10", label: "Amélioration", items: ["Processus d'amélioration continue", "Gestion des non-conformités", "Actions correctives tracées"] }
    ];

    auditChecklist.forEach(section => {
      html += `<div style="margin-top:15px;"><strong style="color:#1e40af;">Clause ${section.ref} - ${section.label}</strong></div>`;
      section.items.forEach(item => {
        // Vérifier si on a des données pour cet item
        const relatedEntries = appData.documentReview.filter(e =>
          e.reference && e.reference.startsWith(section.ref + ".")
        );
        const hasConforme = relatedEntries.some(e => e.status === "conforme");

        html += `
          <div class="checklist-item">
            <div class="checklist-box"></div>
            <div class="checklist-text">${item}</div>
            ${hasConforme ? '<span class="checklist-status" style="background:#dcfce7;color:#166534;">Documenté</span>' : ''}
          </div>
        `;
      });
    });

    html += `
        </div>

        <div class="footer">
          <div style="font-weight:600;">Pack Audit Certification ISO 27001:2022</div>
          <div>Référence: AUDIT-${orgName.replace(/\s+/g, '-').toUpperCase()}-${new Date().toISOString().split('T')[0]}</div>
          <div>Document généré par Cyber-Assistant</div>
          <div style="color:#dc2626;font-weight:600;margin-top:5px;">DOCUMENT CONFIDENTIEL - PRÉPARATION AUDIT</div>
        </div>
      </body>
      </html>
    `;

    const printWindow = window.open('', '_blank');
    printWindow.document.write(html);
    printWindow.document.close();

    setTimeout(() => {
      printWindow.print();
      showToast('Pack Audit Doc Review prêt - Utilisez "Enregistrer en PDF"', 'success');
    }, 300);
  }

  // Exports des fonctions PDF après leur définition
  window.exportSoaPdf = exportSoaPdf;
  window.exportDocReviewPdf = exportDocReviewPdf;

  function loadModuleContent(moduleId) {
    switch (moduleId) {
      case "dashboard":
        updateDashboard();
        break;
      case "controls":
        loadControlsTable();
        break;
      case "nis2-controls":
        loadNis2ControlsTable();
        break;
      case "nis2-program":
        loadNis2Program();
        break;
      case "actions":
        loadActionsTable();
        break;
      case "risks":
        loadRisksTable();
        updateRiskMatrix();
        break;
      case "threats-section":
        loadThreatsTable();
        break;
      case "nonconformities":
        loadNonConformitiesTable();
        break;
      case "soa":
        loadSoaTable();
        break;
      case "docReview":
        loadDocReviewTable();
        break;
      case "audits":
        loadAuditsTable();
        break;
      case "managementReviews":
        loadReviewsTable();
        break;
      case "stakeholders":
        loadStakeholdersTable();
        break;
      case "documents":
        loadDocumentsTable();
        break;
      case "kpis":
        loadKpiTable();
        break;
      case "critical-assets":
        loadCriticalAssetsTable();
        break;
      case "project-risks":
        loadProjectRisksTable();
        break;
      case "nis2-socle":
        renderSocleGrids();
        break;
      case "manual":
        break;
      case "settings":
        renderSettingsPage();
        break;
    }
    refreshReadOnlyLocks();
  }

  // ========================================
  // RACCOURCIS CLAVIER GLOBAUX
  // ========================================
  function setupKeyboardShortcuts() {
    document.addEventListener("keydown", (e) => {
      // Ignorer si on est dans un champ de saisie
      const target = e.target;
      const isInput = target.tagName === "INPUT" || target.tagName === "TEXTAREA" || target.isContentEditable;

      // Escape - Fermer les modals
      if (e.key === "Escape") {
        const openModal = document.querySelector(".modal.show, .modal--visible, [class*='modal'][style*='display: block']");
        if (openModal) {
          closeModal();
          e.preventDefault();
          return;
        }
      }

      // Ctrl/Cmd + S - Sauvegarder
      if ((e.ctrlKey || e.metaKey) && e.key === "s") {
        e.preventDefault();
        if (typeof saveData === "function") {
          const success = saveData();
          if (success) {
            showToast2("Données sauvegardées", "success");
          }
        }
        return;
      }

      // Ctrl/Cmd + E - Export rapide (JSON)
      if ((e.ctrlKey || e.metaKey) && e.key === "e" && !isInput) {
        e.preventDefault();
        if (typeof exportData === "function") {
          exportData();
        }
        return;
      }

      // Ctrl/Cmd + P - Export PDF
      if ((e.ctrlKey || e.metaKey) && e.key === "p" && !isInput) {
        e.preventDefault();
        if (typeof generatePdfReportWithFeedback === "function") {
          generatePdfReportWithFeedback();
        }
        return;
      }

      // Ctrl/Cmd + , - Ouvrir les paramètres
      if ((e.ctrlKey || e.metaKey) && e.key === ",") {
        e.preventDefault();
        if (typeof openSettings === "function") {
          openSettings();
        }
        return;
      }

      // Ctrl/Cmd + D - Aller au Dashboard
      if ((e.ctrlKey || e.metaKey) && e.key === "d" && !isInput) {
        e.preventDefault();
        const dashNav = document.querySelector('[data-module="dashboard"]');
        if (dashNav) dashNav.click();
        return;
      }
    });

    console.log("[Cyber-Assistant] Raccourcis clavier activés: Ctrl+S (sauvegarder), Ctrl+E (export), Ctrl+P (PDF), Ctrl+, (paramètres), Escape (fermer)");
  }

  // app.js
  // Système de sauvegarde automatique
  let _autoSaveInterval = null;
  let _lastSavedHash = null;
  let _autoSavePaused = false;

  function computeDataHash() {
    // Simple hash pour détecter les changements
    const str = JSON.stringify(appData);
    let hash = 0;
    for (let i = 0; i < str.length; i++) {
      const char = str.charCodeAt(i);
      hash = ((hash << 5) - hash) + char;
      hash = hash & hash; // Convert to 32bit integer
    }
    return hash;
  }

  function autoSave() {
    if (_autoSavePaused) return;
    if (typeof isReadOnlyMode === 'function' && isReadOnlyMode()) return;

    const currentHash = computeDataHash();
    if (currentHash !== _lastSavedHash) {
      const success = saveData();
      if (success) {
        _lastSavedHash = currentHash;
        updateAutoSaveIndicator();
      }
    }
  }

  function updateAutoSaveIndicator() {
    const indicator = document.getElementById("autoSaveIndicator");
    if (indicator) {
      const now = new Date();
      indicator.textContent = `Sauvegardé à ${now.toLocaleTimeString()}`;
      indicator.classList.add("autosave-flash");
      setTimeout(() => indicator.classList.remove("autosave-flash"), 1000);
    }
  }

  function initAutoSave(intervalMs = 30000) {
    // Calcul du hash initial
    _lastSavedHash = computeDataHash();

    // Démarrer l'intervalle d'autosave
    if (_autoSaveInterval) clearInterval(_autoSaveInterval);
    _autoSaveInterval = setInterval(autoSave, intervalMs);

    // Sauvegarder aussi avant de quitter la page
    window.addEventListener("beforeunload", () => {
      _autoSavePaused = true; // Éviter double sauvegarde
      autoSave();
    });

    // Sauvegarder quand la page perd le focus (changement d'onglet)
    document.addEventListener("visibilitychange", () => {
      if (document.visibilityState === "hidden") {
        autoSave();
      }
    });

    console.log(`AutoSave initialisé (intervalle: ${intervalMs / 1000}s)`);
  }

  function pauseAutoSave() {
    _autoSavePaused = true;
  }

  function resumeAutoSave() {
    _autoSavePaused = false;
  }

  window.initAutoSave = initAutoSave;
  window.pauseAutoSave = pauseAutoSave;
  window.resumeAutoSave = resumeAutoSave;

  /**
   * Crée et affiche le bandeau DEMO discret en haut de page
   */
  function initDemoBanner() {
    if (!DEMO_MODE) return;

    const banner = document.createElement('div');
    banner.id = 'demoBanner';
    banner.className = 'demo-banner';
    banner.innerHTML = `
      <div class="demo-banner__content">
        <span class="demo-banner__text">
          <i class="fas fa-flask"></i>
          Version de démonstration
        </span>
        <a href="${DEMO_WEBSITE_URL}" target="_blank" rel="noopener" class="demo-banner__link">
          <i class="fas fa-arrow-left"></i>
          Retour Website
        </a>
        <a href="${DEMO_DOWNLOAD_URL}" target="_blank" rel="noopener" class="demo-banner__link">
          <i class="fas fa-download"></i>
          Télécharger l'application
        </a>
        <a href="mailto:${DEMO_CONTACT_EMAIL}" class="demo-banner__link">
          <i class="fas fa-envelope"></i>
          Nous contacter
        </a>
      </div>
    `;

    // Insérer au tout début du body
    document.body.insertBefore(banner, document.body.firstChild);

    // Ajouter les styles inline pour le bandeau
    const style = document.createElement('style');
    style.textContent = `
      .demo-banner {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        z-index: 100000;
        background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
        color: #e8e8e8;
        padding: 8px 16px;
        font-size: 0.85rem;
        box-shadow: 0 2px 8px rgba(0,0,0,0.15);
      }
      .demo-banner__content {
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 24px;
        max-width: 1200px;
        margin: 0 auto;
      }
      .demo-banner__text {
        display: flex;
        align-items: center;
        gap: 8px;
        opacity: 0.9;
      }
      .demo-banner__text i {
        color: #64b5f6;
      }
      .demo-banner__link {
        display: inline-flex;
        align-items: center;
        gap: 6px;
        background: rgba(255,255,255,0.1);
        color: #90caf9;
        padding: 6px 14px;
        border-radius: 20px;
        text-decoration: none;
        font-weight: 500;
        transition: all 0.2s ease;
        border: 1px solid rgba(144,202,249,0.3);
      }
      .demo-banner__link:hover {
        background: rgba(255,255,255,0.15);
        color: #fff;
        border-color: rgba(255,255,255,0.4);
        transform: translateY(-1px);
      }
      .demo-banner__link .fa-external-link-alt {
        font-size: 0.7em;
        opacity: 0.7;
      }
      /* Décaler le contenu principal pour le bandeau */
      body.demo-mode {
        padding-top: 44px !important;
      }
      body.demo-mode .app-header {
        top: 44px;
      }
      body.demo-mode .sidebar {
        top: calc(var(--header-height, 60px) + 44px);
        height: calc(100vh - var(--header-height, 60px) - 44px);
      }
      @media (max-width: 768px) {
        .demo-banner__content {
          flex-direction: column;
          gap: 8px;
        }
        .demo-banner {
          padding: 10px 12px;
        }
      }
    `;
    document.head.appendChild(style);

    // Ajouter classe au body
    document.body.classList.add('demo-mode');
  }

  async function initApp() {
    try {
      // Mode DEMO: charger les données démo si pas de données existantes
      if (DEMO_MODE) {
        const hasCachedData = (() => {
          try {
            return !!localStorage.getItem("smsi_data");
          } catch (e) {
            return false;
          }
        })();

        // En mode DEMO, charger les données démo au premier lancement
        if (!hasCachedData) {
          console.log('[CSI DEMO] Premier lancement - chargement des données de démonstration');
          loadDemoDataSilent();
        }

        // Activer le mode édition par défaut en DEMO (pas de mot de passe)
        editModeEnabled = true;
      }

      const hasCachedData = (() => {
        try {
          return !!localStorage.getItem("smsi_data");
        } catch (e) {
          return false;
        }
      })();
      const showWelcome = shouldShowWelcome(hasCachedData);
      await loadData();
      initThemeToggle();
      // Restore compact mode
      try {
        if (localStorage.getItem("smsi_compact_mode") === "1") {
          document.body.classList.add("compact-mode");
        }
      } catch (e) { /* ignore */ }

      // Mode DEMO: initialiser le bandeau et skip les popups de sécurité
      if (DEMO_MODE) {
        initDemoBanner();
        // Démarrer le rechargement automatique des données (toutes les heures)
        startDemoAutoReload();
        // Marquer le setup serveur comme fait (mais pas le welcome)
        try {
          localStorage.setItem('smsi_setup_completed', '1');
          localStorage.setItem('smsi_deployment_mode', 'local');
          // Note: Ne pas marquer WELCOME_STORAGE_KEY ici, le wizard le fera
        } catch (e) {}
      }

      initReadOnlyBanner();
      updateAppTitle();
      setupNavigation();
      setupEventListeners();
      setupKeyboardShortcuts();
      setupGlobalSearch();
      initDashboard();
      initAutoSave(30000); // Autosave toutes les 30 secondes
      hideLoadingSpinner();
      showToast("Application charg\xE9e avec succ\xE8s", "success");

      // Show setup wizard on first launch (combines language + welcome + demo)
      // En mode DEMO: afficher uniquement le wizard simplifié de bienvenue
      if (DEMO_MODE) {
        showDemoWelcomeWizard();
      } else {
        showSetupWizard();
      }
    } catch (error) {
      console.error("Erreur lors de l'initialisation:", error);
      hideLoadingSpinner();
      showToast("Erreur lors du chargement: " + error.message, "error");
    }
  }

  /**
   * Wizard de bienvenue simplifié pour le mode DEMO
   * Affiche uniquement le choix de langue puis une page d'accueil
   */
  function showDemoWelcomeWizard() {
    // Vérifier si déjà vu (utiliser une clé spécifique pour la démo)
    const DEMO_WELCOME_KEY = 'smsi_demo_welcome_seen';
    try {
      if (localStorage.getItem(DEMO_WELCOME_KEY) === '1') return;
    } catch (e) {}

    if (document.getElementById("demoWelcomeWizard")) return;

    const overlay = document.createElement("div");
    overlay.id = "demoWelcomeWizard";
    overlay.className = "wizard-overlay";
    overlay.innerHTML = `
      <div class="wizard-modal" role="dialog" aria-modal="true" aria-labelledby="demoWizardTitle" style="max-width: 480px; overflow-y: auto;">
        <div style="padding: 28px;">
          <div style="text-align: center; margin-bottom: 20px;">
            <div style="font-size: 2.5rem; margin-bottom: 12px;">
              <i class="fas fa-shield-alt" style="color: var(--color-primary, #3b82f6);"></i>
            </div>
            <h2 id="demoWizardTitle" style="margin: 0 0 6px 0; font-size: 1.35rem; color: var(--color-text);">
              Bienvenue sur Cyber-Assistant
            </h2>
            <p style="margin: 0; color: var(--color-text-secondary); font-size: 0.9rem;">
              Version de démonstration
            </p>
          </div>

          <div style="margin-bottom: 20px;">
            <label style="display: block; margin-bottom: 8px; font-weight: 500; color: var(--color-text); font-size: 0.9rem;">
              <i class="fas fa-globe"></i> Langue / Language / Taal
            </label>
            <div style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 10px;">
              ${window.CSI_i18n ? Object.entries(window.CSI_i18n.getAvailableLanguages()).map(([code, info]) => `
                <button class="wizard-lang-btn ${code === 'fr' ? 'wizard-lang-btn--selected' : ''}" type="button" data-lang="${code}" style="padding: 10px 8px; border: 2px solid var(--color-border); border-radius: 8px; background: var(--color-surface); cursor: pointer; transition: all 0.2s;">
                  <span style="font-size: 1.3rem; display: block;">${info.flag}</span>
                  <span style="font-size: 0.8rem; color: var(--color-text);">${info.name}</span>
                </button>
              `).join("") : ''}
            </div>
          </div>

          <div style="background: var(--color-surface-hover, #f8f9fa); border-radius: 8px; padding: 14px; margin-bottom: 16px;">
            <p style="margin: 0; color: var(--color-text); font-size: 0.85rem;">
              <i class="fas fa-info-circle" style="color: var(--color-primary);"></i>
              <strong>Démonstration</strong> avec données fictives. Explorez librement !
            </p>
          </div>

          <div style="background: linear-gradient(135deg, rgba(59, 130, 246, 0.08), rgba(139, 92, 246, 0.08)); border-radius: 8px; padding: 14px; margin-bottom: 20px; border: 1px solid rgba(59, 130, 246, 0.2);">
            <p style="margin: 0 0 10px 0; color: var(--color-text); font-size: 0.85rem; font-weight: 500;">
              <i class="fas fa-download" style="color: var(--color-primary);"></i> Versions disponibles :
            </p>
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 10px;">
              <div style="background: var(--color-surface); border-radius: 6px; padding: 8px; text-align: center;">
                <i class="fas fa-laptop" style="color: var(--color-primary); font-size: 1rem;"></i>
                <p style="margin: 4px 0 0 0; font-size: 0.75rem; color: var(--color-text);"><strong>Locale</strong></p>
              </div>
              <div style="background: var(--color-surface); border-radius: 6px; padding: 8px; text-align: center;">
                <i class="fas fa-cloud" style="color: var(--color-success, #22c55e); font-size: 1rem;"></i>
                <p style="margin: 4px 0 0 0; font-size: 0.75rem; color: var(--color-text);"><strong>Serveur</strong></p>
              </div>
            </div>
          </div>

          <div style="text-align: center;">
            <button id="demoWizardStart" type="button" style="background: var(--color-primary, #3b82f6); color: white; border: none; padding: 12px 28px; border-radius: 8px; font-size: 0.95rem; font-weight: 500; cursor: pointer; transition: all 0.2s;">
              <i class="fas fa-rocket"></i> Découvrir
            </button>
          </div>
        </div>
      </div>
    `;

    document.body.appendChild(overlay);

    // Afficher l'overlay avec animation
    requestAnimationFrame(() => {
      overlay.classList.add('is-visible');
    });

    // Gestion du choix de langue
    const langBtns = overlay.querySelectorAll('.wizard-lang-btn');
    langBtns.forEach(btn => {
      btn.addEventListener('click', () => {
        langBtns.forEach(b => {
          b.classList.remove('wizard-lang-btn--selected');
          b.style.borderColor = 'var(--color-border)';
          b.style.background = 'var(--color-surface)';
        });
        btn.classList.add('wizard-lang-btn--selected');
        btn.style.borderColor = 'var(--color-primary)';
        btn.style.background = 'var(--color-primary-light, #eff6ff)';

        // Appliquer la langue
        if (window.CSI_i18n) {
          window.CSI_i18n.setLanguage(btn.dataset.lang);
          window.CSI_i18n.markLanguageChosen();
          window.CSI_i18n.translatePage();
        }
      });
    });

    // Bouton démarrer
    document.getElementById('demoWizardStart').addEventListener('click', () => {
      // Marquer comme vu
      try {
        localStorage.setItem(DEMO_WELCOME_KEY, '1');
        if (window.CSI_i18n) {
          window.CSI_i18n.markLanguageChosen();
        }
      } catch (e) {}
      overlay.remove();
      // Rafraîchir le dashboard
      if (typeof initDashboard === 'function') {
        initDashboard();
      }
    });
  }
  document.addEventListener("DOMContentLoaded", initApp);
})();
