/**
 * Cyber-Assistant - Module d'internationalisation (i18n)
 * Gère les traductions multi-langues de l'interface utilisateur.
 *
 * Langues supportées :
 * - fr : Français (défaut)
 * - en : English
 * - nl : Nederlands
 *
 * Utilisation :
 * - CSI_i18n.t('key') ou t('key') : Retourne la traduction
 * - CSI_i18n.setLanguage('en') : Change la langue
 * - CSI_i18n.getLanguage() : Retourne la langue courante
 */
(function() {
  'use strict';

  // Clé localStorage pour stocker la langue
  const STORAGE_KEY = 'smsi_language';

  // Langue par défaut
  const DEFAULT_LANGUAGE = 'fr';

  // Langues disponibles
  const AVAILABLE_LANGUAGES = {
    fr: { name: 'Français', flag: '🇫🇷' },
    en: { name: 'English', flag: '🇬🇧' },
    nl: { name: 'Nederlands', flag: '🇳🇱' }
  };

  // Dictionnaire de traductions (chargé dynamiquement)
  let translations = {};

  // Langue courante
  let currentLanguage = DEFAULT_LANGUAGE;

  /**
   * Initialise le module i18n
   */
  function init() {
    // Charger la langue sauvegardée ou utiliser le défaut
    const savedLang = localStorage.getItem(STORAGE_KEY);
    if (savedLang && AVAILABLE_LANGUAGES[savedLang]) {
      currentLanguage = savedLang;
    } else {
      // Détecter la langue du navigateur
      const browserLang = navigator.language?.substring(0, 2).toLowerCase();
      if (browserLang && AVAILABLE_LANGUAGES[browserLang]) {
        currentLanguage = browserLang;
      }
    }

    console.log('[i18n] Langue initialisée:', currentLanguage);
  }

  /**
   * Charge un fichier de traduction
   * @param {string} lang - Code de la langue
   * @param {object} data - Dictionnaire de traductions
   */
  function loadTranslations(lang, data) {
    if (!AVAILABLE_LANGUAGES[lang]) {
      console.warn('[i18n] Langue non supportée:', lang);
      return false;
    }

    translations[lang] = data;
    console.log('[i18n] Traductions chargées pour:', lang, '(' + Object.keys(data).length + ' clés)');
    return true;
  }

  /**
   * Ajoute une nouvelle langue au système
   * @param {string} code - Code de la langue (ex: 'de')
   * @param {object} info - { name: 'Deutsch', flag: '🇩🇪' }
   * @param {object} data - Dictionnaire de traductions
   */
  function addLanguage(code, info, data) {
    if (AVAILABLE_LANGUAGES[code]) {
      console.warn('[i18n] Langue déjà existante:', code);
      return false;
    }

    AVAILABLE_LANGUAGES[code] = info;
    translations[code] = data;
    console.log('[i18n] Nouvelle langue ajoutée:', code, info.name);
    return true;
  }

  /**
   * Retourne la traduction d'une clé
   * @param {string} key - Clé de traduction (ex: 'nav.dashboard')
   * @param {object} params - Paramètres pour l'interpolation {count: 5}
   * @returns {string} Traduction ou clé si non trouvée
   */
  function translate(key, params) {
    // Si français ou pas de traduction chargée, retourner la clé
    // (le français est dans le HTML/JS original)
    if (currentLanguage === 'fr') {
      return key;
    }

    // Chercher la traduction
    const langDict = translations[currentLanguage];
    if (!langDict) {
      return key;
    }

    // Support des clés imbriquées (nav.dashboard)
    let value = langDict;
    const parts = key.split('.');
    for (let i = 0; i < parts.length; i++) {
      if (value && typeof value === 'object' && parts[i] in value) {
        value = value[parts[i]];
      } else {
        // Clé non trouvée, retourner la clé originale
        return key;
      }
    }

    // Si c'est toujours un objet, retourner la clé
    if (typeof value !== 'string') {
      return key;
    }

    // Interpolation des paramètres {{count}}
    if (params && typeof params === 'object') {
      value = value.replace(/\{\{(\w+)\}\}/g, function(match, paramName) {
        return params[paramName] !== undefined ? params[paramName] : match;
      });
    }

    return value;
  }

  /**
   * Change la langue courante
   * @param {string} lang - Code de la langue
   * @returns {boolean} Succès du changement
   */
  function setLanguage(lang) {
    if (!AVAILABLE_LANGUAGES[lang]) {
      console.error('[i18n] Langue non supportée:', lang);
      return false;
    }

    currentLanguage = lang;
    localStorage.setItem(STORAGE_KEY, lang);

    console.log('[i18n] Langue changée:', lang);

    // Émettre un événement pour que l'UI se mette à jour
    if (typeof CustomEvent !== 'undefined') {
      window.dispatchEvent(new CustomEvent('languageChange', {
        detail: { language: lang }
      }));
    }

    return true;
  }

  /**
   * Retourne la langue courante
   * @returns {string} Code de la langue
   */
  function getLanguage() {
    return currentLanguage;
  }

  /**
   * Retourne les langues disponibles
   * @returns {object} { code: { name, flag } }
   */
  function getAvailableLanguages() {
    return { ...AVAILABLE_LANGUAGES };
  }

  /**
   * Vérifie si c'est le premier lancement (pas de langue sauvegardée)
   * @returns {boolean}
   */
  function isFirstLaunch() {
    return localStorage.getItem(STORAGE_KEY) === null;
  }

  /**
   * Marque le choix de langue comme effectué
   */
  function markLanguageChosen() {
    localStorage.setItem(STORAGE_KEY, currentLanguage);
  }

  /**
   * Traduit tous les éléments avec data-i18n dans le DOM
   */
  function translatePage() {
    if (currentLanguage === 'fr') {
      return; // Pas besoin de traduire, c'est déjà en français
    }

    const elements = document.querySelectorAll('[data-i18n]');
    elements.forEach(function(el) {
      const key = el.getAttribute('data-i18n');
      const translation = translate(key);

      // Ne pas remplacer si la traduction n'existe pas
      if (translation !== key) {
        // Gérer différents types d'éléments
        if (el.tagName === 'INPUT' || el.tagName === 'TEXTAREA') {
          if (el.placeholder) {
            el.placeholder = translation;
          }
        } else if (el.tagName === 'OPTION') {
          el.textContent = translation;
        } else {
          // Préserver les icônes et autres enfants
          const icon = el.querySelector('i.fas, i.far, i.fab');
          if (icon) {
            el.innerHTML = '';
            el.appendChild(icon.cloneNode(true));
            el.appendChild(document.createTextNode(' ' + translation));
          } else {
            el.textContent = translation;
          }
        }
      }
    });

    // Traduire les attributs title et aria-label
    const titledElements = document.querySelectorAll('[data-i18n-title]');
    titledElements.forEach(function(el) {
      const key = el.getAttribute('data-i18n-title');
      const translation = translate(key);
      if (translation !== key) {
        el.title = translation;
        if (el.hasAttribute('aria-label')) {
          el.setAttribute('aria-label', translation);
        }
      }
    });

    // Traduire les placeholders
    const placeholderElements = document.querySelectorAll('[data-i18n-placeholder]');
    placeholderElements.forEach(function(el) {
      const key = el.getAttribute('data-i18n-placeholder');
      const translation = translate(key);
      if (translation !== key) {
        el.placeholder = translation;
      }
    });
  }

  // Exposer le module
  const module = {
    init: init,
    t: translate,
    translate: translate,
    setLanguage: setLanguage,
    getLanguage: getLanguage,
    getAvailableLanguages: getAvailableLanguages,
    loadTranslations: loadTranslations,
    addLanguage: addLanguage,
    isFirstLaunch: isFirstLaunch,
    markLanguageChosen: markLanguageChosen,
    translatePage: translatePage,
    DEFAULT_LANGUAGE: DEFAULT_LANGUAGE
  };

  // Exposer globalement
  window.CSI_i18n = module;

  // Raccourci global pour la fonction de traduction
  window.t = translate;

  // Enregistrer dans le registre CSI si disponible
  if (window.CSI && typeof window.CSI.register === 'function') {
    window.CSI.register('i18n', {
      exports: { CSI_i18n: module, t: translate },
      init: init
    });
  } else {
    // Initialiser directement si le registre n'est pas encore chargé
    init();
  }

  // Load pending translations (loaded before i18n module was ready)
  if (window._pendingTranslations) {
    Object.entries(window._pendingTranslations).forEach(function([lang, data]) {
      loadTranslations(lang, data);
    });
    delete window._pendingTranslations;
  }

})();
