/**
 * Cyber-Assistant - Registre de modules
 * Permet l'enregistrement et la récupération de modules de manière centralisée.
 * Maintient la compatibilité avec les exports window.* existants.
 */
(function() {
  'use strict';

  // Namespace global CSI
  window.CSI = window.CSI || {};

  // Registre des modules
  window.CSI.modules = {};

  // Version du système de modules
  window.CSI.version = '1.0.0';

  /**
   * Enregistre un module dans le registre
   * @param {string} name - Nom unique du module
   * @param {object} module - Le module à enregistrer
   * @param {object} module.exports - Fonctions/objets à exposer globalement
   * @param {function} module.init - Fonction d'initialisation optionnelle
   */
  window.CSI.register = function(name, module) {
    if (!name || typeof name !== 'string') {
      console.error('[CSI] Nom de module invalide:', name);
      return false;
    }

    if (this.modules[name]) {
      console.warn('[CSI] Module déjà enregistré:', name);
      return false;
    }

    this.modules[name] = module;

    // Exposer les exports via window.* pour compatibilité
    if (module.exports && typeof module.exports === 'object') {
      Object.keys(module.exports).forEach(function(key) {
        if (window[key] === undefined) {
          window[key] = module.exports[key];
        } else {
          console.debug('[CSI] Export ignoré (déjà défini):', key);
        }
      });
    }

    // Appeler init si présent
    if (typeof module.init === 'function') {
      try {
        module.init();
      } catch (e) {
        console.error('[CSI] Erreur init module ' + name + ':', e);
      }
    }

    console.log('[CSI] Module enregistré:', name);
    return true;
  };

  /**
   * Récupère un module par son nom
   * @param {string} name - Nom du module
   * @returns {object|undefined} Le module ou undefined
   */
  window.CSI.get = function(name) {
    return this.modules[name];
  };

  /**
   * Vérifie si un module est chargé
   * @param {string} name - Nom du module
   * @returns {boolean}
   */
  window.CSI.has = function(name) {
    return !!this.modules[name];
  };

  /**
   * Liste tous les modules enregistrés
   * @returns {string[]} Noms des modules
   */
  window.CSI.list = function() {
    return Object.keys(this.modules);
  };

  // Log de chargement
  console.log('[CSI] Registre de modules initialisé (v' + window.CSI.version + ')');

})();
