/**
 * Cyber-Assistant - Module de sécurité
 * Fournit des fonctions de hachage sécurisé (PBKDF2) via Web Crypto API.
 * Compatible avec les anciens hash SHA256 pour migration progressive.
 */
(function() {
  'use strict';

  var Security = {
    // Configuration PBKDF2
    ITERATIONS: 100000,
    KEY_LENGTH: 256,
    HASH_ALGORITHM: 'SHA-256',

    /**
     * Génère un sel aléatoire cryptographiquement sûr
     * @param {number} length - Longueur en octets (défaut: 16)
     * @returns {string} Sel en hexadécimal
     */
    generateSalt: function(length) {
      length = length || 16;
      var array = new Uint8Array(length);
      crypto.getRandomValues(array);
      return Array.from(array).map(function(b) {
        return b.toString(16).padStart(2, '0');
      }).join('');
    },

    /**
     * Hash un mot de passe avec PBKDF2
     * @param {string} password - Le mot de passe en clair
     * @param {string} salt - Le sel en hexadécimal
     * @returns {Promise<string>} Hash en hexadécimal
     */
    hashPassword: function(password, salt) {
      var self = this;
      return new Promise(function(resolve, reject) {
        if (!password || !salt) {
          reject(new Error('Password et salt requis'));
          return;
        }

        var encoder = new TextEncoder();

        crypto.subtle.importKey(
          'raw',
          encoder.encode(password),
          'PBKDF2',
          false,
          ['deriveBits']
        ).then(function(key) {
          return crypto.subtle.deriveBits(
            {
              name: 'PBKDF2',
              salt: encoder.encode(salt),
              iterations: self.ITERATIONS,
              hash: self.HASH_ALGORITHM
            },
            key,
            self.KEY_LENGTH
          );
        }).then(function(bits) {
          var hash = Array.from(new Uint8Array(bits)).map(function(b) {
            return b.toString(16).padStart(2, '0');
          }).join('');
          resolve(hash);
        }).catch(function(err) {
          reject(err);
        });
      });
    },

    /**
     * Vérifie un mot de passe contre un hash PBKDF2
     * @param {string} password - Le mot de passe à vérifier
     * @param {string} storedHash - Le hash stocké
     * @param {string} salt - Le sel utilisé
     * @returns {Promise<boolean>}
     */
    verifyPassword: function(password, storedHash, salt) {
      return this.hashPassword(password, salt).then(function(hash) {
        return hash === storedHash;
      });
    },

    /**
     * Hash SHA256 simple (pour compatibilité avec l'ancien système)
     * @param {string} message - Le texte à hasher
     * @returns {Promise<string>} Hash en hexadécimal
     */
    sha256: function(message) {
      return new Promise(function(resolve, reject) {
        var encoder = new TextEncoder();
        crypto.subtle.digest('SHA-256', encoder.encode(message))
          .then(function(buffer) {
            var hash = Array.from(new Uint8Array(buffer)).map(function(b) {
              return b.toString(16).padStart(2, '0');
            }).join('');
            resolve(hash);
          })
          .catch(reject);
      });
    },

    /**
     * Vérifie si le navigateur supporte Web Crypto API
     * @returns {boolean}
     */
    isSupported: function() {
      return !!(
        typeof crypto !== 'undefined' &&
        crypto.subtle &&
        typeof crypto.subtle.importKey === 'function' &&
        typeof crypto.subtle.deriveBits === 'function'
      );
    },

    /**
     * Migre un ancien hash SHA256 vers PBKDF2
     * Retourne les nouvelles informations de hash
     * @param {string} password - Le mot de passe en clair
     * @returns {Promise<{hash: string, salt: string}>}
     */
    migrateToSecure: function(password) {
      var self = this;
      var salt = this.generateSalt();
      return this.hashPassword(password, salt).then(function(hash) {
        return { hash: hash, salt: salt };
      });
    },

    /**
     * Vérifie un mot de passe (compatible ancien et nouveau format)
     * @param {string} password - Mot de passe à vérifier
     * @param {string} storedHash - Hash stocké
     * @param {string|null} salt - Sel (null = ancien format SHA256)
     * @returns {Promise<boolean>}
     */
    verifyAny: function(password, storedHash, salt) {
      if (salt) {
        // Nouveau format PBKDF2
        return this.verifyPassword(password, storedHash, salt);
      } else {
        // Ancien format SHA256 simple
        return this.sha256(password).then(function(hash) {
          return hash === storedHash;
        });
      }
    }
  };

  // Enregistrer dans le registre CSI si disponible
  if (window.CSI && window.CSI.register) {
    window.CSI.register('security', {
      exports: {
        CSI_Security: Security
      }
    });
  }

  // Exposer globalement
  window.CSI_Security = Security;

  // Log de chargement
  if (Security.isSupported()) {
    console.log('[CSI] Module security.js chargé (Web Crypto API disponible)');
  } else {
    console.warn('[CSI] Module security.js chargé (Web Crypto API NON disponible - fonctionnalités limitées)');
  }

})();
