/**
 * Cyber-Assistant - Assistant de démarrage
 *
 * Au premier lancement, guide l'utilisateur pour choisir le mode :
 * - Mode Local : données stockées dans le navigateur uniquement
 * - Mode Serveur : données synchronisées sur le serveur PHP
 *
 * Le choix est définitif et dépend du contexte :
 * - file:// → seul Mode Local disponible
 * - http:// avec PHP → seul Mode Serveur disponible
 *
 * Dépend de: modules/storage.js (doit être chargé avant)
 */
(function() {
  'use strict';

  // Vérifier que storage.js est chargé
  var Storage = window.CSI_Storage;
  if (!Storage) {
    console.error('[CSI] server-setup.js requiert storage.js');
    return;
  }

  // Vérifier que la licence serveur est activée
  if (!Storage.isServerLicenseActivated()) {
    console.log('[CSI] server-setup.js désactivé - licence serveur non activée');
    return;
  }

  var SYNC_URL = './data/sync.php';

  // ============================================================
  // Détection du contexte
  // ============================================================

  function isFileProtocol() {
    return window.location.protocol === 'file:';
  }

  async function checkSyncEndpoint() {
    if (isFileProtocol()) return null;
    try {
      var response = await fetch(SYNC_URL + '?action=status', {
        method: 'GET',
        cache: 'no-store'
      });
      if (!response.ok) return null;
      return await response.json();
    } catch (e) {
      return null;
    }
  }

  // ============================================================
  // Utilitaires
  // ============================================================

  async function sha256(message) {
    if (typeof window.sha256Fallback === 'function') {
      var bytes = new TextEncoder().encode(message);
      return window.sha256Fallback(bytes);
    }
    var encoder = new TextEncoder();
    var data = encoder.encode(message);
    var hashBuffer = await crypto.subtle.digest('SHA-256', data);
    var hashArray = Array.from(new Uint8Array(hashBuffer));
    return hashArray.map(function(b) {
      return b.toString(16).padStart(2, '0');
    }).join('');
  }

  function validatePassword(password) {
    var errors = [];
    if (!password || password.length < 18) errors.push('Au moins 18 caractères');
    if (!/[A-Z]/.test(password)) errors.push('Au moins une majuscule');
    if (!/[a-z]/.test(password)) errors.push('Au moins une minuscule');
    if (!/[^a-zA-Z0-9]/.test(password)) errors.push('Au moins un caractère spécial');
    return errors;
  }

  async function setupServerPassword(password) {
    var hash = await sha256(password);
    var response = await fetch(SYNC_URL + '?action=setup', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ hash: hash })
    });
    var result = await response.json();
    if (!response.ok) {
      throw new Error(result.error || 'Erreur de configuration');
    }
    return result;
  }

  // ============================================================
  // Assistant de démarrage (choix du mode)
  // ============================================================

  function showStartupWizard(phpAvailable, phpConfigured) {
    var isLocal = isFileProtocol();

    // Créer l'overlay
    var overlay = document.createElement('div');
    overlay.id = 'csiStartupWizard';
    overlay.style.cssText = 'position:fixed;top:0;left:0;right:0;bottom:0;background:rgba(0,0,0,0.7);display:flex;align-items:center;justify-content:center;z-index:99999;';

    var localDisabled = !isLocal && phpAvailable;
    var serverDisabled = isLocal || !phpAvailable;

    var localClass = localDisabled ? 'disabled' : '';
    var serverClass = serverDisabled ? 'disabled' : '';

    var html = '<div style="background:var(--color-surface,#fff);border-radius:12px;padding:32px;max-width:500px;width:90%;box-shadow:0 8px 32px rgba(0,0,0,0.3);">' +
      '<h2 style="margin:0 0 8px 0;color:var(--color-text,#333);font-size:1.5rem;text-align:center;">Bienvenue dans Cyber-Assistant</h2>' +
      '<p style="margin:0 0 24px 0;color:var(--color-text-secondary,#666);text-align:center;font-size:0.95rem;">Comment souhaitez-vous utiliser l\'application ?</p>' +

      '<div id="wizardModeLocal" class="wizard-option ' + localClass + '" style="border:2px solid var(--color-border,#ddd);border-radius:8px;padding:16px;margin-bottom:12px;cursor:' + (localDisabled ? 'not-allowed' : 'pointer') + ';transition:all 0.2s;' + (localDisabled ? 'opacity:0.5;' : '') + '">' +
        '<div style="display:flex;align-items:center;gap:12px;">' +
          '<div style="font-size:2rem;">💻</div>' +
          '<div>' +
            '<div style="font-weight:600;color:var(--color-text,#333);margin-bottom:4px;">Mode Local</div>' +
            '<div style="font-size:0.85rem;color:var(--color-text-secondary,#666);">Données stockées uniquement dans ce navigateur</div>' +
            '<div style="font-size:0.8rem;color:var(--color-text-secondary,#888);margin-top:4px;">Idéal pour un usage personnel</div>' +
          '</div>' +
        '</div>' +
        (localDisabled ? '<div style="margin-top:8px;font-size:0.8rem;color:var(--color-warning,#f59e0b);"><i class="fas fa-info-circle"></i> Non disponible sur serveur web</div>' : '') +
      '</div>' +

      '<div id="wizardModeServer" class="wizard-option ' + serverClass + '" style="border:2px solid var(--color-border,#ddd);border-radius:8px;padding:16px;cursor:' + (serverDisabled ? 'not-allowed' : 'pointer') + ';transition:all 0.2s;' + (serverDisabled ? 'opacity:0.5;' : '') + '">' +
        '<div style="display:flex;align-items:center;gap:12px;">' +
          '<div style="font-size:2rem;">🌐</div>' +
          '<div>' +
            '<div style="font-weight:600;color:var(--color-text,#333);margin-bottom:4px;">Mode Serveur</div>' +
            '<div style="font-size:0.85rem;color:var(--color-text-secondary,#666);">Données synchronisées automatiquement sur le serveur</div>' +
            '<div style="font-size:0.8rem;color:var(--color-text-secondary,#888);margin-top:4px;">Partageable entre plusieurs utilisateurs</div>' +
            '<div style="font-size:0.75rem;color:var(--color-text-secondary,#999);margin-top:2px;font-style:italic;">Requiert un hébergement web avec PHP (mutualisé ou dédié)</div>' +
          '</div>' +
        '</div>' +
        (serverDisabled && isLocal ? '<div style="margin-top:8px;font-size:0.8rem;color:var(--color-warning,#f59e0b);"><i class="fas fa-info-circle"></i> Déployez sur un serveur web pour activer</div>' : '') +
        (serverDisabled && !isLocal && !phpAvailable ? '<div style="margin-top:8px;font-size:0.8rem;color:var(--color-warning,#f59e0b);"><i class="fas fa-info-circle"></i> sync.php non détecté sur ce serveur</div>' : '') +
      '</div>' +

      '<p style="margin:16px 0 0 0;font-size:0.75rem;color:var(--color-text-secondary,#999);text-align:center;">Ce choix est définitif pour cette installation</p>' +
    '</div>';

    overlay.innerHTML = html;
    document.body.appendChild(overlay);

    // Événements
    var localBtn = document.getElementById('wizardModeLocal');
    var serverBtn = document.getElementById('wizardModeServer');

    if (!localDisabled) {
      localBtn.addEventListener('mouseenter', function() {
        this.style.borderColor = 'var(--color-primary,#3b82f6)';
        this.style.background = 'var(--color-surface-hover,#f8fafc)';
      });
      localBtn.addEventListener('mouseleave', function() {
        this.style.borderColor = 'var(--color-border,#ddd)';
        this.style.background = 'transparent';
      });
      localBtn.addEventListener('click', function() {
        Storage.markSetupCompleted('local');
        overlay.remove();
        if (typeof window.showToast2 === 'function') {
          window.showToast2('Mode local activé', 'success');
        }
      });
    }

    if (!serverDisabled) {
      serverBtn.addEventListener('mouseenter', function() {
        this.style.borderColor = 'var(--color-primary,#3b82f6)';
        this.style.background = 'var(--color-surface-hover,#f8fafc)';
      });
      serverBtn.addEventListener('mouseleave', function() {
        this.style.borderColor = 'var(--color-border,#ddd)';
        this.style.background = 'transparent';
      });
      serverBtn.addEventListener('click', function() {
        overlay.remove();
        if (phpConfigured) {
          // Serveur déjà configuré, demander le mot de passe
          Storage.markSetupCompleted('server');
          showPasswordEntryModal();
        } else {
          // Premier setup, configurer le mot de passe
          showPasswordSetupModal();
        }
      });
    }
  }

  // ============================================================
  // Modal de configuration du mot de passe (premier setup serveur)
  // ============================================================

  function showPasswordSetupModal() {
    var overlay = document.createElement('div');
    overlay.id = 'csiPasswordSetupModal';
    overlay.style.cssText = 'position:fixed;top:0;left:0;right:0;bottom:0;background:rgba(0,0,0,0.7);display:flex;align-items:center;justify-content:center;z-index:99999;';

    var html = '<div style="background:var(--color-surface,#fff);border-radius:12px;padding:32px;max-width:420px;width:90%;box-shadow:0 8px 32px rgba(0,0,0,0.3);">' +
      '<h2 style="margin:0 0 8px 0;color:var(--color-text,#333);font-size:1.3rem;">Configurer le serveur</h2>' +
      '<p style="margin:0 0 20px 0;color:var(--color-text-secondary,#666);font-size:0.9rem;">Définissez le mot de passe pour sécuriser la synchronisation.</p>' +

      '<label style="display:block;margin-bottom:4px;font-weight:500;color:var(--color-text,#333);font-size:0.9rem;">Mot de passe serveur</label>' +
      '<input type="password" id="csiSetupPassword" placeholder="Minimum 18 caractères" style="width:100%;padding:10px 12px;border:1px solid var(--color-border,#ddd);border-radius:6px;margin-bottom:12px;font-size:0.95rem;box-sizing:border-box;">' +

      '<label style="display:block;margin-bottom:4px;font-weight:500;color:var(--color-text,#333);font-size:0.9rem;">Confirmation</label>' +
      '<input type="password" id="csiSetupConfirm" style="width:100%;padding:10px 12px;border:1px solid var(--color-border,#ddd);border-radius:6px;margin-bottom:16px;font-size:0.95rem;box-sizing:border-box;">' +

      '<div id="csiSetupRequirements" style="font-size:0.8rem;margin-bottom:20px;padding:12px;background:var(--color-surface-hover,#f8f9fa);border-radius:6px;">' +
        '<div style="color:var(--color-text-secondary,#666);margin-bottom:4px;font-weight:500;">Exigences :</div>' +
        '<div class="req-item" data-req="length" style="color:var(--color-danger,#dc3545);">• Au moins 18 caractères</div>' +
        '<div class="req-item" data-req="upper" style="color:var(--color-danger,#dc3545);">• Au moins une majuscule</div>' +
        '<div class="req-item" data-req="lower" style="color:var(--color-danger,#dc3545);">• Au moins une minuscule</div>' +
        '<div class="req-item" data-req="special" style="color:var(--color-danger,#dc3545);">• Au moins un caractère spécial</div>' +
      '</div>' +

      '<div style="display:flex;gap:12px;justify-content:flex-end;">' +
        '<button id="csiSetupCancel" style="padding:10px 20px;border:1px solid var(--color-border,#ddd);background:var(--color-surface,#fff);border-radius:6px;cursor:pointer;font-size:0.95rem;">Retour</button>' +
        '<button id="csiSetupSave" style="padding:10px 20px;border:none;background:var(--color-primary,#3b82f6);color:white;border-radius:6px;cursor:pointer;font-size:0.95rem;font-weight:500;">Configurer</button>' +
      '</div>' +
    '</div>';

    overlay.innerHTML = html;
    document.body.appendChild(overlay);

    var pwdInput = document.getElementById('csiSetupPassword');
    var confirmInput = document.getElementById('csiSetupConfirm');
    var saveBtn = document.getElementById('csiSetupSave');
    var cancelBtn = document.getElementById('csiSetupCancel');

    // Validation en temps réel
    pwdInput.addEventListener('input', function() {
      var val = pwdInput.value || '';
      var checks = {
        length: val.length >= 18,
        upper: /[A-Z]/.test(val),
        lower: /[a-z]/.test(val),
        special: /[^a-zA-Z0-9]/.test(val)
      };

      Object.keys(checks).forEach(function(key) {
        var el = document.querySelector('.req-item[data-req="' + key + '"]');
        if (el) {
          el.style.color = checks[key] ? 'var(--color-success,#22c55e)' : 'var(--color-danger,#dc3545)';
          el.textContent = (checks[key] ? '✓ ' : '• ') + el.textContent.substring(2);
        }
      });
    });

    // Bouton Configurer
    saveBtn.addEventListener('click', async function() {
      var password = pwdInput.value;
      var confirm = confirmInput.value;

      var errors = validatePassword(password);
      if (errors.length > 0) {
        if (typeof window.showToast2 === 'function') {
          window.showToast2('Mot de passe invalide: ' + errors.join(', '), 'error');
        }
        return;
      }

      if (password !== confirm) {
        if (typeof window.showToast2 === 'function') {
          window.showToast2('Les mots de passe ne correspondent pas', 'error');
        }
        return;
      }

      saveBtn.disabled = true;
      saveBtn.textContent = 'Configuration...';

      try {
        await setupServerPassword(password);
        Storage.setPassword(password, true); // Sauvegarder avec remember=true
        Storage.markSetupCompleted('server');

        if (typeof window.showToast2 === 'function') {
          window.showToast2('Serveur configuré avec succès', 'success');
        }

        setTimeout(function() {
          location.reload();
        }, 1000);

      } catch (err) {
        saveBtn.disabled = false;
        saveBtn.textContent = 'Configurer';
        if (typeof window.showToast2 === 'function') {
          window.showToast2('Erreur: ' + err.message, 'error');
        }
      }
    });

    // Bouton Retour
    cancelBtn.addEventListener('click', function() {
      overlay.remove();
      showStartupWizard(true, false);
    });

    // Focus sur le premier champ
    setTimeout(function() { pwdInput.focus(); }, 100);
  }

  // ============================================================
  // Modal de saisie du mot de passe (serveur déjà configuré)
  // ============================================================

  function showPasswordEntryModal() {
    var overlay = document.createElement('div');
    overlay.id = 'csiPasswordEntryModal';
    overlay.style.cssText = 'position:fixed;top:0;left:0;right:0;bottom:0;background:rgba(0,0,0,0.7);display:flex;align-items:center;justify-content:center;z-index:99999;';

    var html = '<div style="background:var(--color-surface,#fff);border-radius:12px;padding:32px;max-width:380px;width:90%;box-shadow:0 8px 32px rgba(0,0,0,0.3);">' +
      '<h2 style="margin:0 0 8px 0;color:var(--color-text,#333);font-size:1.3rem;">Connexion au serveur</h2>' +
      '<p style="margin:0 0 20px 0;color:var(--color-text-secondary,#666);font-size:0.9rem;">Entrez le mot de passe pour activer la synchronisation.</p>' +

      '<label style="display:block;margin-bottom:4px;font-weight:500;color:var(--color-text,#333);font-size:0.9rem;">Mot de passe serveur</label>' +
      '<input type="password" id="csiEntryPassword" style="width:100%;padding:10px 12px;border:1px solid var(--color-border,#ddd);border-radius:6px;margin-bottom:20px;font-size:0.95rem;box-sizing:border-box;">' +

      '<div style="display:flex;gap:12px;justify-content:flex-end;">' +
        '<button id="csiEntryCancel" style="padding:10px 20px;border:1px solid var(--color-border,#ddd);background:var(--color-surface,#fff);border-radius:6px;cursor:pointer;font-size:0.95rem;">Annuler</button>' +
        '<button id="csiEntrySave" style="padding:10px 20px;border:none;background:var(--color-primary,#3b82f6);color:white;border-radius:6px;cursor:pointer;font-size:0.95rem;font-weight:500;">Connecter</button>' +
      '</div>' +
    '</div>';

    overlay.innerHTML = html;
    document.body.appendChild(overlay);

    var pwdInput = document.getElementById('csiEntryPassword');
    var saveBtn = document.getElementById('csiEntrySave');
    var cancelBtn = document.getElementById('csiEntryCancel');

    saveBtn.addEventListener('click', function() {
      var password = pwdInput.value;
      if (!password) {
        if (typeof window.showToast2 === 'function') {
          window.showToast2('Mot de passe requis', 'error');
        }
        return;
      }

      Storage.setPassword(password, true);

      if (typeof window.showToast2 === 'function') {
        window.showToast2('Synchronisation activée', 'success');
      }

      setTimeout(function() {
        location.reload();
      }, 500);
    });

    cancelBtn.addEventListener('click', function() {
      overlay.remove();
    });

    // Focus et Enter pour valider
    setTimeout(function() { pwdInput.focus(); }, 100);
    pwdInput.addEventListener('keypress', function(e) {
      if (e.key === 'Enter') saveBtn.click();
    });
  }

  // ============================================================
  // Point d'entrée
  // ============================================================

  async function init() {
    // Si setup déjà fait, ne rien afficher
    if (Storage.isSetupCompleted()) {
      console.log('[CSI] Setup déjà complété, mode:', Storage.getDeploymentMode());
      return;
    }

    // Si des données utilisateur existent déjà, considérer comme "local" par défaut
    if (Storage.hasUserData()) {
      console.log('[CSI] Données existantes détectées, mode local par défaut');
      Storage.markSetupCompleted('local');
      return;
    }

    // Attendre que le DOM soit prêt
    if (document.readyState === 'loading') {
      document.addEventListener('DOMContentLoaded', function() { init(); });
      return;
    }

    // Attendre que d'autres overlays soient fermés
    var welcomeOverlay = document.querySelector('.welcome-overlay.is-visible');
    if (welcomeOverlay) {
      setTimeout(init, 500);
      return;
    }

    var existingModal = document.getElementById('modalOverlay');
    if (existingModal && existingModal.classList.contains('modal-overlay--visible')) {
      setTimeout(init, 500);
      return;
    }

    // Vérifier si PHP est disponible
    var syncStatus = await checkSyncEndpoint();
    var phpAvailable = syncStatus !== null;
    var phpConfigured = syncStatus && syncStatus.configured;

    console.log('[CSI] Setup wizard - file://', isFileProtocol(), '- PHP:', phpAvailable, '- configured:', phpConfigured);

    // Afficher l'assistant
    showStartupWizard(phpAvailable, phpConfigured);
  }

  // Lancer l'initialisation
  setTimeout(init, 300);

  // Exposer pour usage manuel
  window.CSI_ServerSetup = {
    init: init,
    showWizard: function() {
      checkSyncEndpoint().then(function(status) {
        showStartupWizard(status !== null, status && status.configured);
      });
    },
    showPasswordSetup: showPasswordSetupModal,
    showPasswordEntry: showPasswordEntryModal,
    getMode: function() {
      return Storage.getDeploymentMode();
    }
  };

  console.log('[CSI] Module server-setup.js chargé');

})();
