/**
 * Cyber-Assistant - Module de stockage centralisé
 *
 * Centralise toutes les clés localStorage/sessionStorage et fournit
 * une API unifiée pour l'accès aux données de configuration.
 *
 * Ce module élimine la duplication de code entre save-manager.js,
 * server-sync.js et server-setup.js.
 */
(function() {
  'use strict';

  // ============================================================
  // Constantes - Clés de stockage centralisées
  // ============================================================

  var KEYS = {
    // Données principales
    DATA: 'smsi_data',
    LAST_SAVE: 'smsi_last_save',
    LAST_SERVER_PULL: 'smsi_last_server_pull',
    LAST_SERVER_PUSH: 'smsi_last_server_push',

    // Configuration serveur
    SERVER_CONFIG: 'smsi_server_sync_cfg',
    PASSWORD_SESSION: 'smsi_server_sync_pwd_session',
    PASSWORD_PERSIST: 'smsi_server_sync_pwd_persist',

    // Mode déploiement
    DEPLOYMENT_MODE: 'smsi_deployment_mode',
    SETUP_COMPLETED: 'smsi_setup_completed',

    // Licence serveur
    SERVER_LICENSE: 'smsi_server_license',
    SERVER_LICENSE_ACTIVATED: 'smsi_server_license_activated',

    // UI / Préférences
    THEME: 'smsi_theme',
    WELCOME_SEEN: 'smsi_welcome_seen',
    EDIT_PASSWORD_HASH: 'smsi_edit_password_hash',
    HISTORY_LIMIT: 'smsi_history_limit'
  };

  // Version actuelle du schéma de données
  var DATA_VERSION = 1;

  // ============================================================
  // Utilitaires internes
  // ============================================================

  function safeJsonParse(str, defaultValue) {
    if (!str) return defaultValue;
    try {
      return JSON.parse(str);
    } catch (e) {
      console.warn('[CSI Storage] JSON parse error:', e.message);
      return defaultValue;
    }
  }

  function safeJsonStringify(obj) {
    try {
      return JSON.stringify(obj);
    } catch (e) {
      console.warn('[CSI Storage] JSON stringify error:', e.message);
      return null;
    }
  }

  // ============================================================
  // API Storage
  // ============================================================

  var Storage = {
    // Exposer les clés pour référence externe
    KEYS: KEYS,
    DATA_VERSION: DATA_VERSION,

    // ----------------------------------------------------------
    // Données principales (smsi_data)
    // ----------------------------------------------------------

    /**
     * Récupère les données de l'application
     * @returns {Object|null} Les données ou null si absentes/invalides
     */
    getData: function() {
      return safeJsonParse(localStorage.getItem(KEYS.DATA), null);
    },

    /**
     * Sauvegarde les données de l'application
     * @param {Object} data - Les données à sauvegarder
     * @returns {boolean} Succès de l'opération
     */
    setData: function(data) {
      if (!data) return false;
      // Ajouter la version si absente
      if (!data._version) {
        data._version = DATA_VERSION;
      }
      var json = safeJsonStringify(data);
      if (!json) return false;
      try {
        localStorage.setItem(KEYS.DATA, json);
        localStorage.setItem(KEYS.LAST_SAVE, new Date().toISOString());
        return true;
      } catch (e) {
        console.error('[CSI Storage] Erreur sauvegarde:', e.message);
        return false;
      }
    },

    /**
     * Vérifie si des données utilisateur existent (pas juste les défauts ISO)
     * @returns {boolean}
     */
    hasUserData: function() {
      var data = this.getData();
      if (!data) return false;

      // Actions ou risques = données utilisateur
      if (data.actions && data.actions.length > 0) return true;
      if (data.risks && data.risks.length > 0) return true;

      // Contrôles modifiés
      if (data.controls && data.controls.length > 0) {
        var defaultStatuses = ['non_implemente', 'Non évalué', '', null, undefined];
        var hasModified = data.controls.some(function(c) {
          var isDefault = defaultStatuses.indexOf(c.status) !== -1;
          return (!isDefault) ||
                 (c.notes && c.notes.trim() !== '') ||
                 (c.responsable && c.responsable.trim() !== '');
        });
        if (hasModified) return true;
      }

      return false;
    },

    // ----------------------------------------------------------
    // Configuration serveur
    // ----------------------------------------------------------

    /**
     * Récupère la configuration serveur
     * @returns {Object} Configuration {baseUrl?, rememberPassword?}
     */
    getServerConfig: function() {
      return safeJsonParse(localStorage.getItem(KEYS.SERVER_CONFIG), {});
    },

    /**
     * Sauvegarde la configuration serveur
     * @param {Object} config
     */
    setServerConfig: function(config) {
      localStorage.setItem(KEYS.SERVER_CONFIG, safeJsonStringify(config || {}));
    },

    /**
     * Récupère le mot de passe (session ou persisté)
     * @returns {string} Mot de passe ou chaîne vide
     */
    getPassword: function() {
      // Priorité au mot de passe de session
      var sessionPwd = sessionStorage.getItem(KEYS.PASSWORD_SESSION);
      if (sessionPwd) return sessionPwd;

      // Sinon, mot de passe persisté si autorisé
      var config = this.getServerConfig();
      if (config.rememberPassword) {
        return localStorage.getItem(KEYS.PASSWORD_PERSIST) || '';
      }

      return '';
    },

    /**
     * Sauvegarde le mot de passe
     * @param {string} password
     * @param {boolean} remember - Persister entre sessions
     */
    setPassword: function(password, remember) {
      // Toujours sauver en session
      sessionStorage.setItem(KEYS.PASSWORD_SESSION, password || '');

      // Persister si demandé
      if (remember) {
        localStorage.setItem(KEYS.PASSWORD_PERSIST, password || '');
        var config = this.getServerConfig();
        config.rememberPassword = true;
        this.setServerConfig(config);
      }
    },

    /**
     * Efface le mot de passe
     */
    clearPassword: function() {
      sessionStorage.removeItem(KEYS.PASSWORD_SESSION);
      localStorage.removeItem(KEYS.PASSWORD_PERSIST);
      var config = this.getServerConfig();
      config.rememberPassword = false;
      this.setServerConfig(config);
    },

    // ----------------------------------------------------------
    // Mode déploiement
    // ----------------------------------------------------------

    /**
     * Récupère le mode de déploiement
     * @returns {string} 'local', 'server', ou 'unknown'
     */
    getDeploymentMode: function() {
      return localStorage.getItem(KEYS.DEPLOYMENT_MODE) || 'unknown';
    },

    /**
     * Définit le mode de déploiement
     * @param {string} mode - 'local' ou 'server'
     */
    setDeploymentMode: function(mode) {
      localStorage.setItem(KEYS.DEPLOYMENT_MODE, mode);
    },

    /**
     * Vérifie si le setup initial est terminé
     * @returns {boolean}
     */
    isSetupCompleted: function() {
      return localStorage.getItem(KEYS.SETUP_COMPLETED) === '1';
    },

    /**
     * Marque le setup comme terminé
     * @param {string} mode - Mode choisi ('local' ou 'server')
     */
    markSetupCompleted: function(mode) {
      localStorage.setItem(KEYS.SETUP_COMPLETED, '1');
      this.setDeploymentMode(mode);
    },

    // ----------------------------------------------------------
    // Timestamps de synchronisation
    // ----------------------------------------------------------

    /**
     * Met à jour le timestamp de dernière synchro serveur (push)
     */
    updateLastPush: function() {
      localStorage.setItem(KEYS.LAST_SERVER_PUSH, new Date().toISOString());
    },

    /**
     * Met à jour le timestamp de dernière synchro serveur (pull)
     */
    updateLastPull: function() {
      localStorage.setItem(KEYS.LAST_SERVER_PULL, new Date().toISOString());
    },

    /**
     * Récupère les timestamps de synchronisation
     * @returns {Object} {lastSave, lastPush, lastPull}
     */
    getSyncTimestamps: function() {
      return {
        lastSave: localStorage.getItem(KEYS.LAST_SAVE),
        lastPush: localStorage.getItem(KEYS.LAST_SERVER_PUSH),
        lastPull: localStorage.getItem(KEYS.LAST_SERVER_PULL)
      };
    },

    // ----------------------------------------------------------
    // Préférences UI
    // ----------------------------------------------------------

    /**
     * Récupère le thème actuel
     * @returns {string}
     */
    getTheme: function() {
      return localStorage.getItem(KEYS.THEME) || 'poudre';
    },

    /**
     * Définit le thème
     * @param {string} theme
     */
    setTheme: function(theme) {
      localStorage.setItem(KEYS.THEME, theme);
    },

    /**
     * Vérifie si le welcome a été vu
     * @returns {boolean}
     */
    isWelcomeSeen: function() {
      return localStorage.getItem(KEYS.WELCOME_SEEN) === '1';
    },

    /**
     * Marque le welcome comme vu
     */
    markWelcomeSeen: function() {
      localStorage.setItem(KEYS.WELCOME_SEEN, '1');
    },

    // ----------------------------------------------------------
    // Migration de données
    // ----------------------------------------------------------

    /**
     * Migre les données vers la version actuelle si nécessaire
     * @returns {boolean} true si migration effectuée
     */
    migrateIfNeeded: function() {
      var data = this.getData();
      if (!data) return false;

      var currentVersion = data._version || 0;
      var migrated = false;

      // Migration v0 -> v1 : Ajouter le champ version
      if (currentVersion < 1) {
        data._version = 1;
        data._migratedAt = new Date().toISOString();
        migrated = true;
        console.log('[CSI Storage] Migration v0 -> v1 effectuée');
      }

      // Futures migrations ici...
      // if (currentVersion < 2) { ... }

      if (migrated) {
        this.setData(data);
      }

      return migrated;
    },

    // ----------------------------------------------------------
    // Utilitaires
    // ----------------------------------------------------------

    /**
     * Efface toutes les données CSI
     */
    clearAll: function() {
      Object.values(KEYS).forEach(function(key) {
        localStorage.removeItem(key);
        sessionStorage.removeItem(key);
      });
    },

    /**
     * Exporte toutes les clés CSI pour debug
     * @returns {Object}
     */
    exportAll: function() {
      var result = {};
      Object.keys(KEYS).forEach(function(name) {
        var key = KEYS[name];
        var localValue = localStorage.getItem(key);
        var sessionValue = sessionStorage.getItem(key);
        if (localValue || sessionValue) {
          result[name] = {
            key: key,
            local: localValue,
            session: sessionValue
          };
        }
      });
      return result;
    },

    // ----------------------------------------------------------
    // Licence serveur
    // ----------------------------------------------------------

    /**
     * Vérifie si la licence serveur est activée
     * @returns {boolean}
     */
    isServerLicenseActivated: function() {
      return localStorage.getItem(KEYS.SERVER_LICENSE_ACTIVATED) === '1';
    },

    /**
     * Récupère les informations de licence
     * @returns {Object|null}
     */
    getServerLicense: function() {
      return safeJsonParse(localStorage.getItem(KEYS.SERVER_LICENSE), null);
    },

    /**
     * Enregistre et active une licence serveur
     * @param {Object} licenseData - Données de la licence validée
     */
    activateServerLicense: function(licenseData) {
      localStorage.setItem(KEYS.SERVER_LICENSE, safeJsonStringify(licenseData));
      localStorage.setItem(KEYS.SERVER_LICENSE_ACTIVATED, '1');
    },

    /**
     * Désactive la licence serveur
     */
    deactivateServerLicense: function() {
      localStorage.removeItem(KEYS.SERVER_LICENSE);
      localStorage.removeItem(KEYS.SERVER_LICENSE_ACTIVATED);
    }
  };

  // ============================================================
  // Initialisation
  // ============================================================

  // Exécuter la migration au chargement
  Storage.migrateIfNeeded();

  // Enregistrer dans le registre CSI si disponible
  if (window.CSI && window.CSI.register) {
    window.CSI.register('storage', {
      exports: Storage
    });
  }

  // Exposer globalement
  window.CSI_Storage = Storage;

  console.log('[CSI] Module storage.js chargé (version données: ' + DATA_VERSION + ')');

})();
