/**
 * Cyber-Assistant - Gestionnaire de synchronisation unifié
 *
 * Fusionne les fonctionnalités de save-manager.js et server-sync.js :
 * - Sauvegarde locale avec debounce
 * - Synchronisation serveur automatique
 * - Détection du mode PHP/Node.js
 * - Gestion des retry en cas d'erreur
 * - Indicateur visuel de synchronisation
 *
 * Dépend de: modules/storage.js (doit être chargé avant)
 */
(function() {
  'use strict';

  // Vérifier que storage.js est chargé
  if (!window.CSI_Storage) {
    console.error('[CSI] sync-manager.js requiert storage.js');
    return;
  }

  var Storage = window.CSI_Storage;

  // ============================================================
  // Configuration
  // ============================================================

  var CONFIG = {
    DEBOUNCE_LOCAL_MS: 1000,      // Délai debounce sauvegarde locale
    DEBOUNCE_SYNC_MS: 2000,       // Délai avant sync serveur
    RETRY_DELAY_MS: 5000,         // Délai entre retries
    MAX_RETRIES: 2,               // Nombre max de retries
    SYNC_PHP_URL: './data/sync.php',
    SYNC_NODE_URL: './data/import',
    EXPORT_PHP_URL: './data/sync.php',
    EXPORT_NODE_URL: './data/export'
  };

  // ============================================================
  // État interne
  // ============================================================

  var state = {
    // Timers
    localSaveTimeout: null,
    syncTimeout: null,

    // Flags
    localPending: false,
    syncPending: false,
    syncEnabled: true,

    // Détection mode
    usePHP: null,        // null = pas encore détecté, true/false après détection
    isServerMode: false, // true si http://, false si file://

    // Stats
    saveCount: 0,
    syncCount: 0,
    lastSaveTime: 0,
    lastSyncTime: 0
  };

  // ============================================================
  // Détection du mode (PHP vs Node.js)
  // ============================================================

  /**
   * Détecte si on est en mode serveur (http://)
   */
  function isServerProtocol() {
    return window.location.protocol !== 'file:';
  }

  /**
   * Détecte si PHP est disponible (sync.php répond)
   * @returns {Promise<boolean>}
   */
  async function detectPHP() {
    if (state.usePHP !== null) {
      return state.usePHP;
    }

    if (!isServerProtocol()) {
      state.usePHP = false;
      return false;
    }

    try {
      var response = await fetch(CONFIG.SYNC_PHP_URL + '?action=status', {
        method: 'GET',
        cache: 'no-store'
      });

      if (!response.ok) {
        state.usePHP = false;
        return false;
      }

      var data = await response.json();
      state.usePHP = data && typeof data.configured !== 'undefined';
      return state.usePHP;
    } catch (e) {
      state.usePHP = false;
      return false;
    }
  }

  /**
   * Retourne l'URL d'import selon le mode détecté
   */
  function getImportUrl() {
    var config = Storage.getServerConfig();
    var base = config.baseUrl || window.location.href;

    if (state.usePHP === true) {
      return new URL(CONFIG.SYNC_PHP_URL, base).toString();
    }
    return new URL(CONFIG.SYNC_NODE_URL, base).toString();
  }

  /**
   * Retourne l'URL d'export selon le mode détecté
   */
  function getExportUrl() {
    var config = Storage.getServerConfig();
    var base = config.baseUrl || window.location.href;

    if (state.usePHP === true) {
      return new URL(CONFIG.EXPORT_PHP_URL, base).toString();
    }
    return new URL(CONFIG.EXPORT_NODE_URL, base).toString();
  }

  // ============================================================
  // Sauvegarde locale
  // ============================================================

  /**
   * Demande une sauvegarde locale (debounced)
   */
  function requestSave() {
    state.localPending = true;

    if (state.localSaveTimeout) {
      clearTimeout(state.localSaveTimeout);
    }

    state.localSaveTimeout = setTimeout(function() {
      executeSave();
    }, CONFIG.DEBOUNCE_LOCAL_MS);
  }

  /**
   * Force une sauvegarde immédiate
   * @returns {boolean} Succès
   */
  function forceSave() {
    if (state.localSaveTimeout) {
      clearTimeout(state.localSaveTimeout);
      state.localSaveTimeout = null;
    }
    return executeSave();
  }

  /**
   * Exécute la sauvegarde locale
   * @private
   */
  function executeSave() {
    state.localPending = false;
    state.localSaveTimeout = null;

    // Utiliser saveData() de bundle.js
    if (typeof window.saveData === 'function') {
      var result = window.saveData();
      if (result) {
        state.lastSaveTime = Date.now();
        state.saveCount++;

        // Déclencher sync serveur si mode serveur actif
        if (state.isServerMode && state.syncEnabled) {
          requestServerSync();
        }
      }
      return result;
    }

    console.warn('[CSI Sync] window.saveData non disponible');
    return false;
  }

  // ============================================================
  // Synchronisation serveur
  // ============================================================

  /**
   * Demande une synchronisation serveur (debounced)
   */
  function requestServerSync() {
    state.syncPending = true;

    if (state.syncTimeout) {
      clearTimeout(state.syncTimeout);
    }

    state.syncTimeout = setTimeout(function() {
      executeServerSync(CONFIG.MAX_RETRIES);
    }, CONFIG.DEBOUNCE_SYNC_MS);
  }

  /**
   * Force une synchronisation serveur immédiate
   */
  function forceServerSync() {
    if (state.syncTimeout) {
      clearTimeout(state.syncTimeout);
      state.syncTimeout = null;
    }
    executeServerSync(CONFIG.MAX_RETRIES);
  }

  /**
   * Exécute la synchronisation vers le serveur
   * @param {number} retriesLeft - Nombre de tentatives restantes
   * @private
   */
  function executeServerSync(retriesLeft) {
    state.syncPending = false;
    state.syncTimeout = null;

    var password = Storage.getPassword();
    if (!password) {
      // Pas de mot de passe, skip silencieusement
      return;
    }

    var data = localStorage.getItem(Storage.KEYS.DATA);
    if (!data) {
      return;
    }

    var url = getImportUrl();

    // Afficher indicateur "en cours"
    updateSyncIndicator('syncing');

    fetch(url, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': 'Basic ' + btoa('admin:' + password)
      },
      body: JSON.stringify({ data: data })
    })
    .then(function(response) {
      if (response.ok) {
        state.lastSyncTime = Date.now();
        state.syncCount++;
        Storage.updateLastPush();
        updateSyncIndicator('synced');
        console.log('[CSI Sync] Synchronisation serveur réussie');

        // Notifier les listeners
        if (typeof window.refreshSyncMeta === 'function') {
          window.refreshSyncMeta();
        }
      } else if (response.status === 401) {
        updateSyncIndicator('auth-error');
        console.warn('[CSI Sync] Mot de passe incorrect');
      } else {
        throw new Error('HTTP ' + response.status);
      }
    })
    .catch(function(err) {
      console.warn('[CSI Sync] Erreur:', err.message);

      if (retriesLeft > 0) {
        console.log('[CSI Sync] Retry dans ' + (CONFIG.RETRY_DELAY_MS / 1000) + 's (' + retriesLeft + ' restants)');
        updateSyncIndicator('retrying');
        setTimeout(function() {
          executeServerSync(retriesLeft - 1);
        }, CONFIG.RETRY_DELAY_MS);
      } else {
        updateSyncIndicator('error');
      }
    });
  }

  /**
   * Récupère les données depuis le serveur
   * @returns {Promise<Object|null>}
   */
  async function pullFromServer() {
    var password = Storage.getPassword();
    if (!password) return null;

    var url = getExportUrl();

    try {
      var response = await fetch(url, {
        method: 'GET',
        headers: {
          'Authorization': 'Basic ' + btoa('admin:' + password)
        }
      });

      if (!response.ok) {
        console.warn('[CSI Sync] Pull échoué:', response.status);
        return null;
      }

      var data = await response.json();
      Storage.updateLastPull();
      return data;
    } catch (e) {
      console.warn('[CSI Sync] Pull erreur:', e.message);
      return null;
    }
  }

  // ============================================================
  // Indicateur visuel
  // ============================================================

  /**
   * Met à jour l'indicateur de synchronisation dans l'UI
   * @param {string} status - 'syncing', 'synced', 'retrying', 'auth-error', 'error'
   */
  function updateSyncIndicator(status) {
    var indicator = document.getElementById('serverSyncIndicator');

    if (!indicator) {
      indicator = document.createElement('span');
      indicator.id = 'serverSyncIndicator';
      indicator.style.cssText = 'display:inline-flex;align-items:center;gap:4px;margin-left:12px;font-size:11px;padding:4px 8px;border-radius:4px;transition:all 0.3s;';

      var navbar = document.querySelector('.navbar__left') || document.querySelector('.header__title');
      if (navbar) {
        navbar.appendChild(indicator);
      }
    }

    if (!indicator) return;

    indicator.style.opacity = '1';

    switch (status) {
      case 'syncing':
        indicator.innerHTML = '<i class="fas fa-sync fa-spin" style="color:var(--color-primary,#3b82f6)"></i> <span style="color:var(--color-text-secondary)">Sync...</span>';
        indicator.style.background = 'var(--color-surface-hover,#f0f0f0)';
        indicator.title = 'Synchronisation en cours';
        break;

      case 'synced':
        indicator.innerHTML = '<i class="fas fa-cloud-upload-alt" style="color:var(--color-success,#22c55e)"></i>';
        indicator.style.background = 'transparent';
        indicator.title = 'Synchronisé avec le serveur';
        break;

      case 'retrying':
        indicator.innerHTML = '<i class="fas fa-sync fa-spin" style="color:var(--color-warning,#f59e0b)"></i> <span style="color:var(--color-text-secondary)">Retry...</span>';
        indicator.style.background = 'var(--color-surface-hover,#f0f0f0)';
        indicator.title = 'Nouvelle tentative de synchronisation';
        break;

      case 'auth-error':
        indicator.innerHTML = '<i class="fas fa-exclamation-triangle" style="color:var(--color-warning,#f59e0b)"></i>';
        indicator.style.background = 'transparent';
        indicator.title = 'Erreur d\'authentification - vérifiez le mot de passe';
        break;

      case 'error':
        indicator.innerHTML = '<i class="fas fa-cloud" style="color:var(--color-danger,#dc3545)"></i>';
        indicator.style.background = 'transparent';
        indicator.title = 'Erreur de synchronisation';
        break;
    }

    // Fade out après 5 secondes (sauf pour syncing/retrying)
    if (status !== 'syncing' && status !== 'retrying') {
      setTimeout(function() {
        if (indicator) indicator.style.opacity = '0.5';
      }, 5000);
    }
  }

  // ============================================================
  // API publique
  // ============================================================

  var SyncManager = {
    // Sauvegarde
    requestSave: requestSave,
    forceSave: forceSave,

    // Synchronisation serveur
    requestServerSync: requestServerSync,
    forceServerSync: forceServerSync,
    pullFromServer: pullFromServer,

    // Contrôle
    setAutoSync: function(enabled) {
      state.syncEnabled = !!enabled;
      console.log('[CSI Sync] Auto-sync ' + (state.syncEnabled ? 'activé' : 'désactivé'));
    },

    cancel: function() {
      if (state.localSaveTimeout) {
        clearTimeout(state.localSaveTimeout);
        state.localSaveTimeout = null;
      }
      if (state.syncTimeout) {
        clearTimeout(state.syncTimeout);
        state.syncTimeout = null;
      }
      state.localPending = false;
      state.syncPending = false;
    },

    // État
    isPending: function() {
      return state.localPending || state.syncPending;
    },

    isServerMode: function() {
      return state.isServerMode;
    },

    usePHP: function() {
      return state.usePHP;
    },

    getStats: function() {
      return {
        saveCount: state.saveCount,
        syncCount: state.syncCount,
        lastSaveTime: state.lastSaveTime,
        lastSyncTime: state.lastSyncTime,
        localPending: state.localPending,
        syncPending: state.syncPending,
        isServerMode: state.isServerMode,
        usePHP: state.usePHP,
        syncEnabled: state.syncEnabled
      };
    },

    // Détection
    detectMode: detectPHP,

    // URLs
    getImportUrl: getImportUrl,
    getExportUrl: getExportUrl
  };

  // ============================================================
  // Initialisation
  // ============================================================

  async function init() {
    state.isServerMode = isServerProtocol();

    if (state.isServerMode) {
      await detectPHP();
      console.log('[CSI Sync] Mode serveur détecté, PHP:', state.usePHP);
    } else {
      console.log('[CSI Sync] Mode local (file://)');
    }

    // Exposer pour compatibilité avec ancien code
    window._CSI_usePHP = state.usePHP;
  }

  // Lancer l'initialisation
  init();

  // Enregistrer dans le registre CSI
  if (window.CSI && window.CSI.register) {
    window.CSI.register('syncManager', {
      exports: SyncManager
    });
  }

  // Exposer globalement
  window.CSI_SyncManager = SyncManager;

  // Compatibilité avec ancien code (save-manager.js)
  window.requestSave = requestSave;
  window.forceSave = forceSave;
  window.forceServerSync = forceServerSync;
  window.setAutoSync = SyncManager.setAutoSync;

  console.log('[CSI] Module sync-manager.js chargé');

})();
