// Exporteur HTML pour page NIS2 autonome
// Exporte la page NIS2 avec tous les styles et les graphiques intégrés

(() => {
  const MODULE_ID = 'nis2-program';
  const STYLE_FILE = 'style.css';

  /**
   * Exporte la page NIS2 en HTML autonome avec tous les styles
   */
  async function exportNis2Html() {
    // Bloquer l'export en mode démo
    const isDemo = (() => {
      const meta = document.querySelector('meta[name="csi-mode"]');
      return meta && meta.content === 'demo';
    })();
    if (isDemo) {
      const toast = window.showToast || window.showToast2 || showToast;
      if (toast) {
        toast('L\'export HTML est désactivé en mode démonstration.', 'warning');
      } else {
        alert('L\'export HTML est désactivé en mode démonstration.');
      }
      return;
    }
    try {
      showToast('📋 Préparation de l\'export...', 'info');

      // Obtenir le module NIS2
      const nis2Module = document.getElementById(MODULE_ID);
      if (!nis2Module) {
        throw new Error('Module NIS2 non trouvé');
      }

      // Charger le CSS
      const cssContent = await loadCssContent();

      // Cloner le module
      const clone = nis2Module.cloneNode(true);
      prepareElementForExport(clone);

      // Convertir les canvas en images
      await convertCanvasesToImages(clone);

      // Créer le HTML complet
      const htmlContent = generateStandaloneHtml(clone, cssContent);

      // Télécharger le fichier
      downloadHtml(htmlContent, `nis2-program-${new Date().toISOString().split('T')[0]}.html`);

      showToast('✅ Export HTML réussi!', 'success');

    } catch (error) {
      console.error('Erreur export:', error);
      showToast(`❌ Erreur: ${error.message}`, 'error');
    }
  }

  /**
   * Charge le contenu du CSS
   */
  async function loadCssContent() {
    try {
      const response = await fetch(STYLE_FILE);
      if (!response.ok) throw new Error('Impossible de charger le CSS');
      return await response.text();
    } catch (error) {
      console.warn('CSS non trouvé, utilisation des styles inline');
      return extractInlineCss();
    }
  }

  /**
   * Extrait les styles CSS déjà chargés dans le document
   */
  function extractInlineCss() {
    const styles = [];
    // Styles inline
    document.querySelectorAll('style').forEach(el => styles.push(el.textContent || ''));
    // Styles chargés (même origine) via CSSOM
    Array.from(document.styleSheets || []).forEach(sheet => {
      try {
        const rules = sheet.cssRules || [];
        styles.push(Array.from(rules).map(r => r.cssText).join('\n'));
      } catch (e) {
        // Ignorer les feuilles cross-origin inaccessibles
      }
    });
    return styles.filter(Boolean).join('\n');
  }

  /**
   * Prépare l'élément pour l'export
   */
  function prepareElementForExport(element) {
    // Supprimer les éléments inutiles
    element.querySelectorAll('.module__actions, .icon-btn, button, [data-exclude-export]').forEach(el => {
      if (el.classList.contains('module__actions') || el.classList.contains('icon-btn') || el.tagName === 'BUTTON') {
        el.remove();
      }
    });

    // Nettoyer les styles inline inutiles
    element.querySelectorAll('[style*="display:none"]').forEach(el => {
      el.remove();
    });

    // Ajouter des classes pour l'impression
    element.classList.add('nis2-export-standalone');
    element.style.cssText = `
      width: 100%;
      background: white;
      padding: 20px;
      margin: 0;
      display: block;
      opacity: 1;
      visibility: visible;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    `;
  }

  /**
   * Convertit les canvas en images PNG
   */
  async function convertCanvasesToImages(element) {
    const canvases = element.querySelectorAll('canvas');
    
    for (let canvas of canvases) {
      try {
        const img = document.createElement('img');
        img.src = canvas.toDataURL('image/png');
        img.style.cssText = `
          width: 100%;
          height: auto;
          display: block;
          margin: 12px 0;
          border: 1px solid #e9ecef;
          border-radius: 6px;
        `;
        
        // Remplacer le canvas par l'image
        const parent = canvas.parentNode;
        if (parent) {
          parent.replaceChild(img, canvas);
        }
      } catch (error) {
        console.warn('Erreur conversion canvas:', error);
      }
    }
  }

  /**
   * Génère le HTML complet autonome
   */
  function generateStandaloneHtml(contentElement, cssContent) {
    const title = 'Programme NIS2';
    const date = new Date().toLocaleDateString('fr-FR', {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });

    // Minifier le CSS (optionnel mais recommandé)
    const minifiedCss = minifyCss(cssContent);

    const html = `<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>${title} - Export</title>
    <style>
${minifiedCss}
    </style>
    <style>
        /* Styles spécifiques pour l'export */
        body {
            margin: 0;
            padding: 0;
            background: white;
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            color: #2d3e4a;
        }
        
        .export-header {
            background: white;
            border-bottom: 2px solid #1fb8cd;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        .export-header h1 {
            margin: 0 0 5px 0;
            color: #1fb8cd;
            font-size: 28px;
        }
        
        .export-header p {
            margin: 5px 0;
            color: #666;
            font-size: 14px;
        }
        
        .export-content {
            padding: 20px;
            max-width: 1400px;
            margin: 0 auto;
        }
        
        .export-footer {
            text-align: center;
            padding: 20px;
            color: #999;
            font-size: 12px;
            border-top: 1px solid #e9ecef;
            margin-top: 40px;
        }
        
        /* Masquer les éléments inutiles */
        .nav, .header, .footer, #modeBanner, .global-search {
            display: none !important;
        }
        
        /* Optimiser pour l'impression */
        @media print {
            body {
                background: white;
            }
            .export-header, .export-footer {
                page-break-inside: avoid;
            }
        }
    </style>
</head>
<body>
    <div class="export-header">
        <h1>${title}</h1>
        <p><strong>Exporté le:</strong> ${date}</p>
        <p><strong>Cyber-Assistant - Norme ISO27001 &amp; 27002:2022</strong></p>
    </div>
    
    <div class="export-content">
        ${contentElement.outerHTML}
    </div>
    
    <div class="export-footer">
        <p>Export HTML autonome - Créé avec Cyber-Assistant</p>
        <p>Ce fichier contient tous les styles et données de la section Programme NIS2</p>
    </div>
    
    <script>
        // Ajouter des événements d'interactivité minimaliste (optionnel)
        document.addEventListener('DOMContentLoaded', function() {
            // Rendre les sections cliquables pour cacher/afficher le contenu (optionnel)
            const headers = document.querySelectorAll('.module__header, h2, h3');
            headers.forEach(header => {
                header.style.cursor = 'pointer';
                header.addEventListener('click', function() {
                    const next = this.nextElementSibling;
                    if (next && next !== document.querySelector('.export-footer')) {
                        next.style.display = next.style.display === 'none' ? 'block' : 'none';
                    }
                });
            });
        });
    </script>
</body>
</html>`;

    return html;
  }

  /**
   * Minifie le CSS
   */
  function minifyCss(css) {
    return css
      .replace(/\/\*[\s\S]*?\*\//g, '') // Supprimer les commentaires
      .replace(/\s+/g, ' ')               // Normaliser les espaces
      .replace(/\s*([{};:,>+~])\s*/g, '$1') // Supprimer espaces autour des caractères spéciaux
      .trim();
  }

  /**
   * Télécharge le fichier HTML
   */
  function downloadHtml(htmlContent, filename) {
    const blob = new Blob([htmlContent], { type: 'text/html;charset=utf-8' });
    const url = window.URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.href = url;
    link.download = filename;
    document.body.appendChild(link);
    link.click();
    window.URL.revokeObjectURL(url);
    document.body.removeChild(link);
  }

  /**
   * Affiche une notification toast
   */
  function showToast(message, type = 'info') {
    const container = document.getElementById('toastContainer') || createToastContainer();
    const toast = document.createElement('div');
    toast.className = `toast toast--${type}`;
    toast.textContent = message;
    
    const bgColor = type === 'success' ? '#4caf50' : type === 'error' ? '#f44336' : '#1fb8cd';
    toast.style.cssText = `
      padding: 14px 18px;
      margin: 10px;
      border-radius: 6px;
      background: ${bgColor};
      color: white;
      font-size: 14px;
      box-shadow: 0 4px 6px rgba(0,0,0,0.1);
      animation: slideIn 0.3s ease-in;
    `;
    container.appendChild(toast);
    
    setTimeout(() => {
      toast.style.animation = 'slideOut 0.3s ease-out';
      setTimeout(() => toast.remove(), 300);
    }, 3000);
  }

  function createToastContainer() {
    const container = document.createElement('div');
    container.id = 'toastContainer';
    container.style.cssText = `
      position: fixed;
      top: 20px;
      right: 20px;
      z-index: 9999;
      pointer-events: none;
    `;
    document.body.appendChild(container);
    return container;
  }

  // Ajouter les animations CSS globales si pas déjà présentes
  if (!document.querySelector('style[data-export-animations]')) {
    const style = document.createElement('style');
    style.setAttribute('data-export-animations', 'true');
    style.textContent = `
      @keyframes slideIn {
        from { transform: translateX(400px); opacity: 0; }
        to { transform: translateX(0); opacity: 1; }
      }
      @keyframes slideOut {
        from { transform: translateX(0); opacity: 1; }
        to { transform: translateX(400px); opacity: 0; }
      }
    `;
    document.head.appendChild(style);
  }

  // Exposer la fonction globalement
  window.exportNis2Html = exportNis2Html;
})();
