// Générateur PDF côté client pour Programme NIS2
// Capture fidèle de la page web en PDF A4 paysage sans serveur externe

(() => {
  const TARGET_TAB = 'nis2-program';
  let printButton = null;
  let isGenerating = false;

  // Charger html2canvas dynamiquement depuis CDN
  function loadHtml2Canvas() {
    return new Promise((resolve, reject) => {
      if (window.html2canvas) {
        resolve(window.html2canvas);
      } else {
        const attach = (src, onError) => {
          const script = document.createElement('script');
          script.src = src;
          script.onload = () => resolve(window.html2canvas);
          script.onerror = onError;
          document.head.appendChild(script);
        };
        attach('vendor/html2canvas.min.js', () => {
          attach('https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js', () => {
            reject(new Error('Impossible de charger html2canvas'));
          });
        });
      }
    });
  }

  /**
   * Génère un PDF fidèle en capturant la page NIS2
   */
  async function generateNis2Pdf() {
    // Bloquer l'export en mode démo
    const isDemo = (() => {
      const meta = document.querySelector('meta[name="csi-mode"]');
      return meta && meta.content === 'demo';
    })();
    if (isDemo) {
      const showToast = window.showToast || window.showToast2;
      if (showToast) {
        showToast('L\'export PDF est désactivé en mode démonstration.', 'warning');
      } else {
        alert('L\'export PDF est désactivé en mode démonstration.');
      }
      return;
    }
    if (!printButton || isGenerating) return;
    
    isGenerating = true;
    printButton.disabled = true;
    const originalText = printButton.innerHTML;
    printButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Génération...';

    try {
      // Étape 1: Charger html2canvas
      showToast('📥 Chargement des dépendances...', 'info');
      const html2canvas = await loadHtml2Canvas();

      // Étape 2: Naviguer vers l'onglet NIS2 Program
      showToast('📋 Préparation du contenu...', 'info');
      const previousTab = getActiveTab();
      await switchToTab(TARGET_TAB);
      
      // Charger les données NIS2 si nécessaire
      if (typeof loadNis2Program === 'function') {
        loadNis2Program();
        await wait(800);
      } else {
        await wait(400);
      }

      // Étape 3: Forcer le rendu des charts
      await renderCharts();
      await wait(1000);

      // Étape 4: Préparer le DOM pour la capture
      const nis2Module = document.getElementById(TARGET_TAB);
      if (!nis2Module) throw new Error('Module NIS2 non trouvé');

      // Créer un clone du module pour éviter de modifier la page
      const clone = nis2Module.cloneNode(true);
      prepareElementForCapture(clone);

      // Créer un conteneur temporaire
      const container = document.createElement('div');
      container.style.cssText = `
        position: absolute;
        left: -9999px;
        top: -9999px;
        width: 1920px;
        background: white;
        padding: 20px;
      `;
      container.appendChild(clone);
      document.body.appendChild(container);

      // Étape 5: Capturer en image haute résolution
      showToast('📸 Capture de l\'écran...', 'info');
      const canvas = await html2canvas(container, {
        scale: 2, // Haute résolution
        useCORS: true,
        backgroundColor: '#ffffff',
        logging: false,
        allowTaint: true,
        windowHeight: container.scrollHeight,
        windowWidth: 1920
      });

      // Nettoyer
      document.body.removeChild(container);

      // Étape 6: Créer le PDF
      showToast('📄 Création du PDF...', 'info');
      const { jsPDF } = window.jspdf;
      const pdf = new jsPDF({
        orientation: 'landscape',
        unit: 'mm',
        format: 'a4'
      });

      const imgWidth = 297; // A4 paysage largeur
      const imgHeight = (canvas.height * imgWidth) / canvas.width;
      let heightLeft = imgHeight;
      let position = 0;

      // Ajouter l'image au PDF avec pagination
      const imgData = canvas.toDataURL('image/png');
      const pageHeight = 210; // A4 paysage hauteur

      while (heightLeft > 0) {
        pdf.addImage(imgData, 'PNG', 0, position, imgWidth, imgHeight);
        heightLeft -= pageHeight;
        position -= pageHeight;
        if (heightLeft > 0) {
          pdf.addPage();
        }
      }

      // Étape 7: Télécharger le PDF
      showToast('💾 Téléchargement du PDF...', 'info');
      const filename = `nis2-program-${new Date().toISOString().split('T')[0]}.pdf`;
      pdf.save(filename);

      // Succès
      await wait(500);
      showToast('✅ PDF généré avec succès!', 'success');

    } catch (error) {
      console.error('Erreur PDF:', error);
      showToast(`❌ Erreur: ${error.message}`, 'error');
    } finally {
      // Retourner à l'onglet précédent
      const previousTab = getActiveTab();
      if (previousTab && previousTab !== TARGET_TAB) {
        const navBtn = document.querySelector(`.nav__item[data-tab="${previousTab}"]`);
        if (navBtn) navBtn.click();
      }

      isGenerating = false;
      printButton.disabled = false;
      printButton.innerHTML = originalText;
    }
  }

  /**
   * Prépare le DOM pour une capture PDF optimale
   */
  function prepareElementForCapture(element) {
    // Masquer les boutons et éléments inutiles
    element.querySelectorAll('.module__actions, .icon-btn, [style*="display:none"]').forEach(el => {
      el.style.display = 'none';
    });

    // S'assurer que les canvas sont visibles et dimensionnés
    element.querySelectorAll('canvas').forEach(canvas => {
      canvas.style.display = 'block';
      canvas.style.maxWidth = '100%';
      canvas.style.height = 'auto';
    });

    // Optimiser les styles pour l'impression
    element.style.cssText = `
      width: 100%;
      background: white;
      padding: 0;
      margin: 0;
      display: block;
      opacity: 1;
      visibility: visible;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    `;
  }

  /**
   * Force le rendu des graphiques Chart.js
   */
  async function renderCharts() {
    try {
      const charts = window.charts || {};
      Object.values(charts).forEach(chart => {
        if (chart && typeof chart.resize === 'function') {
          chart.resize();
        }
        if (chart && typeof chart.update === 'function') {
          chart.update('none');
        }
      });
    } catch (e) {
      console.warn('Erreur rendu charts:', e);
    }
  }

  /**
   * Utilitaires
   */
  function getActiveTab() {
    const nav = document.querySelector('.nav__item--active');
    return nav ? nav.getAttribute('data-tab') : null;
  }

  async function switchToTab(tabName) {
    const current = getActiveTab();
    if (current === tabName) return current;
    
    const navBtn = document.querySelector(`.nav__item[data-tab="${tabName}"]`);
    if (navBtn) {
      navBtn.click();
      await wait(600);
    }
    return current;
  }

  function wait(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
  }

  function showToast(message, type = 'info') {
    const container = document.getElementById('toastContainer') || createToastContainer();
    const toast = document.createElement('div');
    toast.className = `toast toast--${type}`;
    toast.textContent = message;
    
    const bgColor = type === 'success' ? '#4caf50' : type === 'error' ? '#f44336' : '#1fb8cd';
    toast.style.cssText = `
      padding: 14px 18px;
      margin: 10px;
      border-radius: 6px;
      background: ${bgColor};
      color: white;
      font-size: 14px;
      box-shadow: 0 4px 6px rgba(0,0,0,0.1);
      animation: slideIn 0.3s ease-in;
    `;
    container.appendChild(toast);
    
    setTimeout(() => {
      toast.style.animation = 'slideOut 0.3s ease-out';
      setTimeout(() => toast.remove(), 300);
    }, 3000);
  }

  function createToastContainer() {
    const container = document.createElement('div');
    container.id = 'toastContainer';
    container.style.cssText = `
      position: fixed;
      top: 20px;
      right: 20px;
      z-index: 9999;
      pointer-events: none;
    `;
    document.body.appendChild(container);
    return container;
  }

  // Ajouter les animations CSS
  const style = document.createElement('style');
  style.textContent = `
    @keyframes slideIn {
      from { transform: translateX(400px); opacity: 0; }
      to { transform: translateX(0); opacity: 1; }
    }
    @keyframes slideOut {
      from { transform: translateX(0); opacity: 1; }
      to { transform: translateX(400px); opacity: 0; }
    }
  `;
  document.head.appendChild(style);

  function init() {
    printButton = document.getElementById('printNis2ProgramBtn');
    if (!printButton) return;

    const handleClick = (event) => {
      if (!window.jspdf || !window.jspdf.jsPDF) {
        return;
      }
      event.preventDefault();
      event.stopImmediatePropagation();
      generateNis2Pdf();
    };

    printButton.addEventListener('click', handleClick, true);
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init, { once: true });
  } else {
    init();
  }
})();
