// Impression dédiée du Programme NIS2 : ouvre une fenêtre propre avec les styles et des canvases figés
(() => {
  const TARGET_TAB = "nis2-program";
  const PRINT_WIDTH = 1400;
  const printButton = document.getElementById("printNis2ProgramBtn");
  if (!printButton) return;

  let isPrinting = false;

  const wait = (ms) => new Promise((resolve) => setTimeout(resolve, ms));

  const getActiveTab = () => {
    const nav = document.querySelector(".nav__item--active");
    return nav ? nav.getAttribute("data-tab") : null;
  };

  const switchToProgram = async () => {
    const current = getActiveTab();
    if (current === TARGET_TAB) return current;
    const navBtn = document.querySelector(`.nav__item[data-tab="${TARGET_TAB}"]`);
    if (navBtn) {
      navBtn.click();
      await wait(500);
    }
    return current;
  };

  const refreshCharts = () => {
    try {
      const charts = window.charts || {};
      Object.values(charts).forEach((chart) => {
        if (!chart || !chart.canvas) return;
        const isNis2Canvas = !!chart.canvas.closest(`#${TARGET_TAB}`);
        if (!isNis2Canvas) return;
        if (typeof chart.resize === "function") chart.resize();
        if (typeof chart.update === "function") chart.update();
      });
    } catch (e) {
      /* noop */
    }
  };

  const snapshotCanvas = (canvas) => {
    try {
      const rect = canvas.getBoundingClientRect();
      const snapshot = document.createElement("canvas");
      const ratio = window.devicePixelRatio || 1;
      snapshot.width = Math.max(1, Math.round(rect.width * ratio));
      snapshot.height = Math.max(1, Math.round(rect.height * ratio));
      const ctx = snapshot.getContext("2d");
      const bg = getComputedStyle(canvas.parentElement || canvas).backgroundColor || "#0f172a";
      ctx.fillStyle = bg;
      ctx.fillRect(0, 0, snapshot.width, snapshot.height);
      ctx.drawImage(canvas, 0, 0, snapshot.width, snapshot.height);
      const img = document.createElement("img");
      img.src = snapshot.toDataURL("image/png");
      img.alt = canvas.getAttribute("aria-label") || "Graphique";
      img.className = "print-canvas-snapshot";
      img.style.display = "block";
      img.style.width = rect.width ? `${rect.width}px` : "100%";
      img.style.height = rect.height ? `${rect.height}px` : "auto";
      return img;
    } catch (e) {
      return null;
    }
  };

  const cloneProgram = () => {
    const program = document.getElementById(TARGET_TAB);
    if (!program) throw new Error("Module NIS2 introuvable.");
    const headerClone = document.querySelector(".header")?.cloneNode(true);
    const programClone = program.cloneNode(true);

    // Remplacer tous les canvases par des images figées
    const sourceCanvases = program.querySelectorAll("canvas");
    const targetCanvases = programClone.querySelectorAll("canvas");
    sourceCanvases.forEach((canvas, idx) => {
      const target = targetCanvases[idx];
      if (!target) return;
      const img = snapshotCanvas(canvas);
      if (img) target.replaceWith(img);
    });

    const main = document.createElement("main");
    main.className = "main";
    main.style.maxWidth = `${PRINT_WIDTH}px`;
    main.style.margin = "0 auto";
    main.style.padding = "0";
    main.appendChild(programClone);

    const wrapper = document.createElement("div");
    wrapper.className = "app";
    if (headerClone) wrapper.appendChild(headerClone);
    wrapper.appendChild(main);
    return wrapper;
  };

  const buildPrintHtml = (contentHtml) => {
    const themeClass = document.body.classList.contains("theme-dark") ? "theme-dark" : "";
    const links = Array.from(document.querySelectorAll('link[rel="stylesheet"]'))
      .map((l) => l.outerHTML)
      .join("");
    const inlineStyles = Array.from(document.querySelectorAll("style"))
      .map((s) => s.textContent || "")
      .join("\n");

    const overrides = `
      <style>
        @page { size: A4 landscape; margin: 0; }
        html, body { margin: 0; padding: 0; width: 100%; min-height: 100%; background: var(--color-background); -webkit-print-color-adjust: exact; print-color-adjust: exact; }
        .header, .main { width: 100%; }
        .main { padding: 0; }
        .module__header { margin: 0; padding: 12px 16px 8px; }
        .nav, .nav--secondary, .nav--tertiary, .global-search, .footer, #modeBanner, #modalOverlay, #toastContainer, #autoSaveStatus { display: none !important; }
        .module { display: none !important; opacity: 1 !important; }
        #${TARGET_TAB} { display: block !important; }
        .chart-card, .socle-card, .nis2-plan-card, .chart-card__content { break-inside: avoid; page-break-inside: avoid; }
        .charts-grid.nis2-top-grid .chart-card__content { min-height: 360px; height: auto; }
        .chart-card--actions-nis2 .chart-card__content,
        .chart-card--actions-domain .chart-card__content { min-height: 280px; height: auto; }
        img.print-canvas-snapshot { width: 100% !important; height: auto !important; display: block; }
        @media print {
          .chart-card, .socle-card, .nis2-plan-card { box-shadow: none; }
        }
        ${inlineStyles}
      </style>
    `;

    return `
      <!doctype html>
      <html lang="fr">
        <head>
          <meta charset="utf-8">
          <title>Programme NIS2</title>
          ${links}
          ${overrides}
        </head>
        <body class="app ${themeClass}">
          ${contentHtml}
        </body>
      </html>
    `;
  };

  const openPrintWindow = (wrapper) => {
    const printWin = window.open("", "_blank", `width=${PRINT_WIDTH},height=900`);
    if (!printWin) throw new Error("Popup bloqué");
    const html = buildPrintHtml(wrapper.outerHTML);
    printWin.document.open();
    printWin.document.write(html);
    printWin.document.close();

    const afterPrint = () => {
      printWin.removeEventListener("afterprint", afterPrint);
      setTimeout(() => printWin.close(), 300);
    };
    printWin.addEventListener("afterprint", afterPrint);

    setTimeout(() => {
      printWin.focus();
      printWin.print();
      setTimeout(afterPrint, 2000); // fallback
    }, 400);
  };

  const handlePrint = async () => {
    if (isPrinting) return;
    isPrinting = true;
    printButton.disabled = true;
    const previousTab = await switchToProgram();
    if (typeof loadNis2Program === "function") {
      loadNis2Program();
      await wait(400);
    } else {
      await wait(200);
    }
    refreshCharts();
    await wait(100);

    try {
      const wrapper = cloneProgram();
      openPrintWindow(wrapper);
    } catch (e) {
      console.error(e);
      alert("Impression NIS2 impossible. Réessayez après avoir rechargé la page.");
    } finally {
      if (previousTab && previousTab !== TARGET_TAB) {
        const navBtn = document.querySelector(`.nav__item[data-tab="${previousTab}"]`);
        if (navBtn) navBtn.click();
      }
      isPrinting = false;
      printButton.disabled = false;
    }
  };

  printButton.addEventListener("click", handlePrint);
})();
