// Construit et imaginé par Jean-François Avart
// pdf-report.js - Version améliorée avec table des matières et graphiques
(function () {
  const { jsPDF } = window.jspdf || {};
  const charts = window.charts || {};
  if (!jsPDF || !(jsPDF.API && jsPDF.API.autoTable)) {
    console.error('jsPDF or autoTable plugin is missing');
    return;
  }

  // Configuration PDF
  const CONFIG = {
    MARGIN_X: 25,
    MARGIN_Y: 20,
    MAX_CHART_HEIGHT: 55,
    CHART_MARGIN: 8,
    LINE_HEIGHT: 5,
    TABLE_FONT_SIZE: 8,
    SUBTITLE_FONT_SIZE: 12,
    TITLE_FONT_SIZE: 12,
    MAIN_TITLE_FONT_SIZE: 16,
    TOC_FONT_SIZE: 10,
    BODY_TOP: 28,
    COLORS: {
      primary: [59, 130, 246],      // Bleu
      success: [34, 197, 94],       // Vert
      warning: [249, 115, 22],      // Orange
      danger: [239, 68, 68],        // Rouge
      muted: [107, 114, 128],       // Gris
      headerBg: [241, 245, 249],    // Fond en-tête
      alternateBg: [248, 250, 252]  // Fond alterné
    }
  };

  const MATRIX_COLORS = {
    insignifiant: [239, 243, 244],
    mineur: [220, 252, 231],
    significatif: [254, 243, 199],
    grave: [254, 215, 170],
    critique: [254, 202, 202]
  };

  const LABELS = {
    impact: { 1: 'Négligeable', 2: 'Mineure', 3: 'Modérée', 4: 'Majeure', 5: 'Grave' },
    probability: { 1: 'Très improbable', 2: 'Improbable', 3: 'Possible', 4: 'Probable', 5: 'Très probable' },
    threatStatus: { reduction: 'Réduction', acceptation: 'Acceptation', transfert: 'Transfert', evitement: 'Évitement' },
    projectRiskStatus: { red: 'Rouge', yellow: 'Jaune', green: 'Vert', todo: 'À réaliser', suspended: 'Suspendue' },
    nis2Status: { todo: 'À réaliser', in_progress: 'En cours', suspended: 'Suspendue', done: 'Réalisé' },
    nis2Type: { service: 'Service', projet: 'Projet', initiative: 'Initiative', programme: 'Programme' },
    nis2Urgency: { high: 'Urgence haute', medium: 'Priorité moyenne', standard: 'Priorité standard' }
  };

  const REPORT_SECTIONS = [
    { id: 'controls', title: 'Contrôles ISO27002', dataKey: 'controls' },
    { id: 'soa', title: "Déclaration d'applicabilité", dataKey: 'soa' },
    { id: 'nis2-controls', title: 'Contrôles NIS2', dataKey: 'nis2Controls' },
    { id: 'nis2-program', title: 'Programme NIS2', dataKey: 'nis2Plan' },
    { id: 'nis2-socle', title: 'Socle de sécurité', dataKey: 'soclePillars' },
    { id: 'actions', title: 'Plan d\'actions', dataKey: 'actions' },
    { id: 'risks', title: 'Risques SMSI', dataKey: 'risks' },
    { id: 'threats-section', title: 'Menaces', dataKey: 'threats' },
    { id: 'nonconformities', title: 'Non-Conformités', dataKey: 'nonconformities' },
    { id: 'audits', title: 'Audits', dataKey: 'audits' },
    { id: 'managementReviews', title: 'Revues de Direction', dataKey: 'reviews' },
    { id: 'docReview', title: 'Revue documentaire 27001', dataKey: 'documentReview' },
    { id: 'documents', title: 'Structure documentaire', dataKey: 'documents' },
    { id: 'stakeholders', title: 'Parties prenantes', dataKey: 'stakeholders' },
    { id: 'kpis', title: 'KPI', dataKey: 'kpis' },
    { id: 'critical-assets', title: 'Actifs critiques', dataKey: 'criticalAssets' },
    { id: 'project-risks', title: 'Risques projets', dataKey: 'projectRisks' }
  ];

  // État global pour la pagination
  let totalPages = 0;
  let tocEntries = [];

  // ============================================================
  // Utilitaires
  // ============================================================

  function formatItem(item) {
    if (item === null || item === undefined) return '';
    if (typeof item === 'string' || typeof item === 'number') return String(item);
    if (item.title) return item.title;
    if (item.name) return item.name;
    try { return JSON.stringify(item); } catch { return String(item); }
  }

  function formatDate(value) {
    if (!value) return '';
    const date = new Date(value);
    return Number.isFinite(date.getTime()) ? date.toLocaleDateString('fr-FR') : String(value);
  }

  function getNis2PhaseLabel(phaseId) {
    if (!phaseId) return 'Hors phase';
    const phases = (window.appData?.nis2Plan?.phases || []);
    const direct = phases.find(p => p.id === String(phaseId));
    if (direct && direct.label) return direct.label;
    const numeric = parseInt(String(phaseId).replace(/[^0-9]/g, ""), 10);
    if (Number.isFinite(numeric)) return `Phase ${numeric}`;
    return `Phase ${phaseId}`;
  }

  function getDataCount(section) {
    const data = window.appData?.[section.dataKey];
    if (!data) return 0;
    if (Array.isArray(data)) return data.length;
    if (section.id === 'nis2-program') {
      const domains = data.domains || [];
      return domains.reduce((sum, d) => sum + (d.actions?.length || 0), 0);
    }
    return 0;
  }

  // ============================================================
  // Dessin d'en-tête et pied de page
  // ============================================================

  function addHeader(doc, title, date) {
    const pageWidth = doc.internal.pageSize.getWidth();

    // Ligne de séparation
    doc.setDrawColor(200);
    doc.setLineWidth(0.3);
    doc.line(CONFIG.MARGIN_X, 14, pageWidth - CONFIG.MARGIN_X, 14);

    // Titre à gauche
    doc.setFont('helvetica', 'normal');
    doc.setFontSize(9);
    doc.setTextColor(100);
    doc.text(title, CONFIG.MARGIN_X, 10);

    // Date à droite
    doc.text(date, pageWidth - CONFIG.MARGIN_X, 10, { align: 'right' });
  }

  function addFooter(doc, pageNum, totalPages) {
    const pageWidth = doc.internal.pageSize.getWidth();
    const pageHeight = doc.internal.pageSize.getHeight();

    // Ligne de séparation
    doc.setDrawColor(200);
    doc.setLineWidth(0.3);
    doc.line(CONFIG.MARGIN_X, pageHeight - 12, pageWidth - CONFIG.MARGIN_X, pageHeight - 12);

    // Numéro de page centré
    doc.setFont('helvetica', 'normal');
    doc.setFontSize(9);
    doc.setTextColor(100);
    const pageText = totalPages > 0 ? `Page ${pageNum} sur ${totalPages}` : `Page ${pageNum}`;
    doc.text(pageText, pageWidth / 2, pageHeight - 7, { align: 'center' });

    // Mention en bas à droite
    doc.setFontSize(7);
    doc.text('Cyber-Assistant', pageWidth - CONFIG.MARGIN_X, pageHeight - 7, { align: 'right' });
  }

  // ============================================================
  // Page de couverture
  // ============================================================

  function drawCoverPage(doc, appTitle, subtitle, date) {
    const pageWidth = doc.internal.pageSize.getWidth();
    const pageHeight = doc.internal.pageSize.getHeight();

    // Fond décoratif en haut
    doc.setFillColor(59, 130, 246);
    doc.rect(0, 0, pageWidth, 60, 'F');

    // Titre principal
    doc.setFont('helvetica', 'bold');
    doc.setFontSize(28);
    doc.setTextColor(255);
    doc.text('Rapport SMSI', pageWidth / 2, 35, { align: 'center' });

    // Sous-titre
    doc.setFontSize(14);
    doc.text(subtitle || 'Système de Management de la Sécurité de l\'Information', pageWidth / 2, 48, { align: 'center' });

    // Titre de l'application
    doc.setTextColor(50);
    doc.setFontSize(20);
    doc.text(appTitle || 'Cyber-Assistant', pageWidth / 2, 100, { align: 'center' });

    // Date de génération
    doc.setFont('helvetica', 'normal');
    doc.setFontSize(12);
    doc.setTextColor(100);
    doc.text('Rapport généré le ' + date, pageWidth / 2, 120, { align: 'center' });

    // Statistiques rapides
    const stats = getQuickStats();
    let y = 160;
    doc.setFontSize(11);
    doc.setTextColor(60);

    const statsLines = [
      `Conformité ISO27002 : ${stats.conformity}%`,
      `Contrôles implémentés : ${stats.implementedControls} / ${stats.totalControls}`,
      `Actions en cours : ${stats.actionsInProgress}`,
      `Risques critiques : ${stats.criticalRisks}`
    ];

    statsLines.forEach(line => {
      doc.text(line, pageWidth / 2, y, { align: 'center' });
      y += 8;
    });

    // Footer
    doc.setFontSize(9);
    doc.setTextColor(150);
    doc.text('Document confidentiel', pageWidth / 2, pageHeight - 20, { align: 'center' });
  }

  function getQuickStats() {
    const controls = window.appData?.controls || [];
    const implemented = controls.filter(c => c.status === 'implemente' || c.status === 'verifie').length;
    const conformity = controls.length > 0 ? Math.round((implemented / controls.length) * 100) : 0;
    const actions = window.appData?.actions || [];
    const inProgress = actions.filter(a => a.progress > 0 && a.progress < 100).length;
    const risks = window.appData?.risks || [];
    const critical = risks.filter(r => (r.impact * r.probability) >= 20).length;

    return {
      conformity,
      implementedControls: implemented,
      totalControls: controls.length,
      actionsInProgress: inProgress,
      criticalRisks: critical
    };
  }

  // ============================================================
  // Table des matières
  // ============================================================

  function drawTableOfContents(doc, sections, date, appTitle) {
    doc.addPage();
    addHeader(doc, appTitle, date);

    const pageWidth = doc.internal.pageSize.getWidth();
    let y = CONFIG.BODY_TOP + 5;

    // Titre
    doc.setFont('helvetica', 'bold');
    doc.setFontSize(16);
    doc.setTextColor(50);
    doc.text('Table des matières', CONFIG.MARGIN_X, y);
    y += 15;

    // Entrées
    doc.setFont('helvetica', 'normal');
    doc.setFontSize(CONFIG.TOC_FONT_SIZE);

    // Dashboard
    doc.setTextColor(59, 130, 246);
    doc.textWithLink('1. Dashboard et indicateurs', CONFIG.MARGIN_X, y, { pageNumber: 3 });
    doc.setTextColor(150);
    const dotsWidth1 = pageWidth - CONFIG.MARGIN_X * 2 - doc.getTextWidth('1. Dashboard et indicateurs') - 15;
    doc.text('.'.repeat(Math.floor(dotsWidth1 / 2)), CONFIG.MARGIN_X + doc.getTextWidth('1. Dashboard et indicateurs') + 5, y);
    doc.text('3', pageWidth - CONFIG.MARGIN_X, y, { align: 'right' });
    y += 8;

    // Sections
    sections.forEach((section, index) => {
      const num = index + 2;
      const count = getDataCount(section);
      const label = `${num}. ${section.title} (${count})`;

      doc.setTextColor(59, 130, 246);
      doc.text(label, CONFIG.MARGIN_X, y);

      // Points de conduite
      doc.setTextColor(150);
      const labelWidth = doc.getTextWidth(label);
      const dotsWidth = pageWidth - CONFIG.MARGIN_X * 2 - labelWidth - 15;
      if (dotsWidth > 10) {
        doc.text('.'.repeat(Math.floor(dotsWidth / 2)), CONFIG.MARGIN_X + labelWidth + 5, y);
      }

      // Numéro de page (sera mis à jour après)
      const pageNum = tocEntries.find(e => e.id === section.id)?.page || '—';
      doc.text(String(pageNum), pageWidth - CONFIG.MARGIN_X, y, { align: 'right' });

      y += 7;

      if (y > doc.internal.pageSize.getHeight() - 30) {
        doc.addPage();
        addHeader(doc, appTitle, date);
        y = CONFIG.BODY_TOP + 5;
      }
    });

    addFooter(doc, 2, totalPages);
  }

  // ============================================================
  // Dashboard avec graphiques
  // ============================================================

  async function drawDashboard(doc, date, appTitle) {
    doc.addPage();
    addHeader(doc, appTitle, date);

    const pageWidth = doc.internal.pageSize.getWidth();
    const pageHeight = doc.internal.pageSize.getHeight();
    let y = CONFIG.BODY_TOP + 5;

    // Titre
    doc.setFont('helvetica', 'bold');
    doc.setFontSize(14);
    doc.setTextColor(50);
    doc.text('Dashboard SMSI', CONFIG.MARGIN_X, y);
    y += 12;

    // KPIs en ligne
    const stats = getQuickStats();
    const kpis = [
      { label: 'Conformité', value: stats.conformity + '%', color: CONFIG.COLORS.primary },
      { label: 'Contrôles', value: `${stats.implementedControls}/${stats.totalControls}`, color: CONFIG.COLORS.success },
      { label: 'Actions', value: String(stats.actionsInProgress), color: CONFIG.COLORS.warning },
      { label: 'Risques critiques', value: String(stats.criticalRisks), color: CONFIG.COLORS.danger }
    ];

    const kpiWidth = (pageWidth - 2 * CONFIG.MARGIN_X - 30) / 4;
    kpis.forEach((kpi, i) => {
      const x = CONFIG.MARGIN_X + i * (kpiWidth + 10);

      // Fond
      doc.setFillColor(...CONFIG.COLORS.headerBg);
      doc.roundedRect(x, y, kpiWidth, 20, 2, 2, 'F');

      // Valeur
      doc.setFont('helvetica', 'bold');
      doc.setFontSize(14);
      doc.setTextColor(...kpi.color);
      doc.text(kpi.value, x + kpiWidth / 2, y + 10, { align: 'center' });

      // Label
      doc.setFont('helvetica', 'normal');
      doc.setFontSize(8);
      doc.setTextColor(100);
      doc.text(kpi.label, x + kpiWidth / 2, y + 17, { align: 'center' });
    });
    y += 30;

    // Graphiques
    const chartWidth = (pageWidth - 2 * CONFIG.MARGIN_X - 10) / 2;

    async function addChartImage(chart, x, yPos, w, h) {
      if (!chart || !chart.canvas) return 0;
      try {
        const canvas = chart.canvas;
        if (!canvas.width || !canvas.height) {
          if (typeof chart.update === 'function') chart.update();
          await new Promise(r => setTimeout(r, 100));
        }

        const img = chart.toBase64Image();
        const ratio = canvas.height / canvas.width;
        let imgW = w;
        let imgH = imgW * ratio;
        if (imgH > h) {
          imgH = h;
          imgW = imgH / ratio;
        }

        doc.addImage(img, 'PNG', x + (w - imgW) / 2, yPos, imgW, imgH);
        return imgH;
      } catch (err) {
        console.warn('Erreur ajout graphique:', err);
        return 0;
      }
    }

    // Radar ISO27002
    if (charts.capabilityRadar) {
      doc.setFont('helvetica', 'bold');
      doc.setFontSize(10);
      doc.setTextColor(60);
      doc.text('Maturité ISO27002', CONFIG.MARGIN_X, y);
      await addChartImage(charts.capabilityRadar, CONFIG.MARGIN_X, y + 5, chartWidth, CONFIG.MAX_CHART_HEIGHT);
    }

    // Radar NIS2
    if (charts.nis2Radar) {
      doc.text('Maturité NIS2', CONFIG.MARGIN_X + chartWidth + 10, y);
      await addChartImage(charts.nis2Radar, CONFIG.MARGIN_X + chartWidth + 10, y + 5, chartWidth, CONFIG.MAX_CHART_HEIGHT);
    }
    y += CONFIG.MAX_CHART_HEIGHT + 15;

    // Nouvelle page si nécessaire
    if (y + CONFIG.MAX_CHART_HEIGHT + 20 > pageHeight - 20) {
      addFooter(doc, doc.internal.getNumberOfPages(), totalPages);
      doc.addPage();
      addHeader(doc, appTitle, date);
      y = CONFIG.BODY_TOP + 5;
    }

    // Graphique des risques et matrice
    if (charts.risks) {
      doc.setFont('helvetica', 'bold');
      doc.setFontSize(10);
      doc.setTextColor(60);
      doc.text('Répartition des risques', CONFIG.MARGIN_X, y);
      await addChartImage(charts.risks, CONFIG.MARGIN_X, y + 5, chartWidth, CONFIG.MAX_CHART_HEIGHT);
    }

    // Matrice de risques
    doc.text('Matrice des risques', CONFIG.MARGIN_X + chartWidth + 10, y);
    drawRiskMatrix(doc, CONFIG.MARGIN_X + chartWidth + 20, y + 8, 45);
    y += CONFIG.MAX_CHART_HEIGHT + 15;

    // Graphique CMM si espace disponible
    if (y + CONFIG.MAX_CHART_HEIGHT < pageHeight - 30 && charts.cmm) {
      doc.setFont('helvetica', 'bold');
      doc.setFontSize(10);
      doc.text('Maturité CMM', CONFIG.MARGIN_X, y);
      await addChartImage(charts.cmm, CONFIG.MARGIN_X, y + 5, chartWidth, CONFIG.MAX_CHART_HEIGHT);
    }

    addFooter(doc, doc.internal.getNumberOfPages(), totalPages);
  }

  function drawRiskMatrix(doc, x, y, size) {
    const cellSize = size / 5;
    const risks = window.appData?.risks || [];

    for (let impact = 5; impact >= 1; impact--) {
      for (let prob = 1; prob <= 5; prob++) {
        const col = prob - 1;
        const row = 5 - impact;
        const px = x + col * cellSize;
        const py = y + row * cellSize;
        const score = impact * prob;

        let level = 'insignifiant';
        if (score >= 20) level = 'critique';
        else if (score >= 15) level = 'grave';
        else if (score >= 10) level = 'significatif';
        else if (score >= 5) level = 'mineur';

        const color = MATRIX_COLORS[level];
        doc.setFillColor(...color);
        doc.rect(px, py, cellSize, cellSize, 'F');
        doc.setDrawColor(180);
        doc.setLineWidth(0.2);
        doc.rect(px, py, cellSize, cellSize);

        // Score
        doc.setTextColor(80);
        doc.setFontSize(7);
        doc.text(String(score), px + cellSize / 2, py + cellSize / 2 - 1, { align: 'center' });

        // Nombre de risques
        const count = risks.filter(r => r.impact === impact && r.probability === prob).length;
        if (count > 0) {
          doc.setFontSize(6);
          doc.setTextColor(59, 130, 246);
          doc.text(`(${count})`, px + cellSize / 2, py + cellSize / 2 + 3, { align: 'center' });
        }
      }
    }

    // Labels axes
    doc.setFontSize(5);
    doc.setTextColor(100);
    for (let prob = 1; prob <= 5; prob++) {
      doc.text(String(prob), x + (prob - 0.5) * cellSize, y + size + 3, { align: 'center' });
    }
    for (let impact = 1; impact <= 5; impact++) {
      doc.text(String(impact), x - 3, y + (5 - impact + 0.5) * cellSize, { align: 'center' });
    }
    doc.text('Probabilité', x + size / 2, y + size + 7, { align: 'center' });
    doc.text('Impact', x - 8, y + size / 2, { align: 'center', angle: 90 });
  }

  // ============================================================
  // Génération des sections
  // ============================================================

  function getSectionTableConfig(section) {
    const data = window.appData?.[section.dataKey];

    const configs = {
      'controls': {
        head: [['#', 'Num', 'Titre', 'Statut', 'CMM']],
        rows: (data || []).map((c, i) => [
          i + 1,
          c.numero || '',
          c.title || '',
          c.status || '',
          c.cmm || ''
        ])
      },
      'soa': {
        head: [['#', 'Référence', 'Titre', 'Applicable', 'Justification']],
        rows: (() => {
          const controls = window.appData?.controls || [];
          const entries = window.appData?.soa || [];
          const entryMap = new Map(entries.map(e => [e.controlId, e]));
          return controls.sort((a, b) => String(a.reference || '').localeCompare(String(b.reference || ''), undefined, { numeric: true }))
            .map((c, i) => {
              const entry = entryMap.get(c.id) || {};
              return [i + 1, c.reference || '', c.title || '', entry.applicable === false ? 'Non' : 'Oui', entry.justification || ''];
            });
        })()
      },
      'nis2-controls': {
        head: [['#', 'Fonction', 'Catégorie', 'Statut', 'CMM']],
        rows: (data || []).map((c, i) => [i + 1, c.fonction || '', c.categorie || '', c.status || '', c.cmm || ''])
      },
      'nis2-program': {
        head: [['Domaine', 'Action', 'Phase', 'Statut', 'Progression']],
        rows: (() => {
          const domains = data?.domains || [];
          const rows = [];
          domains.forEach(d => {
            (d.actions || []).forEach(a => {
              rows.push([
                d.title || '',
                a.text || '',
                getNis2PhaseLabel(a.phase),
                LABELS.nis2Status[a.status] || a.status || '',
                Number.isFinite(a.progress) ? `${Math.round(a.progress)}%` : ''
              ]);
            });
          });
          return rows.length ? rows : [['—', 'Aucune action', '—', '—', '—']];
        })()
      },
      'nis2-socle': {
        head: [['Pilier', 'Titre', 'Élément', 'Responsable']],
        rows: (() => {
          const pillars = data || [];
          const rows = [];
          pillars.forEach((p, idx) => {
            (p.items || ['']).forEach(item => {
              rows.push([`Pilier ${idx + 1}`, p.title || '', item || '', p.owner || '']);
            });
          });
          return rows.length ? rows : [['—', '—', '—', '—']];
        })()
      },
      'actions': {
        head: [['#', 'Action', 'Lien', 'Priorité', 'Progression', 'Échéance', 'Responsable']],
        rows: (data || []).map((a, i) => [
          i + 1,
          a.title || '',
          a.linkType || '',
          a.priority || '',
          Number.isFinite(a.progress) ? `${a.progress}%` : '',
          formatDate(a.dueDate),
          a.owner || ''
        ])
      },
      'risks': {
        head: [['#', 'Risque', 'Impact', 'Prob.', 'Score', 'Niveau', 'Traitement']],
        rows: (data || []).map((r, i) => {
          const score = (r.impact || 0) * (r.probability || 0);
          let level = 'Faible';
          if (score >= 20) level = 'Critique';
          else if (score >= 15) level = 'Majeur';
          else if (score >= 10) level = 'Modéré';
          else if (score >= 5) level = 'Mineur';
          return [i + 1, r.title || '', r.impact || '', r.probability || '', score, level, r.treatment || ''];
        })
      },
      'threats-section': {
        head: [['#', 'Menace', 'Impact', 'Prob.', 'Score', 'Statut']],
        rows: (data || []).map((t, i) => {
          const score = (t.impact || 0) * (t.likelihood || t.probability || 0);
          return [i + 1, t.title || t.name || '', t.impact || '', t.likelihood || t.probability || '', score, LABELS.threatStatus[t.status] || t.status || ''];
        })
      },
      'nonconformities': {
        head: [['#', 'Titre', 'Type', 'Statut', 'Date détection', 'Responsable']],
        rows: (data || []).map((nc, i) => [i + 1, nc.title || '', nc.type || '', nc.status || '', formatDate(nc.detectedDate), nc.owner || ''])
      },
      'audits': {
        head: [['#', 'Titre', 'Type', 'Date', 'Auditeur', 'Statut']],
        rows: (data || []).map((a, i) => [i + 1, a.title || '', a.type || '', formatDate(a.plannedDate), a.auditor || '', a.status || ''])
      },
      'managementReviews': {
        head: [['#', 'Date', 'Participants', 'Décisions']],
        rows: (data || []).map((r, i) => [i + 1, formatDate(r.date), r.participants || '', r.decisions || ''])
      },
      'docReview': {
        head: [['Référence', 'Justification', 'Outil']],
        rows: (data || []).map(d => [d.reference || '', d.justification || '', d.tool || ''])
      },
      'documents': {
        head: [['#', 'Niveau', 'Type', 'Outil', 'Lien']],
        rows: (data || []).map((d, i) => [i + 1, d.category || '', d.type || '', d.tool || '', d.link || ''])
      },
      'stakeholders': {
        head: [['#', 'Nom', 'Rôle', 'Contact']],
        rows: (data || []).map((s, i) => [i + 1, s.name || '', s.role || '', s.contact || ''])
      },
      'kpis': {
        head: [['#', 'Titre', 'Description', 'Progression']],
        rows: (data || []).map((k, i) => [i + 1, k.title || '', k.description || '', Number.isFinite(k.progress) ? `${k.progress}%` : ''])
      },
      'critical-assets': {
        head: [['#', 'Priorité', 'Bénéficiaire', 'Produit', 'RTO', 'RPO', 'Disponibilité']],
        rows: (data || []).map((a, i) => [
          i + 1,
          a.priority || '',
          a.beneficiary || '',
          [a.productCode, a.productName].filter(Boolean).join(' - '),
          a.rto || '',
          a.rpo || '',
          a.availability || ''
        ])
      },
      'project-risks': {
        head: [['#', 'Projet', 'Bénéficiaire', 'Date', 'Statut', 'Recommandations']],
        rows: (data || []).map((r, i) => {
          const recos = r.recommendations || [];
          const pending = recos.filter(rc => !rc.implemented).length;
          return [
            i + 1,
            r.projectName || '',
            r.beneficiary || '',
            formatDate(r.completionDate),
            LABELS.projectRiskStatus[r.status] || r.status || '',
            recos.length ? `${recos.length} (${pending} en attente)` : '0'
          ];
        })
      }
    };

    return configs[section.id] || { head: [['#', 'Donnée']], rows: [['-', 'Aucune donnée']] };
  }

  function drawSection(doc, section, date, appTitle) {
    const pageNum = doc.internal.getNumberOfPages();
    tocEntries.push({ id: section.id, title: section.title, page: pageNum });

    const config = getSectionTableConfig(section);

    if (!config.rows.length) {
      config.rows = [config.head[0].map(() => '—')];
    }

    let firstPage = true;

    doc.autoTable({
      startY: CONFIG.BODY_TOP + 15,
      margin: { top: CONFIG.BODY_TOP + 15, left: CONFIG.MARGIN_X, right: CONFIG.MARGIN_X, bottom: 20 },
      head: config.head,
      body: config.rows,
      styles: {
        fontSize: CONFIG.TABLE_FONT_SIZE,
        font: 'helvetica',
        cellPadding: 2,
        overflow: 'linebreak'
      },
      headStyles: {
        fillColor: CONFIG.COLORS.headerBg,
        textColor: [50, 50, 50],
        fontStyle: 'bold',
        halign: 'center'
      },
      alternateRowStyles: {
        fillColor: CONFIG.COLORS.alternateBg
      },
      columnStyles: {
        0: { cellWidth: 10, halign: 'center' }
      },
      didDrawPage: function(data) {
        addHeader(doc, appTitle, date);
        addFooter(doc, data.pageNumber, totalPages);

        // Titre de section
        doc.setFont('helvetica', 'bold');
        doc.setFontSize(12);
        doc.setTextColor(50);
        const title = firstPage ? section.title : section.title + ' (suite)';
        doc.text(title, CONFIG.MARGIN_X, CONFIG.BODY_TOP + 8);

        // Compteur
        if (firstPage) {
          const count = getDataCount(section);
          doc.setFont('helvetica', 'normal');
          doc.setFontSize(9);
          doc.setTextColor(100);
          doc.text(`(${count} élément${count > 1 ? 's' : ''})`, CONFIG.MARGIN_X + doc.getTextWidth(title) + 5, CONFIG.BODY_TOP + 8);
        }

        firstPage = false;
      }
    });
  }

  // ============================================================
  // Fonction principale
  // ============================================================

  window.generatePdfReport = async function generatePdfReport(selectedIds = null) {
    // Bloquer l'export en mode démo
    const isDemo = (() => {
      const meta = document.querySelector('meta[name="csi-mode"]');
      return meta && meta.content === 'demo';
    })();
    if (isDemo) {
      if (window.showToast) {
        window.showToast('L\'export PDF est désactivé en mode démonstration.', 'warning');
      } else {
        alert('L\'export PDF est désactivé en mode démonstration.');
      }
      return;
    }

    // Réinitialiser l'état
    tocEntries = [];

    // Préparer le dashboard
    const dashboard = document.getElementById('dashboard');
    let restore = null;
    if (dashboard && !dashboard.classList.contains('module--active')) {
      const activeModule = document.querySelector('.module--active');
      if (activeModule) activeModule.classList.remove('module--active');
      dashboard.classList.add('module--active');
      dashboard.style.position = 'absolute';
      dashboard.style.left = '-9999px';
      dashboard.style.visibility = 'hidden';
      restore = activeModule;
    }

    if (window.updateDashboard) {
      window.updateDashboard();
      await new Promise(r => setTimeout(r, 200));
    }

    // Filtrer les sections
    const selected = Array.isArray(selectedIds) && selectedIds.length ? new Set(selectedIds) : null;
    const sections = REPORT_SECTIONS.filter(sec => !selected || selected.has(sec.id));

    // Informations du rapport
    const date = new Date().toLocaleDateString('fr-FR');
    const appTitle = document.getElementById('appTitle')?.innerText.trim() || 'Cyber-Assistant';
    const subtitle = document.querySelector('.header__subtitle')?.innerText.trim() || '';

    // Créer le document
    const doc = new jsPDF({ orientation: 'p', unit: 'mm', format: 'a4' });

    // Estimer le nombre total de pages (approximatif)
    totalPages = 3 + sections.length * 2;

    // Page de couverture
    drawCoverPage(doc, appTitle, subtitle, date);

    // Table des matières (placeholder)
    drawTableOfContents(doc, sections, date, appTitle);

    // Dashboard
    await drawDashboard(doc, date, appTitle);

    // Sections
    for (const section of sections) {
      doc.addPage();
      drawSection(doc, section, date, appTitle);
    }

    // Mettre à jour le nombre total de pages
    totalPages = doc.internal.getNumberOfPages();

    // Regénérer les footers avec le bon total (optionnel - coûteux en performance)
    // Pour une solution parfaite, il faudrait un second passage

    // Sauvegarder
    doc.save('SMSI_Report_' + new Date().toISOString().slice(0, 10) + '.pdf');

    // Restaurer l'interface
    if (restore) {
      dashboard.classList.remove('module--active');
      dashboard.style.position = '';
      dashboard.style.left = '';
      dashboard.style.visibility = '';
      restore.classList.add('module--active');
    }

    return doc;
  };

  window.generatePdfReportWithFeedback = async function generatePdfReportWithFeedback() {
    try {
      if (typeof window.openModal === "function") {
        const fields = REPORT_SECTIONS.map((s) => ({
          name: s.id,
          label: `${s.title} (${getDataCount(s)})`,
          type: "checkbox",
          value: true
        }));

        window.openModal("Sections du rapport PDF", fields, async (data) => {
          const chosen = REPORT_SECTIONS.filter((s) => data[s.id]).map((s) => s.id);
          if (!chosen.length) {
            if (window.showToast) window.showToast("Sélectionnez au moins une section", "warning");
            return false;
          }
          await window.generatePdfReport(chosen);
          if (window.showToast) window.showToast('Rapport PDF généré avec succès', 'success');
        }, null, { allowReadOnly: true });
      } else {
        await window.generatePdfReport();
        if (window.showToast) window.showToast('Rapport PDF généré avec succès', 'success');
      }
    } catch (error) {
      console.error("Erreur lors de l'export PDF:", error);
      if (window.showToast) window.showToast("Erreur: " + error.message, 'error');
    }
  };

  // Exposer les sections pour usage externe
  window.PDF_REPORT_SECTIONS = REPORT_SECTIONS;

  console.log('[CSI] Module pdf-report.js chargé (version améliorée)');
})();
